# Copyright (C) 2011, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import shutil
import logging
import urllib2
import datetime
import subprocess
from os.path import join, exists
from gettext import gettext as _

import dbus
import gobject

from sugar_client import util, env, service
from sugar_client.rest import JSONRest
from sugar_client.util import enforce


INTERFACE = 'org.sugarlabs.client.Backup'
NAME = 'org.sugarlabs.client.Backup'
PATH = '/org/sugarlabs/client/Backup'

_RSYNC = '/usr/bin/rsync'
# Exponential timeout between checks for backup availability
_BACKUP_SERVER_TIMEOUT = lambda a_try: 60 * 2 ** (a_try - 1)

_DS_IFACE = 'org.laptop.sugar.DataStore'
_DS_NAME = 'org.laptop.sugar.DataStore'
_DS_PATH = '/org/laptop/sugar/DataStore'


class Service(service.Service):

    def __init__(self):
        service.Service.__init__(self, dbus.SessionBus, INTERFACE, NAME, PATH)
        self._process = None

    @env.BaseService.method(INTERFACE,
            in_signature='s', out_signature='b')
    def Backup(self, url):
        """Backup datastore to the server.

        The function will ask backup server if it is possible to start backup
        process. If server will reply with reject, function will will raise
        `RuntimeError`.

        :param url:
            rsync url to backup data to, empty value means default url

        """
        return self.backup(url, 1)

    @env.BaseService.method(INTERFACE,
            in_signature='si', out_signature='b')
    def Restore(self, url, date):
        """Restore datastore from the server.

        :param url:
            rsync url to backup data to, empty value means default url
        :param date:
            the date to restore datastore for, ``0`` means the last one

        """
        if not self.acquire():
            return False

        self.info(_('Start Journal restore'))

        if not url:
            url = env.backup_url.value
            if date:
                date = datetime.date.fromtimestamp(date)
                url = url.rstrip('/') + '/' + date.strftime('%Y-%m-%d')
        else:
            enforce(date == 0,
                    _('Backup date might be used only for remote restore'))

        self.info(_('Restore Journal from %s'), url)
        dst_path = _acquire_ds()

        def success_cb():
            _commit_ds(dst_path)
            self.finished(service.STATE_SUCCESS)

        def error_cb(code):
            _commit_ds(dst_path)
            self.finished(service.STATE_FAILED)

        self._rsync(url, dst_path, success_cb, error_cb)

        return True

    @env.BaseService.method(INTERFACE,
            in_signature='', out_signature='')
    def Cancel(self):
        if self.State < 10:
            return
        logging.info(_('Cancel current action'))
        if self._process is None:
            # The process is in timeout, just reset Finished flag
            # to discard it on awake
            self.finished(service.STATE_CANCELED)
        else:
            self._process.kill()

    def backup(self, url, tries, finish_cb=lambda: None):
        if not self.acquire():
            finish_cb()
            return False

        self.info(_('Start Journal backup'))

        def success_cb():
            self.info(_('Journal backed up successfully'))
            self.finished(service.STATE_SUCCESS)
            finish_cb()

        def error_cb(code):
            self.finished(service.STATE_FAILED)
            finish_cb()

        def accepted(url=None):
            if not url:
                url = env.backup_url.value
            self.info(_('Backup Journal to %s'), url)
            self._rsync(env.profile_path('datastore'), url,
                    success_cb, error_cb)

        if not url:
            enforce(env.backup_url.value,
                    _('Registeration was not processed, no backup url'))
            if ':' in env.backup_url.value:
                gobject.idle_add(self._check_backup_server, 1, tries,
                        accepted, error_cb)
            else:
                accepted()
        else:
            accepted(url)

        return True

    def _rsync(self, src, dst, success_cb, error_cb):
        # Add a trailing slash to ensure that we don't generate dst subdir
        src = src.rstrip(os.sep) + os.sep

        ssh_cmd = '/usr/bin/ssh -F /dev/null -o "PasswordAuthentication no" ' \
                '-o "StrictHostKeyChecking no" -i "%s"' % \
                env.profile_path('owner.key')
        rsync_cmd = [_RSYNC, '-z', '-rlt', '--delete', '--timeout=160',
                '--exclude=/index', '--exclude=/index_updated',
                '--partial-dir=.rsync-partial', '--out-format=%i',
                '-e', ssh_cmd, src, dst]
        logging.debug('Backup rsync command: %r', rsync_cmd)

        # At first, calculate number of transferes to have total progress
        self._popen(rsync_cmd + ['--dry-run'], self.increment_total,
                # Then, do a real transfer
                lambda: self._popen(rsync_cmd, self.increment_progress,
                    success_cb, error_cb),
                error_cb)

    def _popen(self, cmd, progress_cb, success_cb, error_cb):
        self._process = subprocess.Popen(cmd,
                stderr=subprocess.PIPE, stdout=subprocess.PIPE)

        def io_cb(fd, condition):
            if condition & gobject.IO_IN:
                fd.readline().rstrip()
                progress_cb()
            if condition & gobject.IO_HUP:
                self._process.wait()
                returncode = self._process.returncode
                stderr = self._process.stderr.read().strip()
                self._process = None
                if returncode == 0:
                    success_cb()
                else:
                    self.info(_('Rsync failed with code %s: %s'),
                            returncode, stderr)
                    error_cb(returncode)
                return False
            else:
                return True

        gobject.io_add_watch(self._process.stdout,
                gobject.IO_IN | gobject.IO_HUP, io_cb)

    def _check_backup_server(self, a_try, tries, success_cb, error_cb):
        if self.State < 10:
            # Canceled
            return False

        try:
            message = _('Check for backup availability on %s') % \
                    env.api_url.value
            if a_try > 1:
                message += ' (%d)' % a_try
            self.info(message)

            rest = JSONRest(env.api_url.value)
            reply = rest.get('/client/backup', {'uid': env.uid.value})
            logging.debug('Got reply from backup server: %r', reply)

            enforce(reply['success'] == 'OK',
                    _('Backup server failed: %s'), dict(reply).get('error'))

            if reply['accepted']:
                success_cb()
            else:
                enforce(a_try < tries, _('Backup server is too busy'))
                timeout = _BACKUP_SERVER_TIMEOUT(a_try)
                self.info(_('Backoff timeout for %d seconds before ' \
                        'trying backup once more'), timeout)
                gobject.timeout_add_seconds(timeout,
                        self._check_backup_server, a_try + 1, tries,
                        success_cb, error_cb)

        except urllib2.HTTPError, error:
            self.info(_('Backup server failed: %s'), error)
            error_cb(1)
        except urllib2.URLError, error:
            self.info(_('Cannot connect to backup server: %s') % error)
            error_cb(1)
        except Exception, error:
            self.send_info(str(error))
            util.exception()
            error_cb(1)

        return False


def _acquire_ds():
    src_path = env.profile_path('datastore')
    dst_path = src_path + '.import'

    if exists(dst_path):
        shutil.rmtree(dst_path, ignore_errors=True)

    if exists(src_path):
        # Make a hardlinked copy of ds to start rsyncing to
        util.cptree(src_path, dst_path)
        # Then, remove all `data` files from the original ds to save space
        # for further rsyncing
        for root, __, files in os.walk(src_path):
            if 'data' in files:
                os.unlink(join(root, 'data'))
    else:
        os.makedirs(dst_path)

    logging.debug('Starting backup restore to %s', dst_path)

    return dst_path


class _Reindexer(object):

    def __init__(self):
        try:
            self._ds = dbus.Interface(
                    dbus.SessionBus().get_object(_DS_NAME, _DS_PATH),
                    _DS_IFACE)
            self._tmp_ds_uid = self._ds.create({
                'title': _('Journal is being restored'),
                }, '', False)
        except Exception, error:
            logging.debug('Running DS was not found, ' \
                    'nothing to inform about restored Journal: %s', error)
            # If there is no running DS instance, no need in reindexing it
            self._ds = None

    def trigger(self):
        if self._ds is None:
            return
        try:
            # DS won't find tmp_ds_uid after renaming dirs
            # and will start reindexing
            self._ds.find({'uid': self._tmp_ds_uid}, [])
            logging.debug('DS was informed about restored Journal')
        except Exception:
            util.exception(_('Cannot inform datastore about restored content'))


def _commit_ds(src_path):
    # Temporary DS item is needed to ask DS, after restoring,
    # for unknow item and, thus, trigger index regenerating
    reindexer = _Reindexer()

    dst_path = env.profile_path('datastore')
    removing_path = env.profile_path('datastore') + '.removing'

    shutil.rmtree(join(src_path, 'index'), ignore_errors=True)
    if exists(dst_path):
        if exists(join(dst_path, 'index')):
            # Preserve current index, even if it is outdated
            # othewise, DS will fail on rebuilding it
            os.rename(join(dst_path, 'index'), join(src_path, 'index'))
        os.renames(dst_path, removing_path)
    os.renames(src_path, dst_path)

    reindexer.trigger()

    if exists(removing_path):
        shutil.rmtree(removing_path, ignore_errors=True)

    logging.debug('Backup restore compeleted and moved to final destination')
