#!/usr/bin/env python

# Copyright (C) 2011, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import sys
import logging
from glob import glob
from gettext import gettext as _

from sugar_client import util
from sugar_client.util import enforce


_logger = logging.getLogger('yum')


class Packager(object):

    reboot_packages = [
            'kernel', 'kernel-firmware',
            ]

    relogin_packages = [
            'sugar', 'sugar-artwork', 'sugar-base', 'sugar-datastore',
            'sugar-presence-service', 'sugar-toolkit',
            ]

    def __init__(self, args, environ):
        self._args = args
        self._environ = environ

    def cleanup(self):
        _logger.info('Compelete transactions')
        code, __ = self._call('/usr/sbin/yum-complete-transaction', '-y')
        if code == 0:
            return

        _logger.info('Transactions completion failed, ' \
                'remove possibly broken rpmdb')
        for i in glob('/var/lib/rpm/__db.*'):
            os.unlink(i)
        code, __ = self._call('/usr/sbin/yum-complete-transaction', '-y')
        if code == 0:
            return

        _logger.info('Transactions still failed, clean them up')
        code, __ = self._call('/usr/sbin/yum-complete-transaction',
                '-y --cleanup-only')

    def update_metadata(self):
        # Clean up all caches to let update() refetch new metadata
        self._call('/usr/bin/yum', 'clean all')

    def to_update(self):
        code, stdout = self._call('/usr/bin/yum', '-e0 check-update')
        if code == 0:
            return []
        elif code == 100:
            result = []
            for line in str(stdout).split('\n'):
                columns = line.split()
                # Include lines only with packages info
                if len(columns) == 3:
                    # Ignore arch suffix
                    package = columns[0].split('.')[0]
                    result.append(package)
            return result
        else:
            raise RuntimeError(_('Cannot get list of packages to update'))

    def update(self):
        code, __ = self._call('/usr/bin/yum', '-y update')
        enforce(code == 0, _('Cannot upgrade system'))

    def last_updated(self):
        return int(os.stat('/var/lib/rpm/Packages').st_mtime)

    def _call(self, command, args):
        cmd = '%s --skip-broken --quiet --tolerant %s %s' % \
                (command, self._args or '', args)
        error = []
        stdout = util.call(cmd, env=self._environ, shell=True,
                error_cb=lambda * args: error.extend(args))
        if stdout is None:
            code, stdout, __ = error
            return code, stdout
        else:
            return 0, stdout


def _main():
    logging.basicConfig(level=logging.DEBUG)

    argv = sys.argv[1:]
    if not len(argv):
        print 'Usage: yum.py <Packager.function> [yum-args]'
        exit(0)

    func = argv.pop(0)
    args = ' '.join(["'%s'" % i for i in argv])
    packager = Packager(args, os.environ)

    print getattr(packager, func)()


if __name__ == '__main__':
    _main()
