# Copyright (C) 2011, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Simple REST implementation.

$Repo: git://git.sugarlabs.org/alsroot/codelets.git$
$File: src/rest.py$
$Data: 2012-01-16$

"""

import os
import urllib
import urllib2
import base64
import httplib
import hashlib
import logging
import datetime
from gettext import gettext as _


_ssl_args = {}


class _Rest(object):

    def __init__(self, url, content_type, username=None, password=None,
            headers=None, **ssl_args):
        self.url = url
        self.username = username
        self.password = password
        self._content_type = content_type
        self._headers = headers

        handlers = []

        if 'http_proxy' in os.environ:
            proxy_support = urllib2.ProxyHandler({
                'http': os.environ['http_proxy'],
                'https': os.environ['http_proxy'],
                })
            handlers.append(proxy_support)

        if ssl_args:
            _ssl_args.clear()
            _ssl_args.update(ssl_args)
            handlers.append(_HTTPSHandler)

        if handlers:
            opener = urllib2.build_opener(*handlers)
            urllib2.install_opener(opener)

    def raw_get(self, path, *args, **kwargs):
        request = self._request(path, *args, **kwargs)
        logging.debug('RESTfull GET %s', request.get_full_url())
        request.add_header('Accept', self._content_type)
        return urllib2.urlopen(request).read()

    def raw_post(self, path, data, *args, **kwargs):
        request = self._request(path, *args, **kwargs)
        logging.debug('RESTfull POST %s', request.get_full_url())
        request.add_header('Content-type', self._content_type)
        request.add_data(data)
        return urllib2.urlopen(request).read()

    def _request(self, path, query=None, **kwargs):
        if not isinstance(query, dict):
            if query is not None:
                kwargs['query'] = query
            query = kwargs
        url = '/'.join([self.url, path.lstrip('/')])
        if query:
            url += '?' + urllib.urlencode(query)
        request = urllib2.Request(url, headers=self._headers or {})
        if self.username:
            password = hashlib.sha1(self.password).hexdigest()
            auth_string = base64.encodestring(
                    ':'.join([self.username, password]))
            request.add_header('Authorization',
                    'Basic %s' % auth_string.rstrip('\n'))
        return request


class JSONRest(_Rest):
    """REST interactions using JSON content type.

    Use this class if server side uses JSON content type for RESTfull
    interactions. This is the most simple usage because content is not being
    converted to Python dictionaries like XMLRest does.

    """

    def __init__(self, url, *args, **kwargs):
        _Rest.__init__(self, url, 'application/json', *args, **kwargs)

        try:
            import json
            if not hasattr(json, 'dumps'):
                raise ImportError()
            self._json = json
        except ImportError:
            import simplejson
            self._json = simplejson

    def get(self, path, *args, **kwargs):
        return self._loads(self.raw_get(path, *args, **kwargs))

    def post(self, path, data, *args, **kwargs):
        data = self._json.dumps(data)
        return self._loads(self.raw_post(path, data, *args, **kwargs))

    def _loads(self, data):
        result = self._json.loads(data or 'none')
        if type(result) is dict:
            # Use dict() typecast to avoid further pylint warnings
            result = dict(result)
        return result


class XMLRest(_Rest):
    """REST interactions using XML content type.

    Use this class if server side uses XML content type for RESTfull
    interactions, e.g., Ruby On Rail application. Though, all external,
    to this class, user's data is Python dictionaries.

    """

    def __init__(self, url, *args, **kwargs):
        _Rest.__init__(self, url, 'application/xml', *args, **kwargs)

    def get(self, path, *args, **kwargs):
        return xml_to_dict(self.raw_get(path, *args, **kwargs))

    def post(self, path, data, *args, **kwargs):
        data = dict_to_xml(data)
        return xml_to_dict(self.raw_post(path, data, *args, **kwargs))


def dict_to_xml(data):
    xml = []

    def value_to_xml(tag, data):
        if type(data) is datetime.date:
            xml.append('<%s type="date">' % tag)
            xml.append(data.isoformat())
        elif type(data) is dict:
            xml.append('<%s>' % tag)
            for key, value in data.items():
                if type(value) in [list, tuple, set]:
                    if not key.endswith('s'):
                        raise RuntimeError(_('Array type XML tag should ' \
                                'ends with "s" that will be cat-off for ' \
                                'array nodes'))
                    xml.append('<%s type="array">' % key)
                    for i in value:
                        value_to_xml(key[:-1], i)
                    xml.append('</%s>' % key)
                else:
                    value_to_xml(key, value)
        else:
            if not type(data) in [str, unicode]:
                raise RuntimeError(_('Unsupported node type'))
            xml.append('<%s>' % tag)
            xml.append(data)
        xml.append('</%s>' % tag)

    value_to_xml('hash', data)
    return ''.join(xml)


def xml_to_dict(xml):
    from xml.etree import ElementTree
    return element_to_dict(ElementTree.fromstring(xml)).values()[0]


def element_to_dict(element):
    result = {}

    if element.text:
        if element.attrib.get('type') == 'date':
            value = datetime.datetime.strptime(element.text, '%Y-%m-%d').date()
        else:
            value = element.text
    else:
        value = {}
    result[element.tag] = value

    children = element.getchildren()
    if not children:
        return result

    parent = result[element.tag] = {}
    for child in children:
        for key, value in element_to_dict(child).items():
            datetime.datetime.strptime('2000-01-01', '%Y-%m-%d')
            if key in parent:
                if not isinstance(parent[key], list):
                    parent[key] = [parent[key]]
                parent[key].append(value)
            elif element.attrib.get('type') == 'array':
                parent[key] = [value]
            else:
                parent[key] = value

    return result


class _HTTPSConnection(httplib.HTTPConnection):

    def connect(self):
        import ssl

        httplib.HTTPConnection.connect(self)
        self.sock = ssl.wrap_socket(self.sock, **_ssl_args)


class _HTTPSHandler(urllib2.HTTPSHandler):

    def do_open(self, http_class, req):
        return urllib2.HTTPSHandler.do_open(self, _HTTPSConnection, req)
