# Copyright (C) 2011, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import logging
from gettext import gettext as _

import dbus
import gobject

from sugar_client import env
from sugar_client.util import enforce


STATE_SUCCESS = 0
STATE_FAILED = 1
STATE_CANCELED = 2
STATE_ONGOING = 10


class Service(env.BaseService):

    def __init__(self, bus_class, interface, name, path):
        env.BaseService.__init__(self, bus_class, name, path)
        self.interface = interface
        self.name = name
        self.path = path
        self._state = STATE_SUCCESS
        self._info = ''
        self._progress = -1
        self._total = None
        self._done = None
        self.on_finished = lambda: None

    @property
    def Progress(self):
        return self._progress

    @property
    def Info(self):
        return self._info

    @property
    def State(self):
        return self._state

    @env.BaseService.method(dbus.PROPERTIES_IFACE,
            in_signature='ss', out_signature='v')
    def Get(self, interface, prop):
        enforce(interface == self.interface,
                _('Interface %s is not supported'), interface)
        enforce(hasattr(self, prop) and \
                isinstance(getattr(self.__class__, prop), property),
                _('Unsupported %s property'), prop)
        return getattr(self, prop)

    @env.BaseService.method(dbus.PROPERTIES_IFACE,
            in_signature='s', out_signature='a{sv}')
    def GetAll(self, interface):
        enforce(interface == self.interface,
                _('Interface %s is not supported'), interface)
        result = {}
        for i in dir(self.__class__):
            if isinstance(getattr(self.__class__, i), property):
                result[i] = getattr(self, i)
        return result

    @env.BaseService.method(dbus.PROPERTIES_IFACE,
            in_signature='ssv')
    def Set(self, interface, prop, value):
        enforce(interface == self.interface,
                _('Interface %s is not supported'), interface)
        enforce(hasattr(self, prop) and \
                isinstance(getattr(self.__class__, prop), property) and \
                getattr(self, prop).fset is not None,
                _('Unsupported %s property'), prop)
        setattr(self, prop, value)
        self.PropertiesChanged(interface, {prop: value}, [])

    @env.BaseService.signal(dbus.PROPERTIES_IFACE,
            signature='sa{sv}as')
    def PropertiesChanged(self, interface, changed_props, invalidated_props):
        pass

    def acquire(self):
        if self._state < STATE_ONGOING:
            self._state = STATE_ONGOING
            self._info = ''
            self._progress = 0.0
            self._total = 0
            self._done = 0
            return True
        else:
            self.info(_('Another action is in progress'))
            return False

    def finished(self, state):
        self._state = state
        gobject.idle_add(self.PropertiesChanged,
                self.interface, {'State': state}, [])
        if self.on_finished is not None:
            self.on_finished()

    def send_info(self, message, *args):
        if args:
            message = message % args
        if message == self._info:
            return False
        self._info = message
        self.PropertiesChanged(self.interface, {'Info': message}, [])
        return True

    def info(self, message, *args):
        if self.send_info(message, *args):
            logging.info('%s', self._info)

    def increment_total(self, steps=1):
        self._total += float(steps)

    def increment_progress(self, done=1, **kwargs):
        self._done += done
        new_percent = min(100, int(self._done / self._total * 100))
        if new_percent == self._progress:
            return
        self._progress = new_percent
        self.PropertiesChanged(self.interface, {'Progress': new_percent}, [])
