# Copyright (C) 2012, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import ssl
import urllib2
import logging
import hashlib
from gettext import gettext as _

import dbus
import gobject
from M2Crypto import DSA

import sugar_stats

from sugar_client import env, service, util
from sugar_client.rest import JSONRest


INTERFACE = 'org.sugarlabs.client.Statistics'
NAME = 'org.sugarlabs.client.Statistics'
PATH = '/org/sugarlabs/client/Statistics'

_UPLOAD_PAGE = 256
_SUGAR_REGISTRATION_TIMEOUT = 30


class Service(service.Service):

    def __init__(self):
        service.Service.__init__(self, dbus.SessionBus, INTERFACE, NAME, PATH)
        self._rrd = None
        self._rest = None
        self._path = None

        if env.stats_url.value:
            if env.has_pubkey():
                self._connect()
            else:
                logging.debug('No Sugar ssh key, will reconnect in %s seconds',
                        _SUGAR_REGISTRATION_TIMEOUT)
                gobject.timeout_add_seconds(_SUGAR_REGISTRATION_TIMEOUT,
                        self._connect)
        else:
            logging.warning(_('Option --stats-url was not specified, ' \
                    'stats are disabled'))

    def ping(self):
        if self._rest is None:
            return

        info = self._rest_call(self._path, cmd='stats-info')
        logging.debug('Got stats info: %r', info)

        session = env.session()
        if info['enable'] != session.get('stats', 'enabled', default=False):
            if info['enable']:
                session['stats', 'step'] = info['step']
                session['stats', 'rras'] = info['rras']
                self._start_monitoring()
            else:
                logging.info(_('Stop monitoring stats'))
                sugar_stats.stop()
            session['stats', 'enabled'] = info['enable']
            env.update_session()

        if self._rrd is not None:
            for name, first, last in self._rrd.dbs:
                upload = {'name': name}
                values = upload['values'] = []

                def send():
                    if not values:
                        return
                    logging.debug('Upload %s stats for %s', len(values), name)
                    self._rest_call(self._path, upload, cmd='stats-upload')
                    del values[:]

                first = info['status'].get(name, first)
                for timestamp, row in self._rrd.get(name, first, last):
                    values.append((timestamp, row))
                    if len(values) >= _UPLOAD_PAGE:
                        send()
                send()

    @property
    def Monitoring(self):
        return env.session().get('stats', 'enabled', default=False)

    def _connect(self):
        if not env.has_pubkey():
            logging.debug('Still no Sugar ssh key, wait for %s seconds',
                    _SUGAR_REGISTRATION_TIMEOUT)
            return True

        uid = env.pubkey_hash()
        headers = {
                'sugar_user': uid,
                'sugar_user_signature': _sign(uid),
                }
        self._path = '/user/%s' % uid
        self._rest = JSONRest(env.stats_url.value, headers=headers,
                ca_certs=env.stats_ca_certs.value,
                cert_reqs=ssl.CERT_REQUIRED)

        if env.session().get('stats', 'enabled', default=False):
            self._start_monitoring()

        if env.stats_timeout.value:
            gobject.timeout_add_seconds(env.stats_timeout.value,
                    self._periodical_ping)

    def _start_monitoring(self):
        logging.info(_('Start monitoring stats'))
        session = env.session()
        self._rrd = sugar_stats.Rrd(
                env.profile_path('stats'), session['stats', 'step'],
                session['stats', 'rras'])
        sugar_stats.start(self._rrd)

    def _rest_call(self, path, request=None, **kwargs):

        while True:
            try:
                if request:
                    return self._rest.post(path, request, **kwargs)
                else:
                    return self._rest.get(path, **kwargs)
            except urllib2.HTTPError, error:
                if error.getcode() == 401 and path != '/user':
                    self._register()
                    continue
                raise RuntimeError('%s: %s' % (error, error.read()))

    def _register(self):
        self._rest_call('/user', {
            'nickname': env.get_nickname() or '',
            'color': '#000000,#000000',
            'machine_sn': env.get_machine_sn() or '',
            'machine_uuid': env.get_machine_uuid() or '',
            'pubkey': env.pubkey(),
            })

    def _periodical_ping(self):
        try:
            self.ping()
        except Exception:
            util.exception(_('Fail to process periodical stats upload'))
        return True


def _sign(data):
    key = DSA.load_key(env.profile_path('owner.key'))
    return key.sign_asn1(hashlib.sha1(data).digest()).encode('hex')
