# Copyright (C) 2011, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import logging
import threading
from os.path import exists, join
from gettext import gettext as _

import dbus
import gobject

from sugar_client import env, pms, util, service


INTERFACE = 'org.sugarlabs.client.System'
NAME = 'org.sugarlabs.client.System'
PATH = '/org/sugarlabs/client/System'

# Backwards compatibility with dextrose-updater
_INTERFACE_COMPAT = 'org.sugarlabs.system'
_PATH_COMPAT = '/org/sugarlabs/system'

_POWERD_INHIBIT_SUSPEND = '/var/run/powerd-inhibit-suspend'


class Service(service.Service):

    def __init__(self):
        service.Service.__init__(self, dbus.SystemBus, INTERFACE, NAME, PATH)
        self._compat = _ServiceCompat()
        self._environ = os.environ.copy()
        self._pms = pms.get(env.update_args.value, self._environ)

    def remove_from_connection(self, connection=None, path=None):
        self._compat.remove_from_connection(connection, path)
        service.Service.remove_from_connection(self, connection, path)

    @env.BaseService.method(INTERFACE,
            in_signature='', out_signature='b', sender_keyword='sender')
    def Update(self, sender=None):
        http_proxy = self.get_http_proxy(sender) or \
                os.environ.get('http_proxy')
        if http_proxy:
            self._environ['http_proxy'] = http_proxy
            logging.debug('Caller\'s http_proxy=%s', http_proxy)
        else:
            if 'http_proxy' in self._environ:
                del self._environ['http_proxy']
            logging.debug('No http_proxy for caller')

        if not self.acquire():
            return False

        def finish_cb(updater, failed):
            updater.join()
            _uninhibit_suspend()
            if not failed:
                for i in updater.signals:
                    getattr(self, i)()
            state = service.STATE_FAILED if failed else service.STATE_SUCCESS
            self.finished(state)

        self.info(_('Check for unattended packages updates.'))
        _inhibit_suspend()
        _Updater(self._pms, finish_cb, self.info).start()
        return True

    @env.BaseService.signal(INTERFACE, signature='')
    def Reboot(self):
        self._compat.Reboot()

    @env.BaseService.signal(INTERFACE, signature='')
    def Relogin(self):
        self._compat.Relogin()

    @property
    def LastUpdate(self):
        return self._pms.last_updated()


class _Updater(threading.Thread):

    def __init__(self, a_pms, finish_cb, info_cb):
        threading.Thread.__init__(self)
        # To not wait for the thread on process exit
        self.daemon = True
        self._pms = a_pms
        self._finish_cb = finish_cb
        self._info_cb = info_cb
        self.signals = []

    def info(self, *args):
        gobject.idle_add(self._info_cb, *args)

    def run(self):
        failed = False
        try:
            self._update()
        except Exception, error:
            self.info(_('Failed to process unattended updates: %s'), error)
            util.exception()
            failed = True
        gobject.idle_add(self._finish_cb, self, failed)

    def _update(self):
        # Let self._pms make sure that environment is ready to start updating
        self._pms.cleanup()

        # Renew metadata on every call to avoid system level caching
        self._pms.update_metadata()

        to_update = set(self._pms.to_update())
        if not to_update:
            self.info(_('No packages to update'))
            return []

        self.info(_('Update %s package(s)'), ', '.join(to_update))
        self._pms.update()

        after_update = set(self._pms.to_update())
        if after_update:
            self.info(_('Not updated packages: %s'), ', '.join(to_update))

        if (to_update & set(self._pms.reboot_packages)) != \
                (after_update & set(self._pms.reboot_packages)):
            self.info(_('Need to reboot'))
            self.signals.append('Reboot')

        if (to_update & set(self._pms.relogin_packages)) != \
                (after_update & set(self._pms.relogin_packages)):
            self.info(_('Need to relogin'))
            self.signals.append('Relogin')


class _ServiceCompat(env.BaseService):

    def __init__(self):
        env.BaseService.__init__(self, dbus.SystemBus, None, _PATH_COMPAT)

    @env.BaseService.signal(_INTERFACE_COMPAT, signature='')
    def Reboot(self):
        pass

    @env.BaseService.signal(_INTERFACE_COMPAT, signature='')
    def Relogin(self):
        pass


def _inhibit_suspend():
    if exists(_POWERD_INHIBIT_SUSPEND):
        # OLPC's powerd daemon,
        # See http://wiki.laptop.org/go/Powerd and `olpc-nosleep` sources
        file(join(_POWERD_INHIBIT_SUSPEND, str(os.getpid())), 'w').close()


def _uninhibit_suspend():
    path = join(_POWERD_INHIBIT_SUSPEND, str(os.getpid()))
    if exists(path):
        os.unlink(path)
