#!/usr/bin/env python

# sugar-lint: disable

import sys
import time
from os.path import exists, relpath, abspath, isfile, isdir

import gobject

from __init__ import tests

from sugar_client import session, env, server


class SessionTest(tests.Test):

    def setUp(self):
        tests.Test.setUp(self)
        self.mainloop = gobject.MainLoop()
        session._STARTUP_DELAY = 0

    def test_Scheduler_Start(self):
        calls = []

        def cb():
            calls.append(True)
            if len(calls) > 3:
                self.mainloop.quit()

        ts = time.time()
        task = session.Scheduler('test', 1, 0, cb)
        task.start()
        self.mainloop.run()
        delay = time.time() - ts
        assert delay >= 2 and delay <= 4

    def test_Scheduler_StartupDelay(self):
        calls = []

        def cb():
            calls.append(True)
            if len(calls) > 3:
                self.mainloop.quit()

        ts = time.time()
        session._STARTUP_DELAY = 3
        task = session.Scheduler('test', 1, 0, cb)
        task.start()
        self.mainloop.run()
        delay = time.time() - ts
        assert delay >= 5 and delay <= 7

    def test_Scheduler_FallbackDelay(self):
        calls = []

        def cb():
            calls.append(True)
            if len(calls) < 3:
                raise RuntimeError()
            elif len(calls) > 5:
                self.mainloop.quit()

        ts = time.time()
        task = session.Scheduler('test', 1, 2, cb)
        task.start()
        self.mainloop.run()
        delay = time.time() - ts
        assert delay >= 6 and delay <= 8

    def test_Scheduler_StartOnceDuringTimeout(self):
        calls = []

        def cb():
            calls.append(True)

        task = session.Scheduler('test', 4, 0, cb)

        task.start()
        gobject.timeout_add_seconds(1, self.mainloop.quit)
        self.mainloop.run()
        self.assertEqual(1, len(calls))
        task.stop()

        task.start()
        gobject.timeout_add_seconds(1, self.mainloop.quit)
        self.mainloop.run()
        self.assertEqual(1, len(calls))
        task.stop()

        task.start()
        gobject.timeout_add_seconds(3, self.mainloop.quit)
        self.mainloop.run()
        self.assertEqual(2, len(calls))

    def test_Scheduler_Stop(self):
        calls = []

        def cb():
            calls.append(True)
            if len(calls) == 3:
                self.mainloop.quit()

        task = session.Scheduler('test', 1, 0, cb)
        task.start()
        self.mainloop.run()
        task.stop()
        gobject.timeout_add_seconds(3, self.mainloop.quit)
        self.mainloop.run()
        self.assertEqual(3, len(calls))

    def test_Scheduler_StopStartupDelay(self):
        calls = []

        def cb():
            calls.append(True)

        session._STARTUP_DELAY = 3
        task = session.Scheduler('test', 1, 0, cb)
        task.start()
        gobject.timeout_add_seconds(2, self.mainloop.quit)
        self.mainloop.run()
        task.stop()
        gobject.timeout_add_seconds(2, self.mainloop.quit)
        self.mainloop.run()
        self.assertEqual(0, len(calls))

    def test_Scheduler_StopFallbackDelay(self):
        calls = []

        def cb():
            calls.append(True)
            raise RuntimeError()

        task = session.Scheduler('test', 1, 3, cb)
        task.start()
        gobject.timeout_add_seconds(2, self.mainloop.quit)
        self.mainloop.run()
        self.assertEqual(1, len(calls))
        task.stop()
        gobject.timeout_add_seconds(3, self.mainloop.quit)
        self.mainloop.run()
        self.assertEqual(1, len(calls))

    def test_start_Connected(self):
        backups = []
        unattended_updates = []

        class Backup(object):

            def backup(self, *args):
                backups.append(True)

        def connection_manager(connected_cb, disconnected_cb):
            gobject.idle_add(connected_cb)
            gobject.timeout_add_seconds(3, disconnected_cb)
            gobject.timeout_add_seconds(4, connected_cb)
            gobject.timeout_add_seconds(5, disconnected_cb)
            gobject.timeout_add_seconds(6, self.mainloop.quit)

        self.override(session, '_connection_manager',
                connection_manager)
        self.override(session, '_unattended_update_cb',
                lambda: unattended_updates.append(True))

        self.httpd(8080,
                ('POST:/client/register', '{"success": "OK", "uid": "UID"}'),
                ('/client/status?uid=UID', '{"registered": true}'),
                ('/client/status?uid=UID', '{"registered": true}'),
                )
        self.touch(('sugar/owner.key.pub', 'ssh-dss pubkey'))

        env.api_url.value = 'http://localhost:8080'
        env.update_timeout.value = 1
        env.backup_timeout.value = 1
        env.auto_register.value = 1
        update_task, backup_task = session.start(server.Service(), Backup())
        self.mainloop.run()

        assert backups
        assert unattended_updates
        assert not update_task.started
        assert not backup_task.started

    def quit_on_backup(self):
        env.backup_timeout.value = 1
        self.quit = None

        class Backup(object):

            def __init__(self):
                self.backuped = []
                self.restored = []
                self.State = 0

            def backup(self_, *args):
                self_.backuped.append(True)
                self.quit()
                self.mainloop.quit()

            def Restore(self_, *args):
                self_.restored.append(True)
                self.quit()
                self.mainloop.quit()

        def connection_manager(con_cb, dis_cb):
            self.quit = dis_cb
            gobject.idle_add(con_cb)

        self.Backup = Backup
        self.override(session, '_connection_manager', connection_manager)

    def test_start_DoNotReregisterWOSeetingOption(self):
        self.quit_on_backup()

        # Run empty httpd to break the test if it will request for registration
        self.httpd(8080)

        self.touch(('sugar/owner.key.pub', 'ssh-dss pubkey'))
        env.api_url.value = 'http://localhost:8080'
        env.auto_register.value = 0
        srv = server.Service()

        session.start(srv, self.Backup())
        gobject.timeout_add_seconds(1, self.mainloop.quit)
        self.mainloop.run()

        session.start(srv, self.Backup())
        gobject.timeout_add_seconds(1, self.mainloop.quit)
        self.mainloop.run()

    def test_start_InitialRegister(self):
        self.quit_on_backup()

        self.httpd(8080,
                ('POST:/client/register', '{"success": "OK", "uid": "UID1"}'),
                ('/client/status?uid=UID1', '{"registered": true}'),
                )

        self.touch(('sugar/owner.key.pub', 'ssh-dss pubkey'))
        env.api_url.value = 'http://localhost:8080'
        env.auto_register.value = 1
        srv = server.Service()

        session.start(srv, self.Backup())
        self.mainloop.run()
        self.assertEqual('UID1', env.uid.value)
        assert env.schoolserver.value

    def test_start_ReregistrationOnResettingUID(self):
        self.quit_on_backup()

        self.httpd(8080,
                ('POST:/client/register', '{"success": "OK", "uid": "UID1"}'),
                ('/client/status?uid=UID1', '{"registered": true}'),

                ('POST:/client/register', '{"success": "OK", "uid": "UID2"}'),
                ('/client/status?uid=UID2', '{"registered": true}'),
                )

        self.touch(('sugar/owner.key.pub', 'ssh-dss pubkey'))
        env.api_url.value = 'http://localhost:8080'
        env.auto_register.value = 1
        srv = server.Service()

        session.start(srv, self.Backup())
        self.mainloop.run()
        self.assertEqual('UID1', env.uid.value)
        assert env.schoolserver.value

        env.uid.value = None
        session.start(srv, self.Backup())
        self.mainloop.run()
        self.assertEqual('UID2', env.uid.value)
        assert env.schoolserver.value

    def test_start_NotRegisteredButFQDNIsTheSame(self):
        self.quit_on_backup()

        self.httpd(8080,
                ('POST:/client/register', '{"success": "OK", "uid": "UID1"}'),
                ('/client/status?uid=UID1', '{"registered": true}'),

                ('/client/status?uid=UID1', '{"registered": false}'),
                ('POST:/client/register', '{"success": "OK", "uid": "UID2"}'),
                ('/client/status?uid=UID2', '{"registered": true}'),
                )

        self.touch(('sugar/owner.key.pub', 'ssh-dss pubkey'))
        env.api_url.value = 'http://localhost:8080'
        env.auto_register.value = 1
        srv = server.Service()

        session.start(srv, self.Backup())
        self.mainloop.run()
        self.assertEqual('UID1', env.uid.value)
        assert env.schoolserver.value

        session.start(srv, self.Backup())
        self.mainloop.run()
        self.assertEqual('UID2', env.uid.value)
        assert env.schoolserver.value

    def test_start_FQDNHasChangedButWeAreStillRegistered(self):
        self.quit_on_backup()

        self.httpd(8080,
                ('POST:/client/register', '{"success": "OK", "uid": "UID1"}'),
                ('/client/status?uid=UID1', '{"registered": true}'),

                ('/client/status?uid=UID1', '{"registered": true}'),
                )

        self.touch(('sugar/owner.key.pub', 'ssh-dss pubkey'))
        env.api_url.value = 'http://localhost:8080'
        env.auto_register.value = 1
        srv = server.Service()

        session.start(srv, self.Backup())
        self.mainloop.run()
        self.assertEqual('UID1', env.uid.value)
        assert env.schoolserver.value

        env.schoolserver.value = 'fake'
        session.start(srv, self.Backup())
        self.mainloop.run()
        self.assertEqual('UID1', env.uid.value)
        self.assertNotEqual('fake', env.schoolserver.value)

    def test_start_FQDNHasChangedAndWeAreNotRegisteredThere(self):
        self.quit_on_backup()

        self.httpd(8080,
                ('POST:/client/register', '{"success": "OK", "uid": "UID1"}'),
                ('/client/status?uid=UID1', '{"registered": true}'),

                ('/client/status?uid=UID1', '{"registered": false}'),
                )

        self.touch(('sugar/owner.key.pub', 'ssh-dss pubkey'))
        env.api_url.value = 'http://localhost:8080'
        env.auto_register.value = 1
        srv = server.Service()

        session.start(srv, self.Backup())
        self.mainloop.run()
        self.assertEqual('UID1', env.uid.value)
        assert env.schoolserver.value

        env.schoolserver.value = 'fake'
        session.start(srv, self.Backup())
        gobject.timeout_add_seconds(2, self.mainloop.quit)
        self.mainloop.run()
        self.assertEqual('UID1', env.uid.value)
        self.assertEqual('fake', env.schoolserver.value)

    def test_start_IncreaseAuto_RegisterToForceRegistration(self):
        self.quit_on_backup()

        self.httpd(8080,
                ('POST:/client/register', '{"success": "OK", "uid": "UID1"}'),
                ('/client/status?uid=UID1', '{"registered": true}'),

                ('/client/status?uid=UID1', '{"registered": false}'),
                ('POST:/client/register', '{"success": "OK", "uid": "UID2"}'),
                ('/client/status?uid=UID2', '{"registered": true}'),
                )

        self.touch(('sugar/owner.key.pub', 'ssh-dss pubkey'))
        env.api_url.value = 'http://localhost:8080'
        env.auto_register.value = 1
        srv = server.Service()

        session.start(srv, self.Backup())
        self.mainloop.run()
        self.assertEqual('UID1', env.uid.value)
        assert env.schoolserver.value

        env.auto_register.value = 2
        session.start(srv, self.Backup())
        self.mainloop.run()
        self.assertEqual('UID2', env.uid.value)
        self.assertNotEqual('fake', env.schoolserver.value)

    def test_start_NoNeedToReregister(self):
        self.quit_on_backup()

        self.httpd(8080,
                ('POST:/client/register', '{"success": "OK", "uid": "UID1"}'),
                ('/client/status?uid=UID1', '{"registered": true}'),

                ('/client/status?uid=UID1', '{"registered": true}'),
                )

        self.touch(('sugar/owner.key.pub', 'ssh-dss pubkey'))
        env.api_url.value = 'http://localhost:8080'
        env.auto_register.value = 1
        srv = server.Service()

        session.start(srv, self.Backup())
        self.mainloop.run()
        self.assertEqual('UID1', env.uid.value)
        assert env.schoolserver.value

        session.start(srv, self.Backup())
        self.mainloop.run()
        self.assertEqual('UID1', env.uid.value)

    def test_start_Restored(self):
        self.quit_on_backup()
        env.auto_restore.value = True

        self.httpd(8080,
                ('POST:/client/register', '{"success": "OK", "uid": "UID1"}'),
                ('/client/status?uid=UID1', '{"registered": true, "pending-restore": true}'),
                )

        self.touch(('sugar/owner.key.pub', 'ssh-dss pubkey'))
        env.api_url.value = 'http://localhost:8080'
        env.auto_register.value = 1
        srv = server.Service()

        backup = self.Backup()
        backup.on_finished = self.mainloop.quit
        session._STARTUP_DELAY = 3

        session.start(srv, backup)
        self.mainloop.run()
        assert backup.restored
        assert not backup.backuped

    def test_start_RestoreOnlyEmptyJoural(self):
        self.quit_on_backup()
        env.auto_restore.value = True

        self.httpd(8080,
                ('POST:/client/register', '{"success": "OK", "uid": "UID1"}'),
                ('/client/status?uid=UID1', '{"registered": true, "pending-restore": true}'),
                )

        self.touch(('sugar/owner.key.pub', 'ssh-dss pubkey'))
        env.api_url.value = 'http://localhost:8080'
        env.auto_register.value = 1
        srv = server.Service()

        self.touch('sugar/datastore/00/')
        backup = self.Backup()
        session.start(srv, backup)
        self.mainloop.run()
        assert not backup.restored
        assert backup.backuped


if __name__ == '__main__':
    tests.main()
