# Copyright (C) 2012, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import logging
from gettext import gettext as _

import dbus
from dbus.lowlevel import HANDLER_RESULT_NOT_YET_HANDLED

from sugar_stats import util
from sugar_stats.util import enforce


_logger = logging.getLogger('sugar_stats')
_session = None


def session():
    global _session
    if _session is None:
        _session = Monitor(dbus.SessionBus())
    return _session


def close():
    global _session
    if _session is not None:
        _session.close()
        _session = None


class Monitor(object):

    def __init__(self, bus):
        self._bus = bus
        self._seqno = 0
        self._connects = {}
        self._signals = []
        self._name_connectors = []

        signal = self._bus.add_signal_receiver(
                self.__NameOwnerChanged_cb,
                signal_name='NameOwnerChanged',
                dbus_interface='org.freedesktop.DBus')
        self._signals.append((signal, None))

        def filter_cb(bus, msg):
            try:
                self.__filter_cb(msg)
            except Exception:
                util.exception(_logger, _('Fail to process DBus filter'))

            if msg.get_destination() == self._bus.get_unique_name():
                # By DBus design, `HANDLER_RESULT_NOT_YET_HANDLED` should be
                # the right return value all time. But even `dbus-monitor` doesn't
                # do that (see its sources). In any case,
                # `HANDLER_RESULT_NOT_YET_HANDLED` should be returned for messages
                # targeting to current process, becuase otherwise, process doesn't
                # process them and requesters fail in timeout.
                return HANDLER_RESULT_NOT_YET_HANDLED

        self._bus.add_message_filter(filter_cb)
        self._filter_cb = filter_cb

    def connect_to_name(self, callback, bus_name=None, path=None):
        if bus_name:
            try:
                obj = self._bus.get_object(bus_name, path)
            except dbus.exceptions.DBusException:
                callback(None)
            else:
                callback(obj)
        self._name_connectors.append((callback, bus_name, path))

    def connect_to_signal(self, interface, bus_name, path, signal, cb):
        obj = dbus.Interface(self._bus.get_object(bus_name, path), interface)
        self._signals.append((obj.connect_to_signal(signal, cb), cb))

    def connect(self, callback, *args, **kwargs):
        match = []
        for key, value in kwargs.items():
            if isinstance(value, str):
                value = "'%s'" % value
            match.append('%s=%s' % (key, value))
        match = ','.join(match)

        getters = []
        for name, value in kwargs.items():
            if name == 'type':
                value = {
                    'method_call': 1,
                    'method_return': 2,
                    'error': 3,
                    'signal': 4,
                    }[value]
            getter = getattr(dbus.lowlevel.Message, 'get_%s' % name)
            getters.append((getter, value))

        error_seqno = None
        if kwargs.get('type') == 'method_return':
            kwargs['type'] = 'error'
            error_seqno = self.connect(self.__error_cb, self._seqno, match,
                    **kwargs)

        self._seqno += 1
        self._connects[self._seqno] = \
                (callback, args, error_seqno, match, getters)
        self._bus.add_match_string(match)

        return self._seqno

    def disconnect(self, seqno):
        enforce(seqno in self._connects)
        __, __, __, match, __ = self._connects.pop(seqno)
        self._bus.remove_match_string(match)

    def disconnect_by_func(self, callback):
        for seqno, (cb, __, __, __, __) in self._connects.items():
            if cb is callback:
                self.disconnect(seqno)

        signals = []
        for signal, cb in self._signals:
            if cb is callback:
                signal.remove()
            else:
                signals.append((signal, cb))
        self._signals = signals

        name_connectors = []
        for cb, bus_name, path in self._name_connectors:
            if cb is not callback:
                name_connectors.append((cb, bus_name, path))
        self._name_connectors = name_connectors

    def close(self):
        self._bus.remove_message_filter(self._filter_cb)
        for seqno in self._connects.keys():
            self.disconnect(seqno)
        while self._signals:
            signal, __ = self._signals.pop()
            signal.remove()
        del self._name_connectors[:]
        self._bus.close()

    def __filter_cb(self, msg):
        for seqno, (callback, args, error_seqno, __, getters) in \
                self._connects.items():
            for getter, value in getters:
                if value != getter(msg):
                    break
            else:
                if callback(msg, *args):
                    self.disconnect(seqno)
                    if error_seqno:
                        self.disconnect(error_seqno)

    def __NameOwnerChanged_cb(self, name, old, new):
        for callback, bus_name, path in self._name_connectors:
            if bus_name and bus_name == name:
                if old and not new:
                    _logger.debug('%s went away', bus_name)
                    callback(None)
                elif new and not old:
                    _logger.debug('%s appeared', bus_name)
                    obj = self._bus.get_object(bus_name, path)
                    callback(obj)
            elif not bus_name:
                callback(name, new, old)

    def __error_cb(self, msg, reply_seqno, match, kwargs):
        _logger.debug('method_return for %s match failed: %s',
                match, msg.get_error_name())
        self.disconnect(reply_seqno)
        return True
