# Copyright (C) 2012, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

from os.path import join
from gettext import gettext as _

import restful_document as rd
enforce = rd.util.enforce
util = rd.util

from sugar_stats.rrd import Rrd


stats = util.Option(
        _('enable stats collecting'),
        default=True, type_cast=util.Option.bool_cast, action='store_true')

stats_root = util.Option(
        _('path to the root directory for placing stats'))

stats_step = util.Option(
        _('step interval in seconds for RRD databases'),
        default=60, type_cast=int)

stats_server_rras = util.Option(
        _('space separated list of RRAs for RRD databases on a server side'),
        default='RRA:AVERAGE:0.5:1:4320 RRA:AVERAGE:0.5:5:2016',
        type_cast=lambda x: [i for i in x.split() if i],
        type_repr=lambda x: ' '.join(x))

stats_client_rras = util.Option(
        _('space separated list of RRAs for RRD databases on client side'),
        default='RRA:AVERAGE:0.5:1:4320 RRA:AVERAGE:0.5:5:2016',
        type_cast=lambda x: [i for i in x.split() if i],
        type_repr=lambda x: ' '.join(x))

stats_keyfile = util.Option(
        _('path to SSL certificate keyfile to serve stats server via HTTPS'))

stats_certfile = util.Option(
        _('path to SSL certificate to serve stats server via HTTPS'))


class User(rd.User):

    _rrd = {}

    @rd.restful_method(method='GET', cmd='stats-info')
    def _stats_info(self):
        enforce(rd.principal.user == self.guid,
                _('Operation is permitted only for owner'))

        status = {}
        rrd = User._get_rrd(self.guid)
        for name, __, last_update in rrd.dbs:
            status[name] = last_update + stats_step.value

        return {'enable': stats.value,
                'step': stats_step.value,
                'rras': stats_client_rras.value,
                'status': status,
                }

    @rd.restful_method(method='POST', cmd='stats-upload')
    def _stats_upload(self):
        enforce(rd.principal.user == self.guid,
                _('Operation is permitted only for owner'))
        rrd = User._get_rrd(self.guid)
        content = rd.request.content
        for timestamp, values in content['values']:
            rrd.put(content['name'], values, timestamp)

    @classmethod
    def _get_rrd(cls, guid):
        rrd = cls._rrd.get(guid)
        if rrd is None:
            rrd = cls._rrd[guid] = Rrd(join(stats_root.value, guid[:2], guid),
                    stats_step.value, stats_server_rras.value)
        return rrd


util.Option.seek('stats')
