# Copyright (C) 2012-2013 Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

"""Main process startup routines."""

import sys
import logging
import textwrap
from optparse import OptionParser
from os.path import basename
from gettext import gettext as _

from .options import Option
from . import printf, util
enforce = util.enforce


debug = Option(
        _('debug logging level; multiple argument'),
        default=0, type_cast=int, short_option='-D', action='count',
        name='debug')

no_hints = Option(
        _('suppress suggesting hints'),
        default=False, short_option='-H', action='store_true',
        name='no-hints')


_LOGFILE_FORMAT = '%(asctime)s %(levelname)s %(name)s: %(message)s'


def command(description='', name=None, hidden=False, **options):

    def decorator(func):
        func.is_command = True
        func.name = name
        func.description = description
        func.options = options
        func.hidden = hidden
        return func

    return decorator


class Application(object):

    def __init__(self, name, description=None, version=None, epilog=None,
            where=None, **parse_args):
        self.args = None
        self.name = name

        self._commands = {}
        for attr in dir(self):
            attr = getattr(self, attr)
            if hasattr(attr, 'is_command') and \
                    (attr.name != 'config' or 'config_files' in parse_args):
                self._commands[attr.name or attr.__name__] = attr

        parser = OptionParser(usage='%prog [OPTIONS]', description=description,
                add_help_option=False)

        if version:
            parser.add_option('-V', '--version',
                    help=_('show version number and exit'),
                    action='version')
            parser.print_version = lambda: sys.stdout.write('%s\n' % version)

        parser.add_option('-h', '--help',
                help=_('show this help message and exit'),
                action='store_true')

        options, self.args = Option.parse_args(parser, **parse_args)
        Option.config.description = \
                _('colon separated list of paths to configuration file(s)')

        def print_desc(term, desc):
            text = []
            for num, line in enumerate(desc.split('\n')):
                if num == 0:
                    for i in line:
                        if i.isalpha() and not i.isupper():
                            break
                    else:
                        term += ' ' + line
                        continue
                text.extend(textwrap.wrap(line, 54))
            if len(term) < 24:
                sys.stdout.write('  %-22s' % term)
            else:
                text.insert(0, '')
                sys.stdout.write('  %s' % term)
            print ('\n' + ' ' * 24).join(text)

        def print_commands():
            if not self._commands:
                return
            print ''
            print _('Commands') + ':'
            for name, attr in sorted(self._commands.items(),
                    lambda x, y: cmp(x[0], y[0])):
                if attr.hidden:
                    continue
                description = attr.description.split('\n')
                name = ' '.join([name] + description[1:])
                print_desc(name, description[0])

        if not self.args and not options.help:
            prog = basename(sys.argv[0])
            usage = _('Usage')
            # TRANS: Usage command-line format
            print usage + ': ' + _('%s [OPTIONS] [COMMAND]') % prog
            print ' ' * len(usage) + '  %s -h|--help' % prog
            print
            print description
            print_commands()
            if epilog:
                print ''
                print epilog
            exit(0)

        if options.help:
            parser.print_help()
            print_commands()
            if where:
                print ''
                # TRANS: Command-line help sub-header
                print _('Where') + ':'
                for term in sorted(where):
                    print_desc(term, where[term])
            if epilog:
                print ''
                print epilog
            exit(0)

        if not debug.value:
            logging_level = logging.WARNING
        elif debug.value == 1:
            logging_level = logging.INFO
        else:
            logging_level = logging.DEBUG
        logging_format = _LOGFILE_FORMAT

        root_logger = logging.getLogger('')
        for i in root_logger.handlers:
            root_logger.removeHandler(i)

        logging.basicConfig(level=logging_level, format=logging_format)

    def start(self):
        cmd_name = self.args.pop(0)
        try:
            cmd = self._commands.get(cmd_name)
            enforce(cmd is not None, _('Unknown command %r') % cmd_name)
            if Option.config_files:
                logging.info('Load configuration from %s file(s)',
                        ', '.join(Option.config_files))
            exit(cmd() or 0)
        except Exception:
            printf.exception('%s %s', _('Aborted'), self.name)
            exit(1)
        finally:
            if not no_hints.value:
                printf.flush_hints()

    @command('output current configuration', name='config')
    def _cmd_config(self):
        if self.args:
            opt = self.args.pop(0)
            enforce(opt in Option.items, _('Unknown option %r'), opt)
            exit(0 if bool(Option.items[opt].value) else 1)
        else:
            print '\n'.join(Option.export())
