# Copyright (C) 2010-2012, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import re
import ftplib
import logging
import tempfile
from os.path import exists
from gettext import gettext as _

from sweets_distribution import lsb_release, util, printf
from sweets_distribution.util import enforce


_PUBKEY = 'C13B 4120 5A82 F877 66D8  F78D B925 F946 3182 B17D'

_REPO_URL_RE = re.compile('[^#]*http://download.sugarlabs.org/packages/' \
        'SweetsDistribution(|:/([^/]+))/([^-]+)-([^/]+)')
_REPO_NAME_RE = re.compile('([^-]+)-(.*)$')

_PMS = {'Fedora': {
            'config_path': '/etc/yum.repos.d/SweetsDistribution.repo',
            'noarch_dir': 'noarch',
            'x86_64_dir': 'x86_64',
            'package_name_re': re.compile('(.+)-([^-]+-.*)\.[^.]+\.rpm$'),
            'query_cmd': \
                    'rpm -q --qf \'%%{epoch} %%{version}-%%{release}\n\' %s',
            'package_rel_re': re.compile('(([^ ]+)) (.+)$'),
            'uninstall_cmd': ['yum', 'remove', '-y', '-t'],
            'install_cmd': ['yum', 'install', '-y', '-t'],
            'direct_install_cmd': ['rpm', '--force', '-i'],
            'list_gpg_cmd': \
                    'rpm -q gpg-pubkey --qf \'%{description}\n\' | ' \
                    'gpg2 --quiet --with-fingerprint',
            'pubkey_path': 'repodata/repomd.xml.key',
            'import_cmd': ['rpm', '--import'],
            'config': [
                '[SweetsDistribution]',
                'name=SweetsDistribution',
                'failovermethod=priority',
                'baseurl=%(prefix)s/%(distro_id)s-%(distro_rel)s/',
                'enabled=1',
                'metadata_expire=1',
                'gpgcheck=1',
                '',
                '[SweetsDistribution:%(repo)s]',
                'name=SweetsDistribution:%(repo)s',
                'failovermethod=priority',
                'baseurl=%(prefix)s:/%(repo)s/%(distro_id)s-%(distro_rel)s/',
                'enabled=1',
                'metadata_expire=1',
                'gpgcheck=1',
                ],
            'disable_all_repos': '--disablerepo=*',
            'enable_repo': '--enablerepo=%s',
            },
        'Debian': {
            'global_conf_path': '/etc/apt/sources.list',
            'config_path': '/etc/apt/sources.list.d/SweetsDistribution.list',
            'noarch_dir': 'all',
            'x86_64_dir': 'amd64',
            'package_name_re': re.compile('(.+)_([^-]+-.*)_[^.]+\.deb$'),
            'query_cmd': 'dpkg-query -W --showformat ' \
                    '\'${Status} ${Version}\n\' %s | grep ^install',
            'package_rel_re': re.compile('(([^: ]):)?([^ ]+)$'),
            'update_cmd': ['apt-get', 'update'],
            'uninstall_cmd': ['apt-get', 'purge', '-y', '-f'],
            'install_cmd': ['apt-get', 'install', '-y', '-f'],
            'direct_install_cmd': ['dpkg', '-i'],
            'list_gpg_cmd': 'apt-key finger',
            'pubkey_path': 'Release.key',
            'import_cmd': ['apt-key', 'add'],
            'config': [
                'deb %(prefix)s/%(distro_id)s-%(distro_rel)s/ ./',
                'deb %(prefix)s:/%(repo)s/%(distro_id)s-%(distro_rel)s/ ./',
                ],
            'rel_map': {'testing': '7.0'},
            },
        'Ubuntu': {
            'global_conf_path': '/etc/apt/sources.list',
            'config_path': '/etc/apt/sources.list.d/SweetsDistribution.list',
            'noarch_dir': 'all',
            'x86_64_dir': 'amd64',
            'package_name_re': re.compile('(.+)_([^-]+-.*)_[^.]+\.deb$'),
            'query_cmd': 'dpkg-query -W --showformat ' \
                    '\'${Status} ${Version}\n\' %s | grep ^install',
            'package_rel_re': re.compile('(([^: ]):)?([^ ]+)$'),
            'update_cmd': ['apt-get', 'update'],
            'uninstall_cmd': ['apt-get', 'purge', '-y', '-f'],
            'install_cmd': ['apt-get', 'install', '-y', '-f'],
            'direct_install_cmd': ['dpkg', '-i'],
            'list_gpg_cmd': 'apt-key finger',
            'pubkey_path': 'Release.key',
            'import_cmd': ['apt-key', 'add'],
            'config': [
                'deb %(prefix)s/%(distro_id)s-%(distro_rel)s/ ./',
                'deb %(prefix)s:/%(repo)s/%(distro_id)s-%(distro_rel)s/ ./',
                ],
            },
        }


class Pms(object):

    def __init__(self):
        self.distro_id = lsb_release.distributor_id()
        self.distro_rel = lsb_release.release()
        self.plugged_repos = []
        self.new_config = None

        self._assert_distro(self.distro_id in _PMS)

        self.__ftp = None
        self._pms = _PMS[self.distro_id]

        if 'rel_map' in self._pms:
            self.distro_rel = self._pms['rel_map'].get(self.distro_rel) or \
                    self.distro_rel

        if 'global_conf_path' in self._pms:
            if exists(self._pms['global_conf_path']):
                new_config_lines = []
                patched = False
                with file(self._pms['global_conf_path']) as f:
                    for line in f.readlines():
                        if _REPO_URL_RE.match(line) is not None:
                            new_config_lines.append('#%s' % line)
                            patched = True
                        else:
                            new_config_lines.append(line)
                if patched:
                    self.new_config = ''.join(new_config_lines)

        if exists(self._pms['config_path']):
            with file(self._pms['config_path']) as f:
                for line in f.readlines():
                    match = _REPO_URL_RE.match(line)
                    if match is None:
                        continue
                    __, repo_version, distro_id, distro_rel = match.groups()
                    if repo_version is None:
                        # Skip common repo
                        continue
                    self.plugged_repos.append(
                            (repo_version, distro_id, distro_rel))

    def fetch_repos(self):
        printf.progress(_('Fetching repositories list..'))

        root = '/packages/SweetsDistribution:'
        result = []

        for repo_version in self._ls(root):
            for distro_id, distro_rel in \
                    self._fetch_repo(root + '/' + repo_version):
                result.append((repo_version, distro_id, distro_rel))

        return result

    def to_sync(self):
        if not self.plugged_repos:
            return []

        repo = self.plugged_repos[0]
        result = []

        for is_factory, name, package_rel, __ in self._fetch_packages_list([
                '/packages/SweetsDistribution:/%s/%s-%s/' % repo,
                '/packages/SweetsDistribution/%s-%s/' % repo[1:]]):
            pkg = _Package(name)
            pkg.to_rel = package_rel

            if name == 'sweets-desktop' and \
                    self.resolve_package('sweets-distribution')[1]:
                name = pkg.from_name = 'sweets-distribution'

            pkg.from_epoch, pkg.from_rel = self.resolve_package(name)
            cmp_epoch = cmp(is_factory, pkg.from_epoch == 1)
            if pkg.from_rel is None or \
                    (pkg.to_rel == pkg.from_rel) and cmp_epoch == 0:
                continue

            if pkg.from_name:
                pkg.downgrade = True
            elif cmp_epoch:
                pkg.downgrade = cmp_epoch < 0
            else:
                pkg.downgrade = \
                        _parse_rel(pkg.to_rel) < _parse_rel(pkg.from_rel)

            result.append(pkg)

        return result

    def has_keys(self):
        output = util.call(self._pms['list_gpg_cmd'], shell=True)

        for line in (output or '').split('\n'):
            if _PUBKEY in line:
                return True

    def update(self):
        if self.new_config:
            confpath = self._pms['global_conf_path']

            printf.info(_('Patch global "%s" to hide ' \
                    'any Sweets Distribution related lines'), confpath)

            with util.new_file(confpath, mode=0644) as f:
                f.write(self.new_config)
            self.new_config = None

        if not self.has_keys() and self.plugged_repos:
            self._import_pubkey(self.plugged_repos[0])

        if 'update_cmd' in self._pms:
            printf.progress(_('Fetching new packages metadata..'))
            util.assert_call(self._pms['update_cmd'])

    def uninstall(self, packages):
        if not packages:
            return
        names = [i.from_name if i.from_name else i.to_name for i in packages]

        printf.info(_('Uninstall %s package(s) to make downgrade possible'),
                ', '.join(names))
        printf.progress(_('Uninstalling packages..'))

        cmd = self._restrict_repos(self._pms['uninstall_cmd'])
        util.assert_call(cmd + names)

    def install(self, packages):
        if not packages:
            return
        names = [i.to_name for i in packages]

        printf.info(_('Update %s package(s)'), ', '.join(names))
        printf.progress(_('Updating packages..'))

        cmd = self._restrict_repos(self._pms['install_cmd'])
        util.assert_call(cmd + names)

    def select(self, repo_version, distro_rel):
        props = {'prefix': 'http://download.sugarlabs.org/packages/' \
                    'SweetsDistribution',
                 'distro_id': self.distro_id,
                 'distro_rel': distro_rel,
                 'repo': repo_version,
                 }

        if self.plugged_repos:
            self_repo, __, self_rel = self.plugged_repos[0]
            if self_repo == repo_version and self_rel == distro_rel:
                printf.info(_('Repository %(repo)s already selected') % props)
                return

        path = '/packages/SweetsDistribution:/' \
                '%(repo)s/%(distro_id)s-%(distro_rel)s/' % props
        enforce([i for i in self._ls(path)],
                _('No such repository %(distro_id)s-%(distro_rel)s ' \
                        'in %(repo)s repository') % props)

        printf.info(_('Create "%s" repository config'),
                self._pms['config_path'])

        with util.new_file(self._pms['config_path'], mode=0644) as f:
            for line in self._pms['config']:
                f.write((line % props) + '\n')

        self.plugged_repos[:] = [(repo_version, self.distro_id, distro_rel)]
        self.update()

    def clean(self):
        if not self.plugged_repos:
            printf.info(_('No registered repositories'))
            return

        if exists(self._pms['config_path']):
            printf.info(_('Unlink "%s" repository config'),
                    self._pms['config_path'])
            os.unlink(self._pms['config_path'])

        del self.plugged_repos[:]
        self.update()

    def self_install(self):
        printf.progress(_('Fetching repositories list..'))

        distro_id, distro_rel = (None, None)
        for distro_id, distro_rel in \
                self._fetch_repo('/packages/SweetsDistribution'):
            if distro_id == self.distro_id and distro_rel == self.distro_rel:
                break
        else:
            self._assert_distro(False)

        root = '/packages/SweetsDistribution/%s-%s/' % (distro_id, distro_rel)
        path = None
        for __, name, __, path in self._fetch_packages_list([root]):
            if name == 'sweets':
                break
        else:
            raise RuntimeError(_('No sweets package for %s') % distro_id)

        printf.progress(_('Fetching sweets package..'))

        with tempfile.NamedTemporaryFile() as tmpfile:
            self._ftp.retrbinary('RETR %s' % path, tmpfile.file.write)
            tmpfile.file.flush()

            printf.info(_('Install sweets package'))
            printf.progress(_('Installing packages..'))

            util.assert_call(self._pms['direct_install_cmd'] + [tmpfile.name])

    @property
    def _ftp(self):
        if self.__ftp is None:
            self.__ftp = ftplib.FTP('download.sugarlabs.org')
            self.__ftp.login()
        return self.__ftp

    def _fetch_repo(self, path):
        for name in self._ls(path):
            match = _REPO_NAME_RE.match(name)
            if match is None:
                continue
            distro_id, distro_rel = match.groups()
            yield distro_id, distro_rel

    def _fetch_packages_list(self, paths):
        printf.progress(_('Fetching packages list..'))

        subdirs = [self._pms['noarch_dir']]
        if util.assert_call('arch') == 'x86_64':
            subdirs.append(self._pms['x86_64_dir'])
        else:
            subdirs.append('i386')

        for subdir in subdirs:
            for path in paths:
                path += '/' + subdir
                for filename in self._ls(path):
                    match = self._pms['package_name_re'].match(filename)
                    if match is None:
                        continue
                    name, rel = match.groups()
                    yield 'Factory' in path, name, rel, path + '/' + filename

    def resolve_package(self, name):
        reply = util.call(self._pms['query_cmd'] % name, shell=True)
        if not reply:
            return None, None

        match = self._pms['package_rel_re'].search(reply)
        if match is None:
            logging.warning(_('Cannot parse "%s" package version ' \
                    'from "%s" string'), name, reply)
            return None, None
        else:
            __, epoch, rel = match.groups()
            if epoch and epoch.isdigit():
                epoch = int(epoch)
            else:
                epoch = 0
            return epoch, rel

    def _import_pubkey(self, repo):
        path = '/packages/SweetsDistribution:/%s/%s-%s/' % repo
        path += self._pms['pubkey_path']

        printf.info(_('Import GPG key from "%s"'), path)

        with tempfile.NamedTemporaryFile() as tmpfile:
            self._ftp.retrbinary('RETR %s' % path, tmpfile.file.write)
            tmpfile.file.flush()

            util.assert_call(self._pms['import_cmd'] + [tmpfile.name])

    def _ls(self, path):
        for i in self._ftp.nlst(path):
            yield i[len(path):].strip('/')

    def _assert_distro(self, value):
        if value:
            return
        printf.hint(_('See %s for the list of supported platforms'),
                'http://wiki.sugarlabs.org/go/Platform_Team/' \
                        'Harmonic_Distribution/Supported_platforms')
        raise RuntimeError(_('Not supported GNU/Linux distribution, %s-%s') % \
                (self.distro_id, self.distro_rel))

    def _restrict_repos(self, cmd):
        if not exists('/ofw') or not self._pms.get('disable_all_repos'):
            return cmd
        cmd.append(self._pms['disable_all_repos'])
        cmd.append(self._pms['enable_repo'] % 'SweetsDistribution')
        for repo, __, __ in self.plugged_repos:
            cmd.append(self._pms['enable_repo'] % 'SweetsDistribution:' + repo)
        return cmd


class _Package(object):

    to_name = None
    to_rel = None
    from_name = None
    from_rel = None
    from_epoch = None
    downgrade = None

    def __init__(self, name):
        self.to_name = name

    def __str__(self):
        return '%s (%s%s => %s)' % (
                self.to_name,
                self.from_name + '-' if self.from_name else '',
                self.from_rel,
                self.to_rel)


def _parse_rel(version_string):
    ver, rel = version_string.split('+')[0].split('-')
    return [[int(i) for i in ver.split('.')], [int(i) for i in rel.split('.')]]
