# Copyright (C) 2011, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU Lesser General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU Lesser General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import re
import os
from os.path import join, exists

import gtk
import dbus
import gobject

from sugar import gui, env


_AUTOSEARCH_TIMEOUT = 1000


class DataWidget(object):

    data = None

    def update_data_field(self):
        pass


class Label(gui.Label, DataWidget):

    __gsignals__ = {
            'data-changed': (
                gobject.SIGNAL_RUN_FIRST, gobject.TYPE_NONE, [str, str]),
            }

    def __init__(self, data_field, **kwargs):
        gobject.GObject.__init__(self, **kwargs)
        DataWidget.__init__(self)
        self._data_field = data_field

    def update_data_field(self):
        if self._data_field in self.data:
            self.props.label = self.data[self._data_field]
        else:
            self.props.label = ''


class ActivityIcon(gui.Icon, DataWidget):

    __gsignals__ = {
            'data-changed': (
                gobject.SIGNAL_RUN_FIRST, gobject.TYPE_NONE, [str, str]),
            }

    def __init__(self, pixel_size=None, **kwargs):
        gobject.GObject.__init__(self, **kwargs)
        DataWidget.__init__(self)
        if not pixel_size:
            pixel_size = env.metrics_get(env.STANDARD_ICON_SIZE)
        self.props.pixel_size = pixel_size

    def update_data_field(self):
        icon = self.data.get('icon_path') or \
                _get_icon_path(self.data.get('url'))
        self.props.file = icon


class Button(gtk.Alignment, DataWidget):

    __gsignals__ = {
            'data-changed': (
                gobject.SIGNAL_RUN_FIRST, gobject.TYPE_NONE, [str, str]),
            'activated': (
                gobject.SIGNAL_RUN_FIRST, gobject.TYPE_NONE, []),
            }

    prelight = False

    def __init__(self):
        gtk.Alignment.__init__(self, 0.5, 0.5, 0, 0)
        DataWidget.__init__(self)

        box = gtk.EventBox()
        box.props.visible_window = False
        box.show()
        self.add(box)

        self.icon = gui.Icon()
        self.icon.show()
        box.add(self.icon)

        box.connect('enter-notify-event', self.__enter_notify_event_cb)
        box.connect('leave-notify-event', self.__leave_notify_event_cb)
        box.connect('button-release-event', self.__button_release_event_cb)

        self.do_colors()

    def do_activate(self):
        self.prelight = False
        self.do_colors()
        self.emit('activated')

    def do_dettach(self):
        self.icon.props.stroke_color = env.color_get(env.COLOR_INACTIVE_STROKE)
        self.icon.props.fill_color = env.color_get(env.COLOR_INACTIVE_FILL)

    def do_colors(self):
        if self.prelight:
            self.icon.props.stroke_color = env.color_get(env.COLOR_BUTTON_GREY)
            self.icon.props.fill_color = env.color_get(env.COLOR_BLACK)
        else:
            self.icon.props.stroke_color = env.color_get(env.COLOR_BUTTON_GREY)
            self.icon.props.fill_color = env.color_get(env.COLOR_BUTTON_GREY)

    def __enter_notify_event_cb(self, widget, event):
        x, y = self.get_pointer()
        if x < 0 or x >= self.allocation.width or \
                y < 0 or y >= self.allocation.height:
            return
        self.prelight = True
        self.do_colors()

    def __leave_notify_event_cb(self, widget, event):
        self.prelight = False
        self.do_colors()

    def __button_release_event_cb(self, widget, event):
        if self.prelight:
            self.do_activate()
            return True
        else:
            return False


class KeepButton(Button):

    def __init__(self):
        self._keep_color = None
        Button.__init__(self)
        self.icon.props.file = 'emblem-favorite'
        self.icon.props.pixel_size = env.metrics_get(env.SMALL_ICON_SIZE)

    def update_data_field(self):
        if 'keep' in self.data and self.data['keep'].isdigit():
            self._set_keep(int(self.data['keep']))
        else:
            self._set_keep(0)

    def do_colors(self):
        if self.prelight:
            if self._keep_color is None:
                self.icon.props.stroke_color = \
                        env.color_get(env.COLOR_BUTTON_GREY)
                self.icon.props.fill_color = \
                        env.color_get(env.COLOR_BUTTON_GREY)
            else:
                self.icon.props.stroke_color = self._keep_color.fill
                self.icon.props.fill_color = self._keep_color.stroke
        else:
            if self._keep_color is None:
                self.icon.props.stroke_color = \
                        env.color_get(env.COLOR_BUTTON_GREY)
                self.icon.props.fill_color = \
                        env.color_get(env.COLOR_TRANSPARENT)
            else:
                self.icon.props.stroke_color = self._keep_color.stroke
                self.icon.props.fill_color = self._keep_color.fill

    def do_activate(self):
        keep = 1 if self._keep_color is None else 0
        self.emit('data-changed', 'keep', str(keep))
        self._set_keep(keep)

    def _set_keep(self, keep):
        if keep:
            self._keep_color = env.profile_get_color()
        else:
            self._keep_color = None
        self.do_colors()



class Search(gtk.ToolItem):

    def __init__(self, datasource, **kwargs):
        gtk.ToolItem.__init__(self, **kwargs)
        self.set_expand(True)

        self._autosearch_timer = None

        self.entry = gui.Entry()
        self.entry.props.primary_icon = 'system-search'
        self.entry.props.has_clear_button = True
        self.entry.show()
        self.entry.connect('activate', self.__activate_cb, datasource)
        self.entry.connect('changed', self.__changed_cb)
        self.add(self.entry)

        datasource.connect('notify::query', self.__notify_query_cb)

    def __notify_query_cb(self, datasource, pspec):
        self.entry.props.text = datasource.query

    def __activate_cb(self, widget, datasource):
        if self._autosearch_timer is not None:
            gobject.source_remove(self._autosearch_timer)
        datasource.query = self.entry.props.text.strip()

    def __changed_cb(self, widget):
        if not self.entry.props.text:
            self.entry.activate()
        else:
            if self._autosearch_timer is not None:
                gobject.source_remove(self._autosearch_timer)
            self._autosearch_timer = gobject.timeout_add(
                    _AUTOSEARCH_TIMEOUT, self.__autosearch_timer_cb)

    def __autosearch_timer_cb(self):
        self._autosearch_timer = None
        self.entry.activate()


def _get_icon_path(url):
    xdg_cache_home = os.environ.get('XDG_CACHE_HOME',
            join(os.environ.get('HOME'), '.cache'))
    path = join(xdg_cache_home, '0install.net', 'interface_icons',
            _escape(url))
    if not path or not exists(path):
        path = gui.default_icon_path()
    return path


def _escape(uri):
    """Convert each space to %20, etc.

    Borrowed from 0install to avoid having 0install as a dependency.

    """
    return re.sub('[^-_.a-zA-Z0-9]',
            lambda match: '%%%02x' % ord(match.group(0)), uri.encode('utf-8'))
