/* channel.vala
 *
 * Copyright (C) 2010, Aleksey Lim
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Programmable data transfer method within sugar connection
 *
 * Transport mechanism via Telepathy DBus Tubes. After getting Channel object,
 * use address property as regular DBus connection address to all interactions
 * within the tube.
 */
public abstract class Sugar.Channel : Object {
    /**
     * Buddy was appeared in the channel
     *
     * @param buddy     Telepathy handle for buddy that was appeared
     * @param bus_name  buddy DBus name to identify it whithin the channel
     */
    public signal void buddy_appeared (uint buddy, string bus_name);

    /**
     * Buddy was disappeared from the channel
     *
     * @param buddy     Telepathy handle for buddy that was disappeared
     */
    public signal void buddy_disappeared (uint buddy);

    /**
     * Was "me" boddy an initiator of channel creation
     */
    public bool initiator {
        get { return _self_handle == _initiator_id; }
    }

    /**
     * A DBus name representing the channel whithin Connection
     */
    public string bus_name { get; protected set; }

    /**
     * This side buddy
     *
     * This property value is null right after channel creation and will be set
     * after getting buddy_appeared signal with appropriate buddy. To be assured
     * about property value, check it for null and either use this value or
     * connect to notify::me (or buddy_appeared) signal.
     */
    public uint me { get; private set; }

    /**
     * An intiator of channel creation
     *
     * This property value is null right after channel creation and will be set
     * after getting buddy_appeared signal with appropriate buddy. To be assured
     * about property value, check it for null and either use this value or
     * connect to notify::owner (or buddy_appeared) signal.
     */
    public uint owner { get; private set; }

    /**
     * DBus connection address associated with the channel
     *
     * All interaction will happen only on DBus connection with this address.
     * http://people.collabora.co.uk/~danni/telepathy-book/sect.tubes.dbus.html
     *
     * @return  DBus connection address
     */
    public string address { get; protected set; }

    protected void _add_buddy (uint buddy, string bus_name) {
        debug (@"Buddy $(buddy) appeared");

        if (buddy == _self_handle)
            me = buddy;
        if (buddy == _initiator_id)
            owner = buddy;

        buddy_appeared (buddy, bus_name);
    }

    protected void _remove_buddies (uint[] removed) {
        foreach (var i in removed) {
            debug (@"Buddy $(i) disappeared");
            buddy_disappeared (i);
        }
    }

    protected uint _self_handle = uint.MAX;
    protected uint _initiator_id;
}
