/* environ.vala
 *
 * Copyright (C) 2010, Aleksey Lim
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Various sugar environment settings
 */
public class Sugar.Environ {
    /**
     * Name of directory to store user's local sugar data
     *
     * This value will be used to find sugar root directory in user's HOME
     * e.g. ~/.sugar.
     */
    public const string ROOT_DIRECTORY = ".sugar";

    /**
     * Width of activity preview image to save in jobject's preview field
     */
    public const int PREVIEW_WIDTH = 300;

    /**
     * Height of activity preview image to save in jobject's preview field
     */
    public const int PREVIEW_HEIGHT = 225;

    /**
     * Is Polyol properly initialized
     *
     * To initialize Polyol environment, call init function.
     */
    public static bool initialized {
        get { return activity_id != null && bundle_id != null; }
    }

    /**
     * Current activity session unique id
     *
     * Property will be used in Polyol when there is need to identify the
     * activity instance. Before invoking Environ.init this value is null.
     */
    public static string activity_id {
        get { return _activity_id; }
    }

    /**
     * Identifier of the activity bundle
     *
     * Property will be used in Polyol when there is need to identify the
     * activity. Before invoking Environ.init this value is null.
     */
    public static string bundle_id {
        get { return _bundle_id; }
    }

    /**
     * Identity of the journal entry associated with current activity session
     *
     * If activity session was just created, object_id will be null. If session
     * was resumed from the Journal, object_id will be identifier value of the
     * resumed entry.
     *
     * Property will be used in Polyol when there is need to identify the object
     * activity is changing. Before invoking Environ.init this value is null.
     */
    public static string object_id {
        get { return _object_id; }
    }

    /**
     * Information about current activity process
     *
     * @return  valid ActivityInfo object or
     *          stub if there is no way to know about current activity
     *
     * Makes sense only being called from activity whose information
     * will be returned by this function.
     */
    public static ActivityInfo activity {
        get {
            if (_activity.bundle_id == null) {
                var path = Environment.get_variable ("SUGAR_BUNDLE_PATH");
                if (path == null || !ActivityInfo.get (path, out _activity)) {
                    _activity.bundle_id = "unknown.activity";
                    _activity.name = "Unknown Activity";
                    _activity.icon = "application-octet-stream";
                }
            }
            return _activity;
        }
    }

    /**
     * Full path to the directory to save activity specific preferences
     *
     * @return  valid path or
     *          "." if there is no way to know about current activity
     *
     * Returns a path to the location in the filesystem where the activity can
     * store activity related data that doesn't pertain to the current
     * execution of the activity and thus cannot go into the DataStore.
     *
     * Currently, this will return something like
     * ~/.sugar/default/MyActivityName/
     *
     * Activities should ONLY save settings, user preferences and other data
     * which isn't specific to a journal item here. If (meta-)data is in
     * anyway specific to a journal entry, it MUST be stored in the DataStore.
     */
    public static string? activity_root {
        get {
            unowned string path =
                    Environment.get_variable ("SUGAR_ACTIVITY_ROOT");
            return path == null ? "." : path;
        }
    }

    /**
     * Activity process main X window
     */
    public static X.ID window {
        get { return _window; }
    }

    /**
     * Is sugar security mode enabled
     *
     * See http://wiki.laptop.org/go/Rainbow for more Information.
     */
    public static bool secure_mode {
        get {
            return FileUtils.test ("/etc/olpc-security", FileTest.EXISTS);
        }
    }

    /**
     * Initialize Polyol environment
     *
     * @param argv  argv strings array that was passed to the main ();
     *              on success, parsed values will be removed from argv
     * @return      true if sugar mode was detected and properly initialized
     *
     * The rest of Polyol can be called after invoking this function and only if
     * it returned true.
     *
     * This function will Initialize internal data basing on command line
     * arguments sugar-activity command passes to the activity process.
     * These options are:
     *
     *  -b BUNDLE_ID, --bundle-id=BUNDLE_ID
     *      identifier of the activity bundle
     *      required for successful parsing
     *
     *  -a ACTIVITY_ID, --activity-id=ACTIVITY_ID
     *      identifier of the activity instance
     *      required for successful parsing
     *
     *  -o OBJECT_ID, --object-id=OBJECT_ID
     *      identifier of the associated journal entry
     *      optional, absence means activity session without or with new
     *      journal entry
     */
    [CCode (cname = "sugar_init")]
    public static bool init (ref unowned string[] argv) {
        if (argv == null)
            return false;

        var options = new OptionContext ("");

        options.set_help_enabled (false);
        options.set_ignore_unknown_options (true);
        options.add_main_entries (_entries, null);

        try {
            options.parse (ref argv);
        } catch (OptionError error) {
            warning ("Command line options error: %s", error.message);
            return false;
        }

        if (activity_id == null || bundle_id == null) {
            message ("Sugar mode was not detected, pass --activity-id and " +
                    "--bundle-id to enable it");
            return false;
        }

        debug ("Activity command line arguments: activity_id=%s object_id=%s " +
                "bundle_id=%s", activity_id, object_id, bundle_id);

        return true;
    }

    /**
     * Setup X properties for activity main window
     *
     * Valid sugar activity should call this function for main X window.
     *
     * @param display   X display to setup properties for
     * @param window    activity main window to setup properties for
     */
    public static void set_window (X.Display display, X.Window window) {
        _window = window;

        if (bundle_id != null)
            display.change_property (window,
                    display.intern_atom ("_SUGAR_BUNDLE_ID", false),
                    display.intern_atom ("STRING", false),
                    8, X.PropMode.Replace,
                    (uchar[]) bundle_id, (int) bundle_id.length);
        else
            warning ("_SUGAR_BUNDLE_ID X property was not set");

        if (activity_id != null)
            display.change_property (window,
                    display.intern_atom ("_SUGAR_ACTIVITY_ID", false),
                    display.intern_atom ("STRING", false),
                    8, X.PropMode.Replace,
                    (uchar[]) activity_id, (int) activity_id.length);
        else
            warning ("_SUGAR_ACTIVITY_ID X property was not set");
    }

    /**
     * Use only synchronous DBus calls
     *
     * Polyol is a set of high-level GObject based libraries thus they use
     * default asynchronous model which is based on GObject signals. All Polyol
     * libraries either make synchronous calls (when these operations should be
     * fast) or make asynchronous calls and emit signals. In special cases when
     * sugar activity (mostly not native one) can't create GObject main loop,
     * setting this property to true, will mean using only synchronous DBus
     * calls for various Polyol components.
     *
     * Call this function only once before using DBus related operations.
     * Default value for this property is false.
     */
    public static bool sync_dbus { get; set; default = false; }

    private Environ () {
    }

    private const OptionEntry[] _entries = {
        { "bundle-id", 'b', 0, OptionArg.STRING, ref _bundle_id,
                null, null },
        { "activity-id", 'a', 0, OptionArg.STRING, ref _activity_id,
                null, null },
        { "object-id", 'o', 0, OptionArg.STRING, ref _object_id,
                null, null },
        { null }
    };

    private static string _bundle_id;
    private static string _activity_id;
    private static string _object_id;
    private static X.ID _window;
    private static ActivityInfo _activity;
}
