/* container.vala
 *
 * Copyright (C) 2010, Aleksey Lim
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Standart Gtk.Container with layout extensions like borders per side
 *
 * Widget could not be used as is and should be inherited in sub class.
 */
public class Sugar.Container : Gtk.Container {
    public int border_left {
        get { return _border_left; }
        set { _whether_to_resize (ref _border_left, value); }
    }

    public int border_right {
        get { return _border_right; }
        set { _whether_to_resize (ref _border_right, value); }
    }

    public int border_top {
        get { return _border_top; }
        set { _whether_to_resize (ref _border_top, value); }
    }

    public int border_bottom {
        get { return _border_bottom; }
        set { _whether_to_resize (ref _border_bottom, value); }
    }

    public int border {
        set {
            if (_border_top != value || _border_bottom != value
                    || _border_left != value || _border_right != value) {
                _border_top = value;
                _border_bottom = value;
                _border_left = value;
                _border_right = value;
                queue_resize ();
            }
        }
    }

    public virtual int child_x {
        get {
            int @value = border_left + (int) border_width;
            if (is_no_window ())
                @value += allocation.x;
            return @value;
        }
    }

    public virtual int child_y {
        get {
            int @value = border_top + (int) border_width;
            if (is_no_window ())
                @value += allocation.y;
            return @value;
        }
    }

    public virtual int child_width {
        get {
            return int.max (0, allocation.width - border_left -
                    border_right - (int) border_width * 2);
        }
    }

    public virtual int child_height {
        get {
            return int.max (0, allocation.height - border_top -
                    border_bottom - (int) border_width * 2);
        }
    }

    construct {
    }

    public void clear () {
        while (get_children () != null)
            remove (get_children ().data);
    }

    public new void get_pointer (out int x, out int y) {
        base.get_pointer (out x, out y);
        x -= child_x;
        y -= child_y;
    }

    protected virtual void child_size_request (ref Gtk.Requisition req) {
        req.width += border_left + border_right + 2 * (int) border_width;
        req.height += border_top + border_bottom + 2 * (int) border_width;
    }

    private void _whether_to_resize (ref int prop, int @value) {
        if (@value != prop) {
            prop = @value;
            queue_resize ();
        }
    }

    private int _border_left = 0;
    private int _border_right = 0;
    private int _border_top = 0;
    private int _border_bottom = 0;
}

/**
 * Base class for containers that can contain several children
 */
public class Sugar.Box : Sugar.Container {
    public int children_count { get; private set; }

    public virtual signal void insert (Gtk.Widget widget, int pos) {
        return_if_fail (widget.parent == null);

        children.insert (widget, pos);
        children_count++;
        widget.parent = this;
    }

    public override void add (Gtk.Widget widget)
            requires (widget.parent == null) {
        insert (widget, -1);
    }

    public override void remove (Gtk.Widget widget)
            requires (widget.parent == this) {
        widget.unparent ();
        children.remove (widget);
        children_count--;
        widget.parent = this;
    }

    public override void forall_internal (bool include_internal,
            Gtk.Callback callback) {
        if (children != null) {
            for (unowned List<Gtk.Widget> i = children.first (); i != null;) {
                var child = i.data;
                i = i.next;
                callback (child);
            }
        }
    }

    protected List<Gtk.Widget> children = null;
}

/**
 * Analog of Gtk.Bin with layout extensions like borders per side
 *
 * In contrast with Gtk.Bin, widget could be used as is
 * without creating subclasses.
 */
public class Sugar.Bin : Sugar.Container {
    public new Gtk.Widget child {
        get {
            return _child;
        }
        set {
            bool widget_was_visible = false;

            if (child != null) {
                child.unparent ();
                widget_was_visible = true;
            }

            _child = value;

            if (child != null) {
                child.parent = this;
                widget_was_visible = true;
            }

            if (widget_was_visible)
                queue_resize ();
        }
    }

    construct {
        set_flags (Gtk.WidgetFlags.NO_WINDOW);
    }

    public override GLib.Type child_type () {
        return child == null ? typeof (Gtk.Widget) : 0;
    }

    public override void add (Gtk.Widget widget)
            requires (child != widget) {
        child = widget;
    }

    public override void remove (Gtk.Widget widget)
            requires (child == widget) {
        child = null;
    }

    public override void forall_internal (bool include_internal,
            Gtk.Callback callback) {
        if (child != null)
            callback (child);
    }

    public override void size_request (out Gtk.Requisition requisition) {
        if (child != null)
            child.size_request (out requisition);
        else {
            requisition.width = 0;
            requisition.height = 0;
        }

        child_size_request (ref requisition);
    }

    public override void size_allocate (Gdk.Rectangle allocation) {
        this.allocation = (Gtk.Allocation) allocation;

        if (child == null)
            return;

        Gdk.Rectangle child_allocation = {
            child_x, child_y, child_width, child_height
        };

        child.size_allocate (child_allocation);
    }

    private Gtk.Widget _child;
}
