/* paintbin.vala
 *
 * Copyright (C) 2010, Aleksey Lim
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;

/**
 * Bin container with rounded corners
 */
public class Sugar.PaintBin : Sugar.Bin {
    construct {
        unset_flags (Gtk.WidgetFlags.NO_WINDOW);
    }

    public bool draw_box {
        get {
            return _draw_box;
        }
        set {
            _draw_box = value;
            queue_draw ();
        }
    }

    public int radius {
        get { return _radius; }
        set { _whether_to_resize (ref _radius, value); }
    }

    public int padding_top {
        get { return _padding_top; }
        set { _whether_to_resize (ref _padding_top, value); }
    }

    public int padding_bottom {
        get { return _padding_bottom; }
        set { _whether_to_resize (ref _padding_bottom, value); }
    }

    public int padding_left {
        get { return _padding_left; }
        set { _whether_to_resize (ref _padding_left, value); }
    }

    public int padding_right {
        get { return _padding_right; }
        set { _whether_to_resize (ref _padding_right, value); }
    }

    public int padding {
        set {
            if (_padding_top != value || _padding_bottom != value ||
                    _padding_left != value || _padding_right != value) {
                _padding_top = value;
                _padding_bottom = value;
                _padding_left = value;
                _padding_right = value;
                queue_resize ();
            }
        }
    }

    public override int child_x {
        get { return base.child_x + padding_left + radius - _inscribing_gap; }
    }

    public override int child_y {
        get { return base.child_y + padding_top + radius - _inscribing_gap; }
    }

    public override int child_width {
        get {
            return int.max (0, base.child_width - padding_left -
                    padding_right - (radius - _inscribing_gap) * 2);
        }
    }

    public override int child_height {
        get {
            return int.max (0, base.child_height - padding_top -
                    padding_bottom - (radius - _inscribing_gap) * 2);
        }
    }

    public override void style_set (Gtk.Style? previous_style) {
        base.style_set (previous_style);
        _reload_corners = true;
        //queue_draw ();
    }

    public override void size_request (out Gtk.Requisition requisition) {
        base.size_request (out requisition);
        var gap = (radius - _inscribing_gap) * 2;
        requisition.width += padding_left + padding_right + gap;
        requisition.height += padding_top + padding_bottom + gap;
    }

    public override void size_allocate (Gdk.Rectangle allocation) {
        this.allocation = (Gtk.Allocation) allocation;

        if (is_realized ())
            window.move_resize (allocation.x, allocation.y,
                    allocation.width, allocation.height);

        if (child != null) {
            Gdk.Rectangle child_allocation = {
                child_x, child_y, child_width, child_height
            };
            child.size_allocate (child_allocation);
        }

        _reload_corners = true;
        queue_draw ();
    }

    public override void realize () {
        set_flags (Gtk.WidgetFlags.REALIZED);

        window = new Gdk.Window (
                get_parent_window (),
                Gdk.WindowAttr () {
                    window_type = Gdk.WindowType.CHILD,
                    x = allocation.x,
                    y = allocation.y,
                    width = allocation.width,
                    height = allocation.height,
                    wclass = Gdk.WindowClass.INPUT_OUTPUT,
                    colormap = get_colormap (),
                    event_mask = get_events () | Gdk.EventMask.EXPOSURE_MASK
                },
                Gdk.WindowAttributesType.X | Gdk.WindowAttributesType.Y |
                Gdk.WindowAttributesType.COLORMAP);
        window.set_user_data (this);

        style.attach (window);
        style.set_background (window, Gtk.StateType.NORMAL);

        if (child != null)
            child.set_parent_window (window);
    }

    public override void map () {
        set_flags (Gtk.WidgetFlags.MAPPED);

        if (child != null && child.visible)
            child.map ();

        window.show ();
    }

    public override bool expose_event (Gdk.EventExpose event) {
        var fg = style.bg_gc[Gtk.StateType.NORMAL];
        var bg = style.bg_gc[Gtk.StateType.INSENSITIVE];

        window.draw_rectangle (bg, true, 0, 0,
                allocation.width, allocation.height);

        if (!draw_box)
            return base.expose_event (event);

        int left = base.child_x;
        int top = base.child_y;
        int right = left + base.child_width - radius;
        int bottom = top + base.child_height - radius;

        if (radius <= 0) {
            window.draw_rectangle (fg, true,
                    left, top, base.child_width, base.child_height);
        } else {
            if (_reload_corners) {
                if (radius > 0) {
                    var key = new _CornerId (this);
                    _corners = _corners_cache[key];
                    if (_corners == null) {
                        _corners = new _Corners (this);
                        _corners_cache[key] = _corners;
                    }
                }
                _reload_corners = false;
            }
            // draw rectangles
            window.draw_rectangle (fg, true,
                    left + radius, top, base.child_width - radius * 2, radius);
            window.draw_rectangle (fg, true,
                    left, top + radius,
                    base.child_width, base.child_height - radius * 2);
            window.draw_rectangle (fg, true,
                    left + radius, bottom,
                    base.child_width - radius * 2, radius);

            // draw corners
            var cr = Gdk.cairo_create (window);
            cr.set_source_surface (_corners.right_bottom, right, bottom);
            cr.paint ();
            cr.set_source_surface (_corners.left_bottom, left, bottom);
            cr.paint ();
            cr.set_source_surface (_corners.left_top, left, top);
            cr.paint ();
            cr.set_source_surface (_corners.right_top, right, top);
            cr.paint ();
        }

        return base.expose_event (event);
    }

    private int _inscribing_gap {
        get { return (int) Math.sqrt (Math.pow (radius, 2) / 2.0); }
    }

    private void _whether_to_resize (ref int padding, int @value) {
        if (padding != @value) {
            padding = @value;
            queue_resize ();
        }
    }

    private class _CornerId {
        public uint color;
        public int radius;

        public _CornerId (PaintBin bin) {
            var gdk_color = bin.style.bg[Gtk.StateType.NORMAL];
            color = (uint) (gdk_color.red & 0xFF) |
                    (uint) ((gdk_color.green & 0xFF) << 8) |
                    (uint) ((gdk_color.blue & 0xFF) << 16);
            radius = bin.radius;
        }

        public static uint hash (_CornerId x) {
            return x.color + x.radius;
        }

        public static bool cmp (_CornerId x, _CornerId y) {
            return x.color == y.color && x.radius == y.radius;
        }
    }

    private class _Corners {
        public Cairo.ImageSurface right_bottom;
        public Cairo.ImageSurface left_bottom;
        public Cairo.ImageSurface left_top;
        public Cairo.ImageSurface right_top;

        public int radius;
        public Gdk.Color color;

        public _Corners (PaintBin bin) {
            radius = bin.radius;
            color = bin.style.bg[Gtk.StateType.NORMAL];

            right_bottom = _setup_serface (0, 0, 0.0, Math.PI / 2.0);
            left_bottom = _setup_serface (radius, 0, Math.PI / 2.0, Math.PI);
            left_top = _setup_serface (radius, radius, Math.PI, Math.PI * 1.5);
            right_top = _setup_serface (0, radius, Math.PI * 1.5,
                    Math.PI * 2.0);
        }

        private Cairo.ImageSurface _setup_serface
                (int x, int y, double angle_start, double angle_stop) {
            var surface = new Cairo.ImageSurface (Cairo.Format.ARGB32,
                    radius, radius);
            var cr = new Cairo.Context (surface);
            Gdk.cairo_set_source_color (cr, color);
            cr.move_to (x, y);
            cr.arc (x, y, radius, angle_start, angle_stop);
            cr.fill ();
            return surface;
        }
    }

    private static LRU<_CornerId, _Corners> _corners_cache =
            new LRU<_CornerId, _Corners> (50, (HashFunc) _CornerId.hash,
            (EqualFunc) _CornerId.cmp);

    private bool _draw_box = true;
    private _Corners _corners;
    private int _radius = Metrics.get (Metrics.DEFAULT_PADDING);
    private int _padding_top = 0;
    private int _padding_bottom = 0;
    private int _padding_left = 0;
    private int _padding_right = 0;
    private bool _reload_corners = true;
}
