/* paletteinvoker.vala
 *
 * Copyright (C) 2010, Aleksey Lim
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * Port from original sugar-toolkit project.
 * File:   src/sugar/graphics/palettewindow.py
 * Commit: c684950ecff34e910e6da5840737a469f95a0e79
 *
 * Copyright (C) 2007, Eduardo Silva <edsiper@gmail.com>
 * Copyright (C) 2008, One Laptop Per Child
 * Copyright (C) 2009, Tomeu Vizoso
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

public class Sugar.Invoker : Object {
    public signal void palette_request ();
    public signal void poped_up ();
    public signal void poped_down ();

    public Connector connector {
        get {
            return _connector;
        }
        construct set {
            if (connector != null) {
                connector.invoker_enter.disconnect (_invoker_enter_cb);
                connector.invoker_leave.disconnect (_invoker_leave_cb);
                connector.popup.disconnect (_force_popup_cb);
                connector.popdown.disconnect (_force_popdown_cb);
            }

            _connector = value;

            if (connector != null) {
                connector.invoker_enter.connect (_invoker_enter_cb);
                connector.invoker_leave.connect (_invoker_leave_cb);
                connector.popup.connect (_force_popup_cb);
                connector.popdown.connect (_force_popdown_cb);
            }
        }
    }

    public string group_id {
        get {
            return _group_id;
        }
        construct set {
            if (group_id != null) {
                var group = PaletteGroups.get (group_id);
                group.remove (this);
            }

            _group_id = value;

            if (group_id != null) {
                var group = PaletteGroups.get (group_id);
                group.add (this);
            }
        }
    }

    public PaletteWindow? palette_window {
        get {
            return _palette;
        }
        set {
            if (palette_window != null) {
                _popdown (true);

                palette_window.show.disconnect (_palette_show_cb);
                palette_window.hide.disconnect (_palette_hide_cb);
                palette_window.enter_notify_event.disconnect (
                        _palette_enter_notify_event_cb);
                palette_window.leave_notify_event.disconnect (
                        _palette_leave_notify_event_cb);

                palette_window.connector = null;
            }

            _palette = value;

            if (palette_window != null) {
                palette_window.connector = connector;

                palette_window.show.connect (_palette_show_cb);
                palette_window.hide.connect (_palette_hide_cb);
                palette_window.enter_notify_event.connect (
                        _palette_enter_notify_event_cb);
                palette_window.leave_notify_event.connect (
                        _palette_leave_notify_event_cb);

                if (is_up)
                    _reveal_palette ();
            }
        }
    }

    public Palette? palette {
        get { return palette_window as Palette; }
        set { palette_window = value; }
    }

    public bool is_up { get; private set; default = false; }

    /**
     * Is cursor under invoker's widget or palette_request
     */
    public bool focused { get; private set; default = false; }

    public Invoker (Connector connector, string group_id = "default") {
        Object (connector: connector, group_id: group_id);
    }

    construct {
        _popup_anim = new Animator (0.5, 10);
        _popup_anim.completed.connect (_popup_completed_cb);

        _popdown_anim = new Animator (0.6, 10);
        _popdown_anim.completed.connect (_popdown_completed_cb);

        _secondary_anim = new Animator (1.5, 10);
        _secondary_anim.completed.connect (_secondary_completed_cb);

        _mouse_detector = new MouseSpeedDetector (200, 5);
        _mouse_detector.motion_slow.connect (_motion_slow_cb);
    }

    public override void dispose () {
        group_id = null;
        base.dispose ();
    }

    public void popup () {
        _popup (true, true);
    }

    public void popdown () {
        _popdown (true);
    }

    private void _popup (bool immediate, bool force_secondary) {
        debug (@"Invoker._popup immediate=$(immediate) " +
                @"force_secondary=$(force_secondary) " +
                @"popup_anim=$(_popup_anim.active) " +
                @"popdown_anim=$(_popdown_anim.active) " +
                @"secondary_anim=$(_secondary_anim.active)" +
                @"secondary_visible=$(_secondary_visible)");

        _popdown_anim.abort ();
        _secondary_visible |= force_secondary;

        if (immediate) {
            is_up = true;
            _popup_anim.abort ();
            _reveal_palette ();
        } else {
            _popup_anim.start ();
        }
    }

    private void _popdown (bool immediate) {
        debug (@"Invoker._popdown immediate=$(immediate) " +
                @"popup_anim=$(_popup_anim.active) " +
                @"popdown_anim=$(_popdown_anim.active) " +
                @"secondary_anim=$(_secondary_anim.active)");

        _popup_anim.abort ();
        _secondary_anim.abort ();
        _mouse_detector.stop ();
        _secondary_visible = false;

        if (immediate || !is_up) {
            is_up = false;
            _popdown_anim.abort ();
            if (palette_window != null) {
                palette_window.hide ();
                // to suppress glitches while later re-opening
                palette_window.secondary_visible = false;
            }
        } else {
            _popdown_anim.start ();
        }
    }

    private void _reveal_palette () {
        if (palette_window == null) {
            palette_request ();
            return;
        }

        if (palette_window.secondary != null) {
            palette_window.secondary_visible = _secondary_visible;
            if (!_secondary_visible)
                _secondary_anim.start ();
        }

        palette_window.set_transient_for (connector.get_toplevel ());
        palette_window.reveal ();
    }

    private void _on_enter () {
        focused = true;

        var parent = PaletteGroups.get (group_id).get_parent (this);
        if (parent != null)
            parent._popdown_anim.abort ();

        _popdown_anim.abort ();
    }

    private void _on_leave () {
        focused = false;
        _popdown (false);
    }

    private void _invoker_enter_cb () {
        _on_enter ();
        _mouse_detector.start ();
    }

    private void _invoker_leave_cb () {
        _on_leave ();
        _mouse_detector.stop ();
    }

    private void _force_popup_cb () {
        _popup (true, true);
    }

    private void _force_popdown_cb () {
        _popdown (true);
    }

    private void _palette_show_cb () {
        connector.on_popup ();
        poped_up ();
    }

    private void _palette_hide_cb () {
        _popdown (true);
        connector.on_popdown ();
        poped_down ();
    }

    private bool _palette_enter_notify_event_cb (Gdk.EventCrossing event) {
        if (event.detail != Gdk.NotifyType.INFERIOR &&
                event.mode == Gdk.CrossingMode.NORMAL)
            _on_enter ();
        return false;
    }

    private bool _palette_leave_notify_event_cb (Gdk.EventCrossing event) {
        if (event.detail != Gdk.NotifyType.INFERIOR &&
                event.mode == Gdk.CrossingMode.NORMAL)
            _on_leave ();
        return false;
    }

    private void _popup_completed_cb (bool aborted) {
        if (!aborted)
            _popup (true, false);
    }

    private void _popdown_completed_cb (bool aborted) {
        if (!aborted)
            _popdown (true);
    }

    private void _motion_slow_cb () {
        _mouse_detector.stop ();

        if (is_up)
            _popdown_anim.abort ();
        else {
            bool immediate = false;
            if (group_id != null)
                immediate = PaletteGroups.get (group_id).is_up;
            _popup (immediate, false);
        }
    }

    private void _secondary_completed_cb (bool aborted) {
        if (!aborted)
            _popup (true, true);
    }

    private string _group_id;
    private Connector _connector;
    private PaletteWindow _palette;
    private Animator _popup_anim;
    private Animator _popdown_anim;
    private Animator _secondary_anim;
    private MouseSpeedDetector _mouse_detector;
    private bool _secondary_visible;
}
