/* activity_info.vala
 *
 * Copyright (C) 2010, Aleksey Lim
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Useful information about activity source code
 *
 * See also http://wiki.sugarlabs.org/go/Development_Team/Almanac/
 * Activity_Bundles#.info_file_format specification.
 */
public struct Sugar.ActivityInfo {
    /**
     * Unique activity identifier
     */
    public string bundle_id;

    /**
     * Full path to a file with activity icon
     */
    public string icon;

    /**
     * Name of activity to display
     */
    public string name;

    /**
     * Get activity info by path to activity sources
     *
     * @param path  full path to activity sources root directory
     * @param info  variable to store result
     *
     * @return      true on success when info contains valid object
     */
    public static bool get (string path, out ActivityInfo result) {
        var activity_info = Path.build_filename (
                path, "activity", "activity.info");

        if (!FileUtils.test (activity_info, FileTest.EXISTS)) {
            warning ("Cannot find activity.info file in %s", path);
            return false;
        }

        ActivityInfo info = { };
        var ini = new KeyFile ();

        try {
            ini.load_from_file (activity_info, KeyFileFlags.NONE);

            if (ini.has_key (_ACTIVITY_GROUP, "bundle_id"))
                info.bundle_id = ini.get_value (_ACTIVITY_GROUP, "bundle_id");
            else if (ini.has_key (_ACTIVITY_GROUP, "service_name")) {
                debug ("Deprecated service_name key in %s, use bundle_id",
                        activity_info);
                info.bundle_id = ini.get_value (_ACTIVITY_GROUP,
                        "service_name");
            } else {
                warning ("Malformed %s: lack of bundle_id key", activity_info);
                return false;
            }

            if (ini.has_key (_ACTIVITY_GROUP, "name"))
                info.name = ini.get_value (_ACTIVITY_GROUP, "name");
            else {
                warning ("Malformed %s: lack of name key", activity_info);
                return false;
            }

            if (ini.has_key (_ACTIVITY_GROUP, "icon"))
                info.icon = Path.build_filename (path, "activity",
                        ini.get_value (_ACTIVITY_GROUP, "icon") + ".svg");
            else {
                warning ("Malformed %s: lack of icon key", activity_info);
                return false;
            }
        } catch (Error error) {
            warning ("Cannot parse %s: %s", activity_info, error.message);
            return false;
        }

        result = info;
        return true;
    }

    private const string _ACTIVITY_GROUP = "Activity";
}
