/* embeddedmenu.vala
 *
 * Copyright (C) 2010, Aleksey Lim
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Menu that could be embeded to a widget
 */
public class Sugar.EmbeddedMenu : Sugar.Bin {
    /*
     * @param palette_connector Connector which is used for palette
     *                          EmbeddedMenu is embeded in
     */
    public EmbeddedMenu (Sugar.Connector? palette_connector) {
        _menu = new _EmbeddedMenu (palette_connector);
        _menu.embed (this);
        _menu.show ();
    }

    public void insert (Gtk.Widget item, int pos = -1) {
        _menu.insert (item, pos);
    }

    private _EmbeddedMenu _menu;
}

/**
 * Sub-menu class to use with EmbeddedMenu
 *
 * If EmbeddedMenu is embeded in Palette, for all sub-menus, use this class.
 * EmbeddedSubMenu is inteded to fix mouse enter/leave related issues.
 */
public class Sugar.EmbeddedSubMenu : Gtk.Menu {
    /*
     * @param palette_connector Connector which is used for palette
     *                          EmbeddedMenu/EmbeddedSubMenu is embeded in
     */
    public EmbeddedSubMenu (Sugar.Connector? palette_connector) {
        _connector = palette_connector;
    }

    public override bool enter_notify_event (Gdk.EventCrossing event) {
        if (event.detail != Gdk.NotifyType.INFERIOR &&
                event.mode == Gdk.CrossingMode.NORMAL) {
            int x, y;
            toplevel = get_toplevel ();
            toplevel.get_pointer (out x, out y);

            if (x < toplevel.allocation.x + toplevel.allocation.width &&
                    y < toplevel.allocation.y + toplevel.allocation.height &&
                    x >= toplevel.allocation.x && y >= toplevel.allocation.y)
                _connector.invoker_enter ();
        }
        return base.enter_notify_event (event);
    }

    public override bool leave_notify_event (Gdk.EventCrossing event) {
        if (event.detail != Gdk.NotifyType.INFERIOR &&
                event.mode == Gdk.CrossingMode.NORMAL)
            _connector.invoker_leave ();
        return base.leave_notify_event (event);
    }

    private Sugar.Connector _connector;
}

public class Sugar._EmbeddedMenu : Gtk.Menu {
    public _EmbeddedMenu (Sugar.Connector? palette_connector) {
        _connector = palette_connector;
    }

    public void embed (Gtk.Container container) {
        toplevel = container.get_toplevel ();
        reparent (container);
    }

    public override void deactivate () {
        // Ignore default deactivating handler to prevent popping down menu
        if (_connector != null)
            _connector.popdown ();
    }

    public override void show () {
        (this as Gtk.MenuShell).active = 1;
        base.show ();
    }

    public override void hide () {
        (this as Gtk.MenuShell).active = 0;
        base.hide ();
    }

    public override bool expose_event (Gdk.EventExpose event) {
        // Ignore the Menu expose to prevent any border from being drawn here.
        // A border is drawn by the palette object around everything.
        foreach (var child in get_children ())
            propagate_expose (child, event);
        return false;
    }

    public override void grab_notify (bool was_grabbed) {
        // Ignore grab_notify as the menu would close otherwise
    }

    private Sugar.Connector _connector;
}
