/* alignment.vala
 *
 * Copyright (C) 2010, Aleksey Lim
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

public class Sugar.Alignment : Sugar.Bin {
    /**
     * The horizontal alignment
     *
     * Value from 0 (left) to 1 (right), reversed for RTL layouts.
     * Makes sense only if xscale < 1.0.
     */
    public float xalign {
        get { return _xalign; }
        set { _whether_to_resize (ref _xalign, value); }
    }

    /**
     * The vertical alignment
     *
     * Value from 0 (left) to 1 (right), reversed for RTL layouts.
     * Makes sense only if yscale < 1.0.
     */
    public float yalign {
        get { return _yalign; }
        set { _whether_to_resize (ref _yalign, value); }
    }

    /**
     * Scale child horizontally
     *
     * If available horizontal space is bigger than needed for the child,
     * how much of it to use for the child.
     *
     *  0.0     use exactly what child requested
     *  1.0     use entirely Bin space for child needs
     */
    public float xscale {
        get { return _xscale; }
        set { _whether_to_resize (ref _xscale, value); }
    }

    /**
     * Scale child vertically
     *
     * If available vertical space is bigger than needed for the child,
     * how much of it to use for the child.
     *
     *  0.0     use exactly what child requested
     *  1.0     use entirely Bin space for child needs
     */
    public float yscale {
        get { return _yscale; }
        set { _whether_to_resize (ref _yscale, value); }
    }

    /**
     * Ignore width requested by child
     *
     * Alignment will request 0 width in spite of child needs.
     * Thus child could get less size then it requested, otherwise regular
     * scheme will be preswerved.
     */
    public bool xignore {
        get {
            return _xignore;
        }
        set {
            if (value == xignore)
                return;
            _xignore = value;
            queue_resize ();
        }
    }

    /**
     * Ignore height requested by child
     *
     * Alignment will request 0 width in spite of child needs.
     * Thus child could get less size then it requested, otherwise regular
     * scheme will be preswerved.
     */
    public bool yignore {
        get {
            return _yignore;
        }
        set {
            if (value == yignore)
                return;
            _yignore = value;
            queue_resize ();
        }
    }

    public override int child_x {
        get { return base.child_x + _xoffset; }
    }

    public override int child_y {
        get { return base.child_y + _yoffset; }
    }

    public override int child_width {
        get { return _child_width; }
    }

    public override int child_height {
        get { return _child_height; }
    }

    public override void size_request (out Gtk.Requisition requisition) {
        base.size_request (out requisition);
        _child_requisition = requisition;

        if (xignore || yignore) {
            Gtk.Requisition container_requisition = { };
            child_size_request (ref container_requisition);

            if (xignore)
                requisition.width = container_requisition.width;
            if (yignore)
                requisition.height = container_requisition.height;
        }
    }

    public override void size_allocate (Gdk.Rectangle allocation) {
        this.allocation = (Gtk.Allocation) allocation;

        if (child == null)
            return;

        if (base.child_width <= _child_requisition.width || xscale == 1.0f) {
            _child_width = base.child_width;
            _xoffset = 0;
        } else {
            if (xscale == 0.0f)
                _child_width = _child_requisition.width;
            else
                _child_width = _child_requisition.width +
                        (int) ((base.child_width - _child_requisition.width) *
                        (1.0f - xscale));
            if (get_direction () == Gtk.TextDirection.RTL)
                _xoffset = (int) ((base.child_width - _child_width) *
                        (1.0f - xalign));
            else
                _xoffset = (int) ((base.child_width - _child_width) * xalign);
        }

        if (base.child_height <= _child_requisition.height || yscale == 1.0f) {
            _child_height = base.child_height;
            _yoffset = 0;
        } else {
            if (yscale == 0.0f)
                _child_height = _child_requisition.height;
            else
                _child_height = _child_requisition.height +
                        (int) ((base.child_height - _child_requisition.height) *
                        (1.0f - yscale));
            _yoffset = (int) ((base.child_height - _child_height) * yalign);
        }

        Gdk.Rectangle child_allocation = {
            child_x, child_y, child_width, child_height
        };

        child.size_allocate (child_allocation);
    }

    private void _whether_to_resize (ref float prop, float @value) {
        if (@value != prop) {
            prop = @value;
            queue_resize ();
        }
    }

    private float _xalign = 0.5f;
    private float _yalign = 0.5f;
    private float _xscale = 1.0f;
    private float _yscale = 1.0f;
    private bool _xignore = true;
    private bool _yignore = true;
    private int _xoffset = 0;
    private int _yoffset = 0;
    private int _child_width = 0;
    private int _child_height = 0;
    private Gtk.Requisition _child_requisition;
}
