/* homogenetable.vala
 *
 * Copyright (C) 2010, Aleksey Lim
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

using Gee;

/**
 * Grid widget with homogeneously placed children of the same class.
 *
 * Grid has fixed number of columns that are visible all time and unlimited
 * rows number (or vice versa if orientation is horizontal). There are frame
 * cells - visible at particular moment - frame cells and virtual (widget is
 * model less itself and only ask callback object about right cell's value)
 * ones - just cells. User can scroll up/down (or left/right if orientation is
 * horizontal) grid to see all virtual cells and the same frame cell will
 * represent content of various virtual cells (widget will call fill_cell to
 * refill frame cell content) in different time moments.
 *
 * There is no way to add cells manually, widget will create them on demand
 * by invoking passed new_cell. By default, widget doesn't have any cells,
 * to make it useful, assign proper value to either frame_width/frame_height or
 * cell_width/cell_height properties. Also set cell_count to set number
 * of virtual cells.
 *
 * Before using widget, inherit it and overload methods:
 *
 *  new_cell
 *  fill_cell
 *  highlight_cell
 *
 * NOTE Since widget is not cell renderers based and uses regular widgets
 * to represent cells, haivng many visible at the same time cells will affect
 * performance, especially with heavy gtk themes.
 *
 * See examples/homogenetable.py for exmaples.
 */
public abstract class Sugar.HomogeneTable : Sugar.Container {
    /**
     * Scroll event used internally by widgets like GtkScrolledWindow
     *
     * XXX Since there is no regualr way to set set_scroll_adjustments_signal
     *     Makefile.am seds resulted homogenetable.c file
     */
    public virtual signal void signal_set_scroll_adjustments (
            Gtk.Adjustment? hadjustment, Gtk.Adjustment? vadjustment) {
        foreach (var i in _adjustments)
            if (i != null)
                i.value_changed.disconnect (_adjustment_value_changed_cb);

        if (vadjustment == null || hadjustment == null) {
            _adjustments = { null, null };
            return;
        }

        _adjustments[_rotate (0, 1)] = vadjustment;
        vadjustment.value_changed.connect (_adjustment_value_changed_cb);
        _adjustments[_rotate (1, 0)] = hadjustment;
        hadjustment.value_changed.connect (_adjustment_value_changed_cb);

        _setup_adjustment (true);
    }

    /**
     * Current cell was changed
     */
    public signal void cursor_changed ();

    /**
     * Visible cells frame was changed
     */
    public signal void frame_scrolled ();

    /**
     * Number of virtual cells
     *
     * Defines maximal number of virtual rows, the minimal has being described
     * by frame_size/cell_size values.
     */
    public int cell_count {
        get { return _cell_count; }
        set {
            if (_cell_count != value) {
                _cell_count = value;
                _setup_adjustment (false);
                _resize_table ();
            }
        }
    }

    /**
     * Persistent number of frame columns
     *
     * Cells will be resized while resizing widget to keep
     * frame column count the same.
     * Mutually exclusive to cell_width.
     */
    public int frame_width {
        get { return _frame_width; }
        set { _set_metric (ref _frame_width, ref _cell_width, value); }
    }

    /**
     * Persistent number of frame rows
     *
     * Cells will be resized while resizing widget to keep
     * frame row count the same.
     * Mutually exclusive to cell_height.
     */
    public int frame_height {
        get { return _frame_height; }
        set { _set_metric (ref _frame_height, ref _cell_height, value); }
    }

    /**
     * Persistent cell width
     *
     * Number of cells will be changed while resizing widget to keep
     * cells width the same.
     * Mutually exclusive to frame_width.
     */
    public int cell_width {
        get { return _cell_width; }
        set { _set_metric (ref _cell_width, ref _frame_width, value); }
    }

    /**
     * Persistent cell height
     *
     * Number of cells will be changed while resizing widget to keep
     * cells height the same.
     * Mutually exclusive to frame_height.
     */
    public int cell_height {
        get { return _cell_height; }
        set { _set_metric (ref _cell_height, ref _frame_height, value); }
    }

    /**
     * Table orientation
     *
     * If orientation is vertical (by default), table will have persistent
     * number of columns and variable row count and vice versa if orientation
     * is horizontal.
     */
    public Gtk.Orientation orientation {
        get { return _orientation; }
        set {
            if (value != _orientation) {
                _orientation = value;
                var tmp = _adjustments[0];
                _adjustments[0] = _adjustments[1];
                _adjustments[1] = tmp;
                foreach (var i in _adjustments)
                    if (i != null) {
                        i.lower = 0;
                        i.upper = 0;
                    }
                _resize_table ();
            }
        }
    }

    /**
     * Highlight cell which is under mouse pointer
     *
     * While moving mouse pointer, highlight_cell function will be invoked.
     */
    public bool hover_selection {
        get { return _hover_selection; }
        set {
            if (value != _hover_selection) {
                if (value)
                    add_events (Gdk.EventMask.POINTER_MOTION_HINT_MASK |
                            Gdk.EventMask.POINTER_MOTION_MASK);
                _hover_selection = value;
                var cell = _get_cell (cursor);
                if (cell != null)
                    highlight_cell (cell.widget, value);
            }
        }
    }

    /**
     * Absolute index of the focused cell
     */
    public int cursor {
        get { return _cursor_index; }
        set {
            var cell_index = int.min (int.max (0, value), cell_count - 1);
            if (cell_index != cursor) {
                scroll_to_cell (cell_index);
                _set_cursor (cell_index);
            }
        }
    }

    /**
     * Is cell focused
     */
    public bool focus_cell {
        get {
            if (cursor < 0 || has_focus)
                return false;

            var cell = _get_cell (cursor);

            if (cell != null) {
                var focus = cell.widget;
                while (focus != null && focus.parent != null) {
                    if (focus == this)
                        return true;
                    focus = focus.parent;
                }
            }

            return false;
        }
        set {
            if (value == focus_cell)
                return;
            if (value) {
                if (!has_focus)
                    grab_focus ();
                var cell = _get_cell (cursor);
                if (cell != null)
                    cell.widget.child_focus (Gtk.DirectionType.TAB_FORWARD);
            } else
                grab_focus ();
        }
    }

    /**
     * Range of visible cells
     */
    public Range frame_range {
        get { return _frame_range; }
        set {
            if (value.is_subset (frame_range))
                /* pass */ ;
            else if (value.size <= frame_range.size) {
                scroll_to_cell (value.last);
                scroll_to_cell (value.start);
            } else if (frame_range.is_empty || frame_range.start < value.start)
                scroll_to_cell (value.start + frame_range.size - 1);
            else if (frame_range.last > value.last)
                scroll_to_cell (value.last - frame_range.size + 1);
        }
    }

    /**
     * Number of created cells
     *
     * This is not about exact number of frame cells visible right now,
     * just the maximal number of cells that might be in the frame.
     */
    public int frame_size {
        get { return _cell_cache.size; }
    }

    /**
     * Where there are no cells
     */
    public bool is_empty {
        get { return _row_cache.size == 0 || _cell_length == 0; }
    }

    construct {
        // when focused cell is out of visible frame,
        // table itslef will be focused
        can_focus = true;
    }

    public override void dispose () {
        _abandon_cells ();
        _cell_cache.clear ();
        signal_set_scroll_adjustments (null, null);
    }

    /**
     * Create new cell
     *
     * @return newly created cell widget
     */
    protected abstract Gtk.Widget new_cell ();

    /**
     * Fill cell content in
     *
     * @param cell  widget to fill in
     * @param index cell index to fill in
     */
    protected abstract void fill_cell (Gtk.Widget cell, int index);

    /**
     * Highlight cell
     *
     * Will be invoked if hover_selection is true
     *
     * @param cell      widget to highlight
     * @param selected  set/unset highlighting
     */
    protected abstract void highlight_cell (Gtk.Widget cell, bool selected);

    /**
     * Get cell widget by index
     *
     * @return widget for visible cell, null otherwise
     */
    public unowned Gtk.Widget? get_cell (int cell_index) {
        var cell = _get_cell (cell_index);
        if (cell == null)
            return null;
        else
            return cell.widget;
    }

    /**
     * Get cell index at postion
     *
     * @param x relative to HomogeneTable widget, x position
     * @param y relative to HomogeneTable widget, y position
     * @return  cell index, -1 otherwise
     */
    public int get_index_at_pos (int x, int y) {
        if (is_empty)
            return -1;
        x = int.min (int.max (0, x), child_width - 1);
        y = int.min (int.max (0, y), child_height - 1);
        return _get_index_at_pos (_rotate (x, y), _rotate (y, x) + _pos);
    }

    /**
     * Scroll to position where cell is visible
     */
    public void scroll_to_cell (int cell_index, bool fully_visible = false) {
        if (is_empty || cell_index == cursor)
            return;

        focus_cell = false;

        var row = cell_index / _column_count;
        var pos = row * _cell_length;

        if (fully_visible && pos <= _pos ||
                !fully_visible && pos + _cell_length <= _pos)
            _pos = pos;
        else {
            pos -= _frame_length;
            if (fully_visible && pos + _cell_length >= _pos ||
                    !fully_visible && pos >= _pos)
                _pos = pos + _cell_length;
        }
    }

    /**
     * Force to run refill cells
     *
     * @param cells range of cells to refill, if empty then all cells
     */
    public void refill () {
        foreach (var cell in _cell_cache)
            cell.index = -1;
        _allocate_rows (false);
    }

    /* Gtk.Widget overrides */

    public override bool scroll_event (Gdk.EventScroll event) {
        if (_adjustments[0] == null ||
                orientation != Gtk.Orientation.HORIZONTAL)
            return false;
        var @value = 0;
        if (event.direction == Gdk.ScrollDirection.UP)
            @value = int.max (0, (int) _adjustments[0].@value - _cell_length);
        else if (event.direction == Gdk.ScrollDirection.DOWN)
            @value = int.min (_max_pos, (int) _adjustments[0].@value +
                    _cell_length);
        else
            return false;
        _adjustments[0].@value = @value;
        return true;
    }

    public override void realize () {
        set_flags (Gtk.WidgetFlags.REALIZED);

        window = new Gdk.Window (
                get_parent_window (),
                Gdk.WindowAttr () {
                    window_type = Gdk.WindowType.CHILD,
                    x = allocation.x,
                    y = allocation.y,
                    width = allocation.width,
                    height = allocation.height,
                    wclass = Gdk.WindowClass.INPUT_OUTPUT,
                    colormap = get_colormap (),
                    event_mask = Gdk.EventMask.VISIBILITY_NOTIFY_MASK
                },
                Gdk.WindowAttributesType.X | Gdk.WindowAttributesType.Y |
                Gdk.WindowAttributesType.COLORMAP);
        window.set_user_data (this);

        _bin_window = new Gdk.Window (
                window,
                Gdk.WindowAttr () {
                    window_type = Gdk.WindowType.CHILD,
                    x = _rotate (child_x, -_pos),
                    y = _rotate (-_pos, child_y),
                    width = _rotate (_thickness, _length),
                    height = _rotate (_length, _thickness),
                    colormap = get_colormap (),
                    wclass = Gdk.WindowClass.INPUT_OUTPUT,
                    event_mask = (get_events () | Gdk.EventMask.EXPOSURE_MASK |
                            Gdk.EventMask.SCROLL_MASK)
                },
                Gdk.WindowAttributesType.X | Gdk.WindowAttributesType.Y |
                Gdk.WindowAttributesType.COLORMAP);
        _bin_window->set_user_data (this);

        style.attach (window);
        style.set_background (window, Gtk.StateType.NORMAL);
        style.set_background (_bin_window, Gtk.StateType.NORMAL);

        foreach (var row in _row_cache)
            foreach (var cell in row)
                cell.widget.set_parent_window (_bin_window);

        if (_pending_allocate != -1) {
            _allocate_rows ((bool) _pending_allocate);
            _pending_allocate = -1;
        }
    }

    public override void size_allocate (Gdk.Rectangle allocation) {
        if (this.allocation.x == allocation.x &&
                this.allocation.y == allocation.y &&
                this.allocation.width == allocation.width &&
                this.allocation.height == allocation.height) {
            _reallocate_cells ();
            return;
        }

        this.allocation = (Gtk.Allocation) allocation;
        _resize_table ();
        if (is_realized ())
            window.move_resize (allocation.x, allocation.y, allocation.width,
                    allocation.height);
    }

    public override void unrealize () {
        _bin_window->set_user_data (null);
        _bin_window->destroy ();
        _bin_window = null;
        base.unrealize ();
    }

    public override void style_set (Gtk.Style? previous_style) {
        base.style_set (previous_style);
        if (is_realized ())
            style.set_background (_bin_window, Gtk.StateType.NORMAL);
    }

    public override bool expose_event (Gdk.EventExpose event) {
        if (event.window != _bin_window)
            return false;
        base.expose_event (event);
        return false;
    }

    public override void map () {
        set_flags (Gtk.WidgetFlags.MAPPED);

        foreach (var row in _row_cache)
            foreach (var cell in row)
                if (cell.widget.visible)
                    cell.widget.map ();

        _bin_window->show ();
        window.show ();
    }

    public override void size_request (out Gtk.Requisition requisition) {
        requisition.width = 0;
        requisition.height = 0;

        foreach (var row in _row_cache)
            foreach (var cell in row) {
                Gtk.Requisition child_requisition;
                cell.widget.size_request (out child_requisition);
            }

        child_size_request (ref requisition);
    }

    public override bool motion_notify_event (Gdk.EventMotion event) {
        if (hover_selection) {
            int x, y;
            get_pointer (out x, out y);
            cursor = get_index_at_pos (x, y);
        }
        return false;
    }

    public override bool key_press_event (Gdk.EventKey event) {
        if (!is_empty && cursor >= 0)
            if (event.keyval == 0xff1b /* XK_Escape */  && focus_cell)
                focus_cell = false;

        return false;
    }

    /* Gtk.Container overrides */

    public override void forall_internal (bool include_internal,
            Gtk.Callback callback) {
        foreach (var row in _row_cache)
            foreach (var cell in row)
                callback (cell.widget);
    }

    public override void add (Gtk.Widget widget) {
        /* container is not intended to add children manually */
        assert (false);
    }

    public override void remove (Gtk.Widget widget) {
        /* container is not intended to remove children manually */
    }

    public override void set_focus_child (Gtk.Widget? widget) {
        if (widget == null)
            return;
        var x = widget.allocation.x;
        var y = widget.allocation.y;
        var new_cursor = _get_index_at_pos (_rotate (x, y), _rotate (y, x));
        if (cursor < 0 || !frame_range.contains (new_cursor))
            cursor = new_cursor;
    }

    public override bool focus (Gtk.DirectionType direction) {
        grab_focus ();
        return true;
    }

    /* Private members */

    private int _column_count {
        get {
            if (_row_cache.size > 0)
                return _row_cache[0].size;
            else
                return 0;
        }
    }

    private int _frame_row_count {
        get { return _row_cache.size - _SPARE_ROWS_COUNT; }
    }

    private int _pos {
        get {
            if (_adjustments[0] == null || _adjustments[0].@value.is_nan ())
                return 0;
            else
                return int.max (0, (int) _adjustments[0].@value);
        }
        set {
            if (_adjustments[0] != null)
                _adjustments[0].@value = value;
        }
    }

    private int _max_pos {
        get {
            if (_adjustments[0] == null)
                return 0;
            else
                return int.max (0, _length - _frame_length);
        }
    }

    private int _thickness {
        get { return _rotate (child_width, child_height); }
    }

    private int _frame_length {
        get { return _rotate (child_height, child_width); }
    }

    private int _length {
        get {
            if (_adjustments[0] == null)
                return _frame_length;
            else
                return (int) _adjustments[0].upper;
        }
    }

    private int _rotate (int x, int y) {
        if (orientation == Gtk.Orientation.HORIZONTAL)
            return y;
        else
            return x;
    }

    private void _set_metric (ref int prop, ref int mutex_prop, int @value) {
        if (@value != prop) {
            mutex_prop = 0;
            prop = @value;
            _resize_table ();
        }
    }

    private int _get_index_at_pos (int x, int y) {
        var cell_row = y / _cell_length;
        var cell_column = x / (_thickness / _column_count);
        var cell_index = cell_row * _column_count + cell_column;
        return int.min (cell_index, cell_count - 1);
    }

    private _Cell? _get_cell (int cell_index) {
        if (cell_index < 0)
            return null;
        var column = cell_index % _column_count;
        var base_index = cell_index - column;
        foreach (var row in _row_cache)
            if (row[0].valid && row[0].index == base_index)
                return row[column];
        return null;
    }

    private int _get_row_pos (_Cells row) {
        Gtk.Allocation allocation = row[0].widget.allocation;
        return _rotate (allocation.y, allocation.x);
    }

    private void _set_cursor (int cell_index) {
        if (hover_selection) {
            var cell = _get_cell (cursor);
            if (cell != null)
                highlight_cell (cell.widget, false);
        }

        _cursor_index = cell_index;

        if (hover_selection) {
            var cell = _get_cell (cursor);
            if (cell != null)
                highlight_cell (cell.widget, true);
        }

        cursor_changed ();
    }

    private void _abandon_cells () {
        foreach (var row in _row_cache)
            foreach (var cell in row)
                cell.widget.unparent ();
        _cell_cache_pos = 0;
        _row_cache = new _Rows ();
    }

    private _Cell _pop_a_cell () {
        _Cell cell;

        if (_cell_cache_pos < _cell_cache.size) {
            cell = _cell_cache[_cell_cache_pos];
            _cell_cache_pos += 1;
        } else {
            cell = new _Cell ();
            cell.widget = new_cell ();
            assert (cell.widget != null);
            _cell_cache.add (cell);
            _cell_cache_pos = _cell_cache.size;
        }

        cell.invalidate_pos ();
        return cell;
    }

    private void _setup_adjustment (bool dry_run) {
        var adj = _adjustments[0];
        if (adj == null)
            return;

        adj.lower = 0;
        if (_column_count == 0)
            adj.upper = 0;
        else
            adj.upper = (int) Math.ceil ((float) cell_count / _column_count) *
                    _cell_length;
        adj.page_size = _frame_length;
        adj.changed ();

        if (_pos > _max_pos) {
            _pos = _max_pos;
            if (!dry_run)
                adj.value_changed ();
        }
    }

    private void _resize_table () {
        if (allocation.x + allocation.width <= 0 ||
                allocation.y + allocation.height <= 0)
            return;

        var row_count = frame_height;
        var column_count = frame_width;

        if (row_count == 0) {
            if (_cell_height == 0)
                return;
            row_count = (int) Math.ceil (_frame_length / (float) _cell_height);
            row_count = int.max (1, row_count);
        }

        if (column_count == 0) {
            if (_cell_width == 0)
                return;
            column_count = int.max (1, _thickness / _cell_width);
        }

        if (column_count != _column_count || row_count != _frame_row_count) {
            _abandon_cells ();
            for (int i = row_count + _SPARE_ROWS_COUNT; i > 0; --i) {
                var row = new _Cells ();
                for (int j = column_count; j > 0; --j) {
                    var cell = _pop_a_cell ();
                    if (is_realized ())
                        cell.widget.set_parent_window (_bin_window);
                    cell.widget.set_parent (this);
                    row.add (cell);
                }
                _row_cache.add (row);
            }
        } else {
            foreach (var row in _row_cache) {
                foreach (var cell in row) {
                    cell.invalidate_pos ();
                    cell.index = -1;
                }
            }
        }

        if (frame_height == 0)
            _cell_length = _cell_height;
        else
            _cell_length = _frame_length / _frame_row_count;

        _setup_adjustment (true);

        if (is_realized ())
            _bin_window->resize (_rotate (_thickness, _length),
                    _rotate (_length, _thickness));

        _allocate_rows (true);
    }

    private void _allocate_rows (bool force) {
        if (!is_realized ()) {
            _pending_allocate = (int) (_pending_allocate > 0 || force);
            return;
        }

        if (is_empty || _pos < 0 || _pos > _max_pos)
            return;

        var old_frame_range = frame_range;
        _frame_range = Range ();

        var spare_rows = new Array<unowned _Cells>.sized (
                false, true, (uint) sizeof (void *), _row_cache.size);
        var visible_rows = new Array<unowned _Cells>.sized (
                false, true, (uint) sizeof (void *), _row_cache.size);

        if (force)
            foreach (var row in _row_cache)
                spare_rows.append_val (row);
        else {
            foreach (var row in _row_cache) {
                var row_pos = _get_row_pos (row);
                if (row_pos < 0 || row_pos > _pos + _frame_length ||
                        (row_pos + _cell_length) < _pos)
                    spare_rows.append_val (row);
                else
                    visible_rows.append_val (row);
            }

            CompareFunc cmp_rows;
            if (orientation == Gtk.Orientation.HORIZONTAL) {
                cmp_rows = (a, b) => {
                    unowned _Cells a_row = *(_Cells **) a;
                    unowned _Cells b_row = *(_Cells **) b;
                    return a_row[0].widget.allocation.x -
                            b_row[0].widget.allocation.x;
                };
            } else {
                cmp_rows = (a, b) => {
                    unowned _Cells a_row = *(_Cells **) a;
                    unowned _Cells b_row = *(_Cells **) b;
                    return a_row[0].widget.allocation.y -
                            b_row[0].widget.allocation.y;
                };
            }
            visible_rows.sort (cmp_rows);
        }

        if (visible_rows.length > 0 || spare_rows.length > 0) {
            var spare_rows_sp = 0;
            var next_row_pos = _pos - _pos % _cell_length;

            _InsertSpareRow insert_spare_row = (pos_begin, pos_end) => {
                while (pos_begin < pos_end) {
                    if (spare_rows_sp >= spare_rows.length) {
                        warning ("spare_rows should not be empty.");
                        return;
                    }
                    unowned _Cells row = spare_rows.index (spare_rows_sp);
                    ++spare_rows_sp;
                    _allocate_cells (row, pos_begin);
                    pos_begin += _cell_length;
                }
            };

            // visible_rows could not be continuous
            // lets try to add spare rows to missed points
            for (int i = 0; i < visible_rows.length; ++i) {
                unowned _Cells row = visible_rows.index (i);
                var row_pos = _get_row_pos (row);
                insert_spare_row (next_row_pos, row_pos);
                _allocate_cells (row, row_pos);
                next_row_pos = row_pos + _cell_length;
            }

            insert_spare_row (next_row_pos, _pos + _frame_length);
        }

        var bin_x = child_x;
        var bin_y = child_y - _pos;
        _bin_window->move (_rotate (bin_x, bin_y), _rotate (bin_y, bin_x));
        _bin_window->process_updates (true);

        if (!frame_range.is_equal (old_frame_range))
            frame_scrolled ();

        if (focus_cell && !frame_range.contains (cursor))
            focus_cell = false;
    }

    private void _allocate_cells (_Cells row, int cell_y) {
        int cell_x = 0;
        var cell_row = cell_y / _cell_length;
        var cell_index = cell_row * _column_count;

        for (int cell_column = 0; cell_column < row.size; cell_column++) {
            var cell = row[cell_column];

            if (cell.index != cell_index) {
                if (cell_index < cell_count) {
                    fill_cell (cell.widget, cell_index);
                    if (hover_selection)
                        highlight_cell (cell.widget, cell_index == cursor);

                    Gtk.Requisition child_requisition;
                    cell.widget.size_request (out child_requisition);

                    cell.widget.show ();
                } else
                    cell.widget.hide ();
                cell.index = cell_index;
            }

            var cell_thickness = _thickness / _column_count;

            Gdk.Rectangle child_allocation = {
                _rotate (cell_x, cell_y),
                _rotate (cell_y, cell_x),
                _rotate (cell_thickness, _cell_length),
                _rotate (_cell_length, cell_thickness)
            };
            cell.widget.size_allocate (child_allocation);

            cell_x += cell_thickness;
            cell_index++;
        }

        if (_frame_range.start < 0)
            _frame_range.start = row[0].index;
        _frame_range.last = row.last ().index;
    }

    private void _reallocate_cells () {
        foreach (var cell in _cell_cache) {
            if (!cell.valid)
                continue;

            var child_allocation = cell.widget.allocation;
            Gtk.Requisition child_requisition;
            cell.widget.size_request (out child_requisition);
            cell.widget.size_allocate ((Gdk.Rectangle) child_allocation);
        }
    }

    private void _adjustment_value_changed_cb () {
        _allocate_rows (false);
        if (hover_selection) {
            int x, y;
            get_pointer (out x, out y);

            if (x >= child_x && y >= child_y &&
                    x < child_x + child_width && y < child_y + child_height)
                cursor = get_index_at_pos (x, y);
        }
    }

    private class _Cell {
        public Gtk.Widget widget = null;
        public int index = -1;

        public void invalidate_pos () {
            Gdk.Rectangle null_allocation = { -1, -1, 0, 0 };
            widget.size_allocate (null_allocation);
        }

        public bool valid {
            get {
                return index >= 0 && widget != null &&
                        widget.allocation.x >= 0 && widget.allocation.y >= 0;
            }
        }
    }

    /* Having spare rows let us making smooth scrolling w/o empty spaces */
    private const int _SPARE_ROWS_COUNT = 2;

    private delegate void _InsertSpareRow (int pos_begin, int pos_end);

    private class _Cells : ArrayList<_Cell> {
    }

    private class _Rows : ArrayList<_Cells> {
    }

    private _Cells _cell_cache = new _Cells ();
    private _Rows _row_cache = new _Rows ();
    private int _cell_cache_pos = 0;
    private Gtk.Adjustment?[] _adjustments = { null, null };
    private Gdk.Window * _bin_window = null;
    private int _cell_count = 0;
    private int _cell_length = 0;
    private int _frame_width = 0;
    private int _frame_height = 0;
    private int _cell_width = 0;
    private int _cell_height = 0;
    private int _cursor_index = -1;
    private int _pending_allocate = -1;
    private Range _frame_range = Range ();
    private Gtk.Orientation _orientation = Gtk.Orientation.VERTICAL;
    private bool _hover_selection = false;
}
