/* palettegroup.vala
 *
 * Copyright (C) 2010, Aleksey Lim
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * Port from original sugar-toolkit project.
 * File:   src/sugar/graphics/palettegroup.py
 * Commit: 2e3061dc1007cd1f11a69174ba868b4726ef1c1c
 *
 * Copyright (C) 2007, Red Hat, Inc.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

using Gee;

namespace Sugar.PaletteGroups {
    public PaletteGroup get (string group_id) {
        if (_groups == null)
            _groups = new HashMap<string, PaletteGroup> ();

        if (_groups.has_key (group_id))
            return _groups[group_id];
        else {
            var group = new PaletteGroup ();
            group.id = group_id;
            _groups[group_id] = group;
            return group;
        }
    }

    public void popdown_all () {
        if (_groups == null)
            return;

        foreach (var group in _groups.values)
            group.popdown ();
    }

    private HashMap<string, PaletteGroup> _groups;
}

public class Sugar.PaletteGroup : Object {
    public signal void poped_up (PaletteGroup group);
    public signal void poped_down (PaletteGroup group);

    public string id { get; set; }
    public bool is_up { get; private set; default = false; }

    public void add (Invoker invoker) {
        if (_invokers.contains (invoker))
            return;
        invoker.poped_up.connect (_palette_poped_up_cb);
        invoker.poped_down.connect (_palette_poped_down_cb);
        _invokers.add (invoker);
    }

    public void remove (Invoker invoker) {
        if (!_invokers.contains (invoker))
            return;
        invoker.poped_up.disconnect (_palette_poped_up_cb);
        invoker.poped_down.disconnect (_palette_poped_down_cb);
        _invokers.remove (invoker);
    }

    public void popdown () {
        foreach (var invoker in _invokers)
            if (invoker.is_up)
                invoker.popdown ();
    }

    public Invoker? get_parent (Invoker child) {
        foreach (var i in _invokers)
            if (_has_relationship (i, child))
                return i;
        return null;
    }

    private bool _has_relationship (Invoker parent, Invoker child) {
        return parent.palette_window != null &&
                parent.palette_window.get_toplevel () ==
                child.connector.get_toplevel ();
    }

    private void _palette_poped_up_cb (Invoker invoker) {
        foreach (var i in _invokers)
            if (i != invoker &&
                    // do not popdown palette
                    // passed invoker's widget is a child of
                    !_has_relationship (i, invoker))
                i.popdown ();

        if (!is_up) {
            poped_up (this);
            is_up = true;
        }
    }

    private void _palette_poped_down_cb (Invoker invoker) {
        bool down = true;

        foreach (var i in _invokers) {
            if (i == invoker || !i.is_up)
                continue;
            if (_has_relationship (i, invoker) && !i.focused) {
                i.popdown ();
                // i.popdown will activate new iteration
                return;
            } else if (_has_relationship (invoker, i)) {
                i.popdown ();
                // i.popdown will activate new iteration
                return;
            } else {
                down = false;
                break;
            }
        }

        if (down) {
            is_up = false;
            poped_down (this);
        }
    }

    private HashSet<Invoker> _invokers = new HashSet<Invoker> ();
}
