/* journal.vala
 *
 * Copyright (C) 2010, Aleksey Lim
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Hight-level interface to interact with sugar Journal process
 */
public class Sugar.Journal : Object {
    /**
     * Object chooser dialog was closed
     *
     * To open chooser dialog, call {@link choose_object}.
     *
     * @param object_id unique id of selected object (@see Jobject.uid)
     *                  or null if object was not selected
     */
    public signal void chooser_response (string? object_id);

    construct {
        try {
            _journal = new JournalService ();
            _journal.object.ObjectChooserResponse.connect (
                    _ObjectChooserResponse_cb);
            _journal.object.ObjectChooserCancelled.connect (
                    _ObjectChooserCancelled_cb);
        } catch (Error error) {
            warning ("Cannot connect to Journal: %s", error.message);
        }
    }

    /**
     * Choose object from Journal
     *
     * @param what  type of objects to choose, could be null to see all objects;
     *              for now, only MIME_* constants make sense
     */
    public void choose_object (string? what) {
        if (_journal == null) {
            _chooser_response (null);
            return;
        }

        try {
            try {
                _chooser_id = _journal.object.ChooseObject (
                        (int) Environ.window, what ?? "");
            } catch (Error error) {
                if (error is DBus.Error.REMOTE_EXCEPTION) {
                    debug ("Use sugar-0.82 ChooseObject signature");
                    _chooser_id = _journal.object.ChooseObject (
                            (int) Environ.window);
                } else {
                    throw error;
                }
            }
        } catch (Error error) {
            warning ("Cannot ChooseObject from journal: %s", error.message);
            _chooser_response (null);
        }
    }

    /**
     * Pop-up Journal and show object with object_id
     *
     * @param object_id journal entry to show
     */
    public void show_object (string object_id) {
        if (_journal == null)
            return;

        try {
            _journal.object.ShowObject (object_id);
        } catch (Error error) {
            warning ("Cannot ShowObject %s in journal: %s",
                    object_id, error.message);
        }
    }

    private void _chooser_response (string? object_id) {
        _chooser_id = null;

        /* Let dbus server close chooser window before processing signal */
        Idle.add (() => {
            chooser_response (object_id);
            return false;
        });
    }

    private void _ObjectChooserResponse_cb (DBus.Object sender,
            string chooser_id, string object_id) {
        if (chooser_id == _chooser_id)
            _chooser_response (object_id);
    }

    private void _ObjectChooserCancelled_cb (DBus.Object sender,
            string chooser_id) {
        if (chooser_id == _chooser_id)
            _chooser_response (null);
    }

    private JournalService _journal;
    private string _chooser_id;
}
