/* alert.vala
 *
 * Copyright (C) 2010, Aleksey Lim
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Container for various user level messages and short iteractions
 *
 * Alerts are used inside the activity window instead of being a
 * separate popup window. They do not hide canvas content. You can use
 * push and pop functions inside your activity to add and remove the
 * alsert. The position of the alert is below the toolbox or top in
 * fullscreen mode.
 *
 * Alerts could be regular widgets that implement Alert interface, but
 * predefined classes like MessageAlert, ConfirmationAlert etc are
 * especially useful. Only one alert will be visible at the same time. If
 * there are other alerts, they will be queued.
 *
 * AlertBin is designed to be used directly from various parts of code since its
 * methods are static, just create one AlertBin instance to shoow all alerts in
 * activity.
 */
public class Sugar.AlertBin : Gtk.EventBox {
    construct {
        modify_bg (Gtk.StateType.NORMAL, color_type_to_rgb (Color.BLACK));
        _queue = new Queue<Alert> ();
        _singleton = this;
    }

    ~AlertBin () {
        _singleton = null;
    }

    /**
     * Push new alert to the queue
     *
     * @param widget alert widget, it will be owned by AlertBin right after
     *               this call (regardless if alsert queue is empty or not)
     *               and could be unreffed on caller side
     */
    public static void push (Alert alert) {
        if (_singleton == null)
            warning ("AlertBin was not created");
        else {
            bool is_new = _singleton._queue.is_empty ();
            _singleton._queue.push_tail (alert);
            if (is_new)
                _singleton._next ();
        }
    }

    /**
     * Remove alert from the queue
     *
     * There is no need to call pop method for every alert, after disappearing,
     * alert widget will be unreffed and removed from the queue automatically.
     *
     * @param widget alert widget to remove
     */
    public static void pop (Alert alert) {
        if (_singleton == null)
            warning ("AlertBin was not created");
        else {
            _singleton._queue.remove (alert);
            if (alert == _singleton._queue.peek_head ())
                _singleton._next ();
        }
    }

    private void _next () {
        if (child != null) {
            remove (child);
            var alert = _queue.pop_head ();
            alert.response.disconnect (_response_cb);
        }

        if (_queue.is_empty ())
            hide ();
        else {
            var alert = _queue.peek_head ();
            alert.response.connect (_response_cb);
            alert.popup ();
            alert.show ();
            add (alert);
            show ();
        }
    }

    private void _response_cb () {
        _next ();
    }

    private Queue<Alert> _queue;

    private static unowned AlertBin _singleton;
}

/**
 * Interface for alerts that should be placed to AlertBin
 */
public interface Sugar.Alert : Gtk.Widget {
    /**
     * Alert was made visible in AlertBin
     */
    public signal void popup ();

    /**
     * Alert button was clicked
     *
     * The Alert class will pass "response" events to your activity when any
     * of its buttons are clicked.
     *
     * @param response_id   button id to help identify what button was clicked
     */
    public signal void response (int response_id);
}

/**
 * Basic notification alert
 *
 * At a high level, MessageAlert and its different variations (NotifyAlert,
 * ConfirmationAlert, etc.) have a title, an alert message and then several
 * buttons that the user can click (response signal will be emited).
 *
 * To make alerts visible in activity, use AlertBin.
 */
public class Sugar.MessageAlert : Gtk.HBox, Sugar.Alert {
    /**
     * Create basic message alert
     *
     * @param title_text    title of the alert, will be shown in bold
     * @param message_text  message of the alert
     * @param icon_name     icon name of full path to icon file that appears at
     *                      the far left; if null then do not use icon
     */
    public MessageAlert (string title_text, string message_text,
            string? icon_name) {
        spacing = Metrics.get (Metrics.DEFAULT_SPACING);
        border_width = Metrics.get (Metrics.DEFAULT_SPACING);

        if (icon_name != null) {
            var icon = new Icon ();
            icon.file = icon_name;
            icon.pixel_size = Metrics.get (Metrics.STANDARD_ICON_SIZE);
            pack_start (icon, false, true, 0);
        }

        var message_box = new Gtk.VBox (false, 0);
        pack_start (message_box, false, true, 0);

        var title = new Gtk.Label (title_text);
        title.attributes = new Pango.AttrList ();
        title.attributes.insert (Pango.attr_weight_new (Pango.Weight.BOLD));
        title.set_alignment (0.0f, 0.5f);
        message_box.pack_start (title, false, true, 0);

        var message = new Gtk.Label (message_text);
        message.set_alignment (0.0f, 0.5f);
        message_box.pack_start (message, false, true, 0);

        _buttons_box = new Gtk.HButtonBox ();
        _buttons_box.layout_style = Gtk.ButtonBoxStyle.END;
        _buttons_box.spacing = Metrics.get (Metrics.DEFAULT_SPACING);
        pack_end (_buttons_box, true, true, 0);

        show_all ();

        popup.connect (_popup_cb);
        response.connect (_response_cb);
    }

    /**
     * Add button to alert
     *
     * @param response_id   will be emitted with the response signal
     * @param label         that will occure right to the buttom
     * @param icon_name     optional icon name or full path to icon image
     * @param timeout       optional timeout to auto send response event
     */
    public void add_button (int response_id, string label,
            string? icon_name = null, int timeout = 0) {
        var button = new Gtk.Button ();
        button.label = label;
        button.show ();
        _buttons_box.pack_start (button, false, true, 0);

        if (timeout > 0) {
            _count_down = new _CountDownLabel (timeout, this);
            button.set_image (_count_down);
        } else if (icon_name != null) {
            var icon = new Icon ();
            icon.icon_name = icon_name;
            button.set_image (icon);
        }

        button.clicked.connect (() => {
            response (response_id);
        });
    }

    private void _popup_cb () {
        if (_count_down != null)
            _count_down.start ();
    }

    private void _response_cb () {
        if (_count_down != null) {
            _count_down.stop ();
            _count_down = null;
        }
    }

    private class _CountDownLabel : Gtk.Label {
        public _CountDownLabel (int count, Alert alert) {
            _count = count;
            _alert = alert;

            modify_fg (Gtk.StateType.NORMAL,
                    color_type_to_rgb (Color.BUTTON_GREY));
            label = _count.to_string ();
            width_chars = 1;
        }

        public void start () {
            if (_sid != 0)
                return;

            _sid = Timeout.add_seconds (1, () => {
                _count -= 1;
                label = _count.to_string ();
                if (_count > 0)
                    return true;
                _alert.response (Gtk.ResponseType.OK);
                return false;
            });
        }

        public void stop () {
            if (_sid != 0) {
                Source.remove (_sid);
                _sid = 0;
            }
        }

        public override bool expose_event (Gdk.EventExpose event) {
            int x = allocation.x + allocation.width / 2;
            int y = parent.allocation.y +
                    (int) Math.ceil (allocation.height / 2.0);
            int d = (int) (allocation.height / 5.0 * 3.0);

            var cr = Gdk.cairo_create (window);
            Gdk.cairo_set_source_color (cr, color_type_to_rgb (Color.WHITE));
            cr.move_to (allocation.x, allocation.y);
            cr.arc (x, y, d, 0, Math.PI * 2);
            cr.fill ();

            return base.expose_event (event);
        }

        private uint _sid;
        private int _count;
        private Alert _alert;
    }

    private Gtk.HButtonBox _buttons_box;
    private _CountDownLabel _count_down;
}

/**
 * A ready-made two button (Cancel, Ok) alert
 *
 * A confirmation alert is a nice shortcut from a standard Alert because it
 * comes with 'OK' and 'Cancel' buttons already built-in. When clicked, the 'OK'
 * button will emit a response with a response_id of Gtk.ResponseType.OK, while
 * the 'Cancel' button will emit Gtk.ResponseType.CANCEL.
 */
public class Sugar.ConfirmationAlert : Sugar.MessageAlert {
    /**
     * {@inheritDoc}
     */
    public ConfirmationAlert (string title_text, string message_text,
            string? icon_name) {
        base (title_text, message_text, icon_name);

        add_button (Gtk.ResponseType.CANCEL, _("Cancel"), "dialog-cancel");
        add_button (Gtk.ResponseType.OK, _("Ok"), "dialog-ok");
    }
}

/**
 * A ready-made two button (Cancel, Continue) alert
 *
 * A confirmation alert is a nice shortcut from a standard Alert because it
 * comes with 'Cancel' and 'Continue' buttons already built-in. When clicked,
 * the 'Cancel' button will emit a response with a response_id of
 * Gtk.ResponseType.Cancel, while the 'Continue' button will emit
 * Gtk.ResponseType.OK.
 *
 * It times out with a positive response after the given amount of seconds.
 */
public class Sugar.TimeoutAlert : Sugar.MessageAlert {
    /**
     * Create an alert
     *
     * @param title_text    title of the alert, will be shown in bold
     * @param message_text  message of the alert
     * @param icon_name     icon name of full path to icon file that appears at
     *                      the far left; if null then do not use icon
     * @param timeout       timeout in seconds to auto close alert
     */
    public TimeoutAlert (string title_text, string message_text,
            string? icon_name, int timeout = 7) {
        base (title_text, message_text, icon_name);

        add_button (Gtk.ResponseType.CANCEL, _("Cancel"), "dialog-cancel");
        add_button (Gtk.ResponseType.OK, _("Continue"), null, timeout);
    }
}

/**
 * Timeout alert with only an "OK" button - just for notifications
 */
public class Sugar.NotifyAlert : Sugar.MessageAlert {
    /**
     * Create an alert
     *
     * @param title_text    title of the alert, will be shown in bold
     * @param message_text  message of the alert
     * @param icon_name     icon name of full path to icon file that appears at
     *                      the far left; if null then do not use icon
     * @param timeout       timeout in seconds to auto close alert
     */
    public NotifyAlert (string title_text, string message_text,
            string? icon_name, int timeout = 7) {
        base (title_text, message_text, icon_name);

        add_button (Gtk.ResponseType.OK, _("Ok"), null, timeout);
    }
}
