#!/usr/bin/env python

# pep8: ignore=E501
# pylint: disable-msg=C0301

import sys
from os.path import abspath, join, dirname

import gtk
import gobject

from __init__ import Test, main
from libjournal import datasource


class DataSource(datasource.DataSource):

    def __init__(self, page_size, total, cache_pages):
        datasource.DataSource.__init__(self, page_size, cache_pages)
        self.size = total
        self.entries = [{'i': str(i)} for i in range(total)]
        self.offsets = []
        self.connect('entry', self.entry_cb)
        self.on_changed()

    def do_search(self, offset, limit):
        self.on_entries(offset, self.size,
                self.entries[offset:offset + limit])

    def entry_cb(self, sender, uid, offset, entry):
        self.offsets.append(offset)
        if len(self.offsets) == self.total:
            gobject.timeout_add(500, lambda: gtk.main_quit())


class DataSourceTest(Test):

    def test_FetchAll(self):
        ds = DataSource(3, 9, 1)

        gobject.idle_add(lambda: ds.request(0, ds.size))
        gtk.main()
        self.assertEqual(ds.size, ds.total)
        self.assertEqual(range(ds.size), ds.offsets)

    def test_FetchForwardByPage(self):
        ds = DataSource(3, 9, 1)

        def cb(offset):
            ds.request(offset, 3)
            if offset + 3 < ds.size:
                gobject.idle_add(cb, offset + 3)

        gobject.idle_add(cb, 0)

        gtk.main()
        self.assertEqual(ds.size, ds.total)
        self.assertEqual(range(ds.size), ds.offsets)

    def test_FetchForward(self):
        ds = DataSource(3, 9, 1)

        def cb(offset):
            ds.request(offset, 1)
            if offset + 1 < ds.size:
                gobject.idle_add(cb, offset + 1)

        gobject.idle_add(cb, 0)

        gtk.main()
        self.assertEqual(ds.size, ds.total)
        self.assertEqual(
                [0, 1, 2, 1, 2, 3, 4, 5, 4, 5, 6, 7, 8, 7, 8],
                ds.offsets)

    def test_FetchBackwardByPage(self):
        ds = DataSource(3, 9, 1)

        def cb(offset):
            ds.request(offset, 3)
            if offset - 3 >= 0:
                gobject.idle_add(cb, offset - 3)

        gobject.idle_add(cb, ds.size - 3)

        gtk.main()
        self.assertEqual(ds.size, ds.total)
        self.assertEqual(
                [6, 7, 8, 3, 4, 5, 0, 1, 2],
                ds.offsets)

    def test_FetchBackward(self):
        ds = DataSource(3, 9, 1)

        def cb(offset):
            ds.request(offset, 1)
            if offset - 1 >= 0:
                gobject.idle_add(cb, offset - 1)

        gobject.idle_add(cb, ds.size - 1)

        gtk.main()
        self.assertEqual(ds.size, ds.total)
        self.assertEqual(
                [6, 7, 8, 7, 6, 3, 4, 5, 4, 3, 0, 1, 2, 1, 0],
                ds.offsets)


if __name__ == '__main__':
    main()
