/* toolbarbox.vala
 *
 * Copyright (C) 2010, Aleksey Lim
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Tool button that could have sub-toolbar
 *
 * Such tool button could be placed only to ToolbarBox.toolbar widget.
 * Sub-toolbar widget should be assigned to page property before using button.
 *
 * Sub-toolbar could be opened in two ways:
 *
 * - Palette mode (popped property is true), activated by regular sugar
 *   workflow (right click on button or after delay) or by setting popped
 *   property directly. Sub-toolbar will appear in poped up palette.
 *
 * - Imbeded mode (expanded property is true), activated by left click or
 *   setting expanded property directly. Sub-toolbar will appear as a sub-widget
 *   in ToolbarBox ToolbarButton is a child of.
 */
public class Sugar.ToolbarButton : Sugar.ToolButton {
    /**
     * Sub-toolbar widget
     *
     * There are not restictions to use only Gtk.ToolBar, page could be
     * any widget.
     */
    public Gtk.Widget page {
        get {
            return _page;
        }
        set {
            if (page == value)
                return;

            if (page != null) {
                popped = false;
                expanded = false;
                if (page.parent != null)
                    page.parent.remove (page);
            }

            _page = value;

            if (page != null) {
                page.show ();
            }
        }
    }

    /**
     * Page is shown and in palette mode
     */
    public bool popped {
        get {
            return invoker.is_up;
        }
        set {
            if (page == null)
                return;

            if (value) {
                expanded = false;
                invoker.popup ();
            } else {
                invoker.popdown ();
                invoker.palette_window = null;
            }
        }
    }

    /**
     * Page is shown and in imbeded mode
     */
    public bool expanded {
        get {
            return page != null && page.get_toplevel ().is_toplevel () &&
                    page.get_toplevel () != _palette;
        }
        set {
            if (page == null || expanded == value)
                return;

            if (value) {
                popped = false;
                if (page.parent != null)
                    page.parent.remove (page);
                if (_toolbox != null)
                    _toolbox.expand_button (this);
            } else {
                if (_toolbox != null)
                    _toolbox.shrink_button (this);
            }
        }
    }

    construct {
        _palette = new _ButtonPalette ();
        invoker.connector = new _Connector (this);
        invoker.palette_request.connect (_palette_request_cb);

        modify_bg (Gtk.StateType.PRELIGHT,
                color_type_to_rgb (Color.BUTTON_GREY));
    }

    public override void clicked () {
        expanded = !expanded;
    }

    public override void parent_set (Gtk.Widget? previous_parent) {
        base.parent_set (previous_parent);

        if (_toolbox != null)
            (invoker.connector as _Connector).orientation =
                    _toolbox.orientation;

        expanded = false;
        popped = false;
    }

    public override bool expose_event (Gdk.EventExpose event) {
        var arrow_type = Gtk.ArrowType.DOWN;

        if (expanded) {
            foreach (var child in get_children ())
                propagate_expose (child, event);
            Gtk.paint_hline (style, event.window,
                    Gtk.StateType.PRELIGHT, event.area, this, null,
                    allocation.x, allocation.x + allocation.width,
                    allocation.y);
            Gtk.paint_vline (style, event.window,
                    Gtk.StateType.PRELIGHT, event.area, this, null,
                    allocation.y, allocation.y + allocation.height,
                    allocation.x);
            Gtk.paint_vline (style, event.window,
                    Gtk.StateType.PRELIGHT, event.area, this, null,
                    allocation.y, allocation.y + allocation.height,
                    allocation.x + allocation.width -
                    Metrics.get (Metrics.FOCUS_LINE_WIDTH));
            arrow_type = Gtk.ArrowType.UP;
        } else {
            base.expose_event (event);
        }

        var x = allocation.x + allocation.width / 2 -
                Metrics.get (Metrics.TOOLBAR_ARROW_SIZE) / 2;
        var y = allocation.y + allocation.height -
                (int) (Metrics.get (Metrics.TOOLBAR_ARROW_SIZE) * 0.85);

        Gtk.paint_arrow (style, event.window,
                Gtk.StateType.NORMAL, Gtk.ShadowType.NONE, event.area,
                this, null, arrow_type, true,
                x, y, Metrics.get (Metrics.TOOLBAR_ARROW_SIZE),
                Metrics.get (Metrics.TOOLBAR_ARROW_SIZE));

        return false;
    }

    private ToolbarBox? _toolbox {
        get {
            if (parent is _ToolbarExpander)
                return (parent as _ToolbarExpander).toolbox;
            else
                return null;
        }
    }

    private void _palette_request_cb () {
        if (!expanded) {
            if (page.parent != null)
                page.parent.remove (page);
            _prepare_page (page, Color.BLACK);
            _palette.bin.add (page);
            invoker.palette_window = _palette;
        }
    }

    private _ButtonPalette _palette;
    private Gtk.Widget _page;
}

/**
 * Container to place ToolbarButton widgets
 *
 * Do not place widgets directly to this container, use toolbar property
 * instead. Toolbar property is a regualr Gtk.Toolbar and can contain not only
 * ToolbarButton widgets.
 *
 * Also see ToolbarButton for more information.
 */
public class Sugar.ToolbarBox : Gtk.VBox {
    /**
     * Gtk.Toolbar to place children to
     */
    public _ToolbarExpander toolbar { get; private set; }

    /**
     * Toolbar orientation
     *
     * This property defines not only orientation of toolbar itself but also
     * how ToolbarButton's pages should appear in imbeded mode.
     */
    public Gtk.PositionType orientation {
        get {
            return _orientation;
        }
        set {
            _orientation = value;

            if (orientation == Gtk.PositionType.TOP ||
                    orientation == Gtk.PositionType.BOTTOM)
                toolbar.orientation = Gtk.Orientation.HORIZONTAL;
            else
                toolbar.orientation = Gtk.Orientation.VERTICAL;
        }
    }

    /**
     * Padding size at the beginning and at the end of toolbar
     *
     * By default it is equal to padding of activity main toolbar.
     */
    public int padding {
        get {
            return _toolbar_bin.padding_left;
        }
        set {
            _toolbar_bin.padding_left = value;
            _toolbar_bin.padding_right = value;
        }
    }

    construct {
        _expanded_button = -1;

        _toolbar_bin = new PaintBin ();
        _toolbar_bin.radius = 0;
        _toolbar_bin.modify_bg (Gtk.StateType.NORMAL,
                color_type_to_rgb (Color.TOOLBAR_GREY));
        _toolbar_bin.show ();

        pack_start (_toolbar_bin, true, true, 0);

        toolbar = new _ToolbarExpander ();
        toolbar.show ();
        _toolbar_bin.add (toolbar);

        _expanded_bin = new _ExpandedBin ();
        _expanded_bin.show ();

        orientation = Gtk.PositionType.TOP;
        padding = Metrics.get (Metrics.TOOLBOX_HORIZONTAL_PADDING);
    }

    /**
     * Used only by ToolbarButton internally
     */
    public void expand_button (ToolbarButton button) {
        var prev = toolbar.get_nth_item (_expanded_button) as ToolbarButton;
        if (prev != null)
            prev.expanded = false;
        _expanded_button = toolbar.get_item_index (button);

        _prepare_page (button.page, Color.TOOLBAR_GREY);
        _expanded_bin.gap_pos = button.allocation.x;
        _expanded_bin.gap_size = button.allocation.width;
        _expanded_bin.add (button.page);
        pack_start (_expanded_bin, true, true, 0);
    }

    /**
     * Used only by ToolbarButton internally
     */
    public void shrink_button (ToolbarButton button) {
        remove (_expanded_bin);
        _expanded_bin.remove (button.page);
        _expanded_button = -1;
        // need to redraw it to erase arrow
        button.queue_draw ();
    }

    private PaintBin _toolbar_bin;
    private Gtk.PositionType _orientation;
    private _ExpandedBin _expanded_bin;
    private int _expanded_button;
}

public class Sugar._ToolbarExpander : Gtk.Toolbar {
    public ToolbarBox toolbox {
        get { return parent.parent as ToolbarBox; }
    }
}

private class Sugar._ExpandedBin : Sugar.PaintBin {
    public int gap_pos;
    public int gap_size;

    construct {
        radius = 0;
        padding_left = Metrics.get (Metrics.TOOLBOX_HORIZONTAL_PADDING);
        padding_right = Metrics.get (Metrics.TOOLBOX_HORIZONTAL_PADDING);

        set_size_request (-1, Metrics.get (Metrics.GRID_CELL_SIZE));
        modify_bg (Gtk.StateType.NORMAL,
                color_type_to_rgb (Color.TOOLBAR_GREY));
        modify_bg (Gtk.StateType.ACTIVE, color_type_to_rgb (Color.BUTTON_GREY));
    }

    public override bool expose_event (Gdk.EventExpose event) {
        base.expose_event (event);
        Gtk.paint_hline (style, event.window,
                Gtk.StateType.ACTIVE, event.area, this, null,
                0, gap_pos + Metrics.get (Metrics.FOCUS_LINE_WIDTH) - 1, 0);
        Gtk.paint_hline (style, event.window,
                Gtk.StateType.ACTIVE, event.area, this, null,
                gap_pos + gap_size - Metrics.get (Metrics.FOCUS_LINE_WIDTH),
                allocation.width, 0);
        return false;
    }
}

private class Sugar._ButtonPalette : Sugar.PaletteWindow {
    public Bin bin;

    construct {
        modify_bg (Gtk.StateType.NORMAL, color_type_to_rgb (Color.BLACK));
        modify_bg (Gtk.StateType.ACTIVE, color_type_to_rgb (Color.BUTTON_GREY));

        bin = new Bin ();
        bin.border_left = Metrics.get (Metrics.TOOLBOX_HORIZONTAL_PADDING);
        bin.border_right = Metrics.get (Metrics.TOOLBOX_HORIZONTAL_PADDING);
        bin.show ();
        add (bin);
    }

    public override void size_request (out Gtk.Requisition requisition) {
        child.size_request (out requisition);
        requisition.width = Gdk.Screen.width ();
        requisition.height = Metrics.get (Metrics.GRID_CELL_SIZE);
    }
}

private class Sugar._Connector : Sugar.ToolConnector {
    public Gtk.PositionType orientation { get; set; }

    public _Connector (Gtk.ToolItem tool_item) {
        base (tool_item);
    }

    protected override unowned Connector.Alignment[] get_alignments () {
        switch (orientation) {
        case Gtk.PositionType.LEFT:
            return _right;
        case Gtk.PositionType.RIGHT:
            return _left;
        case Gtk.PositionType.TOP:
            return _bottom;
        case Gtk.PositionType.BOTTOM:
            return _top;
        default:
            return base.get_alignments ();
        }
    }

    private const Connector.Alignment[] _left = {
        { -1.0f, 0.0f, 0.0f, 0.0f, Gtk.PositionType.LEFT },
        { -1.0f, -1.0f, 0.0f, 1.0f, Gtk.PositionType.LEFT }
    };

    private const Connector.Alignment[] _right = {
        { 0.0f, 0.0f, 1.0f, 0.0f, Gtk.PositionType.RIGHT },
        { 0.0f, -1.0f, 1.0f, 1.0f, Gtk.PositionType.RIGHT }
    };

    private const Connector.Alignment[] _top = {
        { 0.0f, -1.0f, 0.0f, 0.0f, Gtk.PositionType.TOP },
        { -1.0f, -1.0f, 1.0f, 0.0f, Gtk.PositionType.TOP }
    };

    private const Connector.Alignment[] _bottom = {
        { 0.0f, 0.0f, 0.0f, 1.0f, Gtk.PositionType.BOTTOM },
        { -1.0f, 0.0f, 1.0f, 1.0f, Gtk.PositionType.BOTTOM }
    };
}

namespace Sugar {
    private void _prepare_page (Gtk.Widget page, int color) {
        var color_rgb = color_type_to_rgb (color);

        /* XXX modify INSENSITIVE bg for all toolitems to show them nice
         * e.g. Gtk.Entry with sugar gtk theme */
        if (page is Gtk.Container) {
            foreach (var i in (page as Gtk.Container).get_children ()) {
                Gtk.ToolItem item = i as Gtk.ToolItem;
                if (item != null && item.child != null)
                    item.child.modify_bg (Gtk.StateType.INSENSITIVE, color_rgb);
            }
        }

        page.modify_bg (Gtk.StateType.NORMAL, color_rgb);
    }
}
