# Copyright (C) 2012, Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
import logging
from os.path import dirname, join, isdir, isfile, exists


_BWLIST_DIR = '/etc/sugar'

_logger = logging.getLogger('plugins')


def init():
    for mod in _list():
        _logger.info('Initialize %r plugin', mod.__name__)
        mod.init()


def start():
    for mod in _list():
        _logger.info('Start %r plugin', mod.__name__)
        mod.enable()


def blacklisted(category, component):
    global _blacklist

    if _blacklist is None:
        _blacklist = {}
        whitelist = _populate_list('whitelist')
        for category_, components_ in _populate_list('blacklist').items():
            components_ -= whitelist.get(category_, set())
            if components_:
                _blacklist[category_] = components_

    return component in _blacklist.get(category, [])


def _list():
    if _plugins:
        return _plugins

    root = dirname(__file__)
    for filename in os.listdir(root):
        if not isdir(join(root, filename)):
            continue
        mod = __import__('jarabe.plugins.' + filename,
                globals(), locals(), [filename])
        if set(['ORDER', 'init', 'enable', 'disable']) - set(dir(mod)):
            _logger.warning('Skip %r plugin, not all required members',
                    mod.__name__)
            continue
        _plugins.append(mod)

    _plugins.sort(lambda x, y: cmp(x.ORDER, y.ORDER))
    return _plugins


def _populate_list(name):
    result = {}

    root = join(_BWLIST_DIR, name)
    if not exists(root):
        return result

    for filename in os.listdir(root):
        path = join(root, filename)
        if not isfile(path):
            continue

        with file(path) as f:
            for line in f:
                parts = line.split('#', 1)[0].split()
                if not parts:
                    continue
                if len(parts) == 2:
                    category, component = parts
                    result.setdefault(category, set())
                    result[category].add(component)
                else:
                    _logger.warning('Wrong formed %rs line: %r', name, line)

    return result


_plugins = []
_blacklist = None


if __name__ == '__main__':
    logging.basicConfig(level=logging.DEBUG)

    blacklist = _populate_list('blacklist')
    print 'blacklist:', blacklist
    for category, components in blacklist.items():
        for i in components:
            print '\t', category, i, blacklisted(category, i)

    whitelist = _populate_list('whitelist')
    print 'whitelist:', whitelist
    for category, components in whitelist.items():
        for i in components:
            print '\t', category, i, blacklisted(category, i)
