# Copyright (C) 2012 Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import logging

from sugar_network import db, client
from sugar_network.toolkit import coroutine


commands_processor = None
anonymous = False
api_url = None
_logger = logging.getLogger('sugar_network')


class ServerError(RuntimeError):
    pass


class Client(object):

    _subscriptions = {}
    _pull = None

    @classmethod
    def call(cls, method, cmd=None, content=None,
            content_type='application/json', **kwargs):
        request = db.Request(**kwargs)
        request.access_level = db.ACCESS_LOCAL
        request.principal = client.sugar_uid()
        request['method'] = method
        if cmd:
            request['cmd'] = cmd
        request.content = content
        if method != 'GET':
            request.content_type = content_type
        return commands_processor.call(request)

    @classmethod
    def connect(cls, callback, **condition):
        cls._subscriptions[callback] = condition
        if cls._pull is None:
            cls._pull = coroutine.spawn(cls._pull_events)

    @classmethod
    def disconnect(cls, callback):
        if callback in cls._subscriptions:
            del cls._subscriptions[callback]

    @classmethod
    def _pull_events(cls):
        while True:
            event = commands_processor._pooler.wait()
            for callback, condition in cls._subscriptions.items():
                for key, value in condition.items():
                    if event.get(key) != value:
                        break
                else:
                    try:
                        callback(event)
                    except Exception:
                        logging.exception('Failed to dispatch %r', event)

    def __init__(self):
        self._resources = {}

    @property
    def inline(self):
        return self.call('GET', 'inline') or anonymous

    def launch(self, context, command='activity', object_id=None, uri=None,
            args=None):
        """Launch context implementation.

        Function will call fork at the beginning. In forked process,
        it will try to choose proper implementation to execute and launch it.

        Execution log will be stored in `~/.sugar/PROFILE/logs` directory.

        :param context:
            context GUID to look for implementations
        :param command:
            command that selected implementation should support
        :param object_id:
            optional id to restore Journal object
        :param uri:
            optional uri to open; if implementation supports it
        :param args:
            optional list of arguments to pass to launching implementation

        """
        return self.call('GET', cmd='launch',
                document='context', guid=context, object_id=object_id, uri=uri,
                args=args)

    def __getattr__(self, name):
        """Class-like object to access to a resource or call a method.

        :param name:
            resource name started with capital char
        :returns:
            a class-like resource object

        """
        resource = self._resources.get(name)
        if resource is None:
            resource = _Resource(name.lower())
            self._resources[name] = resource
        return resource

    def __enter__(self):
        return self


class _Resource(object):

    def __init__(self, name):
        self.document = name

    def url(self, guid, prop):
        return api_url + '/%s/%s/%s' % (self.document, guid, prop)

    def cursor(self, query=None, order_by=None, reply=None, page_size=18,
            **filters):
        """Query resource objects.

        :param query:
            full text search query string in Xapian format
        :param order_by:
            name of property to sort by; might be prefixed by either `+` or `-`
            to change order's direction
        :param reply:
            list of property names to return for found objects;
            by default, only GUIDs will be returned; for missed properties,
            will be sent additional requests to a server on getting access
            to particular object.
        :param page_size:
            number of items in one cached page, there are might be several
            (at least two) pages
        :param filters:
            a dictionary of properties to filter resulting list

        """
        from cursor import Cursor
        return Cursor(self.document, query, order_by, reply,
                page_size, **filters)

    def delete(self, guid):
        """Delete resource object.

        :param guid:
            resource object's GUID

        """
        return Client.call('DELETE', document=self.document, guid=guid)

    def __call__(self, guid=None, reply=None, **kwargs):
        from .objects import Object
        return Object(self.document, reply or [], guid, **kwargs)
