# Copyright (C) 2012 Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import time

import gobject

from sugar_stats import dbus_monitor
from sugar_stats.util import enforce


class Collector(object):

    def __init__(self, sniffers, backend):
        self._backend = backend
        self._sniffers = {}
        self._stats = {}
        self._started = False
        self._tick_hid = None

        for name in _STATS_META.keys():
            self._stats[name] = {}

        for sniffer in sniffers:
            name = sniffer.__class__.__module__.split('.')[-1]
            self._sniffers[name] = sniffer
            sniffer.update = self.update
            sniffer.get_sniffer = lambda name: self._sniffers.get(name)

    @property
    def started(self):
        return self._started

    def start(self):
        if self._started:
            return
        for sniffer in self._sniffers.values():
            sniffer.start()
        if self._backend.step:
            self._tick_hid = gobject.timeout_add_seconds(
                    self._backend.step, self._tick)
        self._started = True

    def stop(self):
        if not self._started:
            return
        if self._tick_hid:
            gobject.source_remove(self._tick_hid)
            self._tick_hid = None
        for sniffer in self._sniffers.values():
            sniffer.stop()
        dbus_monitor.stop()
        self._started = False

    def update(self, stat, value, instance=None):
        stat_name, stat_key = stat.split('.')

        if stat_name == 'activity' and \
                instance == 'org.laptop.JournalActivity':
            if stat_key not in ['uptime', 'active']:
                # No need in other activity related keys for Journal
                return
            stat_name = 'journal'
            instance = None

        stats = self._stats[stat_name]
        if instance not in stats:
            stats[instance] = {}
            for key, value_class in _STATS_META[stat_name].items():
                stats[instance][key] = value_class()

        stats[instance][stat_key].update(value)

    def _tick(self):
        for sniffer in self._sniffers.values():
            sniffer.trigger()

        for name, instances in self._stats.items():
            for instance, values in instances.items():
                db_name = name
                if instance is not None:
                    db_name += '.' + instance
                db_values = {}
                for key, value in values.items():
                    db_values[key] = int(value.tick())
                self._backend[db_name].put(db_values)

        return True


class _Absolute(object):

    def __init__(self):
        self._value = 0
        self._peak = 0

    def update(self, new):
        self._value = new
        self._peak = max(self._peak, new)

    def tick(self):
        result = self._peak
        self._peak = self._value
        return result


class _Uptime(object):

    def __init__(self):
        self._timestamp = 0
        self._past = 0

    def update(self, enable):
        if enable:
            if not self._timestamp:
                self._timestamp = time.time()
        else:
            if self._timestamp:
                self._past += time.time() - self._timestamp
                self._timestamp = 0

    def tick(self):
        result = self._past
        self._past = 0
        if self._timestamp:
            result += time.time() - self._timestamp
        return result


class _OneTickRegister(object):

    def __init__(self):
        self._value = 0

    def update(self, value):
        enforce(value >= 0)
        self._value += value

    def tick(self):
        result = self._value
        self._value = 0
        return result


class _Register(object):

    def __init__(self):
        self._value = 0

    def update(self, value):
        self._value += value

    def tick(self):
        return self._value


_STATS_META = {
        'shell': {
            'uptime': _Uptime,
            'active': _Uptime,
            'friends': _Absolute,
            },
        'journal': {
            'uptime': _Uptime,
            'active': _Uptime,
            'creates': _OneTickRegister,
            'updates': _OneTickRegister,
            'deletes': _OneTickRegister,
            },
        'activity': {
            'uptime': _Uptime,
            'active': _Uptime,
            'instances': _Register,
            'new': _OneTickRegister,
            'resumed': _OneTickRegister,
            'buddies': _Register,
            },
        'application': {
            'uptime': _Uptime,
            'active': _Uptime,
            },
        'network': {
            'uptime': _Uptime,
            'school': _Uptime,
            },
        'system': {
            'uptime': _Uptime,
            'diskfree': _Absolute,
            },
        }
