# Copyright (C) 2012 Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import gtk
import wnck

from sugar_stats.sniffer import Sniffer


class X11(Sniffer):

    activities = {}
    activity = None
    _shell_alive = False
    _screen = wnck.screen_get_default()

    def start(self):
        self._screen.connect('window-opened', self.__window_opened_cb)
        self._screen.connect('window-closed', self.__window_closed_cb)
        self._screen.connect('active-window-changed',
                self.__active_window_changed_cb)
        self._screen.connect('showing-desktop-changed',
                self.__showing_desktop_changed_cb)

    def stop(self):
        self._screen.disconnect_by_func(self.__window_opened_cb)
        self._screen.disconnect_by_func(self.__window_closed_cb)
        self._screen.disconnect_by_func(self.__active_window_changed_cb)
        self._screen.disconnect_by_func(self.__showing_desktop_changed_cb)
        self.activity = None

    def _get_activity_by_window(self, window):
        xid = window.get_xid()
        for activity in self.activities.values():
            if [i for i in activity if i.get_xid() == xid]:
                return activity

    def _get_activitis_by_bundle_id(self, bundle_id):
        return [i for i in self.activities.values()
                if i.bundle_id == bundle_id]

    def _activity_launched(self, activity):
        if len(self._get_activitis_by_bundle_id(activity.bundle_id)) == 1:
            self.update('%s.uptime' % activity.stat_type, True,
                    activity.bundle_id)
        if self._shell_alive:
            if _is_activity_resumed(activity.pid):
                self.update('%s.resumed' % activity.stat_type, 1,
                        activity.bundle_id)
            else:
                self.update('%s.new' % activity.stat_type, 1,
                        activity.bundle_id)
            self.update('%s.instances' % activity.stat_type, 1,
                    activity.bundle_id)

    def _activity_closed(self, activity):
        if not self._get_activitis_by_bundle_id(activity.bundle_id):
            self.update('%s.uptime' % activity.stat_type, False,
                    activity.bundle_id)
        if self._shell_alive:
            self.update('%s.instances' % activity.stat_type, -1,
                    activity.bundle_id)

    def __window_opened_cb(self, screen, window):
        if window.get_window_type() == wnck.WINDOW_DESKTOP:
            if window.get_name() == 'sugar-session':
                self._shell_alive = True
                self.update('shell.uptime', True)
            return

        if window.get_window_type() != wnck.WINDOW_NORMAL or \
                _get_x11_property(window, '_SUGAR_WINDOW_TYPE') == 'launcher':
            return

        activity_id = _get_x11_property(window, '_SUGAR_ACTIVITY_ID')
        if activity_id:
            bundle_id = _get_x11_property(window, '_SUGAR_BUNDLE_ID')
            stat_type = 'activity'
        elif not self._shell_alive:
            activity_id, bundle_id = _detect_app(window)
            if not activity_id:
                return
            stat_type = 'application'
        else:
            return

        activity = self.activities.get(activity_id)
        if activity is None:
            activity = self.activities[activity_id] = _Activity()
            activity.stat_type = stat_type
            activity.pid = window.get_pid()
            activity.xid = window.get_xid()
            activity.activity_id = activity_id
            activity.bundle_id = bundle_id
            self._activity_launched(activity)

        activity.append(window)

    def __window_closed_cb(self, screen, window):
        if window.get_window_type() == wnck.WINDOW_DESKTOP:
            if window.get_name() == 'sugar-session':
                self._shell_alive = False
                self.update('shell.uptime', False)
            return

        if window.get_window_type() != wnck.WINDOW_NORMAL:
            return

        activity = self._get_activity_by_window(window)
        if activity is None:
            return

        activity.remove(window)
        if not activity:
            del self.activities[activity.activity_id]
            self._activity_closed(activity)

    def __active_window_changed_cb(self, screen, previous_window=None):
        window = screen.get_active_window()
        if window is None:
            return

        if window.get_window_type() != wnck.WINDOW_DIALOG:
            while window.get_transient() is not None:
                window = window.get_transient()

        activity = self._get_activity_by_window(window)
        if self.activity is activity:
            return

        if self.activity is not None:
            self.update('%s.active' % self.activity.stat_type, False,
                    self.activity.bundle_id)
            self.activity = None

        if activity is not None:
            self.update('%s.active' % activity.stat_type, True,
                    activity.bundle_id)
            self.activity = activity

    def __showing_desktop_changed_cb(self, screen):
        if self._shell_alive:
            if screen.get_showing_desktop():
                self.update('shell.active', True)
            else:
                self.update('shell.active', False)


class _Activity(list):

    pid = None
    xid = None
    activity_id = None
    bundle_id = None
    stat_type = None


def _get_x11_property(window, prop):
    gdk_window = gtk.gdk.window_foreign_new(window.get_xid())

    # There is a chance to get X error
    gtk.gdk.error_trap_push()
    prop_info = gdk_window.property_get(prop, 'STRING')
    gtk.gdk.error_trap_pop()

    if prop_info is None:
        return None
    else:
        return prop_info[2]


def _is_activity_resumed(pid):
    cmd_file = file('/proc/%s/cmdline' % pid)
    try:
        cmd = cmd_file.read().split('\0')
        return '-o' in cmd or '--object-id' in cmd
    finally:
        cmd_file.close()


def _detect_app(window):
    class_group = window.get_class_group()
    return class_group.get_res_class(), class_group.get_res_class()
