# Copyright (C) 2012 Aleksey Lim
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <http://www.gnu.org/licenses/>.

import os
from os.path import exists, expanduser
from gettext import gettext as _

import gtk

from sugar_network.toolkit import Option
from sugar.graphics import style

from jarabe.controlpanel.sectionview import SectionView
from jarabe.plugins import plugins, modules


# Only for XO laptops
_XSESSION_PATH = expanduser('~/.xsession')
_XSESSION_FILE = """\
exec sweets-sugar
"""


class Sweets(SectionView):

    def __init__(self, model, alerts):
        SectionView.__init__(self)

        self.set_border_width(style.DEFAULT_SPACING * 2)
        self.set_spacing(style.DEFAULT_SPACING)
        self.setup()

    def setup(self):
        for option in Option.items.values():
            option.default = option.value

        while self.get_children():
            self.remove(self.get_children()[0])

        scrolled_window = gtk.ScrolledWindow()
        scrolled_window.set_policy(gtk.POLICY_NEVER, gtk.POLICY_AUTOMATIC)
        scrolled_window.set_shadow_type(gtk.SHADOW_IN)
        self.pack_start(scrolled_window)

        canvas = gtk.VBox()
        scrolled_window.add_with_viewport(canvas)
        scrolled_window.show_all()

        def section_new(title, container, spacing=style.DEFAULT_SPACING):
            separator = gtk.HSeparator()
            separator.show()
            canvas.pack_start(separator, expand=False)
            label = gtk.Label(title)
            label.set_alignment(0, 0)
            label.show()
            canvas.pack_start(label, expand=False)
            box = container()
            box.props.border_width = style.DEFAULT_SPACING
            box.props.spacing = spacing
            box.show()
            canvas.pack_start(box, expand=False)
            return box

        section = section_new(_('Sweets Distribution features'), gtk.VBox, 0)

        self._xsession = gtk.CheckButton()
        self._xsession.props.label = \
                _('Sweets Desktop is default Sugar session')
        self._xsession.props.active = _xsession_enabled()
        self._xsession.connect('toggled',
                lambda button: _set_xsession(button.props.active))
        if _is_xo():
            section.pack_start(self._xsession, expand=False)

        self._plugins = {}
        for mod in modules:
            plugin_button = gtk.CheckButton()
            plugin_button.props.label = mod.TITLE
            plugin_button.props.active = mod.name in plugins.value
            plugin_button.connect('toggled', self.__plugin_button_toggled_cb)
            section.pack_start(plugin_button, expand=False)
            self._plugins[mod.name] = plugin_button

        section.show_all()

        for mod in modules:
            if hasattr(mod, 'control_panel_section'):
                sub_section = mod.control_panel_section()
                sub_section.show()
                section = section_new(mod.TITLE, gtk.VBox)
                section.add(sub_section)

        self._orig_xsession = _xsession_enabled()

    def undo(self):
        for option in Option.items.values():
            option.value = option.default
        Option.save()
        if self._orig_xsession != _xsession_enabled():
            _set_xsession(self._orig_xsession)
        self.setup()

    def get_needs_restart(self):
        Option.save()
        options_changed = [i for i in Option.items.values() \
                if i.value != i.default]
        return options_changed or \
                self._xsession.props.active != self._orig_xsession

    def set_needs_restart(self, value):
        # needs_restart is fully calculated
        pass

    needs_restart = property(get_needs_restart, set_needs_restart)

    def __plugin_button_toggled_cb(self, button):
        plugins.value = [name for name, button in self._plugins.items() \
                if button.props.active]
        Option.save()


def _xsession_enabled():
    if not exists(_XSESSION_PATH):
        return False
    with file(_XSESSION_PATH) as f:
        return f.read() == _XSESSION_FILE


def _set_xsession(enable):
    if enable:
        with file(_XSESSION_PATH, 'w') as f:
            f.write(_XSESSION_FILE)
    else:
        if exists(_XSESSION_PATH):
            os.unlink(_XSESSION_PATH)


def _is_xo():
    return exists('/ofw') and exists('/usr/bin/sugar')
