# -*- coding: utf-8 -*-
#Copyright (c) 2010, Walter Bender

#Permission is hereby granted, free of charge, to any person obtaining a copy
#of this software and associated documentation files (the "Software"), to deal
#in the Software without restriction, including without limitation the rights
#to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
#copies of the Software, and to permit persons to whom the Software is
#furnished to do so, subject to the following conditions:

#The above copyright notice and this permission notice shall be included in
#all copies or substantial portions of the Software.

#THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
#IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
#FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
#AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
#LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
#OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
#THE SOFTWARE.

"""
This file contains the constants that by-in-large determine the
behavior of Turtle Art. Notably, the block palettes are defined
below. If you want to add a new block to Turtle Art, it is generally a
matter of modifying some tables below and then adding the primitive to
talogo.py. For example, if we want to add a new turtle command,
'uturn', we'd make the following changes:

(1) We'd add 'uturn' to the PALETTES list of lists:

PALETTES = [['forward', 'back', 'clean', 'left', 'right', 'uturn', 'show', 
             'seth', 'setxy', 'heading', 'xcor', 'ycor', 'setscale',
              'arc', 'scale'],
            ['penup','pendown', 'setpensize', 'fillscreen', 'pensize',...

(2) Then we'd add it to one of the block-style definitions. Since it takes
no arguments, we'd add it here:

BASIC_STYLE = ['clean', 'penup', 'pendown', 'stack1', 'stack2', 'vspace',
    'hideblocks', 'showblocks', 'clearheap', 'printheap', 'kbinput', 'uturn']

(3) Then we give it a name (Note the syntax _('string to be
translated') used by the language-internationalization system; also
note that the name is an array, as some blocks contain multiple
strings.):

BLOCK_NAMES = {
...
    'uturn':[_('u-turn')],
...
              }

(4) and a help-menu entry:

HELP_STRINGS = {
...
    'uturn':_('change the heading of the turtle 180 degrees'),
...
               }

(5) Next, we need to define it as a primitive for the Logo command
parser (generally just the same name):

PRIMITIVES = {
...
    'uturn':'uturn',
...
             }

(6) Since there are no default arguments, we don't need to do anything
else here. But we do need to define the actual function in talogo.py

DEFPRIM = {
...
    'uturn':[0, lambda self: self.tw.canvas.seth(self.tw.canvas.heading+180)],
...
          }

That's it. When you next run Turtle Art, you will have a 'uturn' block
on the Turtle Palette.

Adding a new palette is simply a matter of: (1) adding an additional
entry to PALETTE_NAMES; (2) new list of blocks to PALETTES; and (3) an
additional entry in COLORS. However you will have to: (4) create icons
for the palette-selector buttons. These are kept in the icons
subdirectory. You need two icons: yourpalettenameoff.svg and
yourpalettenameon.svg, where yourpalettename is the same string as the
entry you added to the PALETTE_NAMES list. Note that the icons should
be the same size (55x55) as the others. This is the default icon size
for Sugar toolbars.

"""

from gettext import gettext as _

#
# Sprite layers
#

HIDE_LAYER = 100
CANVAS_LAYER = 500
OVERLAY_LAYER = 525
TURTLE_LAYER = 550
BLOCK_LAYER = 600
CATEGORY_LAYER = 700
TAB_LAYER = 710
STATUS_LAYER = 900
TOP_LAYER = 1000

#
# Block-palette categories
#

PALETTE_NAMES = ['turtle', 'pen', 'colors', 'numbers', 'flow', 'blocks',
                 'extras', 'trash']

PALETTES = [['clean', 'forward', 'back', 'show', 'left', 'right',
             'seth', 'setxy', 'heading', 'xcor', 'ycor', 'setscale',
             'arc', 'scale'],
            ['penup','pendown', 'setpensize', 'fillscreen', 'pensize',
             'setcolor', 'setshade', 'startfill', 'color', 'shade',
             'stopfill' ],
            ['red', 'orange', 'yellow', 'green', 'cyan', 'blue', 'purple'],
            ['plus2', 'minus2', 'product2',
             'division2', 'identity2', 'remainder2', 'random',
             'number', 'greater2', 'less2', 'equal2'],
            ['forever', 'repeat', 'if', 'vspace', 'stopstack', 'wait'],
            ['hat', 'stack', 'storein', 'box', 'string', 'start'],
            ['print', 'journal', 'hideblocks'],
            ['empty', 'restoreall']]

#
# Block-style attributes
#

COLORS = [["#00FF00","#00A000"], ["#00FFFF","#00A0A0"], ["#00FFFF","#00A0A0"],
          ["#FF00FF","#A000A0"], ["#FFC000","#A08000"], ["#FFFF00","#A0A000"],
          ["#FF0000","#A00000"], ["#FFFF00","#A0A000"]]

BOX_COLORS = {'red':["#FF0000","#A00000"],'orange':["#FFD000","#AA8000"],
              'yellow':["#FFFF00","#A0A000"],'green':["#00FF00","#008000"],
              'cyan':["#00FFFF","#00A0A0"],'blue':["#0000FF","#000080"],
              'purple':["#FF00FF","#A000A0"]}

#
# Misc. parameters
#
PALETTE_HEIGHT = 120
PALETTE_WIDTH = 175
SELECTOR_WIDTH = 55
ICON_SIZE = 55
SELECTED_COLOR = "#0000FF"
SELECTED_STROKE_WIDTH = 1.0
STANDARD_STROKE_WIDTH = 1.0
BLOCK_SCALE = 2.0
PALETTE_SCALE = 1.5
DEFAULT_TURTLE = 1
HORIZONTAL_PALETTE = 0
VERTICAL_PALETTE = 1

#
# Block-style definitions
#
BASIC_STYLE_HEAD = ['start', 'hat1', 'hat2', 'restore', 'restoreall']
BASIC_STYLE_HEAD_1ARG = ['hat']
BASIC_STYLE_TAIL = ['stopstack', 'empty']
BASIC_STYLE = ['clean', 'penup', 'pendown', 'stack1', 'stack2', 'vspace',
    'hideblocks', 'showblocks', 'clearheap', 'printheap', 'kbinput',
    'fullscreen', 'sandwichcollapsed', 'cartesian', 'polar', 'startfill',
    'stopfill']
BASIC_STYLE_EXTENDED = ['picturelist', 'picture1x1', 'picture2x2',
    'picture2x1', 'picture1x2', 'picture1x1a']
BASIC_STYLE_1ARG = ['forward', 'back', 'left', 'right', 'seth', 'show', 'image',
    'setscale', 'setpensize', 'setcolor', 'setshade', 'print', 'showaligned',
    'settextsize', 'settextcolor', 'print', 'wait', 'storeinbox1', 'savepix',
    'storeinbox2', 'wait', 'stack', 'push', 'nop', 'addturtle', 'comment',
    'savesvg']
BASIC_STYLE_VAR_ARG = ['userdefined', 'userdefined2args', 'userdefined3args']
BULLET_STYLE = ['templatelist', 'list']
BASIC_STYLE_2ARG = ['arc', 'setxy', 'fillscreen', 'storein', 'write']
BOX_STYLE = ['number', 'xcor', 'ycor', 'heading', 'pensize', 'color', 'shade',
    'textcolor', 'textsize', 'box1', 'box2', 'string', 'leftpos', 'scale',
    'toppos', 'rightpos', 'bottompos', 'width', 'height', 'pop', 'keyboard',
    'red', 'orange', 'yellow', 'green', 'cyan', 'blue', 'purple',
    'titlex', 'titley', 'leftx', 'topy', 'rightx', 'bottomy',
    'volume', 'pitch', 'voltage', 'resistance']
BOX_STYLE_MEDIA =  ['description', 'audio', 'journal']
NUMBER_STYLE = ['plus2', 'product2', 'myfunc']
NUMBER_STYLE_VAR_ARG = ['myfunc1arg', 'myfunc2arg', 'myfunc3arg']
NUMBER_STYLE_BLOCK = ['random']
NUMBER_STYLE_PORCH = ['minus2', 'division2', 'remainder2']
NUMBER_STYLE_1ARG = ['sqrt', 'identity2']
NUMBER_STYLE_1STRARG = ['box']
COMPARE_STYLE = ['greater2', 'less2', 'equal2']
BOOLEAN_STYLE = ['and2', 'or2']
NOT_STYLE = ['not']
FLOW_STYLE = ['forever']
FLOW_STYLE_TAIL = ['hspace']
FLOW_STYLE_1ARG = ['repeat']
FLOW_STYLE_BOOLEAN = ['if', 'while', 'until']
FLOW_STYLE_WHILE = ['while2']
FLOW_STYLE_ELSE = ['ifelse']
COLLAPSIBLE_TOP = ['sandwichtop']
COLLAPSIBLE_TOP_NO_ARM = ['sandwichtop2']
COLLAPSIBLE_BOTTOM = ['sandwichbottom']

# Depreciated block styles
PORTFOLIO_STYLE_2x2 = ['template2x2']
PORTFOLIO_STYLE_1x1 = ['template1x1', 'template1x1a']
PORTFOLIO_STYLE_2x1 = ['template2x1']
PORTFOLIO_STYLE_1x2 = ['template1x2']

#
# Blocks that are expandable
#
EXPANDABLE = ['vspace', 'hspace', 'templatelist', 'list', 'identity2',
              'myfunc1arg', 'myfunc2arg', 'myfunc3arg', 'userdefined',
              'userdefined2args', 'userdefined3args']

#
# Blocks that are 'collapsible'
#
COLLAPSIBLE = ['sandwichbottom', 'sandwichcollapsed']

#
# Depreciated block styles that need dock adjustments
#
OLD_DOCK = ['and', 'or', 'plus', 'minus', 'division', 'product', 'remainder']

#
# Blocks that contain media
#
CONTENT_BLOCKS = ['number', 'string', 'description', 'audio', 'journal']

#
# These blocks get a special skin
#
BLOCKS_WITH_SKIN = ['journal', 'audio', 'description', 'nop', 'userdefined',
                    'userdefined2args', 'userdefined3args']

PYTHON_SKIN = ['nop', 'userdefined', 'userdefined2args', 'userdefined3args']

#
# Block-name dictionary used for labels
#
BLOCK_NAMES = {
    'addturtle':[_('turtle')],
    'and2':[' '],
    'arc':[_('arc'), _('angle'), _('radius')],
    'audio':[' '],
    'back':[_('back')],
    'blue':[_('blue')+' = 70'],
    'bottompos':[' '],
    'bottomy':[' '],
    'box':[_('box')],
    'box1':[' '],
    'box2':[' '],
    'cartesian':[_('Cartesian')],
    'clean':[_(' clean ')],
    'clearheap':[' '],
    'color':[_('color')],
    'comment':[' '],
    'cyan':[_('cyan')+' = 50'],
    'decription':[' '],
    'division2':['/'],
    'empty':[_('empty trash')],
    'equal2':['='],
    'fillscreen':[_('fill screen'), _('color'), _('shade')],
    'forever':[_('forever')],
    'forward':[_('forward')],
    'fullscreen':[' '],
    'greater2':[">"],
    'green':[_('green')+' = 30'],
    'hat':[_('action')],
    'hat1':[' '],
    'hat2':[' '],
    'heading':[_('heading')],
    'height':[' '],
    'hideblocks':[_('hide blocks')],
    'hspace':[' '],
    'identity2':['←'],
    'if':[' ', _('if'), _('then')],
    'ifelse':[' ', ' ', ' '],
    'image':[_('show')],
    'journal':[' '],
    'kbinput':[' '],
    'keyboard':[' '],
    'left':[_('left')],
    'leftpos':[_('left')],
    'leftx':[' '],
    'less2':['<'],
    'list':['list'],
    'minus2':['–'],
    'myfunc':[' ', 'f(x)', 'x'],
    'myfunc1arg':[' ', 'f(x)', 'x'],
    'myfunc2arg':[' ', 'f(x,y)', ' '],
    'myfunc3arg':[' ', 'f(x,y,z)', ' '],
    'nop':[' '],
    'not':[' '],
    'number':['100'],
    'orange':[_('orange')+' = 10'],
    'or2':[' '],
    'pendown':[_('pen down')],
    'pensize':[_('pen size')],
    'penup':[_('pen up')],
    'picturelist':[' '],
    'picture1x1':[' '],
    'picture1x1a':[' '],
    'picture2x2':[' '],
    'picture2x1':[' '],
    'picture1x2':[' '],
    'pitch':[_('pitch')],
    'plus2':['+'],
    'polar':[_('polar')],
    'pop':[_('pop')],
    'printheap':[_('show heap')],
    'print':[_('print')],
    'product2':['×'],
    'purple':[_('purple')+' = 90'],
    'push':[_('push')],
    'random':[_('random'), _('min'), _('max')],
    'red':[_('red')+' = 0'],
    'remainder2':[_('mod')],
    'repeat':[' ',_('repeat')],
    'resistance':[' '],
    'restore':[' '],
    'restoreall':[' '],
    'right':[_('right')],
    'rightpos':[_('right')],
    'rightx':[' '],
    'savepix':[' '],
    'savesvg':[' '],
    'sandwichbottom':[' '],
    'sandwichcollapsed':[' '],
    'sandwichtop':[' '],
    'sandwichtop2':[' '],
    'scale':[_('scale')],
    'setcolor':[_('set color')],
    'seth':[_('set heading')],
    'setpensize':[_('set pen size')],
    'setscale':[_('set scale')],
    'setshade':[_('set shade')],
    'settextcolor':[' '],
    'settextsize':[' '],
    'setxy':[_('set xy'), _('x'), _('y')],
    'shade':[_('shade')],
    'show':[_('show')],
    'showblocks':[' '],
    'showaligned':[' '],
    'sqrt':['√'],
    'stack':[_('action')],
    'stack1':[' '],
    'stack2':[' '],
    'start':[_('start')],
    'startfill':[_('start fill')],
    'stopfill':[_('end fill')],
    'stopstack':[_('stop action')],
    'storein':[_('store in'), _('box'), _('value')],
    'storeinbox1':[' '],
    'storeinbox2':[' '],
    'string':[_('text')],
    'template1x1':[' '],
    'template1x1a':[' '],
    'template1x2':[' '],
    'template2x1':[' '],
    'template2x2':[' '],
    'templatelist':[' '],
    'textsize':[' '],
    'titlex':[' '],
    'titley':[' '],
    'toppos':[' '],
    'topy':[' '],
    'turtle':[' '],
    'until':[' '],
    'userdefined':[' '],
    'userdefined2args':[' '],
    'userdefined3args':[' '],
    'voltage':[' '],
    'volume':[' '],
    'vspace':[' '],
    'wait':[_('wait')],
    'while':[_('while')],
    'while2':[_('while')],
    'width':[' '],
    'write':[' '],
    'xcor':[_('xcor')],
    'ycor':[_('ycor')],
    'yellow':[_('yellow')+' = 20']}

#
# Logo primitives
#

PRIMITIVES = {
    'addturtle':'turtle',
    'and2':'and',
    'arc':'arc',
    'back':'back',
    'blue':'blue',
    'bottompos':'bpos',
    'bottomy':'boty',
    'box1':'box1',
    'box2':'box2',
    'box':'box',
    'cartesian':'cartesian',
    'clean':'clean',
    'clearheap':'clearheap',
    'color':'color',
    'comment':'comment',
    'cyan':'cyan',
    'division2':'division',
    'equal2':'equal?',
    'fillscreen':'fillscreen',
    'forever':'forever',
    'forward':'forward',
    'fullscreen':'fullscreen',
    'greater2':'greater?',
    'green':'green',
    'hat':'nop3',
    'hat1':'nop1',
    'hat2':'nop2',
    'heading':'heading',
    'height':'vres',
    'hideblocks':'hideblocks',
    'hspace':'nop',
    'identity2':'id',
    'if':'if',
    'ifelse':'ifelse',
    'image':'show',
    'kbinput':'kbinput',
    'keyboard':'keyboard',
    'left':'left',
    'leftpos':'lpos',
    'leftx':'leftx',
    'less2':'less?',
    'list':'bulletlist',
    'minus2':'minus',
    'myfunc':'myfunction',
    'myfunc1arg':'myfunction',
    'myfunc2arg':'myfunction2',
    'myfunc3arg':'myfunction3',
    'nop':'userdefined',
    'not':'not',
    'orange':'orange',
    'or2':'or',
    'pendown':'pendown',
    'pensize':'pensize',
    'penup':'penup',
    'pitch':'pitch',
    'plus2':'plus',
    'polar':'polar',
    'pop':'pop',
    'printheap':'printheap',
    'print':'print',
    'product2':'product',
    'purple':'purple',
    'push':'push',
    'random':'random',
    'red':'red',
    'remainder2':'mod',
    'repeat':'repeat',
    'resistance':'resistance',
    'right':'right',
    'rightpos':'rpos',
    'rightx':'rightx',
    'sandwichtop':'comment',
    'sandwichtop2':'comment',
    'sandwichbottom':'nop',
    'sandwichcollapsed':'nop',
    'savepix':'savepix',
    'savesvg':'savesvg',
    'scale':'scale',
    'setcolor':'setcolor',
    'seth':'seth',
    'setpensize':'setpensize',
    'setscale':'setscale',
    'setshade':'setshade',
    'settextsize':'settextsize',
    'settextcolor':'settextcolor',
    'setxy':'setxy',
    'shade':'shade',
    'show':'show',
    'showblocks':'showblocks',
    'showaligned':'showaligned',
    'sqrt':'sqrt',
    'stack':'stack',
    'stack1':'stack1',
    'stack2':'stack2',
    'start':'start',
    'startfill':'startfill',
    'stopfill':'stopfill',
    'stopstack':'stopstack',
    'storein':'storeinbox',
    'storeinbox1':'storeinbox1',
    'storeinbox2':'storeinbox2',
    'template1x1':'t1x1',
    'template1x1a':'t1x1a',
    'template1x2':'t1x2',
    'template2x1':'t2x1',
    'template2x2':'t2x2',
    'templatelist':'bullet',
    'textsize':'textsize',
    'titlex':'titlex',
    'titley':'titley',
    'toppos':'tpos',
    'topy':'topy',
    'userdefined':'userdefined',
    'userdefined2args':'userdefined2',
    'userdefined3args':'userdefined3',
    'voltage':'voltage',
    'volume':'volume',
    'vspace':'nop',
    'wait':'wait',
    'while2':'while',
    'width':'hres',
    'write':'write',
    'xcor':'xcor',
    'ycor':'ycor',
    'yellow':'yellow'}

#
# block default values
#

DEFAULTS = {
    'addturtle':[1],
    'arc':[90, 100],
    'audio':[None],
    'back':[100],
    'box':[_('my box')],
    'comment':[_('comment')],
    'description':[None],
    'fillscreen':[60, 80],
    'forever':[None, 'vspace'],
    'forward':[100],
    'hat':[_('action')],
    'if':[None, None, 'vspace'],
    'ifelse':[None, 'vspace', None, 'vspace'],
    'journal':[None],
    'left':[90],
    'list':['∙ ', '∙ '],
    'media':[None],
    'myfunc':['x', 100],
    'myfunc1arg':['x', 100],
    'myfunc2arg':['x+y', 100, 100],
    'myfunc3arg':['x+y+z', 100, 100, 100],
    'nop':[100],
    'number':[100],
    'random':[0, 100],
    'repeat':[4, None, 'vspace'],
    'right':[90],
    'sandwichtop':[_('label')],
    'sandwichtop2':[_('label')],
    'savepix':[_('picture name')],
    'savesvg':[_('picture name')],
    'setcolor':[0],
    'seth':[0],
    'setpensize':[5],
    'setscale':[33],
    'setshade':[50],
    'settextsize':[48],
    'settextcolor':[0],
    'setxy':[0, 0],
    'show':[_('text')],
    'showaligned':[_('text')],
    'stack':[_('action')],
    'storeinbox1':[100],
    'storeinbox2':[100],
    'storein':[_('my box'), 100],
    'string':[_('text')],
    'template1x1':[' ', 'None'],
    'template1x1a':[' ', 'None'],
    'template1x2':[' ', 'None', 'None'],
    'template2x1':[' ', 'None', 'None'],
    'template2x2':[' ', 'None', 'None', 'None', 'None'],
    'templatelist':[' ', '∙ '],
    'userdefined':[100],
    'userdefined2args':[100,100],
    'userdefined3args':[100,100,100],
    'wait':[1],
    'write':[_('text'), 32]}

#
# Blocks that can interchange strings and numbers for their arguments
#
STRING_OR_NUMBER_ARGS = ['plus2', 'equal2', 'less2', 'greater2', 'box',
                         'template1x1', 'template1x2', 'template2x1', 'list',
                         'template2x2', 'template1x1a', 'templatelist', 'nop',
                         'print', 'stack', 'hat', 'addturtle', 'myfunc',
                         'myfunc1arg', 'myfunc2arg', 'myfunc3arg', 'comment',
                         'sandwichtop', 'sandwichtop2', 'userdefined',
                         'userdefined2args', 'userdefined3args', 'storein']

CONTENT_ARGS = ['show', 'showaligned', 'push', 'storein', 'storeinbox1',
                'storeinbox2']

#
# Status blocks
#

MEDIA_SHAPES = ['audiooff', 'audioon', 'audiosmall',
                'journaloff', 'journalon', 'journalsmall',
                'descriptionoff', 'descriptionon', 'descriptionsmall',
                'pythonoff', 'pythonon', 'pythonsmall',
                'list', '1x1', '1x1a', '2x1', '1x2', '2x2']

OVERLAY_SHAPES = ['Cartesian', 'Cartesian_labeled', 'polar']

STATUS_SHAPES = ['status', 'info', 'nostack', 'noinput', 'emptyheap',
                 'emptybox', 'nomedia', 'nocode', 'overflowerror', 'negroot',
                 'syntaxerror', 'nofile', 'nojournal', 'zerodivide']

# 
# Emulate Sugar toolbar when running from outside of Sugar
#
TOOLBAR_SHAPES = ['hideshowoff', 'eraseron', 'run-fastoff',
                  'run-slowoff', 'debugoff', 'stopiton']

#
# Legacy names
#
OLD_NAMES = {'product':'product2', 'storeinbox':'storein', 'minus':'minus2',
             'division':'division2', 'plus':'plus2', 'and':'and2', 'or':'or2',
             'less':'less2', 'greater':'greater2', 'equal':'equal2',
             'remainder':'remainder2', 'identity':'identity2',
             'division':'division2', 'audiooff':'audio', 'endfill':'stopfill',
             'descriptionoff':'description','template3':'templatelist',
             'template1':'template1x1', 'template2':'template2x1',
             'template6':'template1x2', 'template7':'template2x2', 
             'template4':'template1x1a', 'hres':'width', 'vres':'height' }

#
# Define the relative size and postion of media objects
#                    (w,   h,   x,      y,     dx, dy)
#
TITLEXY = (0.9375, 0.875)

#
# Relative placement of portfolio objects (used by depreciated blocks)
#
TEMPLATES = {'t1x1': (0.5, 0.5, 0.0625, 0.125, 1.05, 0),
             't2z1': (0.5, 0.5, 0.0625, 0.125, 1.05, 1.05),
             't1x2': (0.45, 0.45, 0.0625, 0.125, 1.05, 1.05),
             't2x2': (0.45, 0.45, 0.0625, 0.125, 1.05, 1.05),
             't1x1a': (0.9, 0.9, 0.0625, 0.125, 0, 0),
             'bullet': (1, 1, 0.0625, 0.125, 0, 0.1),
             'insertimage': (0.333, 0.333)}

#
# Names for blocks without names for popup help
#
SPECIAL_NAMES = {
    'audio':' ',
    'division2':_('divide'),
    'equal2':_('equal'),
    'greater2':_('greater than'),
    'hspace':' ',
    'identity2':_('identity'),
    'if':_('if then'),
    'ifelse':' ',
    'journal':_('journal'),
    'less2':_('less than'),
    'minus2':_('minus'),
    'nop':' ',
    'number':_('number'),
    'plus2':_('plus'),
    'product2':_('multiply'),
    'sqrt':_('square root'),
    'template1x1':' ',
    'template1x1a':' ',
    'template1x2':' ',
    'template2x1':' ',
    'template2x2':' ',
    'templatelist':' ',
    'textsize':' ',
    'vspace':_('vertical space')}

#
# Help messages
#
HELP_STRINGS = {}

#
# 'dead key' Unicode dictionaries
#

DEAD_KEYS = ['grave','acute','circumflex','tilde','diaeresis','abovering']
DEAD_DICTS = [{'A':192,'E':200,'I':204,'O':210,'U':217,'a':224,'e':232,'i':236,
               'o':242,'u':249},
              {'A':193,'E':201,'I':205,'O':211,'U':218,'a':225,'e':233,'i':237,
               'o':243,'u':250},
              {'A':194,'E':202,'I':206,'O':212,'U':219,'a':226,'e':234,
               'i':238,'o':244,'u':251},
              {'A':195,'O':211,'N':209,'U':360,'a':227,'o':245,'n':241,'u':361},
              {'A':196,'E':203,'I':207,'O':211,'U':218,'a':228,'e':235,
               'i':239,'o':245,'u':252},
              {'A':197,'a':229}]
NOISE_KEYS = ['Shift_L', 'Shift_R', 'Control_L', 'Caps_Lock', 'Pause',
              'Alt_L', 'Alt_R', 'KP_Enter', 'ISO_Level3_Shift', 'KP_Divide',
              'Escape', 'Return', 'KP_Page_Up', 'Up', 'Down', 'Menu',
              'Left', 'Right', 'KP_Home', 'KP_End', 'KP_Up', 'Super_L',
              'KP_Down', 'KP_Left', 'KP_Right', 'KP_Page_Down', 'Scroll_Lock',
              'Page_Down', 'Page_Up']
WHITE_SPACE = ['space','Tab']

CURSOR = '█'
RETURN = '⏎'


#
# Macros (groups of blocks)
#
MACROS = {
    'until':
              [[0, 'forever', 0, 0, [None, 2, 1]],
               [1, 'vspace', 0, 0, [0, None]],
               [2, 'ifelse', 0, 0, [0, None, 3, None, None]],
               [3, 'vspace', 0, 0, [2, 4]],
               [4, 'stopstack', 0, 0, [3, None]]],
    'while':
              [[0, 'forever', 0, 0, [None, 2, 1]],
               [1, 'vspace', 0, 0, [0, None]],
               [2, 'ifelse', 0, 0, [0, None, 3, 4, None]],
               [3, 'vspace', 0, 0, [2, None]],
               [4, 'stopstack', 0, 0, [2, None]]],
    'kbinput':
              [[0, 'forever', 0, 0, [None, 1, None]],
               [1, 'kbinput', 0, 0, [0, 2]],
               [2, 'vspace', 0, 0, [1, 3]],
               [3, 'if', 0, 0, [2, 4, 7, 8]],
               [4, 'greater2', 0, 0, [3, 5, 6, None]],
               [5, 'keyboard', 0, 0, [4, None]],
               [6, ['number', '0'], 0, 0, [4, None]],
               [7, 'stopstack', 0, 0, [3, None]],
               [8, 'vspace', 0, 0, [3, 9]],
               [9, 'wait', 0, 0, [8, 10, None]],
               [10, ['number', '1'], 0, 0, [9, None]]],
    'picturelist':
              [[0, 'sandwichtop', 0, 0, [None, 1, 2]],
               [1, ['string', ' '], 0, 0, [0, None]],
               [2, 'setxy', 0, 0, [0, 3, 4, 5]],
               [3, 'titlex', 0, 0, [2, None]],
               [4, 'titley', 0, 0, [2, None]],
               [5, 'setscale', 0, 0, [2, 6, 7]],
               [6, ['number', '100'], 0, 0, [5, None]],
               [7, 'show', 0, 0, [5, 8, 9]],
               [8, ['string',' '], 0, 0, [7, None]],
               [9, 'setxy', 0, 0, [7, 10, 11, 12]],
               [10, 'leftx', 0, 0, [9, None]],
               [11, 'topy', 0, 0, [9, None]],
               [12, 'setscale', 0, 0, [9, 13, 14]],
               [13, ['number', '67'], 0, 0, [12, None]],
               [14, 'list', 0, 0, [12, 15, 16, 17]],
               [15, ['string','∙ '], 0, 0, [14, None]],
               [16, ['string','∙ '], 0, 0, [14, None]],
               [17, 'sandwichbottom', 0, 0, [14, None]]],
    'picture1x1a':
              [[0, 'sandwichtop', 0, 0, [None, 1, 2]],
               [1, ['string', ' '], 0, 0, [0, None]],
               [2, 'setxy', 0, 0, [0, 3, 4, 5]],
               [3, 'titlex', 0, 0, [2, None]],
               [4, 'titley', 0, 0, [2, None]],
               [5, 'setscale', 0, 0, [2, 6, 7]],
               [6, ['number', '100'], 0, 0, [5, None]],
               [7, 'show', 0, 0, [5, 8, 9]],
               [8, ['string',' '], 0, 0, [7, None]],
               [9, 'setscale', 0, 0, [7, 10, 11]],
               [10, ['number', '90'], 0, 0, [9, None]],
               [11, 'setxy', 0, 0, [9, 12, 13, 14]],
               [12, 'leftx', 0, 0, [11, None]],
               [13, 'topy', 0, 0, [11, None]],
               [14, 'showaligned', 0, 0, [11, 15, 16]],
               [15, 'journal', 0, 0, [14, None]],
               [16, 'sandwichbottom', 0, 0, [14, None]]],
    'picture2x2':
              [[0, 'sandwichtop', 0, 0, [None, 1, 2]],
               [1, ['string', ' '], 0, 0, [0, None]],
               [2, 'setxy', 0, 0, [0, 3, 4, 5]],
               [3, 'titlex', 0, 0, [2, None]],
               [4, 'titley', 0, 0, [2, None]],
               [5, 'setscale', 0, 0, [2, 6, 7]],
               [6, ['number', '100'], 0, 0, [5, None]],
               [7, 'show', 0, 0, [5, 8, 9]],
               [8, ['string',' '], 0, 0, [7, None]],
               [9, 'setscale', 0, 0, [7, 10, 11]],
               [10, ['number', '35'], 0, 0, [9, None]],
               [11, 'setxy', 0, 0, [9, 12, 13, 14]],
               [12, 'leftx', 0, 0, [11, None]],
               [13, 'topy', 0, 0, [11, None]],
               [14, 'showaligned', 0, 0, [11, 15, 16]],
               [15, 'journal', 0, 0, [14, None]],
               [16, 'setxy', 0, 0, [14, 17, 18, 19]],
               [17, 'rightx', 0, 0, [16, None]],
               [18, 'topy', 0, 0, [16, None]],
               [19, 'showaligned', 0, 0, [16, 20, 21]],
               [20, 'journal', 0, 0, [19, None]],
               [21, 'setxy', 0, 0, [19, 22, 23, 24]],
               [22, 'leftx', 0, 0, [21, None]],
               [23, 'bottomy', 0, 0, [21, None]],
               [24, 'showaligned', 0, 0, [21, 25, 26]],
               [25, 'journal', 0, 0, [24, None]],
               [26, 'setxy', 0, 0, [24, 27, 28, 29]],
               [27, 'rightx', 0, 0, [26, None]],
               [28, 'bottomy', 0, 0, [26, None]],
               [29, 'showaligned', 0, 0, [26, 30, 31]],
               [30, 'journal', 0, 0, [29, None]],
               [31, 'sandwichbottom', 0, 0, [29, None]]],
    'picture2x1':
              [[0, 'sandwichtop', 0, 0, [None, 1, 2]],
               [1, ['string', ' '], 0, 0, [0, None]],
               [2, 'setxy', 0, 0, [0, 3, 4, 5]],
               [3, 'titlex', 0, 0, [2, None]],
               [4, 'titley', 0, 0, [2, None]],
               [5, 'setscale', 0, 0, [2, 6, 7]],
               [6, ['number', '100'], 0, 0, [5, None]],
               [7, 'show', 0, 0, [5, 8, 9]],
               [8, ['string',' '], 0, 0, [7, None]],
               [9, 'setscale', 0, 0, [7, 10, 11]],
               [10, ['number', '35'], 0, 0, [9, None]],
               [11, 'setxy', 0, 0, [9, 12, 13, 14]],
               [12, 'leftx', 0, 0, [11, None]],
               [13, 'topy', 0, 0, [11, None]],
               [14, 'showaligned', 0, 0, [11, 15, 16]],
               [15, 'journal', 0, 0, [14, None]],
               [16, 'setxy', 0, 0, [14, 17, 18, 19]],
               [17, 'leftx', 0, 0, [16, None]],
               [18, 'bottomy', 0, 0, [16, None]],
               [19, 'showaligned', 0, 0, [16, 20, 21]],
               [20, 'description', 0, 0, [19, None]],
               [21, 'setxy', 0, 0, [19, 22, 23, 24]],
               [22, 'rightx', 0, 0, [21, None]],
               [23, 'topy', 0, 0, [21, None]],
               [24, 'showaligned', 0, 0, [21, 25, 26]],
               [25, 'journal', 0, 0, [24, None]],
               [26, 'setxy', 0, 0, [24, 27, 28, 29]],
               [27, 'rightx', 0, 0, [26, None]],
               [28, 'bottomy', 0, 0, [26, None]],
               [29, 'showaligned', 0, 0, [26, 30, 31]],
               [30, 'description', 0, 0, [29, None]],
               [31, 'sandwichbottom', 0, 0, [29, None]]],
    'picture1x2':
              [[0, 'sandwichtop', 0, 0, [None, 1, 2]],
               [1, ['string', ' '], 0, 0, [0, None]],
               [2, 'setxy', 0, 0, [0, 3, 4, 5]],
               [3, 'titlex', 0, 0, [2, None]],
               [4, 'titley', 0, 0, [2, None]],
               [5, 'setscale', 0, 0, [2, 6, 7]],
               [6, ['number', '100'], 0, 0, [5, None]],
               [7, 'show', 0, 0, [5, 8, 9]],
               [8, ['string',' '], 0, 0, [7, None]],
               [9, 'setscale', 0, 0, [7, 10, 11]],
               [10, ['number', '35'], 0, 0, [9, None]],
               [11, 'setxy', 0, 0, [9, 12, 13, 14]],
               [12, 'leftx', 0, 0, [11, None]],
               [13, 'topy', 0, 0, [11, None]],
               [14, 'showaligned', 0, 0, [11, 15, 16]],
               [15, 'journal', 0, 0, [14, None]],
               [16, 'setxy', 0, 0, [14, 17, 18, 19]],
               [17, 'rightx', 0, 0, [16, None]],
               [18, 'topy', 0, 0, [16, None]],
               [19, 'showaligned', 0, 0, [16, 20, 21]],
               [20, 'description', 0, 0, [19, None]],
               [21, 'setxy', 0, 0, [19, 22, 23, 24]],
               [22, 'leftx', 0, 0, [21, None]],
               [23, 'bottomy', 0, 0, [21, None]],
               [24, 'showaligned', 0, 0, [21, 25, 26]],
               [25, 'journal', 0, 0, [24, None]],
               [26, 'setxy', 0, 0, [24, 27, 28, 29]],
               [27, 'rightx', 0, 0, [26, None]],
               [28, 'bottomy', 0, 0, [26, None]],
               [29, 'showaligned', 0, 0, [26, 30, 31]],
               [30, 'description', 0, 0, [29, None]],
               [31, 'sandwichbottom', 0, 0, [29, None]]],
    'picture1x1':
              [[0, 'sandwichtop', 0, 0, [None, 1, 2]],
               [1, ['string', ' '], 0, 0, [0, None]],
               [2, 'setxy', 0, 0, [0, 3, 4, 5]],
               [3, 'titlex', 0, 0, [2, None]],
               [4, 'titley', 0, 0, [2, None]],
               [5, 'setscale', 0, 0, [2, 6, 7]],
               [6, ['number', '100'], 0, 0, [5, None]],
               [7, 'show', 0, 0, [5, 8, 9]],
               [8, ['string',' '], 0, 0, [7, None]],
               [9, 'setscale', 0, 0, [7, 10, 11]],
               [10, ['number', '35'], 0, 0, [9, None]],
               [11, 'setxy', 0, 0, [9, 12, 13, 14]],
               [12, 'leftx', 0, 0, [11, None]],
               [13, 'topy', 0, 0, [11, None]],
               [14, 'showaligned', 0, 0, [11, 15, 16]],
               [15, 'journal', 0, 0, [14, None]],
               [16, 'setxy', 0, 0, [14, 17, 18, 19]],
               [17, 'rightx', 0, 0, [16, None]],
               [18, 'topy', 0, 0, [16, None]],
               [19, 'showaligned', 0, 0, [16, 20, 21]],
               [20, 'description', 0, 0, [19, None]],
               [21, 'sandwichbottom', 0, 0, [19, None]]],
         }
