/*
 * Decompiled with CFR 0.152.
 */
package org.torproject.collector.onionperf;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileReader;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.net.MalformedURLException;
import java.net.URL;
import java.nio.file.Files;
import java.nio.file.StandardCopyOption;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.SortedSet;
import java.util.Stack;
import java.util.TimeZone;
import java.util.TreeSet;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.torproject.collector.conf.Configuration;
import org.torproject.collector.conf.ConfigurationException;
import org.torproject.collector.conf.Key;
import org.torproject.collector.cron.CollecTorMain;
import org.torproject.descriptor.Descriptor;
import org.torproject.descriptor.DescriptorParser;
import org.torproject.descriptor.DescriptorSourceFactory;
import org.torproject.descriptor.TorperfResult;

public class OnionperfDownloader
extends CollecTorMain {
    private static final Logger logger = LoggerFactory.getLogger(OnionperfDownloader.class);
    private static final String TORPERF = "torperf";
    private File onionPerfDownloadedFile;
    private SortedSet<String> downloadedTpfFiles = new TreeSet<String>();
    private URL[] onionPerfHosts = null;
    private File archiveDirectory = null;
    private File recentDirectory = null;
    private static final Pattern TPF_FILE_URL_PATTERN = Pattern.compile(".*<a href=\"([^\"]+\\.tpf)\">.*");
    private static final DateFormat DATE_FORMAT = new SimpleDateFormat("yyyy-MM-dd");

    public OnionperfDownloader(Configuration config) {
        super(config);
    }

    @Override
    public String module() {
        return TORPERF;
    }

    @Override
    protected String syncMarker() {
        return "TorperfFiles";
    }

    @Override
    protected void startProcessing() throws ConfigurationException {
        this.onionPerfDownloadedFile = new File(this.config.getPath(Key.StatsPath).toFile(), "onionperf-downloaded");
        this.onionPerfHosts = this.config.getUrlArray(Key.OnionPerfHosts);
        this.readDownloadedOnionPerfTpfFiles();
        this.archiveDirectory = new File(this.config.getPath(Key.OutputPath).toFile(), TORPERF);
        this.recentDirectory = new File(this.config.getPath(Key.RecentPath).toFile(), TORPERF);
        for (URL baseUrl : this.onionPerfHosts) {
            this.downloadFromOnionPerfHost(baseUrl);
        }
        this.writeDownloadedOnionPerfTpfFiles();
        this.cleanUpRsyncDirectory();
    }

    private void readDownloadedOnionPerfTpfFiles() {
        if (!this.onionPerfDownloadedFile.exists()) {
            return;
        }
        try (BufferedReader br = new BufferedReader(new FileReader(this.onionPerfDownloadedFile));){
            String line;
            while ((line = br.readLine()) != null) {
                this.downloadedTpfFiles.add(line);
            }
        }
        catch (IOException e) {
            logger.info("Unable to read download history file '" + this.onionPerfDownloadedFile.getAbsolutePath() + "'.  Ignoring download history and downloading all available .tpf files.");
            this.downloadedTpfFiles.clear();
        }
    }

    private void downloadFromOnionPerfHost(URL baseUrl) {
        logger.info("Downloading from OnionPerf host {}", (Object)baseUrl);
        List<String> tpfFileNames = this.downloadOnionPerfDirectoryListing(baseUrl);
        String source = baseUrl.getHost().split("\\.")[0];
        for (String tpfFileName : tpfFileNames) {
            this.downloadAndParseOnionPerfTpfFile(baseUrl, source, tpfFileName);
        }
    }

    private List<String> downloadOnionPerfDirectoryListing(URL baseUrl) {
        ArrayList<String> tpfFileUrls = new ArrayList<String>();
        try (BufferedReader br = new BufferedReader(new InputStreamReader(baseUrl.openStream()));){
            String line;
            while ((line = br.readLine()) != null) {
                Matcher matcher = TPF_FILE_URL_PATTERN.matcher(line);
                if (!matcher.matches() || matcher.group(1).startsWith("/")) continue;
                tpfFileUrls.add(matcher.group(1));
            }
        }
        catch (IOException e) {
            logger.warn("Unable to download directory listing from '{}'.  Skipping this OnionPerf host.", (Object)baseUrl);
            tpfFileUrls.clear();
        }
        return tpfFileUrls;
    }

    private void downloadAndParseOnionPerfTpfFile(URL baseUrl, String source, String tpfFileName) {
        byte[] rawDescriptorBytes;
        URL tpfFileUrl;
        try {
            tpfFileUrl = new URL(baseUrl, tpfFileName);
        }
        catch (MalformedURLException e1) {
            logger.warn("Unable to put together base URL '{}' and .tpf file path '{}' to a URL.  Skipping.", (Object)baseUrl, (Object)tpfFileName);
            return;
        }
        if (this.downloadedTpfFiles.contains(tpfFileUrl.toString())) {
            return;
        }
        String[] tpfFileNameParts = tpfFileName.split("-");
        if (!tpfFileName.startsWith(source + "-") || tpfFileName.length() < "s-f-yyyy-MM-dd".length() || tpfFileNameParts.length < 5) {
            logger.warn("Invalid .tpf file name '{}{}'.  Skipping.", (Object)baseUrl, (Object)tpfFileName);
            return;
        }
        int fileSize = 0;
        String date = null;
        try {
            fileSize = Integer.parseInt(tpfFileNameParts[tpfFileNameParts.length - 4]);
            date = tpfFileName.substring(tpfFileName.length() - 14, tpfFileName.length() - 4);
            DATE_FORMAT.parse(date);
        }
        catch (NumberFormatException | ParseException e) {
            logger.warn("Invalid .tpf file name '{}{}'.  Skipping.", baseUrl, tpfFileName, e);
            return;
        }
        File tempFile = new File(this.recentDirectory, "." + tpfFileName);
        tempFile.getParentFile().mkdirs();
        try (InputStream is = new URL(baseUrl + tpfFileName).openStream();){
            Files.copy(is, tempFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
        }
        catch (IOException e) {
            logger.warn("Unable to download '{}{}' to temporary file '{}'.  Skipping.", baseUrl, tpfFileName, tempFile, e);
            return;
        }
        DescriptorParser descriptorParser = DescriptorSourceFactory.createDescriptorParser();
        try {
            rawDescriptorBytes = Files.readAllBytes(tempFile.toPath());
        }
        catch (IOException e) {
            logger.warn("OnionPerf file '{}{}' could not be read.  Skipping.", baseUrl, tpfFileName, e);
            tempFile.delete();
            return;
        }
        Iterable<Descriptor> descriptors = descriptorParser.parseDescriptors(rawDescriptorBytes, null, tpfFileName);
        String message = null;
        for (Descriptor descriptor : descriptors) {
            if (!(descriptor instanceof TorperfResult)) {
                message = "File contains descriptors other than Torperf results.";
                break;
            }
            TorperfResult torperf = (TorperfResult)descriptor;
            if (!source.equals(torperf.getSource())) {
                message = "File contains Torperf result from another source.";
                break;
            }
            if (fileSize != torperf.getFileSize()) {
                message = "File contains Torperf result from another file size.";
                break;
            }
            if (date.equals(DATE_FORMAT.format(torperf.getStartMillis()))) continue;
            message = "File contains Torperf result from another date.";
            break;
        }
        if (null != message) {
            logger.warn("OnionPerf file '{}{}' was found to be invalid: {}.  Skipping.", baseUrl, tpfFileName, message);
            tempFile.delete();
            return;
        }
        File archiveFile = new File(this.archiveDirectory, date.replaceAll("-", "/") + "/" + tpfFileName);
        archiveFile.getParentFile().mkdirs();
        try {
            Files.copy(tempFile.toPath(), archiveFile.toPath(), StandardCopyOption.REPLACE_EXISTING);
        }
        catch (IOException e) {
            logger.warn("Unable to copy OnionPerf file {} to {}.  Skipping.", tempFile, archiveFile, e);
            tempFile.delete();
            return;
        }
        File recentFile = new File(this.recentDirectory, tpfFileName);
        tempFile.renameTo(recentFile);
        this.downloadedTpfFiles.add(baseUrl + tpfFileName);
    }

    private void writeDownloadedOnionPerfTpfFiles() {
        this.onionPerfDownloadedFile.getParentFile().mkdirs();
        try (BufferedWriter bw = new BufferedWriter(new FileWriter(this.onionPerfDownloadedFile));){
            for (String line : this.downloadedTpfFiles) {
                bw.write(line);
                bw.newLine();
            }
        }
        catch (IOException e) {
            logger.warn("Unable to write download history file '{}'.  This may result in ignoring history and downloading all available .tpf files in the next execution.", (Object)this.onionPerfDownloadedFile.getAbsolutePath(), (Object)e);
        }
    }

    public void cleanUpRsyncDirectory() throws ConfigurationException {
        long cutOffMillis = System.currentTimeMillis() - 259200000L;
        Stack<File> allFiles = new Stack<File>();
        allFiles.add(new File(this.config.getPath(Key.RecentPath).toFile(), TORPERF));
        while (!allFiles.isEmpty()) {
            File file = (File)allFiles.pop();
            if (file.isDirectory()) {
                allFiles.addAll(Arrays.asList(file.listFiles()));
                continue;
            }
            if (file.lastModified() >= cutOffMillis) continue;
            file.delete();
        }
    }

    static {
        DATE_FORMAT.setLenient(false);
        DATE_FORMAT.setTimeZone(TimeZone.getTimeZone("UTC"));
    }
}

