/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

"use strict";

const xpcshellTestConfig = require("eslint-plugin-mozilla/lib/configs/xpcshell-test.js");
const browserTestConfig = require("eslint-plugin-mozilla/lib/configs/browser-test.js");
const mochitestTestConfig = require("eslint-plugin-mozilla/lib/configs/mochitest-test.js");
const chromeTestConfig = require("eslint-plugin-mozilla/lib/configs/chrome-test.js");
const { testPaths } = require("./.eslintrc-test-paths.js");
const fs = require("fs");
const path = require("path");

/**
 * Some configurations have overrides, which can't be specified within overrides,
 * so we need to remove them.
 */
function removeOverrides(config) {
  config = { ...config };
  delete config.overrides;
  return config;
}

function readFile(path) {
  return fs
    .readFileSync(path, { encoding: "utf-8" })
    .split("\n")
    .filter(p => p && !p.startsWith("#"));
}

const ignorePatterns = [
  ...readFile(
    path.join(__dirname, "tools", "rewriting", "ThirdPartyPaths.txt")
  ),
  ...readFile(path.join(__dirname, "tools", "rewriting", "Generated.txt")),
  ...readFile(
    path.join(
      __dirname,
      "devtools",
      "client",
      "debugger",
      "src",
      ".eslintignore"
    )
  ).map(p => `devtools/client/debugger/src/${p}`),
];
const httpTestingPaths = [
  "**/*mixedcontent",
  "**/*CrossOrigin",
  "**/*crossorigin",
  "**/*cors",
  "**/*downgrade",
  "**/*Downgrade",
];

module.exports = {
  parser: "@babel/eslint-parser",
  parserOptions: {
    sourceType: "script",
    babelOptions: {
      configFile: path.join(__dirname, ".babel-eslint.rc.js"),
    },
  },
  settings: {
    "import/extensions": [".mjs"],
  },
  ignorePatterns,
  // Ignore eslint configurations in parent directories.
  root: true,
  // New rules and configurations should generally be added in
  // tools/lint/eslint/eslint-plugin-mozilla/lib/configs/recommended.js to
  // allow external repositories that use the plugin to pick them up as well.
  extends: ["plugin:mozilla/recommended"],
  plugins: ["mozilla", "import"],
  overrides: [
    {
      // All .eslintrc.js files are in the node environment, so turn that
      // on here.
      // https://github.com/eslint/eslint/issues/13008
      files: [".eslintrc.js"],
      env: {
        node: true,
        browser: false,
      },
    },
    {
      files: ["browser/components/storybook/**"],
      env: {
        "mozilla/privileged": false,
      },
    },
    {
      files: ["*.mjs"],
      rules: {
        "import/default": "error",
        "import/export": "error",
        "import/named": "error",
        "import/namespace": "error",
        "import/newline-after-import": "error",
        "import/no-duplicates": "error",
        "import/no-absolute-path": "error",
        "import/no-named-default": "error",
        "import/no-named-as-default": "error",
        "import/no-named-as-default-member": "error",
        "import/no-self-import": "error",
        "import/no-unassigned-import": "error",
        "import/no-unresolved": [
          "error",
          // Bug 1773473 - Ignore resolver URLs for chrome and resource as we
          // do not yet have a resolver for them.
          { ignore: ["chrome://", "resource://"] },
        ],
        "import/no-useless-path-segments": "error",
      },
    },
    {
      files: [
        // Bug 1773475 - For now, turn off no-unresolved on some paths where we import
        // from node_modules, as the ESLint setup only installs modules at the
        // top-level.
        "devtools/shared/compatibility/**",
        "browser/components/storybook/**",
      ],
      rules: {
        "import/no-unresolved": "off",
      },
    },
    {
      files: ["*.html", "*.xhtml", "*.xml"],
      rules: {
        // Curly brackets are required for all the tree via recommended.js,
        // however these files aren't auto-fixable at the moment.
        curly: "off",
      },
    },
    {
      // TODO: Bug 1515949. Enable no-undef for gfx/
      files: "gfx/layers/apz/test/mochitest/**",
      rules: {
        "no-undef": "off",
      },
    },
    {
      ...removeOverrides(xpcshellTestConfig),
      files: testPaths.xpcshell.map(path => `${path}**`),
      excludedFiles: ["**/*.jsm", "**/*.mjs"],
    },
    {
      // If it is an xpcshell head file, we turn off global unused variable checks, as it
      // would require searching the other test files to know if they are used or not.
      // This would be expensive and slow, and it isn't worth it for head files.
      // We could get developers to declare as exported, but that doesn't seem worth it.
      files: testPaths.xpcshell.map(path => `${path}head*.js`),
      rules: {
        "no-unused-vars": [
          "error",
          {
            args: "none",
            vars: "local",
          },
        ],
      },
    },
    {
      // This section enables errors of no-unused-vars globally for all test*.js
      // files in xpcshell test paths.
      // This is not done in the xpcshell-test configuration as we cannot pull
      // in overrides from there. We should at some stage, aim to enable this
      // for all files in xpcshell-tests.
      files: testPaths.xpcshell.map(path => `${path}test*.js`),
      rules: {
        // No declaring variables that are never used
        "no-unused-vars": [
          "error",
          {
            args: "none",
            vars: "all",
          },
        ],
      },
    },
    {
      ...browserTestConfig,
      files: testPaths.browser.map(path => `${path}**`),
      excludedFiles: ["**/*.jsm", "**/*.mjs"],
    },
    {
      ...removeOverrides(mochitestTestConfig),
      files: testPaths.mochitest.map(path => `${path}**`),
      excludedFiles: [
        "**/*.jsm",
        "**/*.mjs",
        "security/manager/ssl/tests/mochitest/browser/**",
      ],
    },
    {
      ...removeOverrides(chromeTestConfig),
      files: testPaths.chrome.map(path => `${path}**`),
      excludedFiles: ["**/*.jsm", "**/*.mjs"],
    },
    {
      env: {
        // Ideally we wouldn't be using the simpletest env here, but our uses of
        // js files mean we pick up everything from the global scope, which could
        // be any one of a number of html files. So we just allow the basics...
        "mozilla/simpletest": true,
      },
      files: [
        ...testPaths.mochitest.map(path => `${path}/**/*.js`),
        ...testPaths.chrome.map(path => `${path}/**/*.js`),
      ],
    },
    {
      files: ["netwerk/test/mochitests/**", "netwerk/test/unit*/**"],
      rules: {
        "no-shadow": "warn",
      },
    },
    {
      // Bug 877389 - Gradually migrate from Cu.reportError to console.error.
      // Enable these as we fix more areas.
      files: [
        "dom/push/test/mockpushserviceparent.js",
        "browser/components/extensions/**",
        "toolkit/components/extensions/**",
        "toolkit/mozapps/extensions/**",
      ],
      rules: {
        "mozilla/no-cu-reportError": "off",
      },
    },
    {
      files: ["**"],
      excludedFiles: [
        "accessible/tests/**",
        "browser/actors/**",
        "browser/base/content/**",
        "browser/components/Browser*",
        "browser/components/aboutlogins/**",
        "browser/components/attribution/**",
        "browser/components/customizableui/**",
        "browser/components/downloads/**",
        "browser/components/enterprisepolicies/**",
        "browser/components/firefoxview/tests/browser/head.js",
        "browser/components/firefoxview/tests/browser/FirefoxViewTestUtils.sys.mjs",
        "browser/components/ion/content/ion.js",
        "browser/components/newtab/**",
        "browser/components/originattributes/test/browser/**",
        "browser/components/pocket/content/pkt*",
        "browser/components/preferences/**",
        "browser/components/privatebrowsing/**",
        "browser/components/prompts/PromptCollection.sys.mjs",
        "browser/components/protections/content/*-card.mjs",
        "browser/components/protocolhandler/WebProtocolHandlerRegistrar.sys.mjs",
        "browser/components/safebrowsing/content/test/**",
        "browser/components/screenshots/**",
        "browser/components/sessionstore/**",
        "browser/components/storybook/.storybook/**",
        "browser/components/syncedtabs/**",
        "browser/components/tests/browser/**",
        "browser/components/textrecognition/textrecognition.mjs",
        "browser/components/touchbar/tests/browser/browser_touchbar_searchrestrictions.js",
        "browser/components/translation/**",
        "browser/components/uitour/UITour*",
        "browser/extensions/pictureinpicture/**",
        "browser/extensions/screenshots/**",
        "browser/modules/**",
        "browser/themes/BuiltInThemes.sys.mjs",
        "browser/tools/mozscreenshots/**",
        "devtools/**",
        "docshell/base/URIFixup.sys.mjs",
        "dom/**",
        "editor/**",
        "extensions/permissions/test/PermissionTestUtils.sys.mjs",
        "gfx/layers/apz/test/mochitest/apz_test_native_event_utils.js",
        "image/test/**",
        "intl/**",
        "js/xpconnect/**",
        "layout/**",
        "mobile/android/**",
        "modules/**",
        "netwerk/**",
        "parser/htmlparser/**",
        "services/**",
        "storage/**",
        "testing/**",
        "toolkit/actors/**",
        "toolkit/components/aboutmemory/**",
        "toolkit/components/aboutprocesses/content/aboutProcesses.js",
        "toolkit/components/antitracking/**",
        "toolkit/components/asyncshutdown/**",
        "toolkit/components/backgroundtasks/BackgroundTasks*.sys.mjs",
        "toolkit/components/cleardata/**",
        "toolkit/components/contentprefs/ContentPrefService2.sys.mjs",
        "toolkit/components/cookiebanners/**",
        "toolkit/components/corroborator/Corroborate.sys.mjs",
        "toolkit/components/crashes/**",
        "toolkit/components/crashmonitor/CrashMonitor.sys.mjs",
        "toolkit/components/credentialmanagement/IdentityCredentialPromptService.sys.mjs",
        "toolkit/components/downloads/**",
        "toolkit/components/featuregates/FeatureGate*.*",
        "toolkit/components/forgetaboutsite/test/unit/test_removeDataFromDomain.js",
        "toolkit/components/glean/tests/browser/**",
        "toolkit/components/kvstore/kvstore.sys.mjs",
        "toolkit/components/lz4/lz4.js",
        "toolkit/components/messaging-system/**",
        "toolkit/components/mozintl/mozIntl.sys.mjs",
        "toolkit/components/narrate/Narrator.jsm",
        "toolkit/components/nimbus/**",
        "toolkit/components/normandy/**",
        "toolkit/components/passwordmgr/**",
        "toolkit/components/pdfjs/**",
        "toolkit/components/pictureinpicture/**",
        "toolkit/components/places/**",
        "toolkit/components/printing/content/print*.*",
        "toolkit/components/processtools/tests/browser/browser_test_powerMetrics.js",
        "toolkit/components/promiseworker/**/PromiseWorker.*",
        "toolkit/components/prompts/**",
        "toolkit/components/reader/**",
        "toolkit/components/reputationservice/test/unit/test_app_rep_windows.js",
        "toolkit/components/taskscheduler/TaskScheduler*.*",
        "toolkit/components/telemetry/**",
        "toolkit/components/thumbnails/**",
        "toolkit/components/timermanager/UpdateTimerManager.*",
        "toolkit/components/translation/LanguageDetector.*",
        "toolkit/components/translations/**",
        "toolkit/components/url-classifier/**",
        "toolkit/components/utils/**",
        "toolkit/components/viewsource/**",
        "toolkit/components/windowwatcher/**",
        "toolkit/components/workerloader/require.js",
        "toolkit/content/**",
        "toolkit/crashreporter/**",
        "toolkit/modules/**",
        "toolkit/mozapps/downloads/**",
        "toolkit/mozapps/extensions/**",
        "toolkit/mozapps/handling/**",
        "toolkit/mozapps/update/**",
        "toolkit/profile/test/test_create_profile.xhtml",
        "tools/code-coverage/tests/mochitest/test_coverage_specialpowers.html",
        "tools/lint/eslint/**",
        "tools/profiler/tests/**",
        "uriloader/**",
        "widget/tests/window_composition_text_querycontent.xhtml",
      ],
      extends: ["plugin:mozilla/valid-jsdoc"],
    },
    {
      files: ["layout/**"],
      rules: {
        "object-shorthand": "off",
        "mozilla/avoid-removeChild": "off",
        "mozilla/consistent-if-bracing": "off",
        "mozilla/reject-importGlobalProperties": "off",
        "mozilla/no-arbitrary-setTimeout": "off",
        "mozilla/no-define-cc-etc": "off",
        "mozilla/prefer-boolean-length-check": "off",
        "mozilla/use-chromeutils-generateqi": "off",
        "mozilla/use-default-preference-values": "off",
        "mozilla/use-includes-instead-of-indexOf": "off",
        "mozilla/use-services": "off",
        "mozilla/use-ownerGlobal": "off",
        complexity: "off",
        "consistent-return": "off",
        "no-array-constructor": "off",
        "no-caller": "off",
        "no-cond-assign": "off",
        "no-extra-boolean-cast": "off",
        "no-eval": "off",
        "no-func-assign": "off",
        "no-global-assign": "off",
        "no-implied-eval": "off",
        "no-lonely-if": "off",
        "no-nested-ternary": "off",
        "no-new-wrappers": "off",
        "no-redeclare": "off",
        "no-restricted-globals": "off",
        "no-return-await": "off",
        "no-sequences": "off",
        "no-throw-literal": "off",
        "no-useless-concat": "off",
        "no-undef": "off",
        "no-unreachable": "off",
        "no-unsanitized/method": "off",
        "no-unsanitized/property": "off",
        "no-unsafe-negation": "off",
        "no-unused-vars": "off",
        "no-useless-return": "off",
      },
    },
    {
      files: [
        "dom/animation/test/**",
        "dom/base/test/*.*",
        "dom/base/test/unit/test_serializers_entities*.js",
        "dom/base/test/unit_ipc/**",
        "dom/base/test/jsmodules/**",
        "dom/canvas/test/**",
        "dom/encoding/test/**",
        "dom/events/test/**",
        "dom/fetch/tests/**",
        "dom/file/ipc/tests/**",
        "dom/file/tests/**",
        "dom/html/test/**",
        "dom/jsurl/test/**",
        "dom/media/tests/**",
        "dom/media/webaudio/test/**",
        "dom/media/webrtc/tests/**",
        "dom/media/webspeech/recognition/test/**",
        "dom/media/webspeech/synth/test/**",
        "dom/messagechannel/tests/**",
        "dom/midi/tests/**",
        "dom/network/tests/**",
        "dom/payments/test/**",
        "dom/performance/tests/**",
        "dom/quota/test/browser/**",
        "dom/quota/test/common/**",
        "dom/quota/test/mochitest/**",
        "dom/quota/test/xpcshell/**",
        "dom/security/test/cors/**",
        "dom/security/test/csp/**",
        "dom/security/test/mixedcontentblocker/**",
        "dom/serviceworkers/test/**",
        "dom/smil/test/**",
        "dom/tests/mochitest/**",
        "dom/u2f/tests/**",
        "dom/vr/test/**",
        "dom/webauthn/tests/**",
        "dom/webgpu/mochitest/**",
        "dom/websocket/tests/**",
        "dom/workers/test/**",
        "dom/worklet/tests/**",
        "dom/xml/test/**",
        "dom/xslt/tests/**",
        "dom/xul/test/**",
      ],
      rules: {
        "consistent-return": "off",
        "mozilla/avoid-removeChild": "off",
        "mozilla/consistent-if-bracing": "off",
        "mozilla/no-arbitrary-setTimeout": "off",
        "mozilla/no-compare-against-boolean-literals": "off",
        "mozilla/no-define-cc-etc": "off",
        "mozilla/reject-importGlobalProperties": "off",
        "mozilla/use-cc-etc": "off",
        "mozilla/use-chromeutils-generateqi": "off",
        "mozilla/use-includes-instead-of-indexOf": "off",
        "mozilla/use-ownerGlobal": "off",
        "mozilla/use-services": "off",
        "no-array-constructor": "off",
        "no-caller": "off",
        "no-cond-assign": "off",
        "no-control-regex": "off",
        "no-debugger": "off",
        "no-else-return": "off",
        "no-empty": "off",
        "no-eval": "off",
        "no-func-assign": "off",
        "no-global-assign": "off",
        "no-implied-eval": "off",
        "no-lone-blocks": "off",
        "no-lonely-if": "off",
        "no-nested-ternary": "off",
        "no-new-object": "off",
        "no-new-wrappers": "off",
        "no-redeclare": "off",
        "no-return-await": "off",
        "no-restricted-globals": "off",
        "no-self-assign": "off",
        "no-self-compare": "off",
        "no-sequences": "off",
        "no-shadow": "off",
        "no-shadow-restricted-names": "off",
        "no-sparse-arrays": "off",
        "no-throw-literal": "off",
        "no-unreachable": "off",
        "no-unsanitized/method": "off",
        "no-unsanitized/property": "off",
        "no-useless-call": "off",
        "no-useless-concat": "off",
        "no-useless-return": "off",
      },
    },
    {
      // Bug 1792693: Gradually enable no-undef and no-unused-vars on these
      // directories.
      files: [
        "dom/animation/test/**",
        "dom/base/test/*.*",
        "dom/base/test/unit/test_serializers_entities*.js",
        "dom/base/test/unit_ipc/**",
        "dom/base/test/jsmodules/**",
        "dom/canvas/test/**",
        "dom/encoding/test/**",
        "dom/events/test/**",
        "dom/file/ipc/tests/**",
        "dom/file/tests/**",
        "dom/html/test/**",
        "dom/media/tests/**",
        "dom/media/webaudio/test/**",
        "dom/media/webrtc/tests/**",
        "dom/media/webspeech/recognition/test/**",
        "dom/media/webspeech/synth/test/**",
        "dom/payments/test/**",
        "dom/performance/tests/**",
        "dom/quota/test/browser/**",
        "dom/quota/test/common/**",
        "dom/quota/test/mochitest/**",
        "dom/quota/test/xpcshell/**",
        "dom/security/test/cors/**",
        "dom/security/test/csp/**",
        "dom/security/test/mixedcontentblocker/**",
        "dom/serviceworkers/test/**",
        "dom/smil/test/**",
        "dom/tests/mochitest/**",
        "dom/u2f/tests/**",
        "dom/vr/test/**",
        "dom/webauthn/tests/**",
        "dom/webgpu/mochitest/**",
        "dom/websocket/tests/**",
        "dom/workers/test/**",
        "dom/worklet/tests/**",
        "dom/xslt/tests/**",
      ],
      rules: {
        "no-undef": "off",
        "no-unused-vars": "off",
      },
    },
    {
      files: [
        "dom/base/test/chrome/file_bug1139964.xhtml",
        "dom/base/test/chrome/file_bug549682.xhtml",
        "dom/base/test/chrome/file_bug616841.xhtml",
        "dom/base/test/chrome/file_bug990812-1.xhtml",
        "dom/base/test/chrome/file_bug990812-2.xhtml",
        "dom/base/test/chrome/file_bug990812-3.xhtml",
        "dom/base/test/chrome/file_bug990812-4.xhtml",
        "dom/base/test/chrome/file_bug990812-5.xhtml",
        "dom/base/test/chrome/file_bug990812.xhtml",
        "dom/base/test/chrome/test_bug1098074_throw_from_ReceiveMessage.xhtml",
        "dom/base/test/chrome/test_bug339494.xhtml",
        "dom/base/test/chrome/test_bug429785.xhtml",
        "dom/base/test/chrome/test_bug467123.xhtml",
        "dom/base/test/chrome/test_bug683852.xhtml",
        "dom/base/test/chrome/test_bug780529.xhtml",
        "dom/base/test/chrome/test_bug800386.xhtml",
        "dom/base/test/chrome/test_bug884693.xhtml",
        "dom/base/test/chrome/test_document-element-inserted.xhtml",
        "dom/base/test/chrome/test_domparsing.xhtml",
        "dom/base/test/chrome/title_window.xhtml",
        "dom/base/test/chrome/window_nsITextInputProcessor.xhtml",
        "dom/base/test/chrome/window_swapFrameLoaders.xhtml",
        "dom/base/test/test_domrequesthelper.xhtml",
        "dom/bindings/test/test_bug1123516_maplikesetlikechrome.xhtml",
        "dom/console/tests/test_jsm.xhtml",
        "dom/events/test/test_bug1412775.xhtml",
        "dom/events/test/test_bug336682_2.xhtml",
        "dom/events/test/test_bug415498.xhtml",
        "dom/events/test/test_bug602962.xhtml",
        "dom/events/test/test_bug617528.xhtml",
        "dom/events/test/test_bug679494.xhtml",
        "dom/indexedDB/test/test_globalObjects_chrome.xhtml",
        "dom/indexedDB/test/test_wrappedArray.xhtml",
        "dom/ipc/tests/test_process_error.xhtml",
        "dom/notification/test/chrome/test_notification_system_principal.xhtml",
        "dom/security/test/general/test_bug1277803.xhtml",
        "dom/serviceworkers/test/test_serviceworkerinfo.xhtml",
        "dom/serviceworkers/test/test_serviceworkermanager.xhtml",
        "dom/system/tests/test_constants.xhtml",
        "dom/tests/mochitest/chrome/DOMWindowCreated_chrome.xhtml",
        "dom/tests/mochitest/chrome/MozDomFullscreen_chrome.xhtml",
        "dom/tests/mochitest/chrome/sizemode_attribute.xhtml",
        "dom/tests/mochitest/chrome/test_cyclecollector.xhtml",
        "dom/tests/mochitest/chrome/test_docshell_swap.xhtml",
        "dom/tests/mochitest/chrome/window_focus.xhtml",
        "dom/url/tests/test_bug883784.xhtml",
        "dom/workers/test/test_WorkerDebugger.xhtml",
        "dom/workers/test/test_WorkerDebugger_console.xhtml",
        "dom/workers/test/test_fileReadSlice.xhtml",
        "dom/workers/test/test_fileReaderSync.xhtml",
        "dom/workers/test/test_fileSlice.xhtml",
      ],
      rules: {
        "mozilla/no-useless-parameters": "off",
        "mozilla/no-useless-removeEventListener": "off",
        "mozilla/use-chromeutils-generateqi": "off",
        "mozilla/use-services": "off",
        complexity: "off",
        "no-array-constructor": "off",
        "no-caller": "off",
        "no-empty": "off",
        "no-eval": "off",
        "no-lone-blocks": "off",
        "no-redeclare": "off",
        "no-shadow": "off",
        "no-throw-literal": "off",
        "no-unsanitized/method": "off",
        "no-useless-return": "off",
        "object-shorthand": "off",
      },
    },
    {
      // Rules of Hooks broadly checks for camelCase "use" identifiers, so
      // enable only for paths actually using React to avoid false positives.
      extends: ["plugin:react-hooks/recommended"],
      files: [
        "browser/components/newtab/**",
        "browser/components/pocket/**",
        "devtools/**",
      ],
    },
    {
      // Exempt files with these paths since they have to use http for full coverage
      files: httpTestingPaths.map(path => `${path}**`),
      rules: {
        "@microsoft/sdl/no-insecure-url": "off",
      },
    },
    {
      // Exempt all components and test files that explicitly want to test http urls from 'no-insecure-url' rule.
      // Gradually change test cases such that this list gets smaller and more precisely. Bug 1758951
      files: [
        // Re-exempt all xpcshell-test files - some configurations cause rules to be re-applied, so we override them again here.
        "test*/unit*/**",
        "**/test*/unit/**",
        "**/test*/xpcshell/**",
        "test*/*/xpcshell/**",
        "testing/profiles/unittest-required/user.js",
        "testing/profiles/xpcshell/user.js",
        // Xpcshell test that aren't fitting the pattern Bug 1758951
        "toolkit/components/places/tests/bookmarks/**",
        "toolkit/components/places/tests/expiration/**",
        "toolkit/components/places/tests/favicons/**",
        "toolkit/components/places/tests/history/**",
        "toolkit/components/places/tests/legacy/**",
        "toolkit/components/places/tests/maintenance/**",
        "toolkit/components/places/tests/migration/**",
        "toolkit/components/places/tests/queries/**",
        "toolkit/components/places/tests/sync/**",
        // Exempt tests that set https_first_disable to true Bug 1758951
        "browser/components/downloads/test/browser/browser_image_mimetype_issues.js",
        "browser/components/enterprisepolicies/tests/browser/browser_policy_cookie_settings.js",
        "browser/components/enterprisepolicies/tests/browser/browser_policy_extensionsettings.js",
        "devtools/server/tests/browser/browser_storage_listings.js",
        "devtools/server/tests/browser/browser_storage_updates.js",
        "image/test/browser/browser_offscreen_image_in_out_of_process_iframe.js",
        "security/manager/ssl/tests/mochitest/browser/browser_HSTS.js",
        "testing/mochitest/tests/browser/browser_browserLoaded_content_loaded.js",
        "toolkit/components/extensions/test/browser/browser_ext_downloads_referrer.js",
        "toolkit/components/messaging-system/schemas/TriggerActionSchemas/test/browser/browser_asrouter_trigger_listeners.js",
        "toolkit/components/normandy/test/browser/browser_about_studies.js",
        "toolkit/components/remotebrowserutils/tests/browser/browser_RemoteWebNavigation.js",
        "toolkit/components/remotepagemanager/tests/browser/browser_RemotePageManager.js",
        "toolkit/components/viewsource/test/browser/browser_bug464222.js",
        "toolkit/components/viewsource/test/browser/browser_viewsource_newwindow.js",
        "toolkit/content/tests/browser/browser_label_textlink.js",
        "toolkit/crashreporter/test/browser/browser_aboutCrashesResubmit.js",
        // Exempt tests that set pref dom.security.https_first to false Bug 1758951
        "browser/tools/mozscreenshots/controlCenter/browser_controlCenter.js",
        "dom/file/tests/test_agentcluster_bloburl.js",
        "dom/ipc/tests/test_bug1086684.js",
        "dom/manifest/test/browser_ManifestObtainer_obtain.js",
        "dom/media/test/test_access_control.html",
        "dom/websocket/tests/test_bug1384658.html",
        "testing/mochitest/browser-test.js",
        "devtools/client/shared/components/test/node/components/reps/string-with-url.test.js", // no test case
        // Exempt files that have comment which says "not [to] enforce https"
        // https://searchfox.org/mozilla-central/search?q=%2F%2F+We+should+not+enforce+https+for+tests+using+this+page.&path=
        "devtools/client/storage/test/browser_storage_basic_usercontextid_1.js",
        "devtools/client/storage/test/browser_storage_basic_usercontextid_2.js",
        "devtools/client/storage/test/browser_storage_basic_with_fragment.js",
        "devtools/client/storage/test/browser_storage_cookies_delete_all.js",
        "devtools/client/storage/test/browser_storage_delete.js",
        "devtools/client/storage/test/browser_storage_delete_all.js",
        "devtools/client/storage/test/browser_storage_delete_tree.js",
        "devtools/client/storage/test/browser_storage_delete_usercontextid.js",
        "devtools/client/storage/test/browser_storage_dom_cache_disabled.js",
        "devtools/client/storage/test/browser_storage_sidebar.js",
        "devtools/client/storage/test/browser_storage_sidebar_toggle.js",
        "devtools/client/storage/test/head.js",
        "devtools/client/storage/test/storage-complex-values.html",
        "devtools/client/storage/test/storage-sidebar-parsetree.html",
        // No .(s)js or html test files Bug 1780024
        "services/sync/tests/tps/**",
        // Files that test https and http already
        "browser/base/content/test/general/browser_remoteTroubleshoot.js",
        "browser/base/content/test/favicons/browser_mixed_content.js",
        "browser/base/content/test/sanitize/browser_sanitize-history.js",
        "devtools/client/shared/test/shared-head.js",
        "devtools/client/shared/test/head.js",
        "browser/tools/mozscreenshots/mozscreenshots/extension/configurations/ControlCenter.sys.mjs",
        // uses new HttpServer which doesn't support https
        "browser/base/content/test/favicons/browser_favicon_nostore.js",
        // That are all files that produes warnings in the existing test infrastructure.
        // Since our focus is that new tests won't use http without thinking twice we exempt
        // these test files for now. Bug 1758951
        "browser/components/aboutlogins/tests/browser/browser_createLogin.js",
        "browser/components/aboutlogins/tests/chrome/test_login_item.html",
        "browser/components/contextualidentity/test/browser/browser_broadcastchannel.js",
        "browser/components/contextualidentity/test/browser/browser_eme.js",
        "browser/components/contextualidentity/test/browser/browser_favicon.js",
        "browser/components/contextualidentity/test/browser/browser_forgetAPI_EME_forgetThisSite.js",
        "browser/components/contextualidentity/test/browser/browser_forgetAPI_cookie_getCookiesWithOriginAttributes.js",
        "browser/components/contextualidentity/test/browser/browser_forgetAPI_quota_clearStoragesForPrincipal.js",
        "browser/components/contextualidentity/test/browser/browser_forgetaboutsite.js",
        "browser/components/contextualidentity/test/browser/browser_middleClick.js",
        "browser/components/contextualidentity/test/browser/browser_reopenIn.js",
        "browser/components/contextualidentity/test/browser/browser_restore_getCookiesWithOriginAttributes.js",
        "browser/components/contextualidentity/test/browser/browser_usercontextid_tabdrop.js",
        "browser/components/contextualidentity/test/browser/saveLink.sjs",
        "browser/components/customizableui/test/browser_947914_button_history.js",
        "browser/components/customizableui/test/browser_947914_button_print.js",
        "browser/components/customizableui/test/browser_947914_button_zoomReset.js",
        "browser/components/customizableui/test/browser_backfwd_enabled_post_customize.js",
        "browser/components/customizableui/test/browser_customization_context_menus.js",
        "browser/components/customizableui/test/browser_exit_background_customize_mode.js",
        "browser/components/customizableui/test/browser_history_recently_closed.js",
        "browser/components/customizableui/test/browser_history_recently_closed_middleclick.js",
        "browser/components/customizableui/test/browser_synced_tabs_menu.js",
        "browser/components/downloads/test/browser/browser_download_overwrite.js",
        "browser/components/downloads/test/browser/browser_go_to_download_page.js",
        "browser/components/downloads/test/browser/browser_iframe_gone_mid_download.js",
        "browser/components/downloads/test/browser/head.js",
        "browser/components/enterprisepolicies/tests/browser/browser_policy_disable_feedback_commands.js",
        "browser/components/enterprisepolicies/tests/browser/browser_policy_handlers.js",
        "browser/components/enterprisepolicies/tests/browser/browser_policy_search_engine.js",
        "browser/components/enterprisepolicies/tests/browser/browser_policy_set_homepage.js",
        "browser/components/enterprisepolicies/tests/browser/browser_policy_set_startpage.js",
        "browser/components/extensions/test/browser/browser_ext_autocompletepopup.js",
        "browser/components/extensions/test/browser/browser_ext_browserAction_context.js",
        "browser/components/extensions/test/browser/browser_ext_browserAction_contextMenu.js",
        "browser/components/extensions/test/browser/browser_ext_browserAction_popup_preload.js",
        "browser/components/extensions/test/browser/browser_ext_browsingData_history.js",
        "browser/components/extensions/test/browser/browser_ext_chrome_settings_overrides_home.js",
        "browser/components/extensions/test/browser/browser_ext_commands_execute_page_action.js",
        "browser/components/extensions/test/browser/browser_ext_contentscript_cross_docGroup_adoption.js",
        "browser/components/extensions/test/browser/browser_ext_contentscript_cross_docGroup_adoption_xhr.js",
        "browser/components/extensions/test/browser/browser_ext_contextMenus_targetUrlPatterns.js",
        "browser/components/extensions/test/browser/browser_ext_contextMenus_urlPatterns.js",
        "browser/components/extensions/test/browser/browser_ext_devtools_inspectedWindow_targetSwitch.js",
        "browser/components/extensions/test/browser/browser_ext_devtools_network.js",
        "browser/components/extensions/test/browser/browser_ext_devtools_network_targetSwitch.js",
        "browser/components/extensions/test/browser/browser_ext_find.js",
        "browser/components/extensions/test/browser/browser_ext_management.js",
        "browser/components/extensions/test/browser/browser_ext_menus.js",
        "browser/components/extensions/test/browser/browser_ext_menus_replace_menu_context.js",
        "browser/components/extensions/test/browser/browser_ext_menus_targetElement_shadow.js",
        "browser/components/extensions/test/browser/browser_ext_nontab_process_switch.js",
        "browser/components/extensions/test/browser/browser_ext_optionsPage_privileges.js",
        "browser/components/extensions/test/browser/browser_ext_originControls.js",
        "browser/components/extensions/test/browser/browser_ext_originControls_browserAction.js",
        "browser/components/extensions/test/browser/browser_ext_pageAction_context.js",
        "browser/components/extensions/test/browser/browser_ext_pageAction_popup.js",
        "browser/components/extensions/test/browser/browser_ext_pageAction_show_matches.js",
        "browser/components/extensions/test/browser/browser_ext_pageAction_simple.js",
        "browser/components/extensions/test/browser/browser_ext_popup_api_injection.js",
        "browser/components/extensions/test/browser/browser_ext_popup_focus.js",
        "browser/components/extensions/test/browser/browser_ext_port_disconnect_on_crash.js",
        "browser/components/extensions/test/browser/browser_ext_runtime_openOptionsPage.js",
        "browser/components/extensions/test/browser/browser_ext_runtime_openOptionsPage_uninstall.js",
        "browser/components/extensions/test/browser/browser_ext_runtime_setUninstallURL.js",
        "browser/components/extensions/test/browser/browser_ext_search_favicon.js",
        "browser/components/extensions/test/browser/browser_ext_sessions_forgetClosedTab.js",
        "browser/components/extensions/test/browser/browser_ext_sessions_forgetClosedWindow.js",
        "browser/components/extensions/test/browser/browser_ext_sessions_getRecentlyClosed.js",
        "browser/components/extensions/test/browser/browser_ext_sessions_getRecentlyClosed_private.js",
        "browser/components/extensions/test/browser/browser_ext_sessions_getRecentlyClosed_tabs.js",
        "browser/components/extensions/test/browser/browser_ext_sessions_incognito.js",
        "browser/components/extensions/test/browser/browser_ext_sessions_restoreTab.js",
        "browser/components/extensions/test/browser/browser_ext_sessions_window_tab_value.js",
        "browser/components/extensions/test/browser/browser_ext_slow_script.js",
        "browser/components/extensions/test/browser/browser_ext_tabs_attention.js",
        "browser/components/extensions/test/browser/browser_ext_tabs_containerIsolation.js",
        "browser/components/extensions/test/browser/browser_ext_tabs_create.js",
        "browser/components/extensions/test/browser/browser_ext_tabs_detectLanguage.js",
        "browser/components/extensions/test/browser/browser_ext_tabs_discard.js",
        "browser/components/extensions/test/browser/browser_ext_tabs_discard_reversed.js",
        "browser/components/extensions/test/browser/browser_ext_tabs_discarded.js",
        "browser/components/extensions/test/browser/browser_ext_tabs_duplicate.js",
        "browser/components/extensions/test/browser/browser_ext_tabs_events_order.js",
        "browser/components/extensions/test/browser/browser_ext_tabs_executeScript.js",
        "browser/components/extensions/test/browser/browser_ext_tabs_executeScript_bad.js",
        "browser/components/extensions/test/browser/browser_ext_tabs_hide.js",
        "browser/components/extensions/test/browser/browser_ext_tabs_hide_update.js",
        "browser/components/extensions/test/browser/browser_ext_tabs_highlight.js",
        "browser/components/extensions/test/browser/browser_ext_tabs_insertCSS.js",
        "browser/components/extensions/test/browser/browser_ext_tabs_move_array.js",
        "browser/components/extensions/test/browser/browser_ext_tabs_move_discarded.js",
        "browser/components/extensions/test/browser/browser_ext_tabs_move_window.js",
        "browser/components/extensions/test/browser/browser_ext_tabs_move_window_multiple.js",
        "browser/components/extensions/test/browser/browser_ext_tabs_move_window_pinned.js",
        "browser/components/extensions/test/browser/browser_ext_tabs_printPreview.js",
        "browser/components/extensions/test/browser/browser_ext_tabs_query.js",
        "browser/components/extensions/test/browser/browser_ext_tabs_readerMode.js",
        "browser/components/extensions/test/browser/browser_ext_tabs_saveAsPDF.js",
        "browser/components/extensions/test/browser/browser_ext_tabs_sendMessage.js",
        "browser/components/extensions/test/browser/browser_ext_tabs_sharingState.js",
        "browser/components/extensions/test/browser/browser_ext_tabs_update_url.js",
        "browser/components/extensions/test/browser/browser_ext_tabs_warmup.js",
        "browser/components/extensions/test/browser/browser_ext_tabs_zoom.js",
        "browser/components/extensions/test/browser/browser_ext_topSites.js",
        "browser/components/extensions/test/browser/browser_ext_urlbar.js",
        "browser/components/extensions/test/browser/browser_ext_webNavigation_containerIsolation.js",
        "browser/components/extensions/test/browser/browser_ext_webNavigation_getFrames.js",
        "browser/components/extensions/test/browser/browser_ext_webNavigation_urlbar_transitions.js",
        "browser/components/extensions/test/browser/browser_ext_webRequest.js",
        "browser/components/extensions/test/browser/browser_ext_webRequest_error_after_stopped_or_closed.js",
        "browser/components/extensions/test/browser/browser_ext_windows.js",
        "browser/components/extensions/test/browser/browser_ext_windows_allowScriptsToClose.js",
        "browser/components/extensions/test/browser/browser_ext_windows_create_tabId.js",
        "browser/components/firefoxview/tests/browser/browser_recently_closed_tabs.js",
        "browser/components/newtab/content-src/components/DiscoveryStreamComponents/TopSites/TopSites.jsx ",
        "browser/components/newtab/content-src/components/TopSites/TopSiteForm.jsx ",
        "browser/components/newtab/lib/CFRMessageProvider.sys.mjs ",
        "browser/components/newtab/lib/PlacesFeed.jsm ",
        "browser/components/originattributes/test/browser/browser_broadcastChannel.js",
        "browser/components/originattributes/test/browser/browser_cache.js",
        "browser/components/originattributes/test/browser/browser_favicon_firstParty.js",
        "browser/components/originattributes/test/browser/browser_favicon_userContextId.js",
        "browser/components/originattributes/test/browser/browser_firstPartyIsolation.js",
        "browser/components/originattributes/test/browser/browser_firstPartyIsolation_blobURI.js",
        "browser/components/originattributes/test/browser/browser_firstPartyIsolation_js_uri.js",
        "browser/components/originattributes/test/browser/browser_firstPartyIsolation_saveAs.js",
        "browser/components/originattributes/test/browser/browser_permissions.js",
        "browser/components/originattributes/test/browser/browser_postMessage.js",
        "browser/components/originattributes/test/browser/browser_windowOpenerRestriction.js",
        "browser/components/originattributes/test/browser/file_postMessage.html",
        "browser/components/originattributes/test/browser/file_saveAs.sjs",
        "browser/components/originattributes/test/browser/file_thirdPartyChild.iframe.html",
        "browser/components/originattributes/test/browser/file_thirdPartyChild.worker.js",
        "browser/components/originattributes/test/browser/head.js",
        "browser/components/originattributes/test/browser/window2.html",
        "browser/components/originattributes/test/browser/window3.html",
        "browser/components/pagedata/tests/browser/browser_pagedata_background.js",
        "browser/components/pagedata/tests/browser/browser_pagedata_cache.js",
        "browser/components/places/tests/browser/browser_bookmarkMenu_hiddenWindow.js",
        "browser/components/places/tests/browser/browser_bookmarkProperties_addFolderDefaultButton.js",
        "browser/components/places/tests/browser/browser_bookmarkProperties_cancel.js",
        "browser/components/places/tests/browser/browser_bookmarkProperties_editTagContainer.js",
        "browser/components/places/tests/browser/browser_bookmarkProperties_no_user_actions.js",
        "browser/components/places/tests/browser/browser_bookmarkProperties_speculativeConnection.js",
        "browser/components/places/tests/browser/browser_bookmark_add_tags.js",
        "browser/components/places/tests/browser/browser_bookmark_all_tabs.js",
        "browser/components/places/tests/browser/browser_bookmark_backup_export_import.js",
        "browser/components/places/tests/browser/browser_bookmark_context_menu_contents.js",
        "browser/components/places/tests/browser/browser_bookmark_copy_folder_tree.js",
        "browser/components/places/tests/browser/browser_bookmark_folder_moveability.js",
        "browser/components/places/tests/browser/browser_bookmark_titles.js",
        "browser/components/places/tests/browser/browser_bookmarksProperties.js",
        "browser/components/places/tests/browser/browser_bookmarks_sidebar_search.js",
        "browser/components/places/tests/browser/browser_bookmarks_toolbar_context_menu_view_options.js",
        "browser/components/places/tests/browser/browser_bookmarks_toolbar_telemetry.js",
        "browser/components/places/tests/browser/browser_bug485100-change-case-loses-tag.js",
        "browser/components/places/tests/browser/browser_bug631374_tags_selector_scroll.js",
        "browser/components/places/tests/browser/browser_check_correct_controllers.js",
        "browser/components/places/tests/browser/browser_click_bookmarks_on_toolbar.js",
        "browser/components/places/tests/browser/browser_controller_onDrop.js",
        "browser/components/places/tests/browser/browser_controller_onDrop_sidebar.js",
        "browser/components/places/tests/browser/browser_controller_onDrop_tagFolder.js",
        "browser/components/places/tests/browser/browser_cutting_bookmarks.js",
        "browser/components/places/tests/browser/browser_drag_bookmarks_on_toolbar.js",
        "browser/components/places/tests/browser/browser_drag_folder_on_newTab.js",
        "browser/components/places/tests/browser/browser_editBookmark_keywords.js",
        "browser/components/places/tests/browser/browser_editBookmark_tags_liveUpdate.js",
        "browser/components/places/tests/browser/browser_forgetthissite.js",
        "browser/components/places/tests/browser/browser_history_sidebar_search.js",
        "browser/components/places/tests/browser/browser_library_commands.js",
        "browser/components/places/tests/browser/browser_library_delete.js",
        "browser/components/places/tests/browser/browser_library_delete_bookmarks_in_tags.js",
        "browser/components/places/tests/browser/browser_library_delete_tags.js",
        "browser/components/places/tests/browser/browser_library_downloads.js",
        "browser/components/places/tests/browser/browser_library_left_pane_select_hierarchy.js",
        "browser/components/places/tests/browser/browser_library_openFlatContainer.js",
        "browser/components/places/tests/browser/browser_library_open_all.js",
        "browser/components/places/tests/browser/browser_library_panel_leak.js",
        "browser/components/places/tests/browser/browser_library_search.js",
        "browser/components/places/tests/browser/browser_library_views_liveupdate.js",
        "browser/components/places/tests/browser/browser_library_warnOnOpen.js",
        "browser/components/places/tests/browser/browser_paste_bookmarks.js",
        "browser/components/places/tests/browser/browser_paste_into_tags.js",
        "browser/components/places/tests/browser/browser_paste_resets_cut_highlights.js",
        "browser/components/places/tests/browser/browser_remove_bookmarks.js",
        "browser/components/places/tests/browser/browser_stayopenmenu.js",
        "browser/components/places/tests/browser/browser_toolbar_drop_text.js",
        "browser/components/places/tests/browser/browser_toolbar_library_open_recent.js",
        "browser/components/places/tests/browser/browser_toolbar_other_bookmarks.js",
        "browser/components/places/tests/browser/browser_toolbar_overflow.js",
        "browser/components/places/tests/browser/browser_views_iconsupdate.js",
        "browser/components/places/tests/browser/browser_views_liveupdate.js",
        "browser/components/places/tests/chrome/test_bug549192.xhtml",
        "browser/components/places/tests/chrome/test_bug549491.xhtml",
        "browser/components/places/tests/chrome/test_selectItems_on_nested_tree.xhtml",
        "browser/components/places/tests/chrome/test_treeview_date.xhtml",
        "browser/components/pocket/test/unit/panels/browser_pocket_main.js",
        "browser/components/preferences/tests/browser_applications_selection.js",
        "browser/components/preferences/tests/browser_bug410900.js",
        "browser/components/preferences/tests/browser_contentblocking.js",
        "browser/components/preferences/tests/browser_cookie_exceptions_addRemove.js",
        "browser/components/preferences/tests/browser_cookies_exceptions.js",
        "browser/components/preferences/tests/browser_extension_controlled.js",
        "browser/components/preferences/tests/browser_filetype_dialog.js",
        "browser/components/preferences/tests/browser_homepages_use_bookmark.js",
        "browser/components/preferences/tests/browser_https_only_exceptions.js",
        "browser/components/preferences/tests/browser_permissions_checkPermissionsWereAdded.js",
        "browser/components/preferences/tests/browser_permissions_dialog.js",
        "browser/components/preferences/tests/browser_permissions_dialog_default_perm.js",
        "browser/components/preferences/tests/browser_site_login_exceptions.js",
        "browser/components/preferences/tests/siteData/browser_siteData2.js",
        "browser/components/preferences/tests/siteData/browser_siteData3.js",
        "browser/components/preferences/tests/siteData/browser_siteData_multi_select.js",
        "browser/components/privatebrowsing/test/browser/browser_privatebrowsing_DownloadLastDirWithCPS.js",
        "browser/components/privatebrowsing/test/browser/browser_privatebrowsing_about_nimbus.js",
        "browser/components/privatebrowsing/test/browser/browser_privatebrowsing_about_nimbus_dismiss.js",
        "browser/components/privatebrowsing/test/browser/browser_privatebrowsing_about_nimbus_impressions.js",
        "browser/components/privatebrowsing/test/browser/browser_privatebrowsing_about_nimbus_messaging.js",
        "browser/components/privatebrowsing/test/browser/browser_privatebrowsing_cleanup.js",
        "browser/components/privatebrowsing/test/browser/browser_privatebrowsing_downloadLastDir.js",
        "browser/components/privatebrowsing/test/browser/browser_privatebrowsing_downloadLastDir_c.js",
        "browser/components/privatebrowsing/test/browser/browser_privatebrowsing_history_shift_click.js",
        "browser/components/protocolhandler/test/test_registerHandler.html",
        "browser/components/resistfingerprinting/test/browser/browser_bug1369357_site_specific_zoom_level.js",
        "browser/components/resistfingerprinting/test/browser/head.js",
        "browser/components/safebrowsing/content/test/browser_bug400731.js",
        "browser/components/safebrowsing/content/test/browser_bug415846.js",
        "browser/components/safebrowsing/content/test/browser_whitelisted.js",
        "browser/components/screenshots/tests/browser/head.js",
        "browser/components/search/test/browser/browser_483086.js",
        "browser/components/search/test/browser/browser_addKeywordSearch.js",
        "browser/components/search/test/browser/browser_ime_composition.js",
        "browser/components/search/test/browser/browser_search_discovery.js",
        "browser/components/search/test/browser/browser_search_telemetry_searchbar.js",
        "browser/components/search/test/browser/browser_search_telemetry_sources_ads.js",
        "browser/components/sessionstore/test/browser_1284886_suspend_tab.js",
        "browser/components/sessionstore/test/browser_394759_behavior.js",
        "browser/components/sessionstore/test/browser_394759_purge.js",
        "browser/components/sessionstore/test/browser_461634.js",
        "browser/components/sessionstore/test/browser_463205.js",
        "browser/components/sessionstore/test/browser_464199.js",
        "browser/components/sessionstore/test/browser_490040.js",
        "browser/components/sessionstore/test/browser_491168.js",
        "browser/components/sessionstore/test/browser_491577.js",
        "browser/components/sessionstore/test/browser_500328.js",
        "browser/components/sessionstore/test/browser_522545.js",
        "browser/components/sessionstore/test/browser_526613.js",
        "browser/components/sessionstore/test/browser_586068-apptabs.js",
        "browser/components/sessionstore/test/browser_586068-apptabs_ondemand.js",
        "browser/components/sessionstore/test/browser_586068-browser_state_interrupted.js",
        "browser/components/sessionstore/test/browser_586068-cascade.js",
        "browser/components/sessionstore/test/browser_586068-multi_window.js",
        "browser/components/sessionstore/test/browser_586068-reload.js",
        "browser/components/sessionstore/test/browser_586068-select.js",
        "browser/components/sessionstore/test/browser_586068-window_state.js",
        "browser/components/sessionstore/test/browser_586068-window_state_override.js",
        "browser/components/sessionstore/test/browser_589246.js",
        "browser/components/sessionstore/test/browser_590268.js",
        "browser/components/sessionstore/test/browser_595601-restore_hidden.js",
        "browser/components/sessionstore/test/browser_597071.js",
        "browser/components/sessionstore/test/browser_600545.js",
        "browser/components/sessionstore/test/browser_607016.js",
        "browser/components/sessionstore/test/browser_615394-SSWindowState_events_setBrowserState.js",
        "browser/components/sessionstore/test/browser_615394-SSWindowState_events_setTabState.js",
        "browser/components/sessionstore/test/browser_615394-SSWindowState_events_setWindowState.js",
        "browser/components/sessionstore/test/browser_615394-SSWindowState_events_undoCloseWindow.js",
        "browser/components/sessionstore/test/browser_618151.js",
        "browser/components/sessionstore/test/browser_625016.js",
        "browser/components/sessionstore/test/browser_636279.js",
        "browser/components/sessionstore/test/browser_687710.js",
        "browser/components/sessionstore/test/browser_687710_2.js",
        "browser/components/sessionstore/test/browser_701377.js",
        "browser/components/sessionstore/test/browser_819510_perwindowpb.js",
        "browser/components/sessionstore/test/browser_906076_lazy_tabs.js",
        "browser/components/sessionstore/test/browser_async_duplicate_tab.js",
        "browser/components/sessionstore/test/browser_async_flushes.js",
        "browser/components/sessionstore/test/browser_async_window_flushing.js",
        "browser/components/sessionstore/test/browser_background_tab_crash.js",
        "browser/components/sessionstore/test/browser_backup_recovery.js",
        "browser/components/sessionstore/test/browser_broadcast.js",
        "browser/components/sessionstore/test/browser_cleaner.js",
        "browser/components/sessionstore/test/browser_closed_objects_changed_notifications_tabs.js",
        "browser/components/sessionstore/test/browser_cookies.js",
        "browser/components/sessionstore/test/browser_cookies_legacy.js",
        "browser/components/sessionstore/test/browser_cookies_sameSite.js",
        "browser/components/sessionstore/test/browser_duplicate_history.js",
        "browser/components/sessionstore/test/browser_focus_after_restore.js",
        "browser/components/sessionstore/test/browser_forget_async_closings.js",
        "browser/components/sessionstore/test/browser_movePendingTabToNewWindow.js",
        "browser/components/sessionstore/test/browser_old_favicon.js",
        "browser/components/sessionstore/test/browser_parentProcessRestoreHash.js",
        "browser/components/sessionstore/test/browser_remoteness_flip_on_restore.js",
        "browser/components/sessionstore/test/browser_reopen_all_windows.js",
        "browser/components/sessionstore/test/browser_restore_cookies_noOriginAttributes.js",
        "browser/components/sessionstore/test/browser_restore_redirect.js",
        "browser/components/sessionstore/test/browser_restored_window_features.js",
        "browser/components/sessionstore/test/browser_scrollPositions.js",
        "browser/components/sessionstore/test/browser_scrollPositionsReaderMode.js",
        "browser/components/sessionstore/test/browser_sessionHistory.js",
        "browser/components/sessionstore/test/browser_sessionStorage.js",
        "browser/components/sessionstore/test/browser_sessionStoreContainer.js",
        "browser/components/sessionstore/test/browser_switch_remoteness.js",
        "browser/components/sessionstore/test/browser_tab_label_during_restore.js",
        "browser/components/sessionstore/test/browser_tabs_in_urlbar.js",
        "browser/components/sessionstore/test/browser_userTyped_restored_after_discard.js",
        "browser/components/sessionstore/test/browser_windowStateContainer.js",
        "browser/components/sessionstore/test/head.js",
        "browser/components/tests/browser/browser_bug538331.js",
        "browser/components/tests/browser/browser_initial_tab_remoteType.js",
        "browser/components/tests/browser/browser_startup_homepage.js",
        "browser/components/touchbar/tests/browser/browser_touchbar_searchrestrictions.js",
        "browser/components/touchbar/tests/browser/browser_touchbar_tests.js",
        "browser/components/translation/test/bing.sjs",
        "browser/components/translation/test/browser_translation_bing.js",
        "browser/components/translation/test/browser_translation_exceptions.js",
        "browser/components/translation/test/browser_translation_infobar.js",
        "browser/components/translation/test/browser_translation_yandex.js",
        "browser/components/uitour/test/browser_UITour.js",
        "browser/components/uitour/test/head.js",
        "browser/components/urlbar/tests/browser-tips/browser_interventions.js",
        "browser/components/urlbar/tests/browser-tips/browser_picks.js",
        "browser/components/urlbar/tests/browser-tips/browser_searchTips.js",
        "browser/components/urlbar/tests/browser-tips/browser_searchTips_interaction.js",
        "browser/components/urlbar/tests/browser-tips/browser_selection.js",
        "browser/components/urlbar/tests/browser-tips/head.js",
        "browser/components/urlbar/tests/browser-updateResults/browser_appendSpanCount.js",
        "browser/components/urlbar/tests/browser-updateResults/browser_reuse.js",
        "browser/components/urlbar/tests/browser-updateResults/head.js",
        "browser/components/urlbar/tests/browser/browser_UrlbarInput_overflow.js",
        "browser/components/urlbar/tests/browser/browser_UrlbarInput_overflow_resize.js",
        "browser/components/urlbar/tests/browser/browser_UrlbarInput_searchTerms.js",
        "browser/components/urlbar/tests/browser/browser_UrlbarInput_setURI.js",
        "browser/components/urlbar/tests/browser/browser_UrlbarInput_tooltip.js",
        "browser/components/urlbar/tests/browser/browser_UrlbarInput_trimURLs.js",
        "browser/components/urlbar/tests/browser/browser_aboutHomeLoading.js",
        "browser/components/urlbar/tests/browser/browser_autoFill_backspaced.js",
        "browser/components/urlbar/tests/browser/browser_autoFill_canonize.js",
        "browser/components/urlbar/tests/browser/browser_autoFill_caretNotAtEnd.js",
        "browser/components/urlbar/tests/browser/browser_autoFill_firstResult.js",
        "browser/components/urlbar/tests/browser/browser_autoFill_paste.js",
        "browser/components/urlbar/tests/browser/browser_autoFill_placeholder.js",
        "browser/components/urlbar/tests/browser/browser_autoFill_preserve.js",
        "browser/components/urlbar/tests/browser/browser_autoFill_trimURLs.js",
        "browser/components/urlbar/tests/browser/browser_autoFill_typed.js",
        "browser/components/urlbar/tests/browser/browser_autoFill_undo.js",
        "browser/components/urlbar/tests/browser/browser_autoOpen.js",
        "browser/components/urlbar/tests/browser/browser_autocomplete_autoselect.js",
        "browser/components/urlbar/tests/browser/browser_autocomplete_edit_completed.js",
        "browser/components/urlbar/tests/browser/browser_autocomplete_enter_race.js",
        "browser/components/urlbar/tests/browser/browser_autocomplete_no_title.js",
        "browser/components/urlbar/tests/browser/browser_autocomplete_readline_navigation.js",
        "browser/components/urlbar/tests/browser/browser_autocomplete_tag_star_visibility.js",
        "browser/components/urlbar/tests/browser/browser_canonizeURL.js",
        "browser/components/urlbar/tests/browser/browser_copy_during_load.js",
        "browser/components/urlbar/tests/browser/browser_copying.js",
        "browser/components/urlbar/tests/browser/browser_custom_new_window_url.js",
        "browser/components/urlbar/tests/browser/browser_cutting.js",
        "browser/components/urlbar/tests/browser/browser_decode.js",
        "browser/components/urlbar/tests/browser/browser_delete.js",
        "browser/components/urlbar/tests/browser/browser_deleteAllText.js",
        "browser/components/urlbar/tests/browser/browser_downArrowKeySearch.js",
        "browser/components/urlbar/tests/browser/browser_dragdropURL.js",
        "browser/components/urlbar/tests/browser/browser_dynamicResults.js",
        "browser/components/urlbar/tests/browser/browser_engagement.js",
        "browser/components/urlbar/tests/browser/browser_enter.js",
        "browser/components/urlbar/tests/browser/browser_enterAfterMouseOver.js",
        "browser/components/urlbar/tests/browser/browser_groupLabels.js",
        "browser/components/urlbar/tests/browser/browser_helpUrl.js",
        "browser/components/urlbar/tests/browser/browser_heuristicNotAddedFirst.js",
        "browser/components/urlbar/tests/browser/browser_hideHeuristic.js",
        "browser/components/urlbar/tests/browser/browser_ime_composition.js",
        "browser/components/urlbar/tests/browser/browser_inputHistory.js",
        "browser/components/urlbar/tests/browser/browser_inputHistory_autofill.js",
        "browser/components/urlbar/tests/browser/browser_inputHistory_emptystring.js",
        "browser/components/urlbar/tests/browser/browser_keepStateAcrossTabSwitches.js",
        "browser/components/urlbar/tests/browser/browser_keyword_override.js",
        "browser/components/urlbar/tests/browser/browser_keyword_select_and_type.js",
        "browser/components/urlbar/tests/browser/browser_locationBarCommand.js",
        "browser/components/urlbar/tests/browser/browser_oneOffs.js",
        "browser/components/urlbar/tests/browser/browser_oneOffs_settings.js",
        "browser/components/urlbar/tests/browser/browser_pasteAndGo.js",
        "browser/components/urlbar/tests/browser/browser_paste_multi_lines.js",
        "browser/components/urlbar/tests/browser/browser_paste_then_focus.js",
        "browser/components/urlbar/tests/browser/browser_quickactions.js",
        "browser/components/urlbar/tests/browser/browser_remoteness_switch.js",
        "browser/components/urlbar/tests/browser/browser_removeUnsafeProtocolsFromURLBarPaste.js",
        "browser/components/urlbar/tests/browser/browser_remove_match.js",
        "browser/components/urlbar/tests/browser/browser_restoreEmptyInput.js",
        "browser/components/urlbar/tests/browser/browser_resultSpan.js",
        "browser/components/urlbar/tests/browser/browser_result_onSelection.js",
        "browser/components/urlbar/tests/browser/browser_revert.js",
        "browser/components/urlbar/tests/browser/browser_searchMode_autofill.js",
        "browser/components/urlbar/tests/browser/browser_searchMode_clickLink.js",
        "browser/components/urlbar/tests/browser/browser_searchMode_heuristic.js",
        "browser/components/urlbar/tests/browser/browser_searchMode_indicator.js",
        "browser/components/urlbar/tests/browser/browser_searchMode_localOneOffs_actionText.js",
        "browser/components/urlbar/tests/browser/browser_searchMode_no_results.js",
        "browser/components/urlbar/tests/browser/browser_searchMode_pickResult.js",
        "browser/components/urlbar/tests/browser/browser_searchMode_sessionStore.js",
        "browser/components/urlbar/tests/browser/browser_searchMode_setURI.js",
        "browser/components/urlbar/tests/browser/browser_searchMode_suggestions.js",
        "browser/components/urlbar/tests/browser/browser_searchMode_switchTabs.js",
        "browser/components/urlbar/tests/browser/browser_searchSingleWordNotification.js",
        "browser/components/urlbar/tests/browser/browser_selectStaleResults.js",
        "browser/components/urlbar/tests/browser/browser_selectionKeyNavigation.js",
        "browser/components/urlbar/tests/browser/browser_separatePrivateDefault.js",
        "browser/components/urlbar/tests/browser/browser_separatePrivateDefault_differentEngine.js",
        "browser/components/urlbar/tests/browser/browser_stopSearchOnSelection.js",
        "browser/components/urlbar/tests/browser/browser_stop_pending.js",
        "browser/components/urlbar/tests/browser/browser_suggestedIndex.js",
        "browser/components/urlbar/tests/browser/browser_suppressFocusBorder.js",
        "browser/components/urlbar/tests/browser/browser_switchTab_closesUrlbarPopup.js",
        "browser/components/urlbar/tests/browser/browser_switchToTab_closes_newtab.js",
        "browser/components/urlbar/tests/browser/browser_tabKeyBehavior.js",
        "browser/components/urlbar/tests/browser/browser_tabMatchesInAwesomebar.js",
        "browser/components/urlbar/tests/browser/browser_top_sites.js",
        "browser/components/urlbar/tests/browser/browser_top_sites_private.js",
        "browser/components/urlbar/tests/browser/browser_typed_value.js",
        "browser/components/urlbar/tests/browser/browser_urlbar_annotation.js",
        "browser/components/urlbar/tests/browser/browser_urlbar_event_telemetry_abandonment.js",
        "browser/components/urlbar/tests/browser/browser_urlbar_event_telemetry_engagement.js",
        "browser/components/urlbar/tests/browser/browser_urlbar_telemetry_autofill.js",
        "browser/components/urlbar/tests/browser/browser_urlbar_telemetry_places.js",
        "browser/components/urlbar/tests/browser/browser_urlbar_telemetry_remotetab.js",
        "browser/components/urlbar/tests/browser/browser_urlbar_telemetry_searchmode.js",
        "browser/components/urlbar/tests/browser/browser_urlbar_telemetry_sponsored_topsites.js",
        "browser/components/urlbar/tests/browser/browser_view_resultDisplay.js",
        "browser/components/urlbar/tests/browser/browser_view_resultTypes_display.js",
        "browser/components/urlbar/tests/browser/browser_waitForLoadOrTimeout.js",
        "browser/components/urlbar/tests/ext/browser/browser_ext_urlbar_clearInput.js",
        "browser/components/urlbar/tests/quicksuggest/QuickSuggestTestUtils.jsm ",
        "browser/components/urlbar/tests/quicksuggest/browser/browser_quicksuggest.js",
        "browser/components/urlbar/tests/quicksuggest/browser/browser_quicksuggest_bestMatch.js",
        "browser/components/urlbar/tests/quicksuggest/browser/browser_quicksuggest_block.js",
        "browser/components/urlbar/tests/quicksuggest/browser/browser_quicksuggest_configuration.js",
        "browser/components/urlbar/tests/quicksuggest/browser/browser_quicksuggest_indexes.js",
        "browser/components/urlbar/tests/quicksuggest/browser/browser_quicksuggest_telemetry.js",
        "browser/extensions/formautofill/test/browser/browser_dropdown_layout.js",
        "browser/extensions/formautofill/test/browser/creditCard/browser_anti_clickjacking.js",
        "browser/extensions/report-site-issue/test/browser/browser_button_state.js",
        "browser/extensions/report-site-issue/test/browser/browser_disabled_cleanup.js",
        "browser/extensions/report-site-issue/test/browser/browser_report_site_issue.js",
        "browser/extensions/report-site-issue/test/browser/head.js",
        "browser/extensions/screenshots/test/browser/head.js",
        "browser/extensions/webcompat/tests/browser/head.js",
        "browser/modules/test/browser/browser_PageActions.js",
        "browser/modules/test/browser/browser_PageActions_contextMenus.js",
        "browser/modules/test/browser/browser_PageActions_newWindow.js",
        "browser/modules/test/browser/browser_PermissionUI.js",
        "browser/modules/test/browser/browser_PermissionUI_prompts.js",
        "browser/modules/test/browser/browser_SitePermissions_tab_urls.js",
        "browser/modules/test/browser/browser_Telemetry_numberOfSiteOriginsPerDocument.js",
        "browser/modules/test/browser/browser_UnsubmittedCrashHandler.js",
        "browser/modules/test/browser/browser_UsageTelemetry.js",
        "browser/modules/test/browser/browser_UsageTelemetry_domains.js",
        "browser/modules/test/browser/browser_UsageTelemetry_interaction.js",
        "browser/modules/test/browser/browser_UsageTelemetry_private_and_restore.js",
        "browser/modules/test/browser/browser_UsageTelemetry_toolbars.js",
        "browser/modules/test/browser/browser_UsageTelemetry_uniqueOriginsVisitedInPast24Hours.js",
        "caps/tests/mochitest/browser_checkloaduri.js",
        "caps/tests/mochitest/test_addonMayLoad.html",
        "caps/tests/mochitest/test_bug995943.xhtml",
        "caps/tests/mochitest/test_disableScript.xhtml",
        "devtools/client/aboutdebugging/test/browser/browser_aboutdebugging_addons_remote_runtime.js",
        "devtools/client/aboutdebugging/test/browser/browser_aboutdebugging_tab_navigate.js",
        "devtools/client/aboutdebugging/test/browser/browser_aboutdebugging_workers_remote_runtime.js",
        "devtools/client/accessibility/test/browser/head.js",
        "devtools/client/accessibility/test/chrome/test_accessible_learnMoreLink.html",
        "devtools/client/accessibility/test/chrome/test_accessible_openLink.html",
        "devtools/client/application/test/node/fixtures/data/constants.js",
        "devtools/client/debugger/src/components/PrimaryPanes/tests/SourcesTree.spec.js",
        "devtools/client/debugger/src/components/PrimaryPanes/tests/SourcesTreeItem.spec.js",
        "devtools/client/debugger/src/components/SecondaryPanes/Frames/tests/Frames.spec.js",
        "devtools/client/debugger/src/components/SecondaryPanes/Frames/tests/Group.spec.js",
        "devtools/client/debugger/src/reducers/tests/sources.spec.js",
        "devtools/client/debugger/src/utils/sources-tree/tests/addToTree.spec.js",
        "devtools/client/debugger/src/utils/sources-tree/tests/collapseTree.spec.js",
        "devtools/client/debugger/src/utils/sources-tree/tests/getDirectories.spec.js",
        "devtools/client/debugger/src/utils/sources-tree/tests/getUrl.spec.js",
        "devtools/client/debugger/src/utils/sources-tree/tests/utils.spec.js",
        "devtools/client/debugger/src/utils/test-mockup.js",
        "devtools/client/debugger/src/utils/tests/source.spec.js",
        "devtools/client/debugger/src/utils/tests/url.spec.js",
        "devtools/client/debugger/test/mochitest/browser_dbg-project-root.js",
        "devtools/client/framework/browser-toolbox/test/browser_browser_toolbox_debugger.js",
        "devtools/client/framework/test/allocations/browser_allocations_browser_console.js",
        "devtools/client/framework/test/allocations/browser_allocations_reload_no_devtools.js",
        "devtools/client/framework/test/allocations/reload-test.js",
        "devtools/client/framework/test/browser_target_from_url.js",
        "devtools/client/framework/test/browser_toolbox_error_count_reset_on_navigation.js",
        "devtools/client/framework/test/node/components/debug-target-info.test.js",
        "devtools/client/inspector/markup/test/browser_markup_tag_edit_05.js",
        "devtools/client/inspector/shared/test/browser_styleinspector_output-parser.js",
        "devtools/client/inspector/shared/test/head.js",
        "devtools/client/inspector/test/browser_inspector_highlighter-eyedropper-zoom.js",
        "devtools/client/jsonview/test/browser_jsonview_data_blocking.js",
        "devtools/client/memory/test/browser/browser_memory_allocationStackDisplay_01.js",
        "devtools/client/memory/test/browser/browser_memory_clear_snapshots.js",
        "devtools/client/memory/test/browser/browser_memory_diff_01.js",
        "devtools/client/memory/test/browser/browser_memory_displays_01.js",
        "devtools/client/memory/test/browser/browser_memory_dominator_trees_01.js",
        "devtools/client/memory/test/browser/browser_memory_dominator_trees_02.js",
        "devtools/client/memory/test/browser/browser_memory_filter_01.js",
        "devtools/client/memory/test/browser/browser_memory_individuals_01.js",
        "devtools/client/memory/test/browser/browser_memory_keyboard-snapshot-list.js",
        "devtools/client/memory/test/browser/browser_memory_keyboard.js",
        "devtools/client/memory/test/browser/browser_memory_no_allocation_stacks.js",
        "devtools/client/memory/test/browser/browser_memory_no_auto_expand.js",
        "devtools/client/memory/test/browser/browser_memory_percents_01.js",
        "devtools/client/memory/test/browser/browser_memory_refresh_does_not_leak.js",
        "devtools/client/memory/test/browser/browser_memory_simple_01.js",
        "devtools/client/netmonitor/src/har/test/head.js",
        "devtools/client/netmonitor/test/browser_net_cached-status.js",
        "devtools/client/netmonitor/test/browser_net_details_copy.js",
        "devtools/client/netmonitor/test/browser_net_headers-link_clickable.js",
        "devtools/client/netmonitor/test/browser_net_new_request_panel_send_request.js",
        "devtools/client/netmonitor/test/browser_net_resend_headers.js",
        "devtools/client/netmonitor/test/browser_net_resend_hidden_headers.js",
        "devtools/client/netmonitor/test/browser_net_security-state.js",
        "devtools/client/netmonitor/test/browser_net_security-tab-deselect.js",
        "devtools/client/netmonitor/test/browser_net_security-tab-visibility.js",
        "devtools/client/netmonitor/test/browser_net_stacktraces-visibility.js",
        "devtools/client/netmonitor/test/head.js",
        "devtools/client/netmonitor/test/sjs_status-codes-test-server.sjs",
        "devtools/client/performance-new/test/browser/browser_devtools-record-capture.js",
        "devtools/client/performance-new/test/browser/browser_devtools-record-discard.js",
        "devtools/client/performance-new/test/browser/browser_popup-profiler-states.js",
        "devtools/client/performance-new/test/browser/browser_popup-record-capture-view.js",
        "devtools/client/performance-new/test/browser/browser_popup-record-capture.js",
        "devtools/client/performance-new/test/browser/browser_popup-record-discard.js",
        "devtools/client/performance-new/test/browser/browser_split-toolbar-button.js",
        "devtools/client/performance-new/test/browser/helpers.js",
        "devtools/client/responsive/test/browser/browser_cmd_click.js",
        "devtools/client/responsive/test/browser/browser_in_rdm_pane.js",
        "devtools/client/responsive/test/browser/browser_navigation.js",
        "devtools/client/responsive/test/browser/browser_page_state.js",
        "devtools/client/responsive/test/browser/browser_permission_doorhanger.js",
        "devtools/client/responsive/test/browser/browser_screenshot_button_warning.js",
        "devtools/client/responsive/test/browser/browser_tab_close.js",
        "devtools/client/responsive/test/browser/browser_tab_not_selected.js",
        "devtools/client/responsive/test/browser/browser_tab_remoteness_change.js",
        "devtools/client/responsive/test/browser/browser_tab_remoteness_change_fission_switch_target.js",
        "devtools/client/responsive/test/browser/browser_target_blank.js",
        "devtools/client/responsive/test/browser/browser_toolbox_swap_browsers.js",
        "devtools/client/responsive/test/browser/browser_toolbox_swap_inspector.js",
        "devtools/client/responsive/test/browser/browser_viewport_basics.js",
        "devtools/client/responsive/test/browser/head.js",
        "devtools/client/responsive/test/browser/sjs_redirection.sjs",
        "devtools/client/storage/test/browser_storage_basic.js",
        "devtools/client/storage/test/browser_storage_cache_delete.js",
        "devtools/client/storage/test/browser_storage_cookies_add.js",
        "devtools/client/storage/test/browser_storage_cookies_domain.js",
        "devtools/client/storage/test/browser_storage_cookies_domain_port.js",
        "devtools/client/storage/test/browser_storage_cookies_samesite.js",
        "devtools/client/storage/test/browser_storage_values.js",
        "devtools/client/styleeditor/test/browser_styleeditor_inline_friendly_names.js",
        "devtools/client/styleeditor/test/browser_styleeditor_private_perwindowpb.js",
        "devtools/client/styleeditor/test/head.js",
        "devtools/client/webconsole/test/browser/browser_console.js",
        "devtools/client/webconsole/test/browser/browser_console_clear_closed_tab.js",
        "devtools/client/webconsole/test/browser/browser_console_context_menu_entries.js",
        "devtools/client/webconsole/test/browser/browser_console_enable_network_monitoring.js",
        "devtools/client/webconsole/test/browser/browser_console_webconsole_iframe_messages.js",
        "devtools/client/webconsole/test/browser/browser_jsterm_autocomplete_in_debugger_stackframe.js",
        "devtools/client/webconsole/test/browser/browser_jsterm_autocomplete_mapped_variables.js",
        "devtools/client/webconsole/test/browser/browser_jsterm_await_dynamic_import.js",
        "devtools/client/webconsole/test/browser/browser_jsterm_editor_reverse_search_keyboard_navigation.js",
        "devtools/client/webconsole/test/browser/browser_jsterm_multiline.js",
        "devtools/client/webconsole/test/browser/browser_jsterm_screenshot_command_clipboard.js",
        "devtools/client/webconsole/test/browser/browser_jsterm_screenshot_command_file.js",
        "devtools/client/webconsole/test/browser/browser_jsterm_screenshot_command_fixed_header.js",
        "devtools/client/webconsole/test/browser/browser_jsterm_screenshot_command_selector.js",
        "devtools/client/webconsole/test/browser/browser_toolbox_console_new_process.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_batching.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_cached_messages_cross_domain_iframe.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_cached_messages_duplicate_after_target_switching.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_checkloaduri_errors.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_click_function_to_mapped_source.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_click_function_to_prettyprinted_source.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_click_function_to_source.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_clickable_urls.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_close_unfocused_window.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_closing_after_completion.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_console_api_iframe.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_console_group.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_console_logging_workers_api.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_console_table.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_console_trace_duplicates.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_context_menu_copy_link_location.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_context_menu_open_url.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_csp_ignore_reflected_xss_message.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_csp_violation.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_cspro.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_duplicate_errors.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_errors_after_page_reload.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_eval_error.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_eval_in_debugger_stackframe.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_eval_in_debugger_stackframe2.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_eval_sources.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_execution_scope.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_external_script_errors.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_filter_buttons_overflow.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_filter_by_regex_input.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_filter_groups.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_filter_navigation_marker.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_filters.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_filters_persist.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_iframe_wrong_hud.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_init.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_insecure_passwords_about_blank_web_console_warning.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_location_debugger_link.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_location_styleeditor_link.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_network_requests_from_chrome.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_network_unicode.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_object_inspector_custom_formatters.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_object_inspector_local_session_storage.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_optimized_out_vars.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_output_order.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_output_trimmed.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_reopen_closed_tab.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_same_origin_errors.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_select_all.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_sourcemap_error.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_stacktrace_location_debugger_link.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_string.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_time_methods.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_warning_group_cookies.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_warning_group_storage_isolation.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_worker_error.js",
        "devtools/client/webconsole/test/browser/browser_webconsole_worker_evaluate.js",
        "devtools/client/webconsole/test/browser/test-console-filter-by-regex-input.html",
        "devtools/server/tests/browser/browser_inspector-anonymous.js",
        "devtools/server/tests/browser/browser_storage_cookies-duplicate-names.js",
        "devtools/server/tests/browser/browser_storage_dynamic_windows.js",
        "devtools/server/tests/browser/head.js",
        "devtools/shared/commands/resource/tests/browser_resources_sources.js",
        "devtools/shared/commands/target/tests/browser_target_command_bfcache.js",
        "devtools/shared/test-helpers/browser_allocation_tracker.js",
        "dom/animation/test/document-timeline/test_document-timeline.html",
        "dom/animation/test/mozilla/file_restyles.html",
        "dom/base/test/browser_aboutnewtab_process_selection.js",
        "dom/base/test/browser_blocking_image.js",
        "dom/base/test/browser_bug1011748.js",
        "dom/base/test/browser_inputStream_structuredClone.js",
        "dom/base/test/browser_refresh_content.js",
        "dom/base/test/bug435425_redirect.sjs",
        "dom/base/test/bug461735-redirect1.sjs",
        "dom/base/test/chrome/test_bug421622.xhtml",
        "dom/base/test/chrome/test_chromeOuterWindowID.xhtml",
        "dom/base/test/chrome/window_swapFrameLoaders.xhtml",
        "dom/base/test/file_bug1268962.sjs",
        "dom/base/test/file_bug590870.html",
        "dom/base/test/iframe1_bug1640766.html",
        "dom/base/test/iframe_meta_refresh.sjs",
        "dom/base/test/jsmodules/test_import_meta_resolve.html",
        "dom/base/test/referrerHelper.js",
        "dom/base/test/referrer_change_server.sjs",
        "dom/base/test/referrer_testserver.sjs",
        "dom/base/test/test_blocking_image.html",
        "dom/base/test/test_bug1091883.html",
        "dom/base/test/test_bug1222633.html",
        "dom/base/test/test_bug1222633_link_update.html",
        "dom/base/test/test_bug1268962.html",
        "dom/base/test/test_bug282547.html",
        "dom/base/test/test_bug338583.html",
        "dom/base/test/test_bug362391.xhtml",
        "dom/base/test/test_bug364413.xhtml",
        "dom/base/test/test_bug384003.xhtml",
        "dom/base/test/test_bug426308.html",
        "dom/base/test/test_bug435425.html",
        "dom/base/test/test_bug466080.html",
        "dom/base/test/test_bug515401.html",
        "dom/base/test/test_bug564863-2.xhtml",
        "dom/base/test/test_bug574596.html",
        "dom/base/test/test_bug592829.html",
        "dom/base/test/test_bug601803.html",
        "dom/base/test/test_bug694754.xhtml",
        "dom/base/test/test_bug696301-1.html",
        "dom/base/test/test_bug696301-2.html",
        "dom/base/test/test_bug744830.html",
        "dom/base/test/test_current_inner_window.html",
        "dom/base/test/test_embed_xorigin_document.html",
        "dom/base/test/test_fragment_sanitization.xhtml",
        "dom/base/test/test_intersectionobservers.html",
        "dom/base/test/test_messagemanager_send_principal.html",
        "dom/base/test/test_navigator_cookieEnabled.html",
        "dom/base/test/test_postMessage_solidus.html",
        "dom/base/test/test_postMessages_window.html",
        "dom/base/test/test_warning_for_blocked_cross_site_request.html",
        "dom/bindings/test/test_dom_xrays.html",
        "dom/bindings/test/test_setWithNamedGetterNoNamedSetter.html",
        "dom/cache/test/browser/browser_cache_pb_window.js",
        "dom/cache/test/xpcshell/make_profile.js",
        "dom/canvas/test/test_capture.html",
        "dom/canvas/test/test_imagebitmap.html",
        "dom/canvas/test/test_imagebitmap_cropping.html",
        "dom/encoding/test/test_TLD.html",
        "dom/encoding/test/test_in_lk_TLD.html",
        "dom/events/test/browser_beforeinput_by_execCommand_in_contentscript.js",
        "dom/events/test/browser_dragimage.js",
        "dom/events/test/browser_keyboard_event_init_key_event_enabled_in_contentscript.js",
        "dom/events/test/browser_mouse_enterleave_switch_tab.js",
        "dom/events/test/test_DataTransferItemList.html",
        "dom/events/test/test_bug1037990.html",
        "dom/events/test/test_bug1264380.html",
        "dom/events/test/test_bug812744.html",
        "dom/events/test/test_dragstart.html",
        "dom/events/test/test_mouse_capture_iframe.html",
        "dom/events/test/test_mouse_enterleave_iframe.html",
        "dom/events/test/test_use_split_keypress_event_model_on_old_Confluence.html",
        "dom/file/tests/test_mozfiledataurl.html",
        "dom/file/tests/test_nonascii_blob_url.html",
        "dom/html/test/browser_form_post_from_file_to_http.js",
        "dom/html/test/browser_fullscreen-api-keys.js",
        "dom/html/test/browser_fullscreen-contextmenu-esc.js",
        "dom/html/test/browser_fullscreen-newtab.js",
        "dom/html/test/file_fullscreen-api.html",
        "dom/html/test/forms/test_input_url.html",
        "dom/html/test/forms/test_max_attribute.html",
        "dom/html/test/forms/test_min_attribute.html",
        "dom/html/test/forms/test_pattern_attribute.html",
        "dom/html/test/forms/test_required_attribute.html",
        "dom/html/test/forms/test_step_attribute.html",
        "dom/html/test/fullscreen_helpers.js",
        "dom/html/test/test_anchor_href_cache_invalidation.html",
        "dom/html/test/test_bug1292522_same_domain_with_different_port_number.html",
        "dom/html/test/test_bug1315146.html",
        "dom/html/test/test_bug209275.xhtml",
        "dom/html/test/test_bug445004.html",
        "dom/html/test/test_bug448166.html",
        "dom/html/test/test_bug558788-2.html",
        "dom/html/test/test_bug590353-2.html",
        "dom/html/test/test_bug598643.html",
        "dom/html/test/test_bug607145.html",
        "dom/html/test/test_bug839913.html",
        "dom/html/test/test_formSubmission.html",
        "dom/indexedDB/test/browser_forgetThisSite.js",
        "dom/indexedDB/test/browser_private_idb.js",
        "dom/indexedDB/test/test_third_party.html",
        "dom/ipc/tests/JSProcessActor/head.js",
        "dom/ipc/tests/JSWindowActor/browser_observer_notification.js",
        "dom/ipc/tests/JSWindowActor/head.js",
        "dom/ipc/tests/browser_crash_oopiframe.js",
        "dom/ipc/tests/browser_domainPolicy.js",
        "dom/ipc/tests/test_bcg_processes.html",
        "dom/l10n/tests/mochitest/dom_localization/test_overlay.html",
        "dom/l10n/tests/mochitest/dom_localization/test_overlay_missing_children.html",
        "dom/l10n/tests/mochitest/dom_localization/test_overlay_repeated.html",
        "dom/l10n/tests/mochitest/dom_localization/test_repeated_l10nid.html",
        "dom/localstorage/test/browser_private_ls.js",
        "dom/localstorage/test/unit/databaseShadowing-shared.js",
        "dom/manifest/test/browser_ManifestFinder_browserHasManifestLink.js",
        "dom/manifest/test/browser_Manifest_install.js",
        "dom/manifest/test/test_ManifestProcessor_id.html",
        "dom/manifest/test/test_ManifestProcessor_scope.html",
        "dom/manifest/test/test_ManifestProcessor_start_url.html",
        "dom/manifest/test/test_ManifestProcessor_warnings.html",
        "dom/media/autoplay/test/browser/head.js",
        "dom/media/autoplay/test/mochitest/file_autoplay_gv_play_request_window.html",
        "dom/media/autoplay/test/mochitest/file_autoplay_policy_activation_window.html",
        "dom/media/autoplay/test/mochitest/test_autoplay_policy_permission.html",
        "dom/media/doctor/test/browser/browser_decoderDoctor.js",
        "dom/media/mediasession/test/file_trigger_actionhanlder_window.html",
        "dom/media/test/browser/browser_telemetry_video_hardware_decoding_support.js",
        "dom/media/test/file_access_controls.html",
        "dom/media/test/manifest.js",
        "dom/media/test/midflight-redirect.sjs",
        "dom/media/test/redirect.sjs",
        "dom/media/test/test_mixed_principals.html",
        "dom/media/test/test_seek_promise_bug1344357.html",
        "dom/media/test/test_streams_capture_origin.html",
        "dom/media/webrtc/tests/mochitests/test_peerConnection_basicAudioVideoVerifyExtmap.html",
        "dom/media/webrtc/tests/mochitests/test_peerConnection_basicAudioVideoVerifyExtmapSendonly.html",
        "dom/notification/test/mochitest/NotificationTest.js",
        "dom/notification/test/mochitest/test_notification_tag.html",
        "dom/payments/test/test_pmi_validation.html",
        "dom/performance/tests/test_performance_paint_observer_helper.html",
        "dom/performance/tests/test_performance_paint_timing_helper.html",
        "dom/quota/test/xpcshell/make_unknownFiles.js",
        "dom/security/test/cors/file_bug1456721.html",
        "dom/security/test/cors/test_CrossSiteXHR.html",
        "dom/security/test/cors/test_CrossSiteXHR_cache.html",
        "dom/security/test/cors/test_CrossSiteXHR_origin.html",
        "dom/security/test/csp/browser_manifest-src-override-default-src.js",
        "dom/security/test/csp/browser_test_uir_optional_clicks.js",
        "dom/security/test/csp/browser_test_web_manifest.js",
        "dom/security/test/csp/file_blob_uri_blocks_modals.html",
        "dom/security/test/csp/file_blocked_uri_in_violation_event_after_redirects.html",
        "dom/security/test/csp/file_blocked_uri_in_violation_event_after_redirects.sjs",
        "dom/security/test/csp/file_blocked_uri_redirect_frame_src_server.sjs",
        "dom/security/test/csp/file_bug1505412.sjs",
        "dom/security/test/csp/file_bug802872.js",
        "dom/security/test/csp/file_bug941404.html",
        "dom/security/test/csp/file_frameancestors_main.js",
        "dom/security/test/csp/file_main.js",
        "dom/security/test/csp/file_navigate_to.html",
        "dom/security/test/csp/file_navigate_to_request.html",
        "dom/security/test/csp/file_null_baseuri.html",
        "dom/security/test/csp/file_path_matching_redirect_server.sjs",
        "dom/security/test/csp/file_punycode_host_src.sjs",
        "dom/security/test/csp/file_redirect_content.sjs",
        "dom/security/test/csp/file_redirect_report.sjs",
        "dom/security/test/csp/file_redirect_worker.sjs",
        "dom/security/test/csp/file_redirects_main.html",
        "dom/security/test/csp/file_redirects_resource.sjs",
        "dom/security/test/csp/file_sendbeacon.html",
        "dom/security/test/csp/file_strict_dynamic_non_parser_inserted.html",
        "dom/security/test/csp/file_uir_top_nav.html",
        "dom/security/test/csp/file_windowwatcher_subframeB.html",
        "dom/security/test/csp/main_csp_worker.html",
        "dom/security/test/csp/test_301_redirect.html",
        "dom/security/test/csp/test_302_redirect.html",
        "dom/security/test/csp/test_303_redirect.html",
        "dom/security/test/csp/test_307_redirect.html",
        "dom/security/test/csp/test_allow_https_schemes.html",
        "dom/security/test/csp/test_base-uri.html",
        "dom/security/test/csp/test_blocked_uri_in_reports.html",
        "dom/security/test/csp/test_blocked_uri_in_violation_event_after_redirects.html",
        "dom/security/test/csp/test_blocked_uri_redirect_frame_src.html",
        "dom/security/test/csp/test_navigate_to.html",
        "dom/security/test/csp/test_null_baseuri.html",
        "dom/security/test/csp/test_path_matching.html",
        "dom/security/test/csp/test_report_for_import.html",
        "dom/security/test/csp/test_uir_top_nav.html",
        "dom/security/test/csp/test_upgrade_insecure.html",
        "dom/security/test/csp/test_upgrade_insecure_navigation.html",
        "dom/security/test/csp/test_websocket_self.html",
        "dom/security/test/general/browser_same_site_cookies_bug1748693.js",
        "dom/security/test/general/browser_test_data_download.js",
        "dom/security/test/general/browser_test_data_text_csv.js",
        "dom/security/test/general/browser_test_toplevel_data_navigations.js",
        "dom/security/test/general/file_same_site_cookies_redirect.sjs",
        "dom/security/test/general/file_same_site_cookies_toplevel_set_cookie.sjs",
        "dom/security/test/general/test_bug1450853.html",
        "dom/security/test/general/test_same_site_cookies_about.html",
        "dom/security/test/general/test_same_site_cookies_cross_origin_context.html",
        "dom/security/test/general/test_same_site_cookies_from_script.html",
        "dom/security/test/general/test_same_site_cookies_iframe.html",
        "dom/security/test/general/test_same_site_cookies_laxByDefault.html",
        "dom/security/test/general/test_same_site_cookies_redirect.html",
        "dom/security/test/general/test_same_site_cookies_subrequest.html",
        "dom/security/test/general/test_same_site_cookies_toplevel_nav.html",
        "dom/security/test/general/test_same_site_cookies_toplevel_set_cookie.html",
        "dom/security/test/https-first/browser_download_attribute.js",
        "dom/security/test/https-first/browser_httpsfirst.js",
        "dom/security/test/https-first/browser_httpsfirst_console_logging.js",
        "dom/security/test/https-first/browser_httpsfirst_speculative_connect.js",
        "dom/security/test/https-first/browser_mixed_content_console.js",
        "dom/security/test/https-first/browser_mixed_content_download.js",
        "dom/security/test/https-first/browser_navigation.js",
        "dom/security/test/https-first/browser_slow_download.js",
        "dom/security/test/https-first/browser_upgrade_onion.js",
        "dom/security/test/https-first/download_page.html",
        "dom/security/test/https-first/file_multiple_redirection.sjs",
        "dom/security/test/https-first/file_redirect.sjs",
        "dom/security/test/https-first/test_bad_cert.html",
        "dom/security/test/https-first/test_form_submission.html",
        "dom/security/test/https-first/test_fragment.html",
        "dom/security/test/https-first/test_multiple_redirection.html",
        "dom/security/test/https-first/test_referrer_policy.html",
        "dom/security/test/https-first/test_resource_upgrade.html",
        "dom/security/test/https-first/test_toplevel_cookies.html",
        "dom/security/test/https-only/browser_background_redirect.js",
        "dom/security/test/https-only/browser_console_logging.js",
        "dom/security/test/https-only/browser_hsts_host.js",
        "dom/security/test/https-only/browser_httpsonly_prefs.js",
        "dom/security/test/https-only/browser_httpsonly_speculative_connect.js",
        "dom/security/test/https-only/browser_iframe_test.js",
        "dom/security/test/https-only/browser_triggering_principal_exemption.js",
        "dom/security/test/https-only/browser_upgrade_exceptions.js",
        "dom/security/test/https-only/browser_user_gesture.js",
        "dom/security/test/https-only/file_redirect.sjs",
        "dom/security/test/https-only/test_fragment.html",
        "dom/security/test/https-only/test_http_background_auth_request.html",
        "dom/security/test/https-only/test_http_background_request.html",
        "dom/security/test/https-only/test_resource_upgrade.html",
        "dom/security/test/https-only/test_user_suggestion_box.html",
        "dom/security/test/mixedcontentblocker/browser_test_mixed_content_download.js",
        "dom/security/test/mixedcontentblocker/download_page.html",
        "dom/security/test/mixedcontentblocker/file_auth_download_page.html",
        "dom/security/test/mixedcontentblocker/file_frameNavigation.html",
        "dom/security/test/mixedcontentblocker/file_frameNavigation_blankTarget.html",
        "dom/security/test/mixedcontentblocker/file_frameNavigation_secure_grandchild.html",
        "dom/security/test/mixedcontentblocker/file_main.html",
        "dom/security/test/mixedcontentblocker/file_main_bug803225.html",
        "dom/security/test/mixedcontentblocker/file_redirect_handler.sjs",
        "dom/security/test/referrer-policy/browser_referrer_disallow_cross_site_relaxing.js",
        "dom/security/test/referrer-policy/img_referrer_testserver.sjs",
        "dom/security/test/referrer-policy/referrer_testserver.sjs",
        "dom/serviceworkers/test/browser_devtools_serviceworker_interception.js",
        "dom/serviceworkers/test/eventsource/eventsource_opaque_response_intercept_worker.js",
        "dom/serviceworkers/test/fetch/fetch_tests.js",
        "dom/serviceworkers/test/fetch/hsts/image.html",
        "dom/serviceworkers/test/fetch/importscript-mixedcontent/https_test.js",
        "dom/serviceworkers/test/fetch/origin/index.sjs",
        "dom/serviceworkers/test/fetch/requesturl/redirect.sjs",
        "dom/serviceworkers/test/fetch_event_worker.js",
        "dom/serviceworkers/test/opaque_intercept_worker.js",
        "dom/serviceworkers/test/openWindow_worker.js",
        "dom/serviceworkers/test/simpleregister/index.html",
        "dom/serviceworkers/test/test_hsts_upgrade_intercept.html",
        "dom/serviceworkers/test/test_installation_simple.html",
        "dom/serviceworkers/test/test_origin_after_redirect.html",
        "dom/serviceworkers/test/test_origin_after_redirect_cached.html",
        "dom/serviceworkers/test/test_sanitize_domain.html",
        "dom/serviceworkers/test/xslt_worker.js",
        "dom/streams/test/xpcshell/fetch.js",
        "dom/svg/test/test_use_with_hsts.html",
        "dom/tests/browser/browser_ConsoleStorageAPITests.js",
        "dom/tests/browser/browser_ConsoleStoragePBTest_perwindowpb.js",
        "dom/tests/browser/browser_beforeunload_between_chrome_content.js",
        "dom/tests/browser/browser_bug1004814.js",
        "dom/tests/browser/browser_bug1238427.js",
        "dom/tests/browser/browser_bug396843.js",
        "dom/tests/browser/browser_bytecode_cache_asm_js.js",
        "dom/tests/browser/browser_frame_elements.js",
        "dom/tests/browser/browser_hasbeforeunload.js",
        "dom/tests/browser/browser_localStorage_e10s.js",
        "dom/tests/browser/browser_localStorage_snapshotting.js",
        "dom/tests/browser/browser_test_toolbars_visibility.js",
        "dom/tests/browser/browser_windowProxy_transplant.js",
        "dom/tests/mochitest/beacon/file_beaconSafelist.html",
        "dom/tests/mochitest/beacon/test_beaconOriginHeader.html",
        "dom/tests/mochitest/beacon/test_beaconPreflightWithCustomContentType.html",
        "dom/tests/mochitest/beacon/test_beaconRedirect.html",
        "dom/tests/mochitest/beacon/test_beaconWithSafelistedContentType.html",
        "dom/tests/mochitest/bugs/grandchild_bug260264.html",
        "dom/tests/mochitest/bugs/test_bug132255.html",
        "dom/tests/mochitest/bugs/test_bug260264.html",
        "dom/tests/mochitest/bugs/test_bug260264_nested.html",
        "dom/tests/mochitest/bugs/test_bug304459.html",
        "dom/tests/mochitest/bugs/test_bug42976.html",
        "dom/tests/mochitest/bugs/test_bug440572.html",
        "dom/tests/mochitest/bugs/test_bug850517.html",
        "dom/tests/mochitest/bugs/test_bug873229.html",
        "dom/tests/mochitest/bugs/window_bug1171215.html",
        "dom/tests/mochitest/chrome/test_elements_proto.xhtml",
        "dom/tests/mochitest/chrome/test_sandbox_eventhandler.xhtml",
        "dom/tests/mochitest/dom-level0/test_background_loading_iframes.html",
        "dom/tests/mochitest/dom-level0/test_location.html",
        "dom/tests/mochitest/dom-level0/test_location_setters.html",
        "dom/tests/mochitest/dom-level0/test_setting_document.domain_idn.html",
        "dom/tests/mochitest/fetch/test_request.js",
        "dom/tests/mochitest/fetch/test_response.js",
        "dom/tests/mochitest/general/resource_timing_cross_origin.html",
        "dom/tests/mochitest/general/test_frameElementWrapping.html",
        "dom/tests/mochitest/general/test_network_events.html",
        "dom/tests/mochitest/general/window_clipboard_events.html",
        "dom/tests/mochitest/localstorage/test_keySync.html",
        "dom/tests/mochitest/localstorage/test_localStorageFromChrome.xhtml",
        "dom/tests/mochitest/localstorage/test_localStorageOriginsDiff.html",
        "dom/tests/mochitest/localstorage/test_localStorageOriginsDomainDiffs.html",
        "dom/tests/mochitest/localstorage/test_localStorageOriginsEquals.html",
        "dom/tests/mochitest/localstorage/test_localStorageOriginsPortDiffs.html",
        "dom/tests/mochitest/localstorage/test_localStorageOriginsSchemaDiffs.html",
        "dom/tests/mochitest/localstorage/test_localStorageQuota.html",
        "dom/tests/mochitest/localstorage/test_localStorageQuotaPrivateBrowsing_perwindowpb.html",
        "dom/tests/mochitest/localstorage/test_localStorageQuotaSessionOnly.html",
        "dom/tests/mochitest/localstorage/test_localStorageQuotaSessionOnly2.html",
        "dom/tests/mochitest/localstorage/test_localStorageReplace.html",
        "dom/tests/mochitest/sessionstorage/test_sessionStorageHttpHttps.html",
        "dom/tests/mochitest/sessionstorage/test_sessionStorageReplace.html",
        "dom/tests/mochitest/storageevent/frameLocalStorageSlaveEqual.html",
        "dom/tests/mochitest/storageevent/frameSessionStorageMasterNotEqual.html",
        "dom/tests/mochitest/storageevent/frameSessionStorageSlaveEqual.html",
        "dom/tests/mochitest/storageevent/test_storageLocalStorageEventCheckNoPropagation.html",
        "dom/tests/mochitest/storageevent/test_storageLocalStorageEventCheckPropagation.html",
        "dom/tests/mochitest/storageevent/test_storageSessionStorageEventCheckNoPropagation.html",
        "dom/tests/mochitest/storageevent/test_storageSessionStorageEventCheckPropagation.html",
        "dom/tests/mochitest/webcomponents/test_custom_element_htmlconstructor_chrome.html",
        "dom/tests/mochitest/webcomponents/test_custom_element_namespace.html",
        "dom/tests/mochitest/webcomponents/test_custom_element_namespace.xhtml",
        "dom/tests/mochitest/webcomponents/test_custom_element_upgrade_chrome.html",
        "dom/tests/mochitest/webcomponents/test_shadowroot_inert_element.html",
        "dom/tests/mochitest/webcomponents/test_xul_custom_element.xhtml",
        "dom/tests/mochitest/webcomponents/test_xul_shadowdom_accesskey.xhtml",
        "dom/tests/mochitest/whatwg/postMessage_joined_helper.html",
        "dom/tests/mochitest/whatwg/postMessage_joined_helper2.html",
        "dom/tests/mochitest/whatwg/postMessage_onOther.html",
        "dom/tests/mochitest/whatwg/test_MessageEvent.html",
        "dom/tests/mochitest/whatwg/test_MessageEvent_dispatchToOther.html",
        "dom/tests/mochitest/whatwg/test_bug500328.html",
        "dom/tests/mochitest/whatwg/test_postMessage.html",
        "dom/tests/mochitest/whatwg/test_postMessage_chrome.html",
        "dom/tests/mochitest/whatwg/test_postMessage_idn.xhtml",
        "dom/tests/mochitest/whatwg/test_postMessage_joined.html",
        "dom/tests/mochitest/whatwg/test_postMessage_onOther.html",
        "dom/tests/mochitest/whatwg/test_postMessage_origin.xhtml",
        "dom/tests/mochitest/whatwg/test_postMessage_override.html",
        "dom/tests/mochitest/whatwg/test_postMessage_structured_clone.html",
        "dom/tests/mochitest/whatwg/test_postMessage_throw.html",
        "dom/tests/mochitest/whatwg/test_postMessage_transfer.html",
        "dom/tests/mochitest/whatwg/test_postMessage_userpass.html",
        "dom/u2f/tests/frame_appid_facet.html",
        "dom/u2f/tests/frame_appid_facet_insecure.html",
        "dom/u2f/tests/frame_appid_facet_subdomain.html",
        "dom/url/tests/browser_download_after_revoke.js",
        "dom/url/tests/file_url.jsm ",
        "dom/url/tests/protocol_worker.js",
        "dom/url/tests/test_unknown_url_origin.html",
        "dom/url/tests/test_url.html",
        "dom/url/tests/test_urlExceptions.html",
        "dom/url/tests/test_url_data.html",
        "dom/url/tests/test_url_empty_port.html",
        "dom/url/tests/test_url_malformedHost.html",
        "dom/url/tests/test_urlutils_stringify.html",
        "dom/url/tests/urlApi_worker.js",
        "dom/url/tests/urlSearchParams_commons.js",
        "dom/url/tests/url_exceptions_worker.js",
        "dom/websocket/tests/test_websocket_mixed_content_opener.html",
        "dom/websocket/tests/window_websocket_wss.html",
        "dom/workers/test/browser_fileURL.js",
        "dom/workers/test/bug1063538_worker.js",
        "dom/workers/test/importForeignScripts_worker.js",
        "dom/workers/test/importScripts_worker.js",
        "dom/workers/test/redirect_to_foreign.sjs",
        "dom/workers/test/referrer_test_server.sjs",
        "dom/workers/test/sharedWorker_thirdparty_window.html",
        "dom/workers/test/test_multi_sharedWorker.html",
        "dom/workers/test/test_multi_sharedWorker_lifetimes_bfcache.html",
        "dom/workers/test/test_multi_sharedWorker_lifetimes_nobfcache.html",
        "dom/workers/test/test_subworkers_suspended.html",
        "dom/workers/test/test_suspend.html",
        "dom/xhr/tests/file_XHRResponseURL.js",
        "dom/xhr/tests/file_XHRSendData.sjs",
        "dom/xhr/tests/test_XHRDocURI.html",
        "dom/xhr/tests/test_XHR_anon.html",
        "dom/xhr/tests/test_XHR_system.html",
        "dom/xhr/tests/test_bug1070763.html",
        "dom/xhr/tests/test_worker_xhr_headers.html",
        "dom/xhr/tests/test_worker_xhr_system.js",
        "dom/xhr/tests/test_xhr_forbidden_headers.html",
        "dom/xslt/tests/mochitest/test_bug440974.html",
        "editor/libeditor/tests/browser_bug527935.js",
        "editor/libeditor/tests/test_bug1181130-2.html",
        "editor/libeditor/tests/test_bug372345.html",
        "extensions/permissions/test/browser_permmgr_sync.js",
        "gfx/layers/apz/test/mochitest/browser_test_select_popup_position.js",
        "gfx/layers/apz/test/mochitest/helper_wheelevents_handoff_on_non_scrollable_iframe.html",
        "image/test/browser/head.js",
        "image/test/mochitest/test_animated_gif.html",
        "image/test/mochitest/test_bug671906.html",
        "intl/locale/tests/LangPackMatcherTestUtils.jsm ",
        "layout/base/tests/browser_disableDialogs_onbeforeunload.js",
        "layout/base/tests/browser_lazyload_page_load_telemetry_iframe.js",
        "layout/base/tests/browser_lazyload_telemetry.js",
        "layout/base/tests/browser_onbeforeunload_only_after_interaction.js",
        "layout/base/tests/browser_onbeforeunload_only_after_interaction_in_frame.js",
        "layout/forms/test/test_bug536567_perwindowpb.html",
        "layout/generic/test/frame_visibility_in_iframe.html",
        "layout/generic/test/test_movement_by_words.html",
        "layout/style/test/browser_sourcemap.js",
        "layout/style/test/test_bug397427.html",
        "layout/style/test/test_load_events_on_stylesheets.html",
        "layout/style/test/test_root_node_display.html",
        "layout/style/test/test_selectors.html",
        "layout/svg/tests/file_context_fill_fallback_red.html",
        "layout/svg/tests/test_embed_sizing.html",
        "layout/xul/test/test_bug386386.html",
        "mobile/android/components/extensions/test/mochitest/test_ext_tabs_create.html",
        "mobile/android/components/extensions/test/mochitest/test_ext_tabs_events.html",
        "mobile/android/components/extensions/test/mochitest/test_ext_tabs_executeScript.html",
        "mobile/android/components/extensions/test/mochitest/test_ext_tabs_executeScript_bad.html",
        "mobile/android/components/extensions/test/mochitest/test_ext_tabs_query.html",
        "mobile/android/components/extensions/test/mochitest/test_ext_tabs_sendMessage.html",
        "mobile/android/components/extensions/test/mochitest/test_ext_tabs_update_url.html",
        "netwerk/cookie/test/browser/browser_cookies_ipv6.js",
        "netwerk/cookie/test/browser/browser_originattributes.js",
        "netwerk/cookie/test/browser/browser_oversize.js",
        "netwerk/cookie/test/browser/browser_sameSiteConsole.js",
        "netwerk/test/browser/browser_103_assets.js",
        "netwerk/test/browser/browser_103_preload.js",
        "netwerk/test/browser/browser_103_telemetry.js",
        "netwerk/test/browser/browser_cookie_filtering_subdomain.js",
        "netwerk/test/browser/browser_test_favicon.js",
        "netwerk/test/browser/cookie_filtering_helper.jsm",
        "netwerk/test/browser/early_hint_preload_test_helper.jsm ",
        "netwerk/test/httpserver/httpd.js",
        "netwerk/test/mochitests/file_loadinfo_redirectchain.sjs",
        "netwerk/test/mochitests/file_testloadflags_chromescript.js",
        "netwerk/test/mochitests/test_1396395.html",
        "netwerk/test/mochitests/test_loadinfo_redirectchain.html",
        "netwerk/test/mochitests/test_uri_scheme.html",
        "netwerk/test/unit/trr_common.js",
        "remote/cdp/test/browser/browser_httpd.js",
        "remote/cdp/test/browser/head.js",
        "remote/cdp/test/browser/network/browser_deleteCookies.js",
        "remote/cdp/test/browser/network/browser_getAllCookies.js",
        "remote/cdp/test/browser/network/browser_getCookies.js",
        "remote/cdp/test/browser/network/browser_navigationEvents.js",
        "remote/cdp/test/browser/network/browser_setCookie.js",
        "remote/shared/messagehandler/test/browser/browser_session_data.js",
        "security/manager/ssl/tests/mochitest/browser/some_content_framed.html",
        "security/manager/ssl/tests/mochitest/mixedcontent/iframeunsecredirect.sjs",
        "security/manager/ssl/tests/mochitest/mixedcontent/imgunsecredirect.sjs",
        "security/manager/ssl/tests/mochitest/mixedcontent/redirecttoemptyimage.sjs",
        "security/manager/ssl/tests/mochitest/mixedcontent/test_bug329869.html",
        "security/manager/ssl/tests/mochitest/mixedcontent/test_dynDelayedUnsecurePicture.html",
        "security/manager/ssl/tests/mochitest/mixedcontent/test_dynDelayedUnsecureXHR.html",
        "security/manager/ssl/tests/mochitest/mixedcontent/test_dynUnsecureBackground.html",
        "security/manager/ssl/tests/mochitest/mixedcontent/test_dynUnsecurePicture.html",
        "security/manager/ssl/tests/mochitest/mixedcontent/test_dynUnsecurePicturePreload.html",
        "security/sandbox/test/browser_bug1393259.js",
        "testing/mochitest/chrome/test_sanityEventUtils.xhtml",
        "testing/mochitest/server.js",
        "testing/mochitest/tests/Harness_sanity/test_SpecialPowersExtension.html",
        "testing/mochitest/tests/SimpleTest/TestRunner.js",
        "testing/mozbase/mozprofile/tests/files/prefs_with_interpolation.js",
        "testing/raptor/webext/raptor/runner.js",
        "testing/talos/talos/tests/cpstartup/extension/api.js",
        "testing/talos/talos/tests/devtools/addon/content/tests/debugger/custom.js",
        "testing/talos/talos/tests/devtools/addon/content/tests/head.js",
        "testing/talos/talos/tests/tabswitch/api.js",
        "testing/talos/talos/webextensions/dummy/background.js",
        "testing/xpcshell/moz-http2/moz-http2.js",
        "toolkit/components/aboutprocesses/tests/browser/head.js",
        "toolkit/components/alerts/test/browser_bug1682866.js",
        "toolkit/components/antitracking/test/browser/browser_contentBlockingAllowListPrincipal.js",
        "toolkit/components/antitracking/test/browser/browser_partitionedServiceWorkers.js",
        "toolkit/components/antitracking/test/browser/browser_permissionInNormalWindows.js",
        "toolkit/components/antitracking/test/browser/browser_permissionInNormalWindows_alwaysPartition.js",
        "toolkit/components/antitracking/test/browser/browser_socialtracking_save_image.js",
        "toolkit/components/antitracking/test/browser/browser_staticPartition_CORS_preflight.js",
        "toolkit/components/antitracking/test/browser/browser_staticPartition_HSTS.js",
        "toolkit/components/antitracking/test/browser/browser_staticPartition_cache.js",
        "toolkit/components/antitracking/test/browser/browser_staticPartition_websocket.js",
        "toolkit/components/antitracking/test/browser/browser_storageAccessPrivilegeAPI.js",
        "toolkit/components/antitracking/test/browser/browser_storageAccessSandboxed.js",
        "toolkit/components/antitracking/test/browser/browser_storageAccessSandboxed_alwaysPartition.js",
        "toolkit/components/antitracking/test/browser/browser_storageAccessThirdPartyChecks.js",
        "toolkit/components/antitracking/test/browser/browser_storageAccessThirdPartyChecks_alwaysPartition.js",
        "toolkit/components/antitracking/test/browser/browser_storageAccess_TopLevel_Embed.js",
        "toolkit/components/antitracking/test/browser/browser_thirdPartyStorageRejectionForCORS.js",
        "toolkit/components/antitracking/test/browser/browser_urlQueryStringStripping.js",
        "toolkit/components/antitracking/test/browser/cookiesCORS.sjs",
        "toolkit/components/antitracking/test/browser/head.js",
        "toolkit/components/antitracking/test/browser/storageAccessAPIHelpers.js",
        "toolkit/components/certviewer/tests/browser/adjustedCerts.js",
        "toolkit/components/certviewer/tests/chrome/CSoutput.mjs",
        "toolkit/components/certviewer/tests/chrome/parseOutput.mjs",
        "toolkit/components/cleardata/tests/browser/browser_css_cache.js",
        "toolkit/components/cleardata/tests/browser/browser_image_cache.js",
        "toolkit/components/cleardata/tests/browser/browser_preflight_cache.js",
        "toolkit/components/cleardata/tests/browser/browser_quota.js",
        "toolkit/components/cleardata/tests/browser/browser_sessionStorage.js",
        "toolkit/components/cookiebanners/test/browser/browser_cookiebannerservice.js",
        "toolkit/components/downloads/test/unit/common_test_Download.js",
        "toolkit/components/extensions/test/browser/browser_ext_downloads_filters.js",
        "toolkit/components/extensions/test/browser/browser_ext_extension_page_tab_navigated.js",
        "toolkit/components/extensions/test/browser/browser_ext_themes_autocomplete_popup.js",
        "toolkit/components/extensions/test/browser/browser_ext_thumbnails_bg_extension.js",
        "toolkit/components/extensions/test/mochitest/file_third_party.html",
        "toolkit/components/extensions/test/mochitest/test_chrome_ext_permissions.html",
        "toolkit/components/extensions/test/mochitest/test_chrome_ext_trackingprotection.html",
        "toolkit/components/extensions/test/mochitest/test_chrome_ext_webrequest_host_permissions.html",
        "toolkit/components/extensions/test/mochitest/test_ext_contentscript_activeTab.html",
        "toolkit/components/extensions/test/mochitest/test_ext_cookies.html",
        "toolkit/components/extensions/test/mochitest/test_ext_cookies_containers.html",
        "toolkit/components/extensions/test/mochitest/test_ext_cookies_expiry.html",
        "toolkit/components/extensions/test/mochitest/test_ext_cookies_first_party.html",
        "toolkit/components/extensions/test/mochitest/test_ext_cookies_incognito.html",
        "toolkit/components/extensions/test/mochitest/test_ext_cookies_permissions_bad.html",
        "toolkit/components/extensions/test/mochitest/test_ext_cookies_permissions_good.html",
        "toolkit/components/extensions/test/mochitest/test_ext_embeddedimg_iframe_frameAncestors.html",
        "toolkit/components/extensions/test/mochitest/test_ext_optional_permissions.html",
        "toolkit/components/extensions/test/mochitest/test_ext_protocolHandlers.html",
        "toolkit/components/extensions/test/mochitest/test_ext_request_urlClassification.html",
        "toolkit/components/extensions/test/mochitest/test_ext_runtime_disconnect.html",
        "toolkit/components/extensions/test/mochitest/test_ext_storage_manager_capabilities.html",
        "toolkit/components/extensions/test/mochitest/test_ext_streamfilter_multiple.html",
        "toolkit/components/extensions/test/mochitest/test_ext_streamfilter_processswitch.html",
        "toolkit/components/extensions/test/mochitest/test_ext_tabs_executeScript_good.html",
        "toolkit/components/extensions/test/mochitest/test_ext_tabs_query_popup.html",
        "toolkit/components/extensions/test/mochitest/test_ext_tabs_sendMessage.html",
        "toolkit/components/extensions/test/mochitest/test_ext_web_accessible_incognito.html",
        "toolkit/components/extensions/test/mochitest/test_ext_web_accessible_resources.html",
        "toolkit/components/extensions/test/mochitest/test_ext_webrequest_and_proxy_filter.html",
        "toolkit/components/extensions/test/mochitest/test_ext_webrequest_hsts.html",
        "toolkit/components/extensions/test/xpcshell/data/file_permission_xhr.html",
        "toolkit/components/extensions/test/xpcshell/data/file_script_xhr.js",
        "toolkit/components/glean/tests/browser/browser_fog_rdd.js",
        "toolkit/components/httpsonlyerror/tests/browser/browser_errorpage.js",
        "toolkit/components/httpsonlyerror/tests/browser/browser_errorpage_timeout.js",
        "toolkit/components/httpsonlyerror/tests/browser/browser_errorpage_www_suggestion.js",
        "toolkit/components/httpsonlyerror/tests/browser/browser_exception.js",
        "toolkit/components/narrate/test/head.js",
        "toolkit/components/normandy/test/NormandyTestUtils.jsm ",
        "toolkit/components/normandy/test/browser/browser_RecipeRunner.js",
        "toolkit/components/normandy/test/browser/browser_actions_ShowHeartbeatAction.js",
        "toolkit/components/normandy/test/browser/head.js",
        "toolkit/components/passwordmgr/test/LoginTestUtils.jsm ",
        "toolkit/components/passwordmgr/test/browser/browser_autofill_http.js",
        "toolkit/components/passwordmgr/test/browser/browser_context_menu.js",
        "toolkit/components/passwordmgr/test/browser/browser_context_menu_iframe.js",
        "toolkit/components/passwordmgr/test/browser/browser_doorhanger_httpsUpgrade.js",
        "toolkit/components/passwordmgr/test/browser/browser_doorhanger_remembering.js",
        "toolkit/components/passwordmgr/test/browser/browser_localip_frame.js",
        "toolkit/components/passwordmgr/test/browser/head.js",
        "toolkit/components/passwordmgr/test/mochitest/test_autocomplete_basic_form_insecure.html",
        "toolkit/components/passwordmgr/test/mochitest/test_autocomplete_basic_form_subdomain.html",
        "toolkit/components/passwordmgr/test/mochitest/test_autocomplete_highlight_non_login.html",
        "toolkit/components/passwordmgr/test/mochitest/test_autocomplete_https_upgrade.html",
        "toolkit/components/passwordmgr/test/mochitest/test_autofill_different_formActionOrigin.html",
        "toolkit/components/passwordmgr/test/mochitest/test_autofill_https_upgrade.html",
        "toolkit/components/passwordmgr/test/mochitest/test_autofill_password-only.html",
        "toolkit/components/passwordmgr/test/mochitest/test_basic_form_html5.html",
        "toolkit/components/passwordmgr/test/mochitest/test_bug_627616.html",
        "toolkit/components/passwordmgr/test/mochitest/test_formless_submit_form_removal.html",
        "toolkit/components/passwordmgr/test/mochitest/test_formless_submit_form_removal_negative.html",
        "toolkit/components/passwordmgr/test/mochitest/test_formless_submit_navigation.html",
        "toolkit/components/passwordmgr/test/mochitest/test_formless_submit_navigation_negative.html",
        "toolkit/components/passwordmgr/test/mochitest/test_password_field_autocomplete.html",
        "toolkit/components/passwordmgr/test/mochitest/test_prompt.html",
        "toolkit/components/passwordmgr/test/mochitest/test_prompt_async.html",
        "toolkit/components/passwordmgr/test/mochitest/test_prompt_http.html",
        "toolkit/components/passwordmgr/test/mochitest/test_prompt_promptAuth.html",
        "toolkit/components/passwordmgr/test/mochitest/test_prompt_promptAuth_proxy.html",
        "toolkit/components/passwordmgr/test/mochitest/test_username_focus.html",
        "toolkit/components/pdfjs/test/browser_pdfjs_download_button.js",
        "toolkit/components/pdfjs/test/browser_pdfjs_editing_contextmenu.js",
        "toolkit/components/pdfjs/test/browser_pdfjs_editing_telemetry.js",
        "toolkit/components/pdfjs/test/browser_pdfjs_fill_login.js",
        "toolkit/components/pdfjs/test/browser_pdfjs_form.js",
        "toolkit/components/pdfjs/test/browser_pdfjs_hcm.js",
        "toolkit/components/pdfjs/test/browser_pdfjs_js.js",
        "toolkit/components/pdfjs/test/browser_pdfjs_main.js",
        "toolkit/components/pdfjs/test/browser_pdfjs_navigation.js",
        "toolkit/components/pdfjs/test/browser_pdfjs_notification_close_on_navigation.js",
        "toolkit/components/pdfjs/test/browser_pdfjs_saveas.js",
        "toolkit/components/pdfjs/test/browser_pdfjs_savedialog.js",
        "toolkit/components/pdfjs/test/browser_pdfjs_views.js",
        "toolkit/components/pdfjs/test/browser_pdfjs_zoom.js",
        "toolkit/components/pictureinpicture/tests/head.js",
        "toolkit/components/places/tests/browser/browser_bug1601563.js",
        "toolkit/components/places/tests/browser/browser_bug399606.js",
        "toolkit/components/places/tests/browser/browser_bug461710.js",
        "toolkit/components/places/tests/browser/browser_bug646422.js",
        "toolkit/components/places/tests/browser/browser_double_redirect.js",
        "toolkit/components/places/tests/browser/browser_favicon_privatebrowsing_perwindowpb.js",
        "toolkit/components/places/tests/browser/browser_history_post.js",
        "toolkit/components/places/tests/browser/browser_multi_redirect_frecency.js",
        "toolkit/components/places/tests/browser/browser_onvisit_title_null_for_navigation.js",
        "toolkit/components/places/tests/browser/browser_settitle.js",
        "toolkit/components/places/tests/browser/browser_visituri.js",
        "toolkit/components/places/tests/browser/browser_visituri_nohistory.js",
        "toolkit/components/places/tests/browser/browser_visituri_privatebrowsing_perwindowpb.js",
        "toolkit/components/places/tests/browser/previews/browser_thumbnails.js",
        "toolkit/components/places/tests/browser/redirect_once.sjs",
        "toolkit/components/places/tests/chrome/test_371798.xhtml",
        "toolkit/components/places/tests/chrome/test_favicon_annotations.xhtml",
        "toolkit/components/printing/tests/head.js",
        "toolkit/components/processtools/tests/browser/browser_test_powerMetrics.js",
        "toolkit/components/processtools/tests/browser/browser_test_procinfo.js",
        "toolkit/components/prompts/test/test_modal_prompts.html",
        "toolkit/components/prompts/test/test_subresources_prompts.html",
        "toolkit/components/reader/test/browser_bug1453818_samesite_cookie.js",
        "toolkit/components/reader/test/browser_drag_url_readerMode.js",
        "toolkit/components/reader/test/browser_readerMode_bc_reuse.js",
        "toolkit/components/reader/test/browser_readerMode_colorSchemePref.js",
        "toolkit/components/reader/test/browser_readerMode_hidden_nodes.js",
        "toolkit/components/reader/test/browser_readerMode_menu.js",
        "toolkit/components/reader/test/browser_readerMode_pocket.js",
        "toolkit/components/reader/test/browser_readerMode_readingTime.js",
        "toolkit/components/reader/test/browser_readerMode_refresh.js",
        "toolkit/components/reader/test/browser_readerMode_remoteType.js",
        "toolkit/components/reader/test/browser_readerMode_samesite_cookie_redirect.js",
        "toolkit/components/reader/test/browser_readerMode_with_anchor.js",
        "toolkit/components/resistfingerprinting/tests/test_spoof_english.html",
        "toolkit/components/satchel/test/browser/browser_privbrowsing_perwindowpb.js",
        "toolkit/components/startup/tests/browser/browser_bug511456.js",
        "toolkit/components/startup/tests/browser/browser_bug537449.js",
        "toolkit/components/thumbnails/test/browser_thumbnails_bg_basic.js",
        "toolkit/components/thumbnails/test/browser_thumbnails_bg_topsites.js",
        "toolkit/components/url-classifier/tests/mochitest/allowlistAnnotatedFrame.html",
        "toolkit/components/url-classifier/tests/mochitest/classifiedAnnotatedFrame.html",
        "toolkit/components/url-classifier/tests/mochitest/features.js",
        "toolkit/components/url-classifier/tests/mochitest/redirect_tracker.sjs",
        "toolkit/components/url-classifier/tests/mochitest/test_classifier.html",
        "toolkit/components/url-classifier/tests/mochitest/test_classifier_match.html",
        "toolkit/components/url-classifier/tests/mochitest/test_classifier_worker.html",
        "toolkit/components/url-classifier/tests/mochitest/test_classify_ping.html",
        "toolkit/components/url-classifier/tests/mochitest/test_classify_track.html",
        "toolkit/components/url-classifier/tests/mochitest/test_cryptomining.html",
        "toolkit/components/url-classifier/tests/mochitest/test_emailtracking.html",
        "toolkit/components/url-classifier/tests/mochitest/test_fingerprinting.html",
        "toolkit/components/url-classifier/tests/mochitest/test_reporturl.html",
        "toolkit/components/url-classifier/tests/mochitest/test_socialtracking.html",
        "toolkit/components/url-classifier/tests/mochitest/test_threathit_report.html",
        "toolkit/components/url-classifier/tests/mochitest/test_trackingprotection_bug1312515.html",
        "toolkit/components/url-classifier/tests/mochitest/test_trackingprotection_bug1580416.html",
        "toolkit/components/url-classifier/tests/mochitest/test_trackingprotection_whitelist.html",
        "toolkit/components/viewsource/test/browser/browser_contextmenu.js",
        "toolkit/components/viewsource/test/browser/browser_open_docgroup.js",
        "toolkit/components/viewsource/test/browser/browser_validatefilename.js",
        "toolkit/components/windowcreator/test/browser_save_form_input_state.js",
        "toolkit/content/tests/browser/browser_crash_previous_frameloader.js",
        "toolkit/content/tests/browser/browser_isSynthetic.js",
        "toolkit/content/tests/browser/head.js",
        "toolkit/content/tests/chrome/findbar_window.xhtml",
        "toolkit/content/tests/chrome/window_browser_drop.xhtml",
        "toolkit/modules/tests/browser/browser_AsyncPrefs.js",
        "toolkit/modules/tests/browser/browser_BrowserUtils.js",
        "toolkit/modules/tests/browser/browser_web_channel.js",
        "toolkit/modules/tests/browser/file_web_channel_iframe.html",
        "toolkit/modules/tests/chrome/test_bug544442_checkCert.xhtml",
        "toolkit/mozapps/downloads/tests/browser/browser_unknownContentType_extension.js",
        "toolkit/mozapps/downloads/tests/browser/browser_unknownContentType_policy.js",
        "toolkit/mozapps/extensions/test/browser/browser_disco_taar_telemetry.js",
        "toolkit/mozapps/extensions/test/browser/browser_file_xpi_no_process_switch.js",
        "toolkit/mozapps/extensions/test/browser/browser_history_navigation.js",
        "toolkit/mozapps/extensions/test/browser/browser_html_abuse_report_dialog.js",
        "toolkit/mozapps/extensions/test/browser/browser_html_detail_view.js",
        "toolkit/mozapps/extensions/test/browser/browser_html_discover_view.js",
        "toolkit/mozapps/extensions/test/browser/browser_html_recent_updates.js",
        "toolkit/mozapps/extensions/test/browser/browser_html_recommendations.js",
        "toolkit/mozapps/extensions/test/browser/browser_html_updates.js",
        "toolkit/mozapps/extensions/test/browser/browser_installssl.js",
        "toolkit/mozapps/extensions/test/browser/browser_updatessl.js",
        "toolkit/mozapps/extensions/test/browser/browser_webapi_abuse_report.js",
        "toolkit/mozapps/extensions/test/browser/head.js",
        "toolkit/mozapps/extensions/test/browser/head_abuse_report.js",
        "toolkit/mozapps/extensions/test/browser/head_disco.js",
        "toolkit/mozapps/extensions/test/browser/webapi_checknavigatedwindow.html",
        "toolkit/mozapps/extensions/test/xpinstall/browser_amosigned_trigger.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_amosigned_trigger_iframe.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_auth.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_auth2.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_auth3.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_auth4.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_badhash.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_badhashtype.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_block_fullscreen_prompt.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_bug540558.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_bug645699.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_bug645699_postDownload.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_bug672485.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_containers.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_cookies.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_cookies2.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_cookies3.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_cookies4.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_corrupt.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_datauri.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_doorhanger_installs.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_empty.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_hash.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_hash2.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_httphash.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_httphash2.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_httphash3.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_httphash4.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_httphash5.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_httphash6.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_installchrome.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_newwindow.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_offline.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_privatebrowsing.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_relative.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_softwareupdate.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_trigger_redirect.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_unsigned_trigger.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_unsigned_trigger_iframe.js",
        "toolkit/mozapps/extensions/test/xpinstall/browser_unsigned_trigger_xorigin.js",
        "toolkit/mozapps/extensions/test/xpinstall/bug645699.html",
        "toolkit/mozapps/extensions/test/xpinstall/head.js",
        "toolkit/mozapps/update/tests/data/sharedUpdateXML.js",
        "toolkit/mozapps/update/tests/data/xpcshellUtilsAUS.js",
        "toolkit/mozapps/update/tests/unit_aus_update/remoteUpdateXML.js",
        "toolkit/mozapps/update/tests/unit_aus_update/updateManagerXML.js",
        "tools/profiler/tests/browser/head.js",
        "uriloader/exthandler/tests/mochitest/browser_download_idn_blocklist.js",
        "uriloader/exthandler/tests/mochitest/head.js",
        "widget/tests/native_menus_window.xhtml",
        "widget/tests/standalone_native_menu_window.xhtml",
        "widget/tests/test_bug596600.xhtml",
        "widget/tests/test_sizemode_events.xhtml",
        "xpfe/appshell/test/test_windowlessBrowser.xhtml",
      ],
      rules: {
        "@microsoft/sdl/no-insecure-url": "off",
      },
    },
  ],
};
