/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

"use strict";

const EventEmitter = require("devtools/shared/event-emitter");
const { ConnectionManager } =
  require("devtools/shared/client/connection-manager");
const { Devices } = require("devtools/shared/apps/Devices.jsm");
const { dumpn } = require("devtools/shared/DevToolsUtils");
const { RuntimeTypes } =
  require("devtools/client/webide/modules/runtime-types");

const ADBScanner = {

  _runtimes: [],

  enable() {
    this._updateRuntimes = this._updateRuntimes.bind(this);
    Devices.on("register", this._updateRuntimes);
    Devices.on("unregister", this._updateRuntimes);
    Devices.on("addon-status-updated", this._updateRuntimes);
    this._updateRuntimes();
  },

  disable() {
    Devices.off("register", this._updateRuntimes);
    Devices.off("unregister", this._updateRuntimes);
    Devices.off("addon-status-updated", this._updateRuntimes);
  },

  _emitUpdated() {
    this.emit("runtime-list-updated");
  },

  _updateRuntimes() {
    if (this._updatingPromise) {
      return this._updatingPromise;
    }
    this._runtimes = [];
    const promises = [];
    for (const id of Devices.available()) {
      const device = Devices.getByName(id);
      promises.push(this._detectRuntimes(device));
    }
    this._updatingPromise = Promise.all(promises);
    this._updatingPromise.then(() => {
      this._emitUpdated();
      this._updatingPromise = null;
    }, () => {
      this._updatingPromise = null;
    });
    return this._updatingPromise;
  },

  _detectRuntimes: async function(device) {
    const model = await device.getModel();
    const detectedRuntimes =
      await AbrowserOnAndroidRuntime.detect(device, model);
    this._runtimes.push(...detectedRuntimes);
  },

  scan() {
    return this._updateRuntimes();
  },

  listRuntimes() {
    return this._runtimes;
  }

};

EventEmitter.decorate(ADBScanner);

function Runtime(device, model, socketPath) {
  this.device = device;
  this._model = model;
  this._socketPath = socketPath;
}

Runtime.prototype = {
  type: RuntimeTypes.USB,
  connect(connection) {
    const port = ConnectionManager.getFreeTCPPort();
    const local = "tcp:" + port;
    let remote;
    if (this._socketPath.startsWith("@")) {
      remote = "localabstract:" + this._socketPath.substring(1);
    } else {
      remote = "localfilesystem:" + this._socketPath;
    }
    return this.device.forwardPort(local, remote).then(() => {
      connection.host = "localhost";
      connection.port = port;
      connection.connect();
    });
  },
  get id() {
    return this.device.id + "|" + this._socketPath;
  },
};

function AbrowserOnAndroidRuntime(device, model, socketPath) {
  Runtime.call(this, device, model, socketPath);
}

// This requires Unix socket support from Abrowser for Android (35+)
AbrowserOnAndroidRuntime.detect = async function(device, model) {
  const runtimes = [];
  // A matching entry looks like:
  // 00000000: 00000002 00000000 00010000 0001 01 6551588
  //  /data/data/org.mozilla.fennec/abrowser-debugger-socket
  const query = "cat /proc/net/unix";
  const rawSocketInfo = await device.shell(query);
  let socketInfos = rawSocketInfo.split(/\r?\n/);
  // Filter to lines with "abrowser-debugger-socket"
  socketInfos = socketInfos.filter(l => l.includes("abrowser-debugger-socket"));
  // It's possible to have multiple lines with the same path, so de-dupe them
  const socketPaths = new Set();
  for (const socketInfo of socketInfos) {
    const socketPath = socketInfo.split(" ").pop();
    socketPaths.add(socketPath);
  }
  for (const socketPath of socketPaths) {
    const runtime = new AbrowserOnAndroidRuntime(device, model, socketPath);
    dumpn("Found " + runtime.name);
    runtimes.push(runtime);
  }
  return runtimes;
};

AbrowserOnAndroidRuntime.prototype = Object.create(Runtime.prototype);

Object.defineProperty(AbrowserOnAndroidRuntime.prototype, "name", {
  get() {
    // If using abstract socket address, it is "@org.mozilla.abrowser/..."
    // If using path base socket, it is "/data/data/<package>...""
    // Until Fennec 62 only supports path based UNIX domain socket, but
    // Fennec 63+ supports both path based and abstract socket.
    const packageName = this._socketPath.startsWith("@") ?
                        this._socketPath.substr(1).split("/")[0] :
                        this._socketPath.split("/")[3];
    let channel;
    switch (packageName) {
      case "org.mozilla.abrowser":
        channel = "";
        break;
      case "org.mozilla.abrowser_beta":
        channel = " Beta";
        break;
      case "org.mozilla.fennec_aurora":
        // This package name is now the one for Abrowser Nightly distributed
        // through the Google Play Store since "dawn project"
        // cf. https://bugzilla.mozilla.org/show_bug.cgi?id=1357351#c8
        channel = " Nightly";
        break;
      case "org.mozilla.fennec":
        channel = " Nightly";
        break;
      default:
        channel = " Custom";
    }
    return "Abrowser" + channel + " on Android (" +
           (this._model || this.device.id) + ")";
  }
});

exports.ADBScanner = ADBScanner;
