/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim: set ts=8 sts=2 et sw=2 tw=80: */
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this file,
 * You can obtain one at http://mozilla.org/MPL/2.0/. */

#ifndef mozilla_dom_WebIDLGlobalNameHash_h__
#define mozilla_dom_WebIDLGlobalNameHash_h__

#include "js/RootingAPI.h"
#include "nsTArray.h"
#include "mozilla/dom/BindingDeclarations.h"

namespace mozilla {
namespace dom {

struct WebIDLNameTableEntry;

namespace constructors {
namespace id {
enum ID : uint16_t;
} // namespace id
} // namespace constructors

class WebIDLGlobalNameHash
{
public:
  static void Init();
  static void Shutdown();

  // Check whether a constructor should be enabled for the given object.
  // Note that the object should NOT be an Xray, since Xrays will end up
  // defining constructors on the underlying object.
  // This is a typedef for the function type itself, not the function
  // pointer, so it's more obvious that pointers to a ConstructorEnabled
  // can be null.
  typedef bool
  (*ConstructorEnabled)(JSContext* cx, JS::Handle<JSObject*> obj);

  static void Register(uint16_t aNameOffset, uint16_t aNameLength,
                       CreateInterfaceObjectsMethod aCreate,
                       ConstructorEnabled aEnabled,
                       constructors::id::ID aConstructorId);

  static void Remove(const char* aName, uint32_t aLength);

  // Returns false if something failed. aFound is set to true if the name is in
  // the hash, whether it's enabled or not.
  static bool DefineIfEnabled(JSContext* aCx, JS::Handle<JSObject*> aObj,
                              JS::Handle<jsid> aId,
                              JS::MutableHandle<JS::PropertyDescriptor> aDesc,
                              bool* aFound);

  static bool MayResolve(jsid aId);

  // The type of names we're asking for.
  enum NameType {
    // All WebIDL names enabled for aObj.
    AllNames,
    // Only the names that are enabled for aObj and have not been resolved for
    // aObj in the past (and therefore can't have been deleted).
    UnresolvedNamesOnly
  };
  // Returns false if an exception has been thrown on aCx.
  static bool GetNames(JSContext* aCx, JS::Handle<JSObject*> aObj,
                       NameType aNameType,
                       JS::AutoIdVector& aNames);

private:
  friend struct WebIDLNameTableEntry;

  // The total number of names that we will add to the hash.
  // The value of sCount is generated by Codegen.py in RegisterBindings.cpp.
  static const uint32_t sCount;

  // The names that will be registered in the hash, concatenated as one big
  // string with \0 as a separator between names.
  // The value of sNames is generated by Codegen.py in RegisterBindings.cpp.
  static const char sNames[];
};

} // namespace dom
} // namespace mozilla

#endif // mozilla_dom_WebIDLGlobalNameHash_h__
