use std::path::Path;
use std::fs::File;
use std::io::Read;


pub static INDEX: &'static [u8] = include_bytes!("index.hbs");
pub static CSS: &'static [u8] = include_bytes!("book.css");
pub static FAVICON: &'static [u8] = include_bytes!("favicon.png");
pub static JS: &'static [u8] = include_bytes!("book.js");
pub static TOMORROW_NIGHT_CSS: &'static [u8] = include_bytes!("tomorrow-night.css");

/// The `Theme` struct should be used instead of the static variables because the `new()` method
/// will look if the user has a theme directory in his source folder and use the users theme instead
/// of the default.
///
/// You should exceptionnaly use the static variables only if you need the default theme even if the
/// user has specified another theme.
pub struct Theme {
    pub index: Vec<u8>,
    pub css: Vec<u8>,
    pub favicon: Vec<u8>,
    pub js: Vec<u8>,
    pub tomorrow_night_css: Vec<u8>,
}

impl Theme {
    pub fn new(src: &Path) -> Self {

        // Default theme
        let mut theme = Theme {
            index: INDEX.to_owned(),
            css: CSS.to_owned(),
            favicon: FAVICON.to_owned(),
            js: JS.to_owned(),
            tomorrow_night_css: TOMORROW_NIGHT_CSS.to_owned(),
        };

        // Check if the given path exists
        if !src.exists() || !src.is_dir() {
            return theme;
        }

        // Check for individual files if they exist

        // index.hbs
        if let Ok(mut f) = File::open(&src.join("index.hbs")) {
            theme.index.clear(); // Reset the value, because read_to_string appends...
            let _ = f.read_to_end(&mut theme.index);
        }

        // book.js
        if let Ok(mut f) = File::open(&src.join("book.js")) {
            theme.js.clear();
            let _ = f.read_to_end(&mut theme.js);
        }

        // book.css
        if let Ok(mut f) = File::open(&src.join("book.css")) {
            theme.css.clear();
            let _ = f.read_to_end(&mut theme.css);
        }

        // favicon.png
        if let Ok(mut f) = File::open(&src.join("favicon.png")) {
            theme.favicon.clear();
            let _ = f.read_to_end(&mut theme.favicon);
        }

        // tomorrow-night.css
        if let Ok(mut f) = File::open(&src.join("tomorrow-night.css")) {
            theme.tomorrow_night_css.clear();
            let _ = f.read_to_end(&mut theme.tomorrow_night_css);
        }

        theme
    }
}
