# Copyright 2019 Fortinet, Inc.
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Ansible.  If not, see <https://www.gnu.org/licenses/>.

# Make coding more python3-ish
from __future__ import (absolute_import, division, print_function)
__metaclass__ = type

import os
import json
import pytest
from mock import ANY
from ansible.module_utils.network.fortios.fortios import FortiOSHandler

try:
    from ansible.modules.network.fortios import fortios_firewall_ssh_setting
except ImportError:
    pytest.skip("Could not load required modules for testing", allow_module_level=True)


@pytest.fixture(autouse=True)
def connection_mock(mocker):
    connection_class_mock = mocker.patch('ansible.modules.network.fortios.fortios_firewall_ssh_setting.Connection')
    return connection_class_mock


fos_instance = FortiOSHandler(connection_mock)


def test_firewall_ssh_setting_creation(mocker):
    schema_method_mock = mocker.patch('ansible.module_utils.network.fortios.fortios.FortiOSHandler.schema')

    set_method_result = {'status': 'success', 'http_method': 'POST', 'http_status': 200}
    set_method_mock = mocker.patch('ansible.module_utils.network.fortios.fortios.FortiOSHandler.set', return_value=set_method_result)

    input_data = {
        'username': 'admin',
        'state': 'present',
        'firewall_ssh_setting': {
            'caname': 'test_value_3',
            'host_trusted_checking': 'enable',
            'hostkey_dsa1024': 'myhostname5',
            'hostkey_ecdsa256': 'myhostname6',
            'hostkey_ecdsa384': 'myhostname7',
            'hostkey_ecdsa521': 'myhostname8',
            'hostkey_ed25519': 'myhostname9',
            'hostkey_rsa2048': 'myhostname10',
            'untrusted_caname': 'test_value_11'
        },
        'vdom': 'root'}

    is_error, changed, response = fortios_firewall_ssh_setting.fortios_firewall_ssh(input_data, fos_instance)

    expected_data = {
        'caname': 'test_value_3',
        'host-trusted-checking': 'enable',
        'hostkey-dsa1024': 'myhostname5',
        'hostkey-ecdsa256': 'myhostname6',
        'hostkey-ecdsa384': 'myhostname7',
        'hostkey-ecdsa521': 'myhostname8',
        'hostkey-ed25519': 'myhostname9',
        'hostkey-rsa2048': 'myhostname10',
        'untrusted-caname': 'test_value_11'
    }

    set_method_mock.assert_called_with('firewall.ssh', 'setting', data=expected_data, vdom='root')
    schema_method_mock.assert_not_called()
    assert not is_error
    assert changed
    assert response['status'] == 'success'
    assert response['http_status'] == 200


def test_firewall_ssh_setting_creation_fails(mocker):
    schema_method_mock = mocker.patch('ansible.module_utils.network.fortios.fortios.FortiOSHandler.schema')

    set_method_result = {'status': 'error', 'http_method': 'POST', 'http_status': 500}
    set_method_mock = mocker.patch('ansible.module_utils.network.fortios.fortios.FortiOSHandler.set', return_value=set_method_result)

    input_data = {
        'username': 'admin',
        'state': 'present',
        'firewall_ssh_setting': {
            'caname': 'test_value_3',
            'host_trusted_checking': 'enable',
            'hostkey_dsa1024': 'myhostname5',
            'hostkey_ecdsa256': 'myhostname6',
            'hostkey_ecdsa384': 'myhostname7',
            'hostkey_ecdsa521': 'myhostname8',
            'hostkey_ed25519': 'myhostname9',
            'hostkey_rsa2048': 'myhostname10',
            'untrusted_caname': 'test_value_11'
        },
        'vdom': 'root'}

    is_error, changed, response = fortios_firewall_ssh_setting.fortios_firewall_ssh(input_data, fos_instance)

    expected_data = {
        'caname': 'test_value_3',
        'host-trusted-checking': 'enable',
        'hostkey-dsa1024': 'myhostname5',
        'hostkey-ecdsa256': 'myhostname6',
        'hostkey-ecdsa384': 'myhostname7',
        'hostkey-ecdsa521': 'myhostname8',
        'hostkey-ed25519': 'myhostname9',
        'hostkey-rsa2048': 'myhostname10',
        'untrusted-caname': 'test_value_11'
    }

    set_method_mock.assert_called_with('firewall.ssh', 'setting', data=expected_data, vdom='root')
    schema_method_mock.assert_not_called()
    assert is_error
    assert not changed
    assert response['status'] == 'error'
    assert response['http_status'] == 500


def test_firewall_ssh_setting_idempotent(mocker):
    schema_method_mock = mocker.patch('ansible.module_utils.network.fortios.fortios.FortiOSHandler.schema')

    set_method_result = {'status': 'error', 'http_method': 'DELETE', 'http_status': 404}
    set_method_mock = mocker.patch('ansible.module_utils.network.fortios.fortios.FortiOSHandler.set', return_value=set_method_result)

    input_data = {
        'username': 'admin',
        'state': 'present',
        'firewall_ssh_setting': {
            'caname': 'test_value_3',
            'host_trusted_checking': 'enable',
            'hostkey_dsa1024': 'myhostname5',
            'hostkey_ecdsa256': 'myhostname6',
            'hostkey_ecdsa384': 'myhostname7',
            'hostkey_ecdsa521': 'myhostname8',
            'hostkey_ed25519': 'myhostname9',
            'hostkey_rsa2048': 'myhostname10',
            'untrusted_caname': 'test_value_11'
        },
        'vdom': 'root'}

    is_error, changed, response = fortios_firewall_ssh_setting.fortios_firewall_ssh(input_data, fos_instance)

    expected_data = {
        'caname': 'test_value_3',
        'host-trusted-checking': 'enable',
        'hostkey-dsa1024': 'myhostname5',
        'hostkey-ecdsa256': 'myhostname6',
        'hostkey-ecdsa384': 'myhostname7',
        'hostkey-ecdsa521': 'myhostname8',
        'hostkey-ed25519': 'myhostname9',
        'hostkey-rsa2048': 'myhostname10',
        'untrusted-caname': 'test_value_11'
    }

    set_method_mock.assert_called_with('firewall.ssh', 'setting', data=expected_data, vdom='root')
    schema_method_mock.assert_not_called()
    assert not is_error
    assert not changed
    assert response['status'] == 'error'
    assert response['http_status'] == 404


def test_firewall_ssh_setting_filter_foreign_attributes(mocker):
    schema_method_mock = mocker.patch('ansible.module_utils.network.fortios.fortios.FortiOSHandler.schema')

    set_method_result = {'status': 'success', 'http_method': 'POST', 'http_status': 200}
    set_method_mock = mocker.patch('ansible.module_utils.network.fortios.fortios.FortiOSHandler.set', return_value=set_method_result)

    input_data = {
        'username': 'admin',
        'state': 'present',
        'firewall_ssh_setting': {
            'random_attribute_not_valid': 'tag',
            'caname': 'test_value_3',
            'host_trusted_checking': 'enable',
            'hostkey_dsa1024': 'myhostname5',
            'hostkey_ecdsa256': 'myhostname6',
            'hostkey_ecdsa384': 'myhostname7',
            'hostkey_ecdsa521': 'myhostname8',
            'hostkey_ed25519': 'myhostname9',
            'hostkey_rsa2048': 'myhostname10',
            'untrusted_caname': 'test_value_11'
        },
        'vdom': 'root'}

    is_error, changed, response = fortios_firewall_ssh_setting.fortios_firewall_ssh(input_data, fos_instance)

    expected_data = {
        'caname': 'test_value_3',
        'host-trusted-checking': 'enable',
        'hostkey-dsa1024': 'myhostname5',
        'hostkey-ecdsa256': 'myhostname6',
        'hostkey-ecdsa384': 'myhostname7',
        'hostkey-ecdsa521': 'myhostname8',
        'hostkey-ed25519': 'myhostname9',
        'hostkey-rsa2048': 'myhostname10',
        'untrusted-caname': 'test_value_11'
    }

    set_method_mock.assert_called_with('firewall.ssh', 'setting', data=expected_data, vdom='root')
    schema_method_mock.assert_not_called()
    assert not is_error
    assert changed
    assert response['status'] == 'success'
    assert response['http_status'] == 200
