/* Copyright 2011,2012,2013 David Cleaver
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef __JACOBI_SUM__
#define __JACOBI_SUM__

#ifndef HAVE_U64_T
#define HAVE_U64_T
typedef long long s64_t;
typedef unsigned long long u64_t;
#endif

/* The idea to hard code the J(p,q) values is thanks to Jason Moxham.
 * The computation of J(p,q) takes about 4*q bytes of memory, which
 * for large q primes can be substantial.  These J values, where J is
 * in reference to the Jacobi Sum, only need to be computed once, which
 * greatly reduces the run-time memory requirements of this program
 * The following sls and jpqs arrays are taken from his APRCL v1.15 code
 * I have also used a smaller set of his t values, since in testing they
 * proved to be very efficient for proving primality.
 *
 * On 2013/04/14 Toby Moxham has approved the APR-CL code and data tables, 
 * originally written by his brother Jason Moxham on 2003/01/01, to be released 
 * under the LGPL.
 */

#define SLSMAXVAL 33949
#define SLSMINVAL -44056
#define SLSMAX 36148
#define JPQSMAX 3493

struct jpq_t {int q;int index;char p;};

static const int sls[SLSMAX+1]={
/*a is 0 t is 60    */
/*    J(2,5)=    */    -1,-2,
/*    J(3,7)=    */    -1,-3,
/*    J(2,13)=    */    3,-2,
/*    J(3,13)=    */    -4,-3,
/*    J(5,11)=    */    0,2,-2,-1,
/*    J(3,31)=    */    5,6,
/*    J(5,31)=    */    2,-4,-1,2,
/*    J(2,61)=    */    -5,6,
/*    J(3,61)=    */    -4,-9,
/*    J(5,61)=    */    -2,-2,-8,1,
/*    J(3,19)=    */    2,1,4,0,-1,2,
/*    J(2,37)=    */    -1,6,
/*    J(3,37)=    */    2,5,-4,0,1,-2,
/*    J(2,181)=    */    -9,-10,
/*    J(3,181)=    */    8,-1,2,-6,-5,4,
/*    J(5,181)=    */    -4,-16,-4,-7,
/*    J(2,29)=    */    -5,-2,
/*    J(7,29)=    */    -2,-2,-6,-2,0,-3,
/*    J(2,41)=    */    -3,-4,0,-4,
/*    J(1,41)=    */    -5,0,4,0,
/*    J(4,41)=    */    -3,-4,0,-4,
/*    J(5,41)=    */    2,6,4,7,
/*    J(3,43)=    */    -1,6,
/*    J(7,43)=    */    -4,-6,-7,-4,-2,-6,
/*    J(5,71)=    */    -2,8,2,1,
/*    J(7,71)=    */    6,0,5,6,8,2,
/*    J(3,211)=    */    14,15,
/*    J(5,211)=    */    -6,0,-10,-15,
/*    J(7,211)=    */    -8,0,10,-2,2,-3,
/*    J(2,281)=    */    -9,-10,0,-10,
/*    J(1,281)=    */    5,0,16,0,
/*    J(4,281)=    */    9,10,0,10,
/*    J(5,281)=    */    4,-10,5,10,
/*    J(7,281)=    */    16,6,10,16,4,3,
/*    J(2,421)=    */    15,14,
/*    J(3,421)=    */    20,21,
/*    J(5,421)=    */    -6,10,0,-15,
/*    J(7,421)=    */    -10,-4,-4,14,-4,-7,
/*    J(3,127)=    */    -1,-8,4,-6,-10,2,
/*    J(7,127)=    */    6,0,6,3,4,-6,
/*    J(3,631)=    */    14,-14,-17,6,-10,2,
/*    J(5,631)=    */    -17,13,7,1,
/*    J(7,631)=    */    3,-6,-12,18,4,6,
/*    J(2,17)=    */    0,-2,2,0,-1,0,2,2,
/*    J(1,17)=    */    0,-1,0,-1,1,-3,-2,1,
/*    J(4,17)=    */    3,0,2,0,0,0,2,0,
/*    J(2,113)=    */    1,2,-4,-6,0,-6,4,2,
/*    J(1,113)=    */    3,-6,0,-4,-4,4,4,2,
/*    J(4,113)=    */    -9,0,4,0,0,0,4,0,
/*    J(7,113)=    */    2,6,8,8,5,12,
/*    J(2,241)=    */    0,-10,2,4,-1,-4,2,10,
/*    J(1,241)=    */    -8,1,4,5,1,7,-6,7,
/*    J(4,241)=    */    -13,0,-6,0,0,0,-6,0,
/*    J(3,241)=    */    -16,-15,
/*    J(5,241)=    */    15,12,18,14,
/*    J(2,337)=    */    7,8,8,-4,0,-4,-8,8,
/*    J(1,337)=    */    5,14,-2,-2,8,-2,6,-2,
/*    J(4,337)=    */    7,0,12,0,0,0,12,0,
/*    J(3,337)=    */    8,21,
/*    J(7,337)=    */    10,15,6,20,16,16,
/*    J(2,73)=    */    -1,6,0,6,
/*    J(1,73)=    */    -3,0,-8,0,
/*    J(4,73)=    */    1,-6,0,-6,
/*    J(3,73)=    */    2,6,-3,6,6,0,
/*    J(2,2521)=    */    37,-24,0,-24,
/*    J(1,2521)=    */    35,0,36,0,
/*    J(4,2521)=    */    37,-24,0,-24,
/*    J(3,2521)=    */    -4,16,-8,42,11,-22,
/*    J(5,2521)=    */    -29,-51,-39,-57,
/*    J(7,2521)=    */    33,5,1,-23,21,25,
/*a is 1 t is 5040    */
/*    J(2,1009)=    */    0,-12,-18,6,-1,-6,-18,12,
/*    J(1,1009)=    */    0,3,-8,-3,9,27,-6,9,
/*    J(4,1009)=    */    19,0,-18,0,0,0,-18,0,
/*    J(3,1009)=    */    -10,-15,30,0,0,12,
/*    J(7,1009)=    */    -15,-21,-25,-9,-33,-3,
/*    J(2,109)=    */    3,-10,
/*    J(3,109)=    */    6,0,6,3,-4,2,0,2,-4,4,2,4,-1,-2,0,2,2,-2,
/*    J(3,271)=    */    -6,4,-4,4,-6,6,8,-2,0,-4,-2,-4,8,2,4,4,1,4,
/*    J(5,271)=    */    6,-6,-9,8,
/*    J(3,379)=    */    0,8,0,12,0,0,-4,-5,0,-2,4,4,-4,-4,6,0,-3,-4,
/*    J(7,379)=    */    4,10,8,6,18,-5,
/*    J(2,433)=    */    0,4,14,2,-1,-2,14,-4,
/*    J(1,433)=    */    0,-11,4,11,1,13,-2,-1,
/*    J(4,433)=    */    19,0,6,0,0,0,6,0,
/*    J(3,433)=    */    -10,6,2,8,-2,0,3,10,6,-4,2,2,6,-4,-10,8,6,0,
/*    J(2,541)=    */    -21,-10,
/*    J(3,541)=    */    6,-4,14,-2,-4,-2,-4,-13,-4,10,-4,10,-8,-8,4,0,-9,-4,
/*    J(5,541)=    */    -12,8,12,-9,
/*    J(2,757)=    */    -9,-26,
/*    J(3,757)=    */    6,8,0,8,0,-2,-18,7,-4,2,12,-2,-2,-4,0,0,-5,-10,
/*    J(7,757)=    */    -8,14,-10,-12,-16,3,
/*    J(2,2161)=    */    0,22,-14,-16,-17,16,-14,-22,
/*    J(1,2161)=    */    -8,-5,-24,-5,17,-31,-10,-11,
/*    J(4,2161)=    */    19,0,-30,0,0,0,-30,0,
/*    J(3,2161)=    */    20,-8,9,14,-16,-20,12,-4,4,20,-2,-8,6,-8,-8,8,-28,-4,
/*    J(5,2161)=    */    -13,21,9,-33,
/*    J(2,7561)=    */    -19,-60,0,-60,
/*    J(1,7561)=    */    75,0,-44,0,
/*    J(4,7561)=    */    -19,-60,0,-60,
/*    J(3,7561)=    */    -18,-18,-40,-6,-28,-7,28,-2,26,-34,-22,-22,48,-36,-16,20,-8,14,
/*    J(5,7561)=    */    -35,7,-77,-71,
/*    J(7,7561)=    */    20,42,-40,-13,-8,54,
/*    J(2,15121)=    */    -41,-64,32,-40,0,-40,-32,-64,
/*    J(1,15121)=    */    -67,-76,10,4,32,4,42,-44,
/*    J(4,15121)=    */    -89,0,60,0,0,0,60,0,
/*    J(3,15121)=    */    -30,-8,52,-58,-24,-58,0,-20,26,4,-14,-12,-54,9,-72,-28,6,-2,
/*    J(5,15121)=    */    -70,-3,78,-36,
/*    J(7,15121)=    */    55,140,26,90,50,30,
/*a is 2 t is 55440    */
/*    J(11,23)=    */    0,2,4,2,2,4,0,3,2,2,
/*    J(3,67)=    */    2,9,
/*    J(11,67)=    */    4,6,0,2,4,2,4,-2,2,-1,
/*    J(2,89)=    */    -9,2,0,2,
/*    J(1,89)=    */    5,0,-8,0,
/*    J(4,89)=    */    9,-2,0,-2,
/*    J(11,89)=    */    -2,-1,-2,4,-6,-2,2,2,2,2,
/*    J(3,199)=    */    -4,-2,4,-12,5,8,
/*    J(11,199)=    */    2,4,4,-2,6,8,10,6,-3,8,
/*    J(3,331)=    */    -10,-21,
/*    J(5,331)=    */    10,-8,-2,-11,
/*    J(11,331)=    */    -1,6,-2,10,-2,2,12,4,-4,-4,
/*    J(2,397)=    */    19,-6,
/*    J(3,397)=    */    -1,2,14,12,-8,10,
/*    J(11,397)=    */    -4,-2,-22,-10,-6,-14,-6,-8,-8,-9,
/*    J(3,463)=    */    -22,-21,
/*    J(7,463)=    */    6,0,15,18,-4,6,
/*    J(11,463)=    */    0,-8,-4,6,-8,10,2,-2,-10,-9,
/*    J(2,617)=    */    15,14,0,14,
/*    J(1,617)=    */    -19,0,16,0,
/*    J(4,617)=    */    -15,-14,0,-14,
/*    J(7,617)=    */    -4,-6,14,-4,12,15,
/*    J(11,617)=    */    -4,0,-9,-8,-12,-4,-10,8,8,-14,
/*    J(2,661)=    */    -25,6,
/*    J(3,661)=    */    20,-9,
/*    J(5,661)=    */    -14,-6,-24,-27,
/*    J(11,661)=    */    -4,-8,-10,-14,-22,-10,0,-24,-8,-11,
/*    J(2,881)=    */    7,4,-16,12,0,12,16,4,
/*    J(1,881)=    */    -3,0,6,20,-8,4,-10,-16,
/*    J(4,881)=    */    -9,0,20,0,0,0,20,0,
/*    J(5,881)=    */    14,10,-20,5,
/*    J(11,881)=    */    14,8,10,14,27,18,18,12,22,32,
/*    J(3,991)=    */    2,6,6,24,-15,-6,
/*    J(5,991)=    */    -6,5,0,30,
/*    J(11,991)=    */    -8,-4,18,10,-2,10,2,21,-4,0,
/*    J(2,1321)=    */    13,-24,0,-24,
/*    J(1,1321)=    */    -5,0,36,0,
/*    J(4,1321)=    */    13,-24,0,-24,
/*    J(3,1321)=    */    -31,9,
/*    J(5,1321)=    */    -30,-23,-42,-36,
/*    J(11,1321)=    */    17,-4,14,-2,0,14,-2,-20,-6,10,
/*    J(3,2311)=    */    -55,-21,
/*    J(5,2311)=    */    -32,13,22,-14,
/*    J(7,2311)=    */    34,0,-2,-8,29,30,
/*    J(11,2311)=    */    -38,-32,-49,-26,-16,-30,-42,-12,-14,-28,
/*    J(2,3697)=    */    0,34,-14,-4,31,4,-14,-34,
/*    J(1,3697)=    */    -24,-5,-28,23,17,13,-22,29,
/*    J(4,3697)=    */    -13,0,42,0,0,0,42,0,
/*    J(3,3697)=    */    -7,57,
/*    J(7,3697)=    */    47,33,27,-17,29,-1,
/*    J(11,3697)=    */    2,-4,-30,26,30,10,16,-10,28,-3,
/*    J(2,4621)=    */    -61,30,
/*    J(3,4621)=    */    -76,-21,
/*    J(5,4621)=    */    -36,0,-20,45,
/*    J(7,4621)=    */    16,6,-26,-44,8,-45,
/*    J(11,4621)=    */    10,8,12,10,10,-4,52,-22,-24,-9,
/*    J(2,9241)=    */    23,-66,0,-66,
/*    J(1,9241)=    */    5,0,-96,0,
/*    J(4,9241)=    */    -23,66,0,66,
/*    J(3,9241)=    */    -55,-111,
/*    J(5,9241)=    */    -36,-30,-85,30,
/*    J(7,9241)=    */    -36,-42,-96,15,-8,-30,
/*    J(11,9241)=    */    6,22,52,-10,-2,38,39,-34,14,-38,
/*    J(2,18481)=    */    127,2,-4,-34,0,-34,4,2,
/*    J(1,18481)=    */    -95,28,62,14,-8,14,66,-4,
/*    J(4,18481)=    */    7,0,-96,0,0,0,-96,0,
/*    J(3,18481)=    */    56,-99,
/*    J(5,18481)=    */    -126,-45,-90,20,
/*    J(7,18481)=    */    -18,120,28,66,96,15,
/*    J(11,18481)=    */    -20,-32,34,-32,7,-56,46,88,-8,16,
/*    J(2,55441)=    */    47,-2,-124,-106,0,-106,124,-2,
/*    J(1,55441)=    */    -15,154,94,-64,-32,16,58,-118,
/*    J(4,55441)=    */    -233,0,-24,0,0,0,-24,0,
/*    J(3,55441)=    */    -73,-6,-114,144,84,-18,
/*    J(5,55441)=    */    222,146,-36,87,
/*    J(7,55441)=    */    149,279,177,229,155,221,
/*    J(11,55441)=    */    -28,-208,-62,-136,-182,-45,-146,-116,-154,-24,
/*    J(2,97)=    */    2,-2,0,-4,0,-4,0,-2,-2,2,-2,-2,5,2,-2,-2,
/*    J(1,97)=    */    1,2,2,-2,1,2,-5,4,3,5,0,-1,0,1,-1,-1,
/*    J(4,97)=    */    -5,0,0,0,6,0,0,0,0,0,0,0,6,0,0,0,
/*    J(3,97)=    */    8,-3,
/*    J(2,353)=    */    0,-2,-10,-4,0,-6,-2,4,1,-4,-2,6,0,4,-10,2,
/*    J(1,353)=    */    -2,0,3,1,2,7,1,-2,3,12,-7,-1,-8,-3,1,-2,
/*    J(4,353)=    */    15,0,0,0,-8,0,0,0,0,0,0,0,-8,0,0,0,
/*    J(11,353)=    */    4,2,-1,2,-16,0,4,2,-2,4,
/*    J(2,673)=    */    -10,4,-2,6,-3,-6,-2,-4,-10,-12,4,4,0,4,-4,-12,
/*    J(1,673)=    */    -9,4,1,-6,-4,-4,4,0,-9,9,-11,-5,-7,-3,-2,11,
/*    J(4,673)=    */    -5,0,0,0,18,0,0,0,0,0,0,0,18,0,0,0,
/*    J(3,673)=    */    29,21,
/*    J(7,673)=    */    23,20,28,18,30,20,
/*    J(2,2017)=    */    -2,-2,-8,18,-5,-18,-8,2,-2,4,-22,10,0,10,22,4,
/*    J(1,2017)=    */    13,7,9,22,-4,-8,4,17,-11,-6,-5,-9,19,-5,10,-10,
/*    J(4,2017)=    */    -37,0,0,0,-18,0,0,0,0,0,0,0,-18,0,0,0,
/*    J(3,2017)=    */    8,-2,4,21,-40,-10,
/*    J(7,2017)=    */    -44,-30,-14,1,-10,-30,
/*    J(2,3169)=    */    -10,14,16,6,-21,-6,16,-14,-10,-24,-14,2,0,2,14,-24,
/*    J(1,3169)=    */    -3,9,23,10,-8,-12,-10,-13,-15,-4,1,-9,-1,7,-32,24,
/*    J(4,3169)=    */    -37,0,0,0,30,0,0,0,0,0,0,0,30,0,0,0,
/*    J(3,3169)=    */    -40,35,2,-30,40,22,
/*    J(11,3169)=    */    -14,-4,36,8,15,8,20,40,-2,24,
/*    J(2,3361)=    */    0,-10,2,-6,-16,4,-10,32,-17,-32,-10,-4,-16,6,2,10,
/*    J(1,3361)=    */    -22,23,-27,-22,4,24,13,13,1,5,-1,10,-2,8,1,3,
/*    J(4,3361)=    */    47,0,0,0,-24,0,0,0,0,0,0,0,-24,0,0,0,
/*    J(3,3361)=    */    -49,15,
/*    J(5,3361)=    */    63,28,42,6,
/*    J(7,3361)=    */    -56,-26,2,0,-12,-35,
/*    J(2,5281)=    */    42,4,4,-8,0,-8,-4,4,-42,-20,-10,14,13,-14,-10,20,
/*    J(1,5281)=    */    1,-19,-16,25,15,7,-1,19,3,10,-20,16,-6,-38,1,-34,
/*    J(4,5281)=    */    59,0,0,0,30,0,0,0,0,0,0,0,30,0,0,0,
/*    J(3,5281)=    */    -79,-15,
/*    J(5,5281)=    */    -21,15,-65,-45,
/*    J(11,5281)=    */    -16,28,16,12,-2,27,26,32,-40,4,
/*    J(2,7393)=    */    0,16,18,2,8,16,-46,26,1,-26,-46,-16,8,-2,18,-16,
/*    J(1,7393)=    */    -26,-8,-39,23,18,5,-33,-20,-1,-34,11,13,-8,23,23,-14,
/*    J(4,7393)=    */    79,0,0,0,24,0,0,0,0,0,0,0,24,0,0,0,
/*    J(3,7393)=    */    56,99,
/*    J(7,7393)=    */    43,60,28,-2,-22,60,
/*    J(11,7393)=    */    4,20,-20,-32,24,-34,32,39,24,-14,
/*    J(2,110881)=    */    -22,54,-170,-72,-21,72,-170,-54,-22,14,104,-82,0,-82,-104,14,
/*    J(1,110881)=    */    109,52,-41,108,-16,36,-132,-18,-103,3,-53,-91,-53,-91,102,143,
/*    J(4,110881)=    */    -37,0,0,0,234,0,0,0,0,0,0,0,234,0,0,0,
/*    J(3,110881)=    */    53,274,-86,-24,-70,-4,
/*    J(5,110881)=    */    -89,-36,-114,258,
/*    J(7,110881)=    */    96,-16,28,-230,81,166,
/*    J(11,110881)=    */    162,128,-42,244,52,142,180,158,-32,-25,
/*a is 3 t is 720720    */
/*    J(2,53)=    */    7,-2,
/*    J(13,53)=    */    0,2,2,-2,2,-2,2,2,-2,-4,0,-1,
/*    J(3,79)=    */    -10,-3,
/*    J(13,79)=    */    2,0,-2,4,0,-1,2,2,-4,-2,-4,2,
/*    J(5,131)=    */    2,-4,-6,7,
/*    J(13,131)=    */    -4,-2,-4,-4,-2,0,-2,2,0,2,-10,-3,
/*    J(2,157)=    */    11,6,
/*    J(3,157)=    */    -13,-12,
/*    J(13,157)=    */    8,0,6,4,5,0,-2,4,0,4,0,-4,
/*    J(2,313)=    */    5,12,0,12,
/*    J(1,313)=    */    -13,0,-12,0,
/*    J(4,313)=    */    5,12,0,12,
/*    J(3,313)=    */    -16,3,
/*    J(13,313)=    */    -8,-20,-8,-16,-10,-14,-8,-8,-12,-8,-12,-7,
/*    J(2,521)=    */    -3,-16,0,-16,
/*    J(1,521)=    */    11,0,-20,0,
/*    J(4,521)=    */    -3,-16,0,-16,
/*    J(5,521)=    */    -1,-5,-5,-25,
/*    J(13,521)=    */    -4,3,12,0,8,-8,-6,4,4,-10,0,-4,
/*    J(3,547)=    */    14,27,
/*    J(7,547)=    */    0,2,-24,-6,-6,-9,
/*    J(13,547)=    */    16,8,10,8,2,-8,6,8,10,-2,6,13,
/*    J(3,859)=    */    -10,-33,
/*    J(11,859)=    */    -6,-14,-16,0,-10,-6,-28,-16,-14,-23,
/*    J(13,859)=    */    30,16,12,8,14,4,2,6,18,4,6,9,
/*    J(5,911)=    */    21,14,-14,8,
/*    J(7,911)=    */    -4,-6,-22,-22,9,-12,
/*    J(13,911)=    */    2,-3,0,-10,-18,-2,6,-2,4,0,12,-16,
/*    J(2,937)=    */    -17,18,0,18,
/*    J(1,937)=    */    -19,0,-24,0,
/*    J(4,937)=    */    17,-18,0,-18,
/*    J(3,937)=    */    2,-25,-4,18,-8,10,
/*    J(13,937)=    */    13,17,-1,19,11,23,7,3,-1,-1,17,9,
/*    J(2,1093)=    */    -33,2,
/*    J(3,1093)=    */    29,36,
/*    J(7,1093)=    */    24,36,24,36,28,33,
/*    J(13,1093)=    */    16,6,0,6,3,14,16,-2,-4,12,-16,0,
/*    J(3,1171)=    */    -10,-13,26,-6,-11,34,
/*    J(5,1171)=    */    28,18,42,21,
/*    J(13,1171)=    */    -2,12,28,-2,4,-2,12,2,-2,8,-10,3,
/*    J(2,1873)=    */    0,-12,18,18,17,-18,18,12,
/*    J(1,1873)=    */    0,-15,-10,-21,-9,-3,24,-21,
/*    J(4,1873)=    */    35,0,18,0,0,0,18,0,
/*    J(3,1873)=    */    8,-35,10,-6,-40,-10,
/*    J(13,1873)=    */    6,32,21,24,18,36,20,18,40,16,38,16,
/*    J(7,2003)=    */    21,2,18,50,8,12,
/*    J(11,2003)=    */    9,13,7,-9,23,9,-15,-7,-17,19,
/*    J(13,2003)=    */    4,-14,-4,-18,21,4,-2,12,10,-22,-8,-10,
/*    J(2,2341)=    */    15,-46,
/*    J(3,2341)=    */    -4,48,-6,-15,6,-12,
/*    J(5,2341)=    */    -42,-27,-58,-24,
/*    J(13,2341)=    */    -14,10,-8,6,-16,6,-12,-15,-4,-18,12,26,
/*    J(3,2731)=    */    -49,6,
/*    J(5,2731)=    */    -29,34,16,-2,
/*    J(7,2731)=    */    -3,-18,24,0,10,42,
/*    J(13,2731)=    */    -6,-20,4,16,22,-6,6,16,-30,2,-9,4,
/*    J(2,2861)=    */    19,-50,
/*    J(5,2861)=    */    28,8,52,51,
/*    J(11,2861)=    */    -38,-16,-40,-38,-38,-40,-4,-6,-32,-35,
/*    J(13,2861)=    */    -12,-24,-14,8,6,-34,8,-14,12,16,0,-5,
/*    J(2,3121)=    */    1,10,-28,-26,0,-26,28,10,
/*    J(1,3121)=    */    19,8,-16,34,4,10,12,-32,
/*    J(4,3121)=    */    23,0,-36,0,0,0,-36,0,
/*    J(3,3121)=    */    -64,-39,
/*    J(5,3121)=    */    21,24,36,68,
/*    J(13,3121)=    */    30,24,35,20,18,28,52,6,42,38,10,8,
/*    J(2,3433)=    */    29,36,0,36,
/*    J(1,3433)=    */    27,0,52,0,
/*    J(4,3433)=    */    29,36,0,36,
/*    J(3,3433)=    */    -64,-51,
/*    J(11,3433)=    */    34,-2,8,12,24,26,11,10,44,52,
/*    J(13,3433)=    */    -32,-18,-22,-8,-24,-20,-4,-2,-54,-40,-24,-39,
/*    J(3,6007)=    */    77,78,
/*    J(7,6007)=    */    -47,-57,-71,7,-43,-21,
/*    J(11,6007)=    */    0,-30,4,14,-30,-62,-8,-37,10,-16,
/*    J(13,6007)=    */    22,0,18,-10,44,-8,2,40,22,-20,-11,-22,
/*    J(2,6553)=    */    55,-42,0,-42,
/*    J(1,6553)=    */    37,0,72,0,
/*    J(4,6553)=    */    -55,42,0,42,
/*    J(3,6553)=    */    -28,0,36,0,-21,84,
/*    J(7,6553)=    */    19,68,-28,-6,-10,12,
/*    J(13,6553)=    */    -14,-52,38,-6,-12,-4,10,-2,8,14,32,13,
/*    J(2,8009)=    */    -51,-52,0,-52,
/*    J(1,8009)=    */    -85,0,28,0,
/*    J(4,8009)=    */    -51,-52,0,-52,
/*    J(7,8009)=    */    -40,48,-4,23,60,24,
/*    J(11,8009)=    */    -18,38,22,18,1,-2,-34,-2,-18,60,
/*    J(13,8009)=    */    -18,-2,14,-22,12,-14,38,28,35,0,-46,26,
/*    J(3,8191)=    */    -31,6,-66,-42,60,-18,
/*    J(5,8191)=    */    57,-34,-6,-48,
/*    J(7,8191)=    */    11,38,-8,68,-2,-38,
/*    J(13,8191)=    */    -6,30,-6,0,-12,12,30,48,-12,39,22,-42,
/*    J(2,8581)=    */    -65,66,
/*    J(3,8581)=    */    20,-81,
/*    J(5,8581)=    */    24,30,45,110,
/*    J(11,8581)=    */    72,69,40,44,64,100,86,54,32,76,
/*    J(13,8581)=    */    37,-8,24,34,4,46,6,10,62,80,14,28,
/*    J(2,16381)=    */    91,-90,
/*    J(3,16381)=    */    -88,41,-46,-126,-11,-20,
/*    J(5,16381)=    */    -156,-80,-60,-45,
/*    J(7,16381)=    */    -16,-30,-54,-12,-144,-39,
/*    J(13,16381)=    */    -26,-8,58,0,34,-38,0,-2,20,14,-88,-17,
/*    J(2,20021)=    */    -89,-110,
/*    J(5,20021)=    */    55,-28,8,-116,
/*    J(7,20021)=    */    -130,-48,-85,-148,-138,-96,
/*    J(11,20021)=    */    48,118,44,147,92,110,76,104,126,102,
/*    J(13,20021)=    */    -6,10,56,-52,60,16,-46,-62,-40,30,-7,14,
/*    J(2,20593)=    */    0,46,-14,-40,-113,40,-14,-46,
/*    J(1,20593)=    */    -120,-13,40,-53,1,-31,22,13,
/*    J(4,20593)=    */    -109,0,66,0,0,0,66,0,
/*    J(3,20593)=    */    -115,-90,18,-75,0,-36,
/*    J(11,20593)=    */    26,142,103,74,74,108,78,112,24,94,
/*    J(13,20593)=    */    69,45,-75,9,-49,-15,-39,3,33,-15,21,-27,
/*    J(2,21841)=    */    49,-70,68,14,0,14,-68,-70,
/*    J(1,21841)=    */    -77,4,-28,-34,28,-106,0,44,
/*    J(4,21841)=    */    -121,0,-60,0,0,0,-60,0,
/*    J(3,21841)=    */    -64,105,
/*    J(5,21841)=    */    67,36,174,42,
/*    J(7,21841)=    */    69,84,72,-6,-50,96,
/*    J(13,21841)=    */    80,50,10,36,100,-22,14,-28,-8,76,47,60,
/*    J(2,25741)=    */    -45,-154,
/*    J(3,25741)=    */    65,88,-56,36,116,-124,
/*    J(5,25741)=    */    -5,-128,58,34,
/*    J(11,25741)=    */    26,-61,50,0,-20,78,24,28,-82,-44,
/*    J(13,25741)=    */    -141,-69,-81,-131,-73,-103,-23,-33,-11,-93,-105,-61,
/*    J(2,36037)=    */    111,-154,
/*    J(3,36037)=    */    62,-65,88,12,-181,2,
/*    J(7,36037)=    */    160,-14,10,-44,16,-59,
/*    J(11,36037)=    */    -38,-68,-28,-40,-140,-60,-88,90,-28,-63,
/*    J(13,36037)=    */    92,68,90,46,-8,60,42,122,-68,8,36,-21,
/*    J(2,48049)=    */    0,50,-14,-52,-193,52,-14,-50,
/*    J(1,48049)=    */    24,43,-92,55,17,-19,-182,-19,
/*    J(4,48049)=    */    211,0,42,0,0,0,42,0,
/*    J(3,48049)=    */    -253,-120,
/*    J(7,48049)=    */    -71,-35,-211,-11,53,-55,
/*    J(11,48049)=    */    -136,57,76,88,18,58,14,86,-34,58,
/*    J(13,48049)=    */    -72,-30,52,28,-57,-24,36,110,-48,56,-98,-58,
/*    J(2,51481)=    */    53,-156,0,-156,
/*    J(1,51481)=    */    195,0,116,0,
/*    J(4,51481)=    */    53,-156,0,-156,
/*    J(3,51481)=    */    -79,-98,70,-216,-106,-4,
/*    J(5,51481)=    */    -57,-267,-63,-109,
/*    J(11,51481)=    */    29,-20,-108,10,-36,130,-38,-28,86,84,
/*    J(13,51481)=    */    -102,52,20,44,-24,-24,108,32,4,102,72,-77,
/*    J(2,65521)=    */    127,146,4,58,0,58,-4,146,
/*    J(1,65521)=    */    81,-34,50,-56,8,104,-178,-98,
/*    J(4,65521)=    */    -217,0,-96,0,0,0,-96,0,
/*    J(3,65521)=    */    230,-123,-24,84,-51,-126,
/*    J(5,65521)=    */    -273,-189,-21,-113,
/*    J(7,65521)=    */    -39,36,-30,-54,-246,38,
/*    J(13,65521)=    */    200,120,212,172,-4,132,176,104,52,110,47,82,
/*    J(2,72073)=    */    125,168,0,168,
/*    J(1,72073)=    */    267,0,-28,0,
/*    J(4,72073)=    */    125,168,0,168,
/*    J(3,72073)=    */    -154,-7,8,-294,-56,-62,
/*    J(7,72073)=    */    -156,-303,-208,-264,-186,-270,
/*    J(11,72073)=    */    -132,68,48,-128,-68,-176,-3,-64,4,32,
/*    J(13,72073)=    */    10,-6,-116,34,22,110,-52,61,64,-158,-54,-46,
/*    J(2,120121)=    */    103,-234,0,-234,
/*    J(1,120121)=    */    85,0,-336,0,
/*    J(4,120121)=    */    -103,234,0,234,
/*    J(3,120121)=    */    -400,-189,
/*    J(5,120121)=    */    418,198,132,111,
/*    J(7,120121)=    */    -270,-83,-210,-228,-408,-216,
/*    J(11,120121)=    */    102,-46,60,-26,51,-38,-32,-22,314,76,
/*    J(13,120121)=    */    -154,-124,-158,-169,-6,-96,-26,-288,6,66,-54,-38,
/*    J(2,180181)=    */    215,-366,
/*    J(3,180181)=    */    -316,74,59,102,136,-116,
/*    J(5,180181)=    */    -48,-54,-111,362,
/*    J(7,180181)=    */    457,300,292,52,248,120,
/*    J(11,180181)=    */    -64,-82,-2,46,186,-225,154,194,114,74,
/*    J(13,180181)=    */    -125,-67,-77,109,-221,-37,55,-77,65,-209,-267,-177,
/*    J(2,1249)=    */    1,8,14,-14,0,-8,-10,-2,0,-2,10,-8,0,-14,-14,8,
/*    J(1,1249)=    */    -11,8,0,-7,-8,13,8,8,4,-2,12,-17,4,7,6,10,
/*    J(4,1249)=    */    31,0,0,0,-12,0,0,0,0,0,0,0,-12,0,0,0,
/*    J(3,1249)=    */    -40,-27,
/*    J(13,1249)=    */    14,-2,18,-8,7,16,6,2,14,4,-8,-12,
/*    J(2,2081)=    */    -18,-10,16,-10,11,10,16,10,-18,0,2,14,0,14,-2,0,
/*    J(1,2081)=    */    -1,3,-7,14,-18,0,-8,1,7,-28,-19,1,5,-3,12,8,
/*    J(4,2081)=    */    27,0,0,0,-26,0,0,0,0,0,0,0,-26,0,0,0,
/*    J(5,2081)=    */    -5,-13,-37,19,
/*    J(13,2081)=    */    4,-11,-8,16,4,-22,2,-6,14,-6,22,16,
/*    J(2,8737)=    */    -2,20,-4,-20,-77,20,-4,-20,-2,-2,-2,24,0,24,2,-2,
/*    J(1,8737)=    */    23,-29,-19,18,24,-6,-44,-5,-53,-8,11,-13,-5,25,-10,14,
/*    J(4,8737)=    */    -5,0,0,0,-66,0,0,0,0,0,0,0,-66,0,0,0,
/*    J(3,8737)=    */    77,-27,
/*    J(7,8737)=    */    -28,-12,-88,-52,19,-36,
/*    J(13,8737)=    */    -22,-46,16,3,-36,-42,-26,-60,20,-26,-2,12,
/*    J(2,13729)=    */    -18,-44,28,4,0,4,-28,-44,18,8,10,14,83,-14,10,-8,
/*    J(1,13729)=    */    53,-23,36,19,37,1,-15,-1,-1,2,-20,46,50,-32,-33,2,
/*    J(4,13729)=    */    -101,0,0,0,-42,0,0,0,0,0,0,0,-42,0,0,0,
/*    J(3,13729)=    */    -88,45,
/*    J(11,13729)=    */    70,76,40,4,100,58,28,46,-17,34,
/*    J(13,13729)=    */    34,-32,58,-6,18,14,-8,10,20,22,-47,-58,
/*    J(2,14561)=    */    30,-4,-42,26,-67,-26,-42,4,30,-32,-20,16,0,16,20,-32,
/*    J(1,14561)=    */    25,2,-5,12,-14,22,70,18,49,-51,13,11,-13,-15,38,27,
/*    J(4,14561)=    */    -69,0,0,0,-70,0,0,0,0,0,0,0,-70,0,0,0,
/*    J(5,14561)=    */    -28,-84,64,7,
/*    J(7,14561)=    */    14,-19,42,20,24,-96,
/*    J(13,14561)=    */    -24,-68,-74,-46,-46,-34,-94,-62,-112,0,-28,-63,
/*    J(2,26209)=    */    0,-14,76,-64,-24,-28,8,10,-55,-10,8,28,-24,64,76,14,
/*    J(1,26209)=    */    48,65,67,-83,58,5,-17,-5,-21,-47,-25,-13,-12,-1,31,7,
/*    J(4,26209)=    */    -161,0,0,0,-12,0,0,0,0,0,0,0,-12,0,0,0,
/*    J(3,26209)=    */    17,16,154,-15,38,158,
/*    J(7,26209)=    */    -30,-65,30,100,-32,80,
/*    J(13,26209)=    */    -72,38,30,52,-28,-10,50,-4,76,-34,-59,-14,
/*    J(2,96097)=    */    30,28,-76,-32,-77,32,-76,-28,30,178,70,4,0,4,-70,178,
/*    J(1,96097)=    */    -67,-49,169,94,38,22,108,-13,1,38,-3,-91,89,-133,0,52,
/*    J(4,96097)=    */    -133,0,0,0,198,0,0,0,0,0,0,0,198,0,0,0,
/*    J(3,96097)=    */    -337,-273,
/*    J(7,96097)=    */    -58,-30,-78,-318,36,-15,
/*    J(11,96097)=    */    2,82,8,118,-64,62,207,222,-20,130,
/*    J(13,96097)=    */    74,68,68,153,-124,-44,-78,120,2,-4,160,46,
/*    J(2,131041)=    */    -145,-128,82,70,96,92,74,62,0,62,-74,92,-96,70,-82,-128,
/*    J(1,131041)=    */    3,-112,32,173,-122,85,14,40,60,26,118,35,66,-97,-82,-154,
/*    J(4,131041)=    */    287,0,0,0,-156,0,0,0,0,0,0,0,-156,0,0,0,
/*    J(3,131041)=    */    -85,320,164,-120,310,70,
/*    J(5,131041)=    */    -104,169,-104,-272,
/*    J(7,131041)=    */    118,-301,-94,-180,-184,-144,
/*    J(13,131041)=    */    -119,-79,77,117,59,-115,29,119,21,-55,97,-191,
/*a is 4 t is 4324320    */
/*    J(2,2377)=    */    -35,-24,0,-24,
/*    J(1,2377)=    */    -21,0,44,0,
/*    J(4,2377)=    */    -35,-24,0,-24,
/*    J(3,2377)=    */    12,6,3,-4,8,18,-2,10,28,4,8,11,-18,-4,-14,16,8,14,
/*    J(11,2377)=    */    -30,-14,-26,-6,-20,-30,-28,17,-8,-10,
/*    J(3,2971)=    */    -9,14,-10,14,-14,-2,6,10,-2,14,30,2,16,4,-18,-6,-8,24,
/*    J(5,2971)=    */    -10,17,8,-46,
/*    J(11,2971)=    */    -46,-14,-18,0,-6,5,-8,2,-20,16,
/*    J(3,3511)=    */    28,-4,30,6,-20,-16,-12,-1,6,18,-32,22,8,-20,-2,2,-19,-4,
/*    J(5,3511)=    */    5,-13,-7,-61,
/*    J(13,3511)=    */    28,-17,10,30,20,2,22,4,42,36,20,10,
/*    J(3,4159)=    */    6,-15,-26,30,18,6,16,-2,-20,-2,-21,2,-22,4,-2,-2,-8,-14,
/*    J(7,4159)=    */    -6,-66,-9,-24,-28,6,
/*    J(11,4159)=    */    2,4,30,-28,-10,-5,-4,-2,-48,-6,
/*    J(3,7723)=    */    4,44,16,-24,-22,-36,5,-20,18,-6,44,4,-22,-30,-36,3,-52,34,
/*    J(11,7723)=    */    -60,-10,-32,-6,-52,-46,-38,-16,-35,30,
/*    J(13,7723)=    */    46,18,60,0,15,22,30,24,-16,34,2,-28,
/*    J(2,8317)=    */    91,-6,
/*    J(3,8317)=    */    -5,-2,22,-40,10,-18,-32,6,2,-12,26,44,12,-24,-2,4,38,-12,
/*    J(7,8317)=    */    -10,94,34,54,40,39,
/*    J(11,8317)=    */    4,20,24,-60,-12,10,14,37,-10,38,
/*    J(2,9829)=    */    15,98,
/*    J(3,9829)=    */    34,-4,-18,28,-10,-18,2,21,0,18,2,-28,22,-72,-30,-30,36,22,
/*    J(7,9829)=    */    -55,-24,-40,-106,-2,-12,
/*    J(13,9829)=    */    44,56,12,16,32,24,18,52,-27,40,-6,-28,
/*    J(2,16633)=    */    -89,-66,0,-66,
/*    J(1,16633)=    */    -107,0,-72,0,
/*    J(4,16633)=    */    89,66,0,66,
/*    J(3,16633)=    */    -10,29,32,2,-60,-40,-46,-6,-16,-28,-3,66,6,-86,26,-42,18,-2,
/*    J(7,16633)=    */    -24,80,60,18,111,6,
/*    J(11,16633)=    */    8,34,-102,-22,11,22,2,46,-12,12,
/*    J(2,23761)=    */    79,-50,28,74,0,74,-28,-50,
/*    J(1,23761)=    */    33,8,94,22,-56,86,34,40,
/*    J(4,23761)=    */    -73,0,96,0,0,0,96,0,
/*    J(3,23761)=    */    16,46,29,-14,18,-12,-16,6,40,90,-54,45,34,44,36,18,-18,12,
/*    J(5,23761)=    */    53,108,-78,6,
/*    J(11,23761)=    */    10,44,-2,53,106,16,78,24,140,80,
/*    J(3,24571)=    */    -20,48,-8,-76,20,72,-27,38,-6,52,28,-44,-44,-22,50,-30,-10,-70,
/*    J(5,24571)=    */    -174,-166,-129,-132,
/*    J(7,24571)=    */    144,52,-42,30,-6,45,
/*    J(13,24571)=    */    -63,101,53,7,-11,-45,5,-49,17,-21,-1,19,
/*    J(2,28081)=    */    0,40,-62,-74,-79,74,-62,-40,
/*    J(1,28081)=    */    48,-71,50,-41,7,89,92,11,
/*    J(4,28081)=    */    -109,0,90,0,0,0,90,0,
/*    J(3,28081)=    */    -22,-16,69,-4,-86,0,72,-14,86,-10,72,53,0,-42,-38,26,14,34,
/*    J(5,28081)=    */    -93,-119,69,-33,
/*    J(13,28081)=    */    0,82,-54,48,44,30,-34,28,85,104,42,66,
/*    J(2,30241)=    */    17,-2,38,62,0,-8,98,4,0,4,-98,-8,0,62,-38,-2,
/*    J(1,30241)=    */    -27,35,8,-20,8,68,8,-11,12,11,98,-70,-12,-58,-68,-7,
/*    J(4,30241)=    */    127,0,0,0,84,0,0,0,0,0,0,0,84,0,0,0,
/*    J(3,30241)=    */    32,70,-18,90,-40,-18,-26,-50,-5,-10,62,-26,-40,-30,66,-14,-36,-32,
/*    J(5,30241)=    */    34,-15,30,180,
/*    J(7,30241)=    */    27,168,78,60,-32,90,
/*    J(3,38611)=    */    36,-114,14,-5,-62,-6,14,42,28,34,22,-30,13,10,90,-24,6,-42,
/*    J(5,38611)=    */    64,-35,10,-170,
/*    J(11,38611)=    */    -46,12,-92,-62,-14,20,-66,-14,-43,128,
/*    J(13,38611)=    */    -57,-25,-49,41,-61,-131,-61,49,-57,65,-11,-3,
/*    J(2,39313)=    */    0,6,-126,-60,-17,60,-126,-6,
/*    J(1,39313)=    */    -24,9,44,-99,33,15,18,159,
/*    J(4,39313)=    */    35,0,138,0,0,0,138,0,
/*    J(3,39313)=    */    -6,88,-24,-24,-54,-22,88,33,-100,38,94,-32,-36,-28,-34,20,8,-148,
/*    J(7,39313)=    */    -52,-67,152,8,2,-58,
/*    J(13,39313)=    */    -96,52,-80,-80,-40,-74,-148,-60,5,28,-44,-62,
/*    J(2,47521)=    */    -30,-32,110,-50,0,-50,-110,-32,30,-70,16,34,43,-34,16,70,
/*    J(1,47521)=    */    75,-46,118,107,-3,7,-17,2,1,79,-38,-46,30,52,-5,25,
/*    J(4,47521)=    */    -197,0,0,0,66,0,0,0,0,0,0,0,66,0,0,0,
/*    J(3,47521)=    */    -53,18,36,18,136,-72,-78,100,26,0,-30,52,2,112,-24,20,84,72,
/*    J(5,47521)=    */    -220,-78,33,-66,
/*    J(11,47521)=    */    -26,-91,-106,108,14,16,-14,-78,0,-110,
/*    J(2,66529)=    */    -62,-36,-100,-48,0,-48,100,-36,62,-64,-82,-26,93,26,-82,64,
/*    J(1,66529)=    */    147,-47,-46,-103,-7,-69,61,-45,-51,12,-40,42,-68,-52,37,-68,
/*    J(4,66529)=    */    251,0,0,0,-42,0,0,0,0,0,0,0,-42,0,0,0,
/*    J(3,66529)=    */    -54,-144,-12,20,-110,134,0,22,-24,-44,-76,94,6,-130,30,56,36,51,
/*    J(7,66529)=    */    -19,-85,199,23,-17,-95,
/*    J(11,66529)=    */    185,96,78,88,-46,60,-68,106,52,-46,
/*    J(2,108109)=    */    -253,-210,
/*    J(3,108109)=    */    -158,102,-176,-32,42,10,2,-35,88,-198,-44,-212,40,50,44,-24,-55,102,
/*    J(7,108109)=    */    -252,-166,-126,-400,-256,-215,
/*    J(11,108109)=    */    228,98,2,168,104,282,22,64,44,-23,
/*    J(13,108109)=    */    184,124,158,68,42,186,148,374,152,92,184,185,
/*    J(2,123553)=    */    -14,-72,62,168,0,168,-62,-72,14,70,20,-4,195,4,20,-70,
/*    J(1,123553)=    */    -47,-39,38,-34,101,-74,-207,23,-1,-8,-154,45,-86,81,129,42,
/*    J(4,123553)=    */    251,0,0,0,-174,0,0,0,0,0,0,0,-174,0,0,0,
/*    J(3,123553)=    */    52,-110,-85,30,-40,72,-86,-18,-38,148,-162,3,138,80,158,-64,88,-146,
/*    J(11,123553)=    */    88,80,120,-96,-156,-46,127,-130,128,16,
/*    J(13,123553)=    */    62,132,259,-36,170,112,-88,-16,-4,156,32,52,
/*    J(2,196561)=    */    0,104,-158,250,1,-250,-158,-104,
/*    J(1,196561)=    */    192,-265,134,49,-1,-97,-136,-203,
/*    J(4,196561)=    */    -413,0,114,0,0,0,114,0,
/*    J(3,196561)=    */    -119,-62,-140,18,46,126,10,56,18,-15,-148,-192,-28,292,12,136,164,-88,
/*    J(5,196561)=    */    -90,-78,388,-21,
/*    J(7,196561)=    */    -372,-252,-348,-174,-393,-2,
/*    J(13,196561)=    */    -76,-4,162,112,250,-10,308,62,-68,132,212,11,
/*    J(2,216217)=    */    343,-222,0,-222,
/*    J(1,216217)=    */    -91,0,-456,0,
/*    J(4,216217)=    */    -343,222,0,222,
/*    J(3,216217)=    */    16,118,28,80,236,-146,248,108,36,-60,222,64,-81,12,-136,252,96,88,
/*    J(7,216217)=    */    -393,-168,-180,36,-204,124,
/*    J(11,216217)=    */    -244,-48,-262,-202,-260,-23,44,66,-306,-20,
/*    J(13,216217)=    */    -74,176,146,-194,132,-30,240,252,158,15,102,64,
/*    J(3,270271)=    */    -82,-156,-22,62,110,308,-106,152,-150,-72,-138,-62,28,-122,34,49,-86,-252,
/*    J(5,270271)=    */    -414,-296,171,-162,
/*    J(7,270271)=    */    -133,-390,-258,-18,-372,120,
/*    J(11,270271)=    */    -222,-79,-232,166,-58,4,126,-296,-278,-100,
/*    J(13,270271)=    */    118,-182,-50,-154,234,-164,-313,-102,-160,-108,48,0,
/*    J(2,332641)=    */    0,240,-246,-94,-64,184,-14,-26,33,26,-14,-184,-64,94,-246,-240,
/*    J(1,332641)=    */    -254,-44,-37,-37,102,3,69,-114,-297,80,97,79,-220,235,9,146,
/*    J(4,332641)=    */    271,0,0,0,-360,0,0,0,0,0,0,0,-360,0,0,0,
/*    J(3,332641)=    */    99,-16,62,-24,-174,-44,58,-12,122,-181,270,-34,-282,-274,-126,-106,14,170,
/*    J(5,332641)=    */    404,-250,-85,-190,
/*    J(7,332641)=    */    -102,141,354,-324,-74,-150,
/*    J(11,332641)=    */    302,-224,68,250,49,38,32,-156,-254,48,
/*    J(2,393121)=    */    0,-256,98,0,128,46,230,-94,-287,94,230,-46,128,0,98,256,
/*    J(1,393121)=    */    -2,113,-371,60,202,58,21,277,95,45,11,116,-76,94,-107,-259,
/*    J(4,393121)=    */    -593,0,0,0,144,0,0,0,0,0,0,0,144,0,0,0,
/*    J(3,393121)=    */    -70,64,128,-42,306,246,-112,22,14,320,-60,-24,-76,-64,-18,81,-64,110,
/*    J(5,393121)=    */    12,-444,-256,-633,
/*    J(7,393121)=    */    -14,114,63,230,346,-404,
/*    J(13,393121)=    */    306,103,130,-12,200,236,174,-90,330,-70,-284,68,
/*    J(2,432433)=    */    0,-118,-434,116,-31,-116,-434,118,
/*    J(1,432433)=    */    152,187,-18,-317,-17,-503,68,125,
/*    J(4,432433)=    */    595,0,198,0,0,0,198,0,
/*    J(3,432433)=    */    202,5,-264,52,118,-8,92,286,-184,102,104,-130,286,-218,-106,-222,68,-130,
/*    J(7,432433)=    */    184,160,-76,-6,117,670,
/*    J(11,432433)=    */    17,-344,-224,-170,-562,12,-272,-366,-346,32,
/*    J(13,432433)=    */    -400,-416,-270,-377,-638,-76,-88,-120,-280,-354,-78,-336,
/*    J(2,540541)=    */    379,-630,
/*    J(3,540541)=    */    -18,140,-138,-226,398,-86,-132,-162,-350,76,30,62,-168,266,-42,23,72,224,
/*    J(5,540541)=    */    -213,351,-141,-553,
/*    J(7,540541)=    */    85,473,185,213,533,-279,
/*    J(11,540541)=    */    258,58,596,265,612,482,26,200,486,250,
/*    J(13,540541)=    */    -212,-412,-480,6,-614,-296,-298,-511,-402,-632,-518,-338,
/*    J(2,617761)=    */    118,256,-140,256,0,256,140,256,-118,-24,82,-330,-237,330,82,24,
/*    J(1,617761)=    */    189,-211,68,-103,-127,-285,-107,-273,-57,330,292,-312,122,46,59,154,
/*    J(4,617761)=    */    667,0,0,0,294,0,0,0,0,0,0,0,294,0,0,0,
/*    J(3,617761)=    */    -68,-186,-110,-60,-73,-202,-272,-414,416,16,-212,-62,210,-160,-270,-130,-440,444,
/*    J(5,617761)=    */    -487,-297,-933,-219,
/*    J(11,617761)=    */    176,388,446,90,-98,10,112,-264,500,3,
/*    J(13,617761)=    */    349,-43,101,31,137,-49,-21,-139,187,221,669,-79,
/*    J(2,4324321)=    */    -154,100,796,-1046,0,-1046,-796,100,154,-276,30,532,-285,-532,30,276,
/*    J(1,4324321)=    */    -207,-486,-214,-316,-47,-898,1317,446,-577,-211,-470,183,-18,347,315,553,
/*    J(4,4324321)=    */    -2053,0,0,0,-234,0,0,0,0,0,0,0,-234,0,0,0,
/*    J(3,4324321)=    */    42,110,80,434,1204,804,42,-790,141,290,-642,116,916,202,1072,-354,-174,-440,
/*    J(5,4324321)=    */    -1660,-1033,-2452,-736,
/*    J(7,4324321)=    */    -168,1752,288,-480,440,-573,
/*    J(11,4324321)=    */    -2081,-1513,-1003,-1959,-757,-1217,-907,-1451,-761,-441,
/*    J(13,4324321)=    */    1492,818,1816,1594,1029,306,548,918,740,344,1418,1586,
/*    J(3,103)=    */    11,9,
/*    J(17,103)=    */    2,2,0,0,2,4,-2,-4,2,-6,0,0,2,0,-1,-2,
/*    J(2,137)=    */    -3,8,0,8,
/*    J(1,137)=    */    11,0,4,0,
/*    J(4,137)=    */    -3,8,0,8,
/*    J(17,137)=    */    0,6,8,6,4,6,6,4,2,4,2,0,8,0,7,4,
/*    J(7,239)=    */    4,-11,-2,-6,-12,-2,
/*    J(17,239)=    */    2,-4,2,0,7,2,0,-2,-4,-2,-4,-2,6,-4,-4,6,
/*    J(3,307)=    */    2,6,6,-3,18,6,
/*    J(17,307)=    */    -2,-3,-8,2,0,-2,0,-12,-4,-2,0,-10,2,2,0,2,
/*    J(2,409)=    */    -11,-12,0,-12,
/*    J(1,409)=    */    3,0,20,0,
/*    J(4,409)=    */    -11,-12,0,-12,
/*    J(3,409)=    */    8,-15,
/*    J(17,409)=    */    8,2,-2,-6,-4,0,7,-4,-4,-2,2,-10,4,4,-2,6,
/*    J(13,443)=    */    4,18,18,6,20,6,12,8,8,10,10,9,
/*    J(17,443)=    */    -10,0,-6,2,-4,-8,-4,-2,-10,-14,-4,-4,-6,-4,-14,-15,
/*    J(2,613)=    */    -17,-18,
/*    J(3,613)=    */    2,-15,-6,6,-27,-6,
/*    J(17,613)=    */    -4,-4,-6,-16,-4,0,2,-10,4,0,4,-6,-8,0,-16,-5,
/*    J(2,953)=    */    21,-16,0,-16,
/*    J(1,953)=    */    -13,0,28,0,
/*    J(4,953)=    */    21,-16,0,-16,
/*    J(7,953)=    */    5,12,-4,-22,6,-12,
/*    J(17,953)=    */    -18,-16,-18,-15,-10,-12,-6,-16,-4,-16,-12,2,-10,-20,4,-4,
/*    J(2,1021)=    */    11,30,
/*    J(3,1021)=    */    11,36,
/*    J(5,1021)=    */    5,-3,33,9,
/*    J(17,1021)=    */    -18,-2,-12,-4,4,-12,-4,8,4,0,-10,-2,8,-6,9,2,
/*    J(3,1123)=    */    -34,-33,
/*    J(11,1123)=    */    2,-10,-10,16,6,16,-12,0,2,-11,
/*    J(17,1123)=    */    10,22,14,30,18,2,8,14,12,8,0,20,8,12,6,19,
/*    J(3,1327)=    */    23,42,
/*    J(13,1327)=    */    8,4,-2,-14,2,-16,-24,10,-5,0,-14,-2,
/*    J(17,1327)=    */    -12,-18,-4,4,-10,-16,-10,-10,-12,-2,6,6,-4,-15,10,-16,
/*    J(2,1361)=    */    0,-24,-2,10,-1,-10,-2,24,
/*    J(1,1361)=    */    -16,-13,4,1,9,15,-18,17,
/*    J(4,1361)=    */    3,0,-26,0,0,0,-26,0,
/*    J(5,1361)=    */    40,32,8,19,
/*    J(17,1361)=    */    2,-16,-22,-3,-18,4,-10,0,-4,-12,-8,-12,-20,-8,-24,-20,
/*    J(2,1429)=    */    23,-30,
/*    J(3,1429)=    */    -43,-15,
/*    J(7,1429)=    */    -20,-24,-30,11,-14,-8,
/*    J(17,1429)=    */    13,8,-6,10,22,-10,-2,-10,-2,-6,2,10,2,2,-10,10,
/*    J(3,1531)=    */    2,33,6,6,-9,6,
/*    J(5,1531)=    */    -42,-12,-18,1,
/*    J(17,1531)=    */    10,4,8,-16,-18,-2,4,-10,-12,8,-22,-2,-10,-4,-2,-5,
/*    J(5,1871)=    */    14,0,10,-35,
/*    J(11,1871)=    */    -3,-21,1,-35,-23,-21,-17,-13,-39,-17,
/*    J(17,1871)=    */    25,11,31,23,17,-5,13,15,1,23,21,23,17,17,33,23,
/*    J(3,2143)=    */    -10,-14,-20,-15,32,-10,
/*    J(7,2143)=    */    -11,-6,-50,-8,2,-12,
/*    J(17,2143)=    */    -20,8,0,-10,19,8,-6,-6,-22,-14,2,-6,-6,-20,8,-4,
/*    J(2,2381)=    */    35,-34,
/*    J(5,2381)=    */    7,56,14,22,
/*    J(7,2381)=    */    8,18,56,5,20,18,
/*    J(17,2381)=    */    16,4,8,-14,6,-4,-2,8,-18,19,14,8,14,24,-6,-10,
/*    J(2,2857)=    */    47,-18,0,-18,
/*    J(1,2857)=    */    -51,0,16,0,
/*    J(4,2857)=    */    -47,18,0,18,
/*    J(3,2857)=    */    8,57,
/*    J(7,2857)=    */    63,44,48,30,42,24,
/*    J(17,2857)=    */    20,-16,6,8,4,6,10,8,8,-22,24,28,24,8,15,4,
/*    J(2,3061)=    */    55,-6,
/*    J(3,3061)=    */    17,28,34,21,2,50,
/*    J(5,3061)=    */    65,32,38,14,
/*    J(17,3061)=    */    -32,6,-18,-22,-26,-18,18,-6,-9,-6,-2,-14,-6,-26,-30,-14,
/*    J(3,3571)=    */    -34,-69,
/*    J(5,3571)=    */    24,-30,-10,-45,
/*    J(7,3571)=    */    -42,-12,-18,-24,-58,-57,
/*    J(17,3571)=    */    -6,-10,10,-44,4,-2,-16,-6,8,14,14,-4,-12,-8,10,13,
/*    J(2,4421)=    */    -65,-14,
/*    J(5,4421)=    */    16,14,71,-2,
/*    J(13,4421)=    */    -9,-20,-2,32,-4,-24,24,-4,36,12,12,-2,
/*    J(17,4421)=    */    -10,-4,-30,-1,-14,-28,2,4,-28,-10,-54,-32,-20,2,-30,-20,
/*    J(2,5237)=    */    71,14,
/*    J(7,5237)=    */    -23,12,-50,4,36,6,
/*    J(11,5237)=    */    -32,-52,6,-49,-22,-20,-28,-36,6,6,
/*    J(17,5237)=    */    8,34,26,30,56,12,30,34,12,56,46,58,10,28,32,37,
/*    J(2,6121)=    */    -17,54,0,54,
/*    J(1,6121)=    */    45,0,64,0,
/*    J(4,6121)=    */    17,-54,0,-54,
/*    J(3,6121)=    */    -22,-56,34,-48,-55,38,
/*    J(5,6121)=    */    20,2,-67,14,
/*    J(17,6121)=    */    2,22,-24,-10,8,24,32,-19,32,0,-4,26,36,18,30,-4,
/*    J(2,6733)=    */    3,-82,
/*    J(3,6733)=    */    32,35,-16,-48,31,-32,
/*    J(11,6733)=    */    -56,-66,-46,-12,-4,-22,-38,-20,-4,-63,
/*    J(17,6733)=    */    -8,28,16,52,10,-12,-10,22,-8,-30,28,-12,16,12,16,15,
/*    J(2,7481)=    */    -57,-46,0,-46,
/*    J(1,7481)=    */    85,0,-16,0,
/*    J(4,7481)=    */    57,46,0,46,
/*    J(5,7481)=    */    -33,1,-91,-53,
/*    J(11,7481)=    */    40,18,18,11,34,-6,16,-4,66,70,
/*    J(17,7481)=    */    6,56,22,44,-6,36,14,-26,20,14,-16,6,24,26,20,-3,
/*    J(2,8161)=    */    0,-2,12,-22,40,14,28,-18,-33,18,28,-14,40,22,12,2,
/*    J(1,8161)=    */    -24,46,3,-14,-18,-34,29,-4,17,0,-9,-30,-4,-26,-31,-2,
/*    J(4,8161)=    */    31,0,0,0,-60,0,0,0,0,0,0,0,-60,0,0,0,
/*    J(3,8161)=    */    104,45,
/*    J(5,8161)=    */    3,8,12,-84,
/*    J(17,8161)=    */    4,-2,20,4,34,-46,5,6,-30,-28,-16,-22,20,4,-20,-36,
/*    J(3,9283)=    */    62,111,
/*    J(7,9283)=    */    18,10,108,28,10,35,
/*    J(13,9283)=    */    38,54,86,40,22,32,54,-24,48,36,22,33,
/*    J(17,9283)=    */    -10,-32,-14,-16,8,-30,-44,-38,-24,-38,-26,56,-30,-8,-18,-9,
/*    J(2,9521)=    */    -7,-24,-64,8,0,8,64,-24,
/*    J(1,9521)=    */    -1,-24,20,-16,12,-16,-12,-88,
/*    J(4,9521)=    */    -57,0,-56,0,0,0,-56,0,
/*    J(5,9521)=    */    -43,-104,4,-28,
/*    J(7,9521)=    */    -71,-63,-41,-101,-95,-99,
/*    J(17,9521)=    */    -44,-56,-4,-4,-12,-32,-34,-46,-28,-46,-19,-26,-24,22,-46,24,
/*    J(3,10711)=    */    -40,-68,1,-66,-46,-61,
/*    J(5,10711)=    */    -31,50,20,-70,
/*    J(7,10711)=    */    18,-60,30,66,7,36,
/*    J(17,10711)=    */    -30,2,-56,-17,-26,-56,0,26,-42,-34,-44,4,6,-14,20,-12,
/*    J(2,12241)=    */    7,16,-56,-52,0,-52,56,16,
/*    J(1,12241)=    */    -27,34,-10,74,32,10,-34,50,
/*    J(4,12241)=    */    71,0,60,0,0,0,60,0,
/*    J(3,12241)=    */    -40,94,28,-21,8,-34,
/*    J(5,12241)=    */    -30,-3,-122,-36,
/*    J(17,12241)=    */    24,33,28,56,40,2,10,56,32,46,94,68,16,46,14,80,
/*    J(2,12377)=    */    -105,-26,0,-26,
/*    J(1,12377)=    */    -91,0,64,0,
/*    J(4,12377)=    */    105,26,0,26,
/*    J(7,12377)=    */    -23,61,-3,45,93,1,
/*    J(13,12377)=    */    -42,-32,34,32,26,14,44,10,-16,42,-28,45,
/*    J(17,12377)=    */    66,28,40,60,52,-2,20,30,44,50,48,-33,16,38,-12,30,
/*    J(2,14281)=    */    13,84,0,84,
/*    J(1,14281)=    */    -85,0,-84,0,
/*    J(4,14281)=    */    13,84,0,84,
/*    J(3,14281)=    */    119,120,
/*    J(5,14281)=    */    18,-117,-18,-4,
/*    J(7,14281)=    */    -16,-72,36,-2,-7,74,
/*    J(17,14281)=    */    32,8,-2,-22,6,2,38,-14,68,20,-36,38,4,-13,-48,-14,
/*    J(2,15913)=    */    125,12,0,12,
/*    J(1,15913)=    */    123,0,28,0,
/*    J(4,15913)=    */    125,12,0,12,
/*    J(3,15913)=    */    56,-66,42,105,18,42,
/*    J(13,15913)=    */    -6,54,6,26,11,62,22,-30,2,-68,36,40,
/*    J(17,15913)=    */    86,82,78,80,40,22,114,30,68,70,62,44,17,68,34,22,
/*    J(3,16831)=    */    -40,42,-84,-6,63,42,
/*    J(5,16831)=    */    -12,73,-78,-54,
/*    J(11,16831)=    */    41,42,8,-32,94,76,66,56,10,78,
/*    J(17,16831)=    */    11,-17,19,15,-11,-29,37,29,23,47,85,-13,1,21,-7,77,
/*    J(2,17137)=    */    119,-20,-8,-32,0,-32,8,-20,
/*    J(1,17137)=    */    -43,2,-26,-10,-96,38,62,2,
/*    J(4,17137)=    */    55,0,-84,0,0,0,-84,0,
/*    J(3,17137)=    */    32,-8,136,24,-40,35,
/*    J(7,17137)=    */    48,-84,-66,23,-30,24,
/*    J(17,17137)=    */    -44,-48,-8,-14,-53,-42,-28,-68,-60,-60,-82,-30,-8,-88,-66,-118,
/*    J(2,17681)=    */    0,24,14,78,63,-78,14,-24,
/*    J(1,17681)=    */    72,9,72,-37,-15,69,-6,-29,
/*    J(4,17681)=    */    3,0,94,0,0,0,94,0,
/*    J(5,17681)=    */    -145,-68,-2,-46,
/*    J(13,17681)=    */    54,-10,-72,8,-42,-6,-68,-8,-42,-71,-30,-52,
/*    J(17,17681)=    */    14,6,90,10,54,34,22,14,48,58,36,82,-21,-30,32,26,
/*    J(3,19891)=    */    -70,19,10,90,5,-10,
/*    J(5,19891)=    */    -18,116,54,117,
/*    J(13,19891)=    */    -18,-64,-76,2,-68,-10,-80,-102,-62,-40,-106,-1,
/*    J(17,19891)=    */    20,30,-50,-10,68,72,20,70,-10,14,32,-12,48,-6,34,-15,
/*    J(2,22441)=    */    79,90,0,90,
/*    J(1,22441)=    */    -115,0,-96,0,
/*    J(4,22441)=    */    -79,-90,0,-90,
/*    J(3,22441)=    */    155,144,
/*    J(5,22441)=    */    36,-126,-9,38,
/*    J(11,22441)=    */    -32,-22,-36,-62,82,20,-26,-40,-12,-93,
/*    J(17,22441)=    */    -20,-36,-56,-44,-80,-76,-38,-100,-60,-52,-103,-76,-52,-92,-82,-156,
/*    J(3,23563)=    */    -130,51,6,-42,-3,90,
/*    J(7,23563)=    */    24,-20,-6,154,22,35,
/*    J(11,23563)=    */    -88,-18,40,-2,40,50,-12,-38,74,-25,
/*    J(17,23563)=    */    -18,8,36,2,-10,-98,-8,-66,20,12,-48,14,-10,-44,-14,53,
/*    J(2,24481)=    */    -18,14,-40,-46,0,-46,40,14,18,62,22,-44,59,44,22,-62,
/*    J(1,24481)=    */    31,11,48,-31,1,-19,9,35,41,56,0,86,-46,-50,31,26,
/*    J(4,24481)=    */    91,0,0,0,-90,0,0,0,0,0,0,0,-90,0,0,0,
/*    J(3,24481)=    */    -76,102,48,30,6,-39,
/*    J(5,24481)=    */    -42,138,52,81,
/*    J(17,24481)=    */    8,-30,-62,-6,32,-70,-20,-51,-68,-34,0,18,2,54,-76,-4,
/*    J(3,27847)=    */    44,-63,-162,54,-66,-42,
/*    J(7,27847)=    */    78,-108,-36,-41,-102,-30,
/*    J(13,27847)=    */    22,-146,-12,16,10,18,-28,16,-2,46,11,-4,
/*    J(17,27847)=    */    38,90,132,70,64,122,124,58,108,32,46,144,48,99,46,70,
/*    J(2,29173)=    */    -137,102,
/*    J(3,29173)=    */    -196,-117,
/*    J(11,29173)=    */    -148,-52,4,-82,-90,-40,-24,-24,40,-47,
/*    J(13,29173)=    */    -24,-194,-68,-106,-44,-60,-34,-40,-46,-52,-60,-79,
/*    J(17,29173)=    */    -2,-26,-12,-8,-62,-4,22,-10,-56,30,56,44,-2,-66,24,-99,
/*    J(2,29921)=    */    -38,36,30,12,3,-12,30,-36,-38,52,68,62,0,62,-68,52,
/*    J(1,29921)=    */    -25,-9,53,11,4,79,-12,75,15,30,-51,-60,3,46,38,-58,
/*    J(4,29921)=    */    123,0,0,0,-86,0,0,0,0,0,0,0,-86,0,0,0,
/*    J(5,29921)=    */    35,157,-47,-1,
/*    J(11,29921)=    */    -76,110,16,-49,-72,-30,-60,-12,-32,-16,
/*    J(17,29921)=    */    10,12,52,34,38,128,52,30,104,32,56,-6,21,-22,116,56,
/*    J(2,30941)=    */    -85,-154,
/*    J(5,30941)=    */    -182,-182,-168,-169,
/*    J(7,30941)=    */    136,22,102,124,192,81,
/*    J(13,30941)=    */    82,122,74,50,44,-12,94,134,70,122,84,149,
/*    J(17,30941)=    */    38,62,130,52,38,88,76,28,84,-54,12,62,30,78,60,99,
/*    J(2,34273)=    */    66,-2,-32,-34,0,-34,32,-2,-66,-2,82,-16,85,16,82,2,
/*    J(1,34273)=    */    -1,11,-36,-31,-89,85,3,-29,-7,-28,-48,-58,34,-26,61,62,
/*    J(4,34273)=    */    91,0,0,0,-114,0,0,0,0,0,0,0,-114,0,0,0,
/*    J(3,34273)=    */    -10,-126,-63,-54,66,-48,
/*    J(7,34273)=    */    -168,-40,-144,-136,-121,-8,
/*    J(17,34273)=    */    -25,64,14,78,50,64,-4,48,12,48,88,104,64,66,50,162,
/*    J(2,42841)=    */    37,144,0,144,
/*    J(1,42841)=    */    35,0,204,0,
/*    J(4,42841)=    */    37,144,0,144,
/*    J(3,42841)=    */    59,130,-86,96,146,-148,
/*    J(5,42841)=    */    -153,-204,-186,-238,
/*    J(7,42841)=    */    -14,51,-150,-132,-6,-114,
/*    J(17,42841)=    */    -62,4,10,8,-70,-30,3,14,-78,-72,-22,8,-66,128,12,8,
/*    J(3,43759)=    */    -22,-92,148,123,-76,116,
/*    J(11,43759)=    */    -4,-26,-14,64,-93,98,40,-48,-90,-82,
/*    J(13,43759)=    */    -22,18,18,84,84,98,3,-72,142,2,16,-8,
/*    J(17,43759)=    */    -112,-112,-26,-114,-106,-170,-84,-184,-124,-110,-11,-68,-158,-94,-98,-96,
/*    J(3,46411)=    */    -91,-246,
/*    J(5,46411)=    */    -97,83,17,161,
/*    J(7,46411)=    */    -132,-66,126,-30,41,-24,
/*    J(13,46411)=    */    138,112,2,4,-6,-36,129,10,88,54,116,38,
/*    J(17,46411)=    */    -102,-96,-162,-52,-80,-96,6,-82,-10,-40,-26,-108,-26,-122,-159,-36,
/*    J(2,52361)=    */    -147,-124,0,-124,
/*    J(1,52361)=    */    -181,0,140,0,
/*    J(4,52361)=    */    -147,-124,0,-124,
/*    J(5,52361)=    */    -272,-192,-208,-139,
/*    J(7,52361)=    */    -196,-222,-36,-75,-28,-130,
/*    J(11,52361)=    */    -72,-32,-148,-14,14,-137,-38,-134,-172,-148,
/*    J(17,52361)=    */    64,92,0,40,80,72,24,-120,68,-34,42,54,-19,-38,18,-38,
/*    J(2,53857)=    */    -98,-30,32,-36,0,-36,-32,-30,98,-98,-46,2,-69,-2,-46,98,
/*    J(1,53857)=    */    -43,-60,72,156,-17,12,35,2,83,-59,54,-13,8,1,59,15,
/*    J(4,53857)=    */    -37,0,0,0,-162,0,0,0,0,0,0,0,-162,0,0,0,
/*    J(3,53857)=    */    -46,-68,-56,-78,-58,-235,
/*    J(11,53857)=    */    50,218,30,118,76,176,196,93,136,72,
/*    J(17,53857)=    */    -138,-134,-100,-160,-146,-46,-60,-134,-188,-128,-114,-177,-206,-180,-80,-186,
/*    J(2,63649)=    */    10,-80,-98,-112,51,112,-98,80,10,24,20,30,0,30,-20,24,
/*    J(1,63649)=    */    145,-9,-31,-3,-6,93,-44,-21,-115,-56,-85,20,-3,-38,44,56,
/*    J(4,63649)=    */    251,0,0,0,18,0,0,0,0,0,0,0,18,0,0,0,
/*    J(3,63649)=    */    -88,-248,82,-117,-154,20,
/*    J(13,63649)=    */    -48,-61,-40,120,20,4,176,-12,20,-40,84,36,
/*    J(17,63649)=    */    -58,70,-114,-14,-60,-64,-30,62,-40,-142,-15,-38,-100,-114,-138,-56,
/*    J(3,72931)=    */    101,306,
/*    J(5,72931)=    */    234,0,65,-90,
/*    J(11,72931)=    */    184,10,-49,108,38,14,-98,-12,46,132,
/*    J(13,72931)=    */    228,156,176,67,146,206,158,298,140,176,190,86,
/*    J(17,72931)=    */    -6,144,142,66,16,80,28,178,74,226,112,84,126,12,86,127,
/*    J(2,74257)=    */    73,-116,-112,-92,0,-92,112,-116,
/*    J(1,74257)=    */    -113,-8,-120,148,-4,-44,-8,152,
/*    J(4,74257)=    */    23,0,-192,0,0,0,-192,0,
/*    J(3,74257)=    */    272,273,
/*    J(7,74257)=    */    -122,-270,-252,-222,-111,-270,
/*    J(13,74257)=    */    58,120,68,250,29,62,128,192,158,4,120,32,
/*    J(17,74257)=    */    -68,22,-160,34,90,-42,-20,-43,-60,74,78,18,12,14,-12,-108,
/*    J(2,78541)=    */    275,54,
/*    J(3,78541)=    */    -196,-321,
/*    J(5,78541)=    */    -14,-126,216,63,
/*    J(7,78541)=    */    38,78,-78,-50,-40,233,
/*    J(11,78541)=    */    -22,116,-6,-10,46,166,4,122,72,-137,
/*    J(17,78541)=    */    -40,-58,44,102,86,-132,46,46,-20,130,26,96,62,108,0,47,
/*    J(2,79561)=    */    269,60,0,60,
/*    J(1,79561)=    */    235,0,156,0,
/*    J(4,79561)=    */    269,60,0,60,
/*    J(3,79561)=    */    -40,60,150,-105,-30,-144,
/*    J(5,79561)=    */    -138,-54,-156,157,
/*    J(13,79561)=    */    107,-21,-43,-61,107,-55,57,-29,-133,1,-125,-105,
/*    J(17,79561)=    */    20,-41,-76,-104,-122,-164,86,-6,-146,16,-56,38,-70,-18,-52,14,
/*    J(2,87517)=    */    -181,-234,
/*    J(3,87517)=    */    50,-111,6,282,-3,90,
/*    J(11,87517)=    */    -84,-64,48,98,58,-44,-84,16,-208,65,
/*    J(13,87517)=    */    92,-118,10,-70,-118,110,14,-62,-82,72,72,79,
/*    J(17,87517)=    */    198,80,-46,40,22,136,-20,-12,76,22,-8,62,-108,-18,-28,45,
/*    J(2,92821)=    */    -105,286,
/*    J(3,92821)=    */    -196,-351,
/*    J(5,92821)=    */    154,330,0,105,
/*    J(7,92821)=    */    15,-108,24,-144,-76,204,
/*    J(13,92821)=    */    116,-8,168,120,76,-6,160,194,139,-46,112,-38,
/*    J(17,92821)=    */    34,154,82,-22,18,200,-11,104,-46,-6,-34,-50,86,112,54,106,
/*    J(2,97241)=    */    21,220,0,220,
/*    J(1,97241)=    */    -221,0,-220,0,
/*    J(4,97241)=    */    21,220,0,220,
/*    J(5,97241)=    */    -184,-116,-359,-52,
/*    J(11,97241)=    */    -122,174,110,-4,80,134,46,90,128,-87,
/*    J(13,97241)=    */    132,18,138,-34,-84,-59,12,106,-170,-14,8,-28,
/*    J(17,97241)=    */    -30,-112,177,16,70,32,16,32,78,-24,-32,-48,112,42,-36,-124,
/*    J(3,102103)=    */    -274,-351,
/*    J(7,102103)=    */    39,-109,159,-139,-187,-107,
/*    J(11,102103)=    */    296,198,205,112,86,290,112,44,70,126,
/*    J(13,102103)=    */    80,158,116,6,182,-8,-36,-102,89,152,-14,130,
/*    J(17,102103)=    */    -40,262,47,76,66,10,-4,106,56,66,72,6,-112,42,44,-18,
/*    J(2,116689)=    */    0,-192,-126,30,97,-30,-126,192,
/*    J(1,116689)=    */    -144,-207,34,-153,135,57,-84,3,
/*    J(4,116689)=    */    -317,0,90,0,0,0,90,0,
/*    J(3,116689)=    */    365,312,
/*    J(11,116689)=    */    124,132,56,128,-92,28,-136,76,101,-132,
/*    J(13,116689)=    */    118,44,288,73,124,204,58,72,56,234,50,-48,
/*    J(17,116689)=    */    -156,74,-62,46,38,-190,-174,18,-46,6,-50,70,50,-28,36,-7,
/*    J(3,117811)=    */    -250,-37,-16,36,157,-122,
/*    J(5,117811)=    */    36,-6,26,-327,
/*    J(7,117811)=    */    -188,-192,-266,-2,62,-213,
/*    J(11,117811)=    */    36,132,132,-42,120,18,6,194,150,309,
/*    J(17,117811)=    */    2,42,80,-126,-46,52,212,18,178,28,-16,154,12,80,98,81,
/*    J(2,145861)=    */    319,-210,
/*    J(3,145861)=    */    -220,-441,
/*    J(5,145861)=    */    -32,388,52,31,
/*    J(11,145861)=    */    164,66,412,192,48,104,226,40,116,83,
/*    J(13,145861)=    */    -224,-222,-52,-288,-142,6,60,-128,-118,-132,16,-155,
/*    J(17,145861)=    */    -114,-72,-90,-4,-34,-102,-130,-38,-382,-146,20,-78,-24,-146,-16,-73,
/*    J(2,148513)=    */    31,38,98,-2,-40,48,158,-184,0,-184,-158,48,40,-2,-98,38,
/*    J(1,148513)=    */    -85,-37,136,112,-22,38,-160,123,-16,121,150,16,70,54,-68,-127,
/*    J(4,148513)=    */    -385,0,0,0,12,0,0,0,0,0,0,0,12,0,0,0,
/*    J(3,148513)=    */    -169,-441,
/*    J(7,148513)=    */    -108,-60,-108,102,237,-190,
/*    J(13,148513)=    */    334,228,218,82,175,94,82,92,286,188,356,204,
/*    J(17,148513)=    */    -4,38,102,222,150,144,89,144,50,34,30,66,72,-124,288,92,
/*    J(2,157081)=    */    133,264,0,264,
/*    J(1,157081)=    */    259,0,-300,0,
/*    J(4,157081)=    */    133,264,0,264,
/*    J(3,157081)=    */    440,111,
/*    J(5,157081)=    */    48,168,157,456,
/*    J(7,157081)=    */    -440,-192,-120,8,-137,-128,
/*    J(11,157081)=    */    -294,-76,-26,-82,-212,-74,-120,128,-77,84,
/*    J(17,157081)=    */    -108,-118,86,172,34,62,-18,191,2,30,62,120,-10,10,192,-96,
/*    J(2,185641)=    */    -113,294,0,294,
/*    J(1,185641)=    */    429,0,-40,0,
/*    J(4,185641)=    */    113,-294,0,-294,
/*    J(3,185641)=    */    491,315,
/*    J(5,185641)=    */    -21,0,-20,420,
/*    J(7,185641)=    */    92,60,420,-44,-64,53,
/*    J(13,185641)=    */    26,12,-86,-60,-178,-105,-150,-80,90,180,-266,44,
/*    J(17,185641)=    */    56,92,60,-58,220,112,316,69,60,104,124,282,228,12,240,156,
/*    J(2,209441)=    */    22,-6,70,-220,0,-220,-70,-6,-22,136,-68,-156,-83,156,-68,-136,
/*    J(1,209441)=    */    -81,-32,72,103,109,5,-69,-190,17,-117,98,-278,-120,118,-59,-25,
/*    J(4,209441)=    */    -453,0,0,0,-46,0,0,0,0,0,0,0,-46,0,0,0,
/*    J(5,209441)=    */    -554,-406,-364,-287,
/*    J(7,209441)=    */    -8,0,49,448,-56,0,
/*    J(11,209441)=    */    -204,-78,-354,-158,-328,-279,-208,-250,-506,-166,
/*    J(17,209441)=    */    -99,-1,-207,-57,-187,41,-149,-363,-95,-137,-199,29,47,-11,-109,-187,
/*    J(2,235621)=    */    -289,390,
/*    J(3,235621)=    */    -34,-369,162,-228,-444,48,
/*    J(5,235621)=    */    489,300,520,180,
/*    J(7,235621)=    */    289,-148,-52,300,224,240,
/*    J(11,235621)=    */    88,-50,-64,-174,-154,-316,238,-12,-128,21,
/*    J(17,235621)=    */    127,-12,6,-82,144,136,212,144,184,172,-244,180,148,56,22,30,
/*    J(2,269281)=    */    26,-164,-188,22,0,22,188,-164,-26,-76,34,-228,157,228,34,76,
/*    J(1,269281)=    */    21,32,-364,-94,-5,28,-59,-84,-89,69,4,83,-146,-65,-193,181,
/*    J(4,269281)=    */    -37,0,0,0,366,0,0,0,0,0,0,0,366,0,0,0,
/*    J(3,269281)=    */    320,24,-408,111,-168,-24,
/*    J(5,269281)=    */    212,66,339,582,
/*    J(11,269281)=    */    -268,-60,-92,-56,-58,-338,213,138,14,-88,
/*    J(17,269281)=    */    -42,-88,-8,142,240,-294,64,-112,-154,-112,-51,32,-38,-36,108,110,
/*    J(2,291721)=    */    493,156,0,156,
/*    J(1,291721)=    */    11,0,540,0,
/*    J(4,291721)=    */    493,156,0,156,
/*    J(3,291721)=    */    -559,-519,
/*    J(5,291721)=    */    80,-528,-192,-291,
/*    J(11,291721)=    */    368,230,249,158,132,328,-56,302,58,-120,
/*    J(13,291721)=    */    -167,43,-9,-183,-99,-137,-97,325,45,75,-193,149,
/*    J(17,291721)=    */    472,220,372,562,312,368,272,232,246,360,228,168,208,440,295,208,
/*    J(2,314161)=    */    23,124,376,-8,0,-8,-376,124,
/*    J(1,314161)=    */    85,-146,86,2,-32,-526,-18,14,
/*    J(4,314161)=    */    23,0,396,0,0,0,396,0,
/*    J(3,314161)=    */    560,561,
/*    J(5,314161)=    */    -75,-548,78,-6,
/*    J(7,314161)=    */    -133,-180,-600,-182,-418,-448,
/*    J(11,314161)=    */    -552,-142,-324,-308,-338,-340,-472,-476,-409,-116,
/*    J(17,314161)=    */    -78,172,35,152,182,200,190,144,-70,188,18,418,164,-154,76,62,
/*    J(2,371281)=    */    0,26,338,260,79,-260,338,-26,
/*    J(1,371281)=    */    -208,91,28,-481,65,-91,162,221,
/*    J(4,371281)=    */    259,0,-390,0,0,0,-390,0,
/*    J(3,371281)=    */    -280,-699,
/*    J(5,371281)=    */    528,528,-33,256,
/*    J(7,371281)=    */    -270,22,-420,-242,271,-34,
/*    J(13,371281)=    */    109,177,369,-111,261,235,-9,269,185,409,425,345,
/*    J(17,371281)=    */    94,160,289,4,162,-124,38,340,120,-86,166,-162,-34,-152,148,158,
/*    J(2,388961)=    */    175,-140,-184,44,-272,-192,-72,88,0,88,72,-192,272,44,184,-140,
/*    J(1,388961)=    */    -271,-22,-6,120,106,240,124,54,120,46,-220,200,118,-272,-22,-142,
/*    J(4,388961)=    */    207,0,0,0,-416,0,0,0,0,0,0,0,-416,0,0,0,
/*    J(5,388961)=    */    292,206,569,-158,
/*    J(11,388961)=    */    -400,-212,-336,-758,-320,-208,-245,-432,-216,-306,
/*    J(13,388961)=    */    -367,-399,-215,-327,-129,-537,-357,-557,-167,-19,-271,-283,
/*    J(17,388961)=    */    -188,-328,114,-290,-114,-104,-120,-122,-59,-174,20,48,186,-210,-290,-308,
/*    J(2,445537)=    */    -126,-212,-202,-56,0,-56,202,-212,126,-226,-212,124,115,-124,-212,226,
/*    J(1,445537)=    */    165,-191,-4,-176,105,-184,-229,319,47,-106,-124,-263,-222,41,-49,20,
/*    J(4,445537)=    */    667,0,0,0,18,0,0,0,0,0,0,0,18,0,0,0,
/*    J(3,445537)=    */    -16,-30,-768,15,-12,-342,
/*    J(7,445537)=    */    234,372,-24,-114,-200,459,
/*    J(13,445537)=    */    252,322,48,-258,177,66,76,-162,-28,32,-284,200,
/*    J(17,445537)=    */    190,200,372,204,196,408,432,290,20,304,494,300,393,332,-94,344,
/*    J(2,471241)=    */    397,396,0,396,
/*    J(1,471241)=    */    555,0,-404,0,
/*    J(4,471241)=    */    397,396,0,396,
/*    J(3,471241)=    */    -712,144,-36,-594,93,114,
/*    J(5,471241)=    */    189,-105,35,-615,
/*    J(7,471241)=    */    490,-68,35,468,486,436,
/*    J(11,471241)=    */    -658,-623,-374,-172,-220,-324,-328,-168,-140,-492,
/*    J(17,471241)=    */    274,34,-134,-138,174,110,68,-136,-88,-158,-334,197,-190,162,124,-68,
/*    J(2,612613)=    */    -33,782,
/*    J(3,612613)=    */    -508,-497,-230,-318,59,-424,
/*    J(7,612613)=    */    -768,-498,-738,-540,-840,-313,
/*    J(11,612613)=    */    -484,58,170,-24,-248,322,-2,216,284,-51,
/*    J(13,612613)=    */    658,386,550,130,448,496,638,730,478,282,596,249,
/*    J(17,612613)=    */    510,440,192,536,484,-18,324,72,326,0,140,144,392,470,252,189,
/*    J(2,680681)=    */    477,476,0,476,
/*    J(1,680681)=    */    91,0,820,0,
/*    J(4,680681)=    */    477,476,0,476,
/*    J(5,680681)=    */    116,-276,-464,493,
/*    J(7,680681)=    */    62,14,-634,306,212,207,
/*    J(11,680681)=    */    -225,-493,-559,-311,-105,-117,-227,225,-589,-97,
/*    J(13,680681)=    */    -496,-257,-316,144,-528,-486,-294,-230,-114,170,-282,-68,
/*    J(17,680681)=    */    -131,-576,-750,-196,-706,-240,-336,-338,-352,-484,-430,-208,-352,-130,-400,-322,
/*    J(2,700129)=    */    -58,-202,-184,-252,0,-252,184,-202,58,-34,-358,274,91,-274,-358,34,
/*    J(1,700129)=    */    -133,-202,-254,-2,-151,386,-119,356,289,-243,-138,267,122,77,-85,59,
/*    J(4,700129)=    */    827,0,0,0,-90,0,0,0,0,0,0,0,-90,0,0,0,
/*    J(3,700129)=    */    338,-725,-464,276,-493,-58,
/*    J(11,700129)=    */    31,93,-33,35,-565,259,-89,-457,-351,-233,
/*    J(13,700129)=    */    -378,-484,-436,-598,-568,-924,-240,-568,-249,-368,-720,-448,
/*    J(17,700129)=    */    -326,-464,-482,-478,-294,-478,-210,130,-390,-434,-208,-438,84,-182,-189,-470,
/*    J(2,816817)=    */    7,-216,-288,528,0,528,288,-216,
/*    J(1,816817)=    */    141,-204,174,300,-376,204,366,564,
/*    J(4,816817)=    */    -425,0,564,0,0,0,564,0,
/*    J(3,816817)=    */    1043,552,
/*    J(7,816817)=    */    -975,-436,-144,6,-246,-432,
/*    J(11,816817)=    */    -76,116,780,192,214,-40,115,376,-270,176,
/*    J(13,816817)=    */    44,386,-30,-68,-156,-110,406,248,544,-200,301,376,
/*    J(17,816817)=    */    166,136,232,30,216,530,374,-32,150,276,448,458,664,-96,527,204,
/*    J(2,1633633)=    */    82,-108,-738,352,-189,-352,-738,108,82,176,212,-190,0,-190,-212,176,
/*    J(1,1633633)=    */    409,-163,87,-115,590,-187,-194,57,-87,38,45,436,-351,174,-218,-770,
/*    J(4,1633633)=    */    -805,0,0,0,-702,0,0,0,0,0,0,0,-702,0,0,0,
/*    J(3,1633633)=    */    -1456,-519,
/*    J(7,1633633)=    */    -279,347,-387,585,903,-183,
/*    J(11,1633633)=    */    364,-72,750,-66,258,450,180,372,1164,735,
/*    J(13,1633633)=    */    -560,-16,-317,288,-416,-580,-582,-934,-64,-160,-782,-610,
/*    J(17,1633633)=    */    -676,-68,200,-564,172,-140,-240,-414,-324,105,476,78,264,-360,76,20,
/*    J(2,4084081)=    */    -521,4,952,-1000,0,-1000,-952,4,
/*    J(1,4084081)=    */    501,1178,-806,-362,-392,-1114,466,-230,
/*    J(4,4084081)=    */    887,0,1284,0,0,0,1284,0,
/*    J(3,4084081)=    */    -400,1791,
/*    J(5,4084081)=    */    -954,-566,-2364,-447,
/*    J(7,4084081)=    */    584,-1532,-944,-147,-1112,84,
/*    J(11,4084081)=    */    1311,1625,1057,115,861,757,911,253,-283,1235,
/*    J(13,4084081)=    */    -1162,240,636,-694,504,178,280,-20,513,476,960,90,
/*    J(17,4084081)=    */    110,578,690,1432,756,628,1130,1469,198,1072,728,744,834,422,1710,248,
/*    J(2,8168161)=    */    0,-656,-8,-1424,928,684,-448,96,-417,-96,-448,-684,928,1424,-8,656,
/*    J(1,8168161)=    */    -316,1098,433,-1332,-594,-1038,349,-588,-511,-472,-301,362,168,1160,-845,458,
/*    J(4,8168161)=    */    2207,0,0,0,-1284,0,0,0,0,0,0,0,-1284,0,0,0,
/*    J(3,8168161)=    */    3296,1791,
/*    J(5,8168161)=    */    3204,1670,2355,690,
/*    J(7,8168161)=    */    1368,-1070,-1837,278,-1068,200,
/*    J(11,8168161)=    */    786,602,1112,1786,256,1032,19,-684,-176,2174,
/*    J(13,8168161)=    */    695,556,1108,122,340,1042,662,-378,750,940,1186,2908,
/*    J(17,8168161)=    */    -624,1684,981,664,-144,528,1414,446,804,668,1778,82,1264,892,-254,50,
/*a is 5 t is 73513440    */
/*    J(3,919)=    */    12,-2,0,-9,6,2,-2,6,8,2,10,-2,-3,-4,2,8,-10,-10,
/*    J(17,919)=    */    2,12,6,2,8,6,-12,20,11,10,2,0,6,0,0,-6,
/*    J(2,3673)=    */    55,18,0,18,
/*    J(1,3673)=    */    37,0,-48,0,
/*    J(4,3673)=    */    -55,-18,0,-18,
/*    J(3,3673)=    */    -16,15,-16,32,-10,-4,12,30,16,-16,8,-16,12,-2,24,20,4,20,
/*    J(17,3673)=    */    21,23,51,17,9,11,-1,9,25,-13,27,21,21,11,35,21,
/*    J(3,4591)=    */    2,-11,2,-6,-6,10,-2,-66,2,-16,-10,4,-10,10,18,4,-52,0,
/*    J(5,4591)=    */    58,-2,-23,16,
/*    J(17,4591)=    */    6,20,6,-3,-16,-22,16,2,2,30,12,-8,-10,38,0,28,
/*    J(3,6427)=    */    6,-8,16,2,-12,32,6,46,-4,10,-20,0,-3,36,36,-16,36,22,
/*    J(7,6427)=    */    8,39,-22,50,52,54,
/*    J(17,6427)=    */    -14,-26,12,-30,-3,8,-18,-8,8,-26,16,-6,-4,-14,40,30,
/*    J(2,9181)=    */    91,30,
/*    J(3,9181)=    */    -22,-36,20,-32,-16,38,16,17,-66,-24,-30,10,-10,-2,10,38,13,-12,
/*    J(5,9181)=    */    -60,-48,-92,-111,
/*    J(17,9181)=    */    6,34,22,-18,0,22,16,8,10,-24,-18,42,-18,54,36,31,
/*    J(3,10099)=    */    30,26,-8,16,10,14,-14,-39,58,-14,-20,-46,-2,-6,32,-2,-25,22,
/*    J(11,10099)=    */    38,-20,-8,-18,-44,-58,26,-48,-30,-37,
/*    J(17,10099)=    */    44,54,6,-26,56,4,34,48,38,54,12,40,20,22,-6,7,
/*    J(2,12853)=    */    -57,98,
/*    J(3,12853)=    */    -12,20,-12,22,-26,50,14,4,56,-7,12,4,14,-26,2,34,76,20,
/*    J(7,12853)=    */    -10,-32,97,12,-2,-24,
/*    J(17,12853)=    */    -46,6,-18,22,-42,-30,18,-34,2,-22,-32,20,60,5,-4,-8,
/*    J(2,23869)=    */    -37,150,
/*    J(3,23869)=    */    26,52,-80,20,32,-10,14,13,-2,12,38,62,46,34,-38,22,-31,14,
/*    J(13,23869)=    */    -50,-40,56,-34,50,-30,-10,32,64,38,-56,-21,
/*    J(17,23869)=    */    6,-120,-40,-30,-64,10,-54,-30,-98,-76,-110,-56,-48,-42,-72,-61,
/*    J(2,36721)=    */    -7,-124,16,-52,0,-52,-16,-124,
/*    J(1,36721)=    */    -17,-104,60,76,92,-20,76,-40,
/*    J(4,36721)=    */    -89,0,-120,0,0,0,-120,0,
/*    J(3,36721)=    */    24,-82,-124,24,84,14,-12,-16,24,10,-60,-82,17,54,6,38,-40,108,
/*    J(5,36721)=    */    36,54,171,-62,
/*    J(17,36721)=    */    108,58,40,48,34,46,28,96,174,40,32,106,10,85,-6,18,
/*    J(2,47737)=    */    -217,-18,0,-18,
/*    J(1,47737)=    */    -171,0,-136,0,
/*    J(4,47737)=    */    217,18,0,18,
/*    J(3,47737)=    */    -33,82,22,90,-24,22,-36,-14,-78,-89,8,36,74,-22,14,-10,-144,8,
/*    J(13,47737)=    */    -128,-152,-26,-208,-92,-93,-28,-24,-38,-120,-88,-44,
/*    J(17,47737)=    */    -66,-48,-46,-184,11,-76,-18,-16,-58,-44,-2,20,-66,-66,-12,58,
/*    J(3,59671)=    */    76,10,-94,70,31,110,64,38,-22,30,-24,-2,16,122,36,58,132,-58,
/*    J(5,59671)=    */    -293,-129,-111,-63,
/*    J(13,59671)=    */    48,-30,-41,24,2,-110,-2,78,24,170,-52,-8,
/*    J(17,59671)=    */    -62,28,54,36,-90,6,56,-42,-48,18,-113,80,18,80,70,44,
/*    J(3,70687)=    */    42,-70,-7,-200,50,18,80,84,-34,62,72,11,-142,90,52,4,32,-16,
/*    J(7,70687)=    */    -97,95,83,113,-151,5,
/*    J(11,70687)=    */    -68,170,118,116,14,94,103,132,-64,66,
/*    J(17,70687)=    */    54,-76,8,8,-98,102,-2,-88,-50,12,-38,-84,-141,-6,-16,74,
/*    J(2,100981)=    */    215,234,
/*    J(3,100981)=    */    -104,-100,-44,58,-76,-78,-60,124,-60,-114,-13,-106,136,-140,-160,-46,124,22,
/*    J(5,100981)=    */    -144,-146,201,-42,
/*    J(11,100981)=    */    -33,0,-42,246,66,-30,142,78,162,-18,
/*    J(17,100981)=    */    -24,-34,24,161,-72,94,48,-56,-4,52,10,126,-26,-70,-152,24,
/*    J(2,128521)=    */    -289,150,0,150,
/*    J(1,128521)=    */    125,0,-336,0,
/*    J(4,128521)=    */    289,-150,0,-150,
/*    J(3,128521)=    */    218,-70,-20,72,-75,-2,76,44,-238,38,-152,12,26,-67,20,64,-10,-18,
/*    J(5,128521)=    */    111,4,-294,78,
/*    J(7,128521)=    */    -380,-254,-188,-359,-140,-122,
/*    J(17,128521)=    */    -182,-30,24,-82,-12,30,-169,-42,-160,-158,-56,18,-70,-268,-118,-154,
/*    J(2,161569)=    */    114,-166,88,62,0,62,-88,-166,-114,-86,58,-20,-187,20,58,86,
/*    J(1,161569)=    */    -53,7,138,-115,95,-91,93,47,-47,38,30,88,206,160,-101,-88,
/*    J(4,161569)=    */    251,0,0,0,222,0,0,0,0,0,0,0,222,0,0,0,
/*    J(3,161569)=    */    -34,112,238,50,98,0,-92,18,-90,32,-83,142,42,122,-212,-68,126,-54,
/*    J(11,161569)=    */    -260,-232,-114,-371,-86,-244,-164,-256,-180,36,
/*    J(17,161569)=    */    242,-36,188,186,28,26,72,218,116,194,3,118,96,74,-8,-90,
/*    J(2,167077)=    */    271,306,
/*    J(3,167077)=    */    -162,94,-254,-41,-140,-42,218,94,-46,-124,88,-186,-4,10,4,16,46,68,
/*    J(7,167077)=    */    -267,-168,-62,-318,-354,-414,
/*    J(13,167077)=    */    -76,164,186,212,138,142,45,286,102,0,278,4,
/*    J(17,167077)=    */    33,73,-69,139,-33,-11,97,19,23,177,-43,111,91,99,-35,-247,
/*    J(2,201961)=    */    431,90,0,90,
/*    J(1,201961)=    */    269,0,-360,0,
/*    J(4,201961)=    */    -431,-90,0,-90,
/*    J(3,201961)=    */    227,212,-144,136,144,0,76,-48,-76,48,16,84,98,168,-120,2,-84,52,
/*    J(5,201961)=    */    536,269,296,128,
/*    J(11,201961)=    */    287,383,407,417,367,431,421,411,399,535,
/*    J(17,201961)=    */    -295,0,-124,-156,-24,54,-46,-44,-44,-124,-272,-186,-34,8,-172,-276,
/*    J(2,238681)=    */    343,246,0,246,
/*    J(1,238681)=    */    -91,0,480,0,
/*    J(4,238681)=    */    -343,-246,0,-246,
/*    J(3,238681)=    */    -38,246,-144,6,-66,-118,88,52,-28,108,96,-185,-10,-178,-320,-20,180,-102,
/*    J(5,238681)=    */    -248,-104,-416,-533,
/*    J(13,238681)=    */    -24,148,-86,-1,-174,248,-20,-248,-134,-130,-118,70,
/*    J(17,238681)=    */    -360,-360,-304,-301,-280,-328,-384,-360,-346,-300,-324,-396,-412,-472,-492,-566,
/*    J(3,417691)=    */    -32,34,84,322,324,-34,-8,-232,-4,-24,-262,318,97,130,-4,88,-312,-84,
/*    J(5,417691)=    */    197,-279,-261,387,
/*    J(7,417691)=    */    606,528,562,45,336,330,
/*    J(13,417691)=    */    180,162,18,-114,462,-78,-78,-143,-192,132,132,168,
/*    J(17,417691)=    */    -214,74,84,-132,-10,356,90,-4,2,32,-205,14,68,42,88,-354,
/*    J(2,477361)=    */    0,362,254,-196,97,196,254,-362,
/*    J(1,477361)=    */    -320,-361,122,95,-9,365,-248,161,
/*    J(4,477361)=    */    -269,0,-450,0,0,0,-450,0,
/*    J(3,477361)=    */    -182,220,214,-70,-83,-42,58,90,26,258,-262,-46,-126,-171,-182,-74,-40,114,
/*    J(5,477361)=    */    736,169,256,-32,
/*    J(13,477361)=    */    486,130,132,-138,-42,192,126,354,-24,24,363,372,
/*    J(17,477361)=    */    -88,4,287,-240,48,-24,-136,-40,-100,-352,-2,18,32,-420,-224,-192,
/*    J(2,514081)=    */    -186,-148,58,62,0,62,-58,-148,186,-86,-344,166,-283,-166,-344,86,
/*    J(1,514081)=    */    163,-214,326,-115,39,133,109,-74,201,355,62,-224,-186,-154,49,-67,
/*    J(4,514081)=    */    667,0,0,0,186,0,0,0,0,0,0,0,186,0,0,0,
/*    J(3,514081)=    */    -6,405,108,-26,-148,-322,-32,242,-416,-44,157,-78,-38,-38,54,-206,42,-246,
/*    J(5,514081)=    */    -484,-826,-139,-382,
/*    J(7,514081)=    */    -402,-81,-414,-380,-744,-24,
/*    J(17,514081)=    */    -240,0,-194,-380,188,-148,-311,-228,108,52,178,0,-136,-200,-48,-308,
/*    J(2,565489)=    */    -1,-58,508,146,0,146,-508,-58,
/*    J(1,565489)=    */    -111,712,86,46,-8,-178,-70,8,
/*    J(4,565489)=    */    -89,0,528,0,0,0,528,0,
/*    J(3,565489)=    */    -126,-83,-18,126,-122,18,-8,-352,-320,-158,293,-38,-58,-38,-150,-170,-154,-566,
/*    J(7,565489)=    */    -297,-228,-26,-186,-378,-846,
/*    J(11,565489)=    */    478,-114,112,58,140,110,-389,-138,216,230,
/*    J(17,565489)=    */    -56,238,444,178,268,130,148,132,285,-356,28,130,132,270,24,350,
/*    J(3,656371)=    */    56,298,80,-152,-342,-552,406,-199,-236,24,224,-80,-64,-258,-276,116,-65,-250,
/*    J(5,656371)=    */    206,-556,86,413,
/*    J(11,656371)=    */    -108,-190,-208,-512,66,298,-320,86,-114,275,
/*    J(13,656371)=    */    84,-546,202,120,-196,-266,-18,-176,32,10,258,209,
/*    J(17,656371)=    */    -154,-218,-308,-56,-76,-202,-464,-182,-268,-286,-346,74,-464,-592,-322,-625,
/*    J(2,1670761)=    */    -1169,390,0,390,
/*    J(1,1670761)=    */    -819,0,1000,0,
/*    J(4,1670761)=    */    1169,-390,0,-390,
/*    J(3,1670761)=    */    288,-300,528,362,-158,104,-240,-764,-165,-14,68,532,-156,-154,-300,320,-570,-381,
/*    J(5,1670761)=    */    704,645,-720,240,
/*    J(7,1670761)=    */    -666,42,-657,-570,654,20,
/*    J(13,1670761)=    */    124,164,-54,430,148,22,671,698,-568,2,-378,248,
/*    J(17,1670761)=    */    -299,79,-123,-247,-735,779,325,185,201,-91,15,37,-155,109,-165,-35,
/*    J(2,1837837)=    */    -429,1286,
/*    J(3,1837837)=    */    -436,-352,-506,-448,830,-602,-452,-112,269,-88,-84,70,-246,608,-510,-104,-224,148,
/*    J(7,1837837)=    */    597,181,591,1143,-423,45,
/*    J(11,1837837)=    */    -138,-222,-432,-90,-857,-546,-600,126,510,-744,
/*    J(13,1837837)=    */    338,-602,-59,322,470,316,-420,56,-682,94,-140,-396,
/*    J(17,1837837)=    */    69,-472,-138,416,184,-720,-190,-892,-6,-270,-232,-62,-200,200,68,-306,
/*    J(2,2625481)=    */    541,1080,0,1080,
/*    J(1,2625481)=    */    1259,0,-1020,0,
/*    J(4,2625481)=    */    541,1080,0,1080,
/*    J(3,2625481)=    */    262,1134,158,-526,62,134,-312,-626,146,-128,1216,-348,-500,-88,60,87,-354,-354,
/*    J(5,2625481)=    */    -1308,-799,-1836,-408,
/*    J(11,2625481)=    */    690,-75,-466,340,498,-374,-292,488,262,1106,
/*    J(13,2625481)=    */    -719,140,18,-336,150,1080,-142,-114,-340,-562,280,-314,
/*    J(17,2625481)=    */    296,1298,160,366,-200,544,866,390,1229,746,586,588,604,170,664,362,
/*    J(2,5250961)=    */    0,1070,-1058,256,769,-256,-1058,-1070,
/*    J(1,5250961)=    */    -416,-145,406,-1157,-521,-487,-1092,1361,
/*    J(4,5250961)=    */    1187,0,-1386,0,0,0,-1386,0,
/*    J(3,5250961)=    */    34,-206,38,-1026,-1455,-16,744,-1014,-48,-402,-536,722,-814,-192,-438,230,-956,-126,
/*    J(5,5250961)=    */    655,687,1983,-801,
/*    J(11,5250961)=    */    294,820,784,-271,-40,-788,1194,-664,-924,56,
/*    J(13,5250961)=    */    -881,-157,-1087,-1429,-1445,-871,81,-437,-821,125,-157,-1749,
/*    J(17,5250961)=    */    806,1386,-42,1498,1236,1448,954,1210,1042,670,140,1486,1496,973,24,802,
/*    J(2,5654881)=    */    -842,334,-234,-388,0,-388,234,334,842,80,1004,652,-851,-652,1004,-80,
/*    J(1,5654881)=    */    -221,1124,-140,121,257,179,679,1114,-779,-319,-270,-218,-1116,106,573,-307,
/*    J(4,5654881)=    */    667,0,0,0,-1614,0,0,0,0,0,0,0,-1614,0,0,0,
/*    J(3,5654881)=    */    672,-1042,-96,128,-12,-301,-606,-1284,-732,626,-96,-20,-92,362,584,-570,-1168,724,
/*    J(5,5654881)=    */    2087,2816,1124,1772,
/*    J(7,5654881)=    */    2559,2523,1189,1545,915,1467,
/*    J(11,5654881)=    */    1748,342,-734,102,-894,464,-200,724,443,-16,
/*    J(17,5654881)=    */    406,-643,-1330,140,424,-110,-764,250,-1146,-190,-1190,-670,-522,-1048,14,-116,
/*    J(2,9189181)=    */    91,-3030,
/*    J(3,9189181)=    */    -786,-92,-1784,880,-116,1522,830,-99,30,-538,1120,-462,50,-452,1452,142,-319,1006,
/*    J(5,9189181)=    */    167,476,764,-2608,
/*    J(7,9189181)=    */    -1047,36,474,2256,-1184,-270,
/*    J(11,9189181)=    */    1704,1408,2730,1878,-306,1002,864,564,912,2157,
/*    J(13,9189181)=    */    -118,-1338,-272,-2148,-799,-508,-1612,694,394,-184,-1522,-310,
/*    J(17,9189181)=    */    1358,1038,1532,2126,744,-96,696,542,1056,-371,88,2102,380,1520,1056,1222,
/*    J(2,10501921)=    */    966,-854,824,1432,0,1432,-824,-854,-966,170,170,-274,-1205,274,170,-170,
/*    J(1,10501921)=    */    1875,638,-718,-130,417,398,-23,1304,-1311,1009,758,547,418,-79,155,-445,
/*    J(4,10501921)=    */    -2213,0,0,0,-1674,0,0,0,0,0,0,0,-1674,0,0,0,
/*    J(3,10501921)=    */    -470,-2714,162,493,532,-844,708,-504,-478,-936,-2088,-168,-347,278,354,-562,230,-130,
/*    J(5,10501921)=    */    3420,3252,1088,1959,
/*    J(11,10501921)=    */    1033,811,3363,549,539,1075,-439,287,1191,1237,
/*    J(13,10501921)=    */    -757,-781,-105,-1023,1029,-2399,-1987,-563,-219,-653,147,-9,
/*    J(17,10501921)=    */    -1110,-192,-1844,-298,-560,-488,-1754,-1092,-188,-1010,-1337,-2558,-600,-1680,-1334,472,
/*a is 6 t is 367567200    */
/*    J(2,101)=    */    -1,-10,
/*    J(5,101)=    */    -4,0,-2,3,-2,-6,-4,-2,-1,-2,-2,-6,2,3,-2,-4,-2,-2,-1,-2,
/*    J(3,151)=    */    14,9,
/*    J(5,151)=    */    -2,4,6,0,2,-4,0,0,-2,0,-3,0,2,-4,4,-2,2,4,-6,8,
/*    J(2,401)=    */    0,6,-2,4,17,-4,-2,-6,
/*    J(1,401)=    */    8,-7,-14,1,-9,3,0,-1,
/*    J(4,401)=    */    3,0,14,0,0,0,14,0,
/*    J(5,401)=    */    10,2,-6,-1,-2,2,0,-10,3,0,-4,-6,-4,3,-2,6,-4,-6,-9,2,
/*    J(2,601)=    */    23,6,0,6,
/*    J(1,601)=    */    5,0,-24,0,
/*    J(4,601)=    */    -23,-6,0,-6,
/*    J(3,601)=    */    -1,24,
/*    J(5,601)=    */    14,4,2,-6,0,4,2,10,-4,-4,4,13,2,4,0,12,14,2,0,6,
/*    J(2,701)=    */    -5,-26,
/*    J(5,701)=    */    2,6,8,3,14,0,10,0,9,6,2,12,-2,1,-4,0,18,6,-5,8,
/*    J(7,701)=    */    -12,20,8,8,12,5,
/*    J(3,1051)=    */    35,6,
/*    J(5,1051)=    */    16,-3,-2,-14,0,10,-7,12,4,6,16,3,-2,2,12,2,-5,8,2,14,
/*    J(7,1051)=    */    -6,18,9,20,12,30,
/*    J(2,1201)=    */    -23,-4,8,-16,0,-16,-8,-4,
/*    J(1,1201)=    */    -9,2,8,14,-20,-2,-16,-14,
/*    J(4,1201)=    */    7,0,24,0,0,0,24,0,
/*    J(3,1201)=    */    -40,-21,
/*    J(5,1201)=    */    10,4,-2,8,2,16,6,4,-6,-12,12,9,10,4,-14,26,-6,-6,-2,-4,
/*    J(2,1301)=    */    -25,-26,
/*    J(5,1301)=    */    12,-8,8,-5,-2,10,-16,6,-3,2,10,-10,-10,5,22,12,-2,14,-5,4,
/*    J(13,1301)=    */    -6,10,-2,8,-6,6,-8,-30,2,2,0,-3,
/*    J(2,1801)=    */    -1,-30,0,-30,
/*    J(1,1801)=    */    -35,0,-24,0,
/*    J(4,1801)=    */    1,30,0,30,
/*    J(3,1801)=    */    11,-14,22,0,-16,-22,
/*    J(5,1801)=    */    16,-18,-14,-14,6,-8,-6,1,0,-10,4,-8,-10,-4,-18,2,6,-4,10,-2,
/*    J(3,1951)=    */    26,51,
/*    J(5,1951)=    */    10,-2,0,16,16,12,4,-18,22,22,8,4,-22,0,4,14,-16,-5,12,18,
/*    J(13,1951)=    */    38,21,12,-2,4,14,16,14,8,2,20,34,
/*    J(3,2551)=    */    -1,-51,
/*    J(5,2551)=    */    -20,-6,-27,14,6,-10,-8,-22,12,-18,-20,0,-8,-6,-4,-26,-14,6,16,-6,
/*    J(17,2551)=    */    14,0,20,10,4,-14,2,-8,-16,10,28,-10,6,-4,6,19,
/*    J(2,2801)=    */    0,-14,-2,0,49,0,-2,14,
/*    J(1,2801)=    */    0,1,-42,-11,7,-9,28,-1,
/*    J(4,2801)=    */    51,0,-10,0,0,0,-10,0,
/*    J(5,2801)=    */    -2,-14,-12,-14,24,16,-18,-8,0,-3,8,-4,-18,0,-16,2,-36,-6,-2,-8,
/*    J(7,2801)=    */    -20,18,-20,-47,-12,-18,
/*    J(2,3301)=    */    -49,-30,
/*    J(3,3301)=    */    65,21,
/*    J(5,3301)=    */    -28,-2,8,-40,12,-15,18,-4,-12,16,-16,14,16,-4,14,-12,-18,4,-8,16,
/*    J(11,3301)=    */    38,38,10,25,26,18,30,64,36,44,
/*    J(5,3851)=    */    6,-18,-16,-9,14,4,-18,0,23,6,6,4,-14,-11,20,-12,22,10,17,0,
/*    J(7,3851)=    */    -18,-20,10,36,-14,33,
/*    J(11,3851)=    */    -14,30,-10,22,16,4,-8,-36,2,15,
/*    J(2,4201)=    */    -49,30,0,30,
/*    J(1,4201)=    */    -51,0,-40,0,
/*    J(4,4201)=    */    49,-30,0,-30,
/*    J(3,4201)=    */    71,15,
/*    J(5,4201)=    */    -26,-10,18,-5,4,14,6,22,-13,36,-22,-12,4,-13,30,-12,-28,8,-21,14,
/*    J(7,4201)=    */    -36,-14,27,-18,-24,-48,
/*    J(3,4951)=    */    -46,30,-6,-18,9,48,
/*    J(5,4951)=    */    -8,32,-16,22,24,-4,40,0,30,-10,-4,18,-16,-1,32,10,28,-22,-12,16,
/*    J(11,4951)=    */    46,58,18,12,-1,30,32,40,32,62,
/*    J(2,5101)=    */    51,-50,
/*    J(3,5101)=    */    -19,60,
/*    J(5,5101)=    */    28,38,19,-28,-18,46,22,19,-8,6,18,32,15,-6,22,22,16,23,-34,12,
/*    J(17,5101)=    */    0,14,32,38,-6,18,26,20,-10,8,0,2,44,-15,4,-6,
/*    J(3,5851)=    */    -46,3,-24,-36,-51,18,
/*    J(5,5851)=    */    8,12,2,9,14,14,10,-22,-39,22,-22,30,-10,-29,2,14,10,-6,15,30,
/*    J(13,5851)=    */    -16,-20,-38,-6,-14,-16,-16,24,-22,-10,-34,-67,
/*    J(2,6301)=    */    75,26,
/*    J(3,6301)=    */    -52,-36,-36,-3,-60,-12,
/*    J(5,6301)=    */    6,-10,34,-8,2,22,6,42,-20,30,8,6,32,-38,-40,18,10,21,-8,6,
/*    J(7,6301)=    */    90,42,60,72,47,24,
/*    J(5,7151)=    */    16,8,-28,18,6,-14,37,-48,-6,-30,-10,-10,-26,36,20,-18,2,-14,18,2,
/*    J(11,7151)=    */    -23,34,-16,-30,-16,-48,-46,2,8,-42,
/*    J(13,7151)=    */    28,46,0,14,0,62,20,34,64,46,-2,25,
/*    J(2,9901)=    */    99,-10,
/*    J(3,9901)=    */    -40,99,18,-6,63,36,
/*    J(5,9901)=    */    54,-22,30,3,-4,16,28,60,-33,2,28,2,70,-13,0,46,22,60,-17,-18,
/*    J(11,9901)=    */    2,30,-16,-80,0,-48,-26,-14,-4,-43,
/*    J(3,11551)=    */    119,90,
/*    J(5,11551)=    */    56,32,0,16,-2,32,12,-72,24,22,42,-12,0,-24,-9,24,16,-12,-22,-4,
/*    J(7,11551)=    */    -96,-50,-54,-114,-27,-24,
/*    J(11,11551)=    */    -46,-74,-24,-18,-73,-96,-70,-68,-8,-74,
/*    J(2,11701)=    */    -105,-26,
/*    J(3,11701)=    */    -76,20,35,24,-20,55,
/*    J(5,11701)=    */    35,4,28,-8,0,37,-82,6,28,-2,37,-24,10,-18,-36,35,-44,14,-30,-16,
/*    J(13,11701)=    */    11,-67,7,-7,11,-61,-49,-5,-43,-21,21,7,
/*    J(2,12601)=    */    -107,-24,0,-24,
/*    J(1,12601)=    */    51,0,-100,0,
/*    J(4,12601)=    */    -107,-24,0,-24,
/*    J(3,12601)=    */    -76,4,73,-84,44,56,
/*    J(5,12601)=    */    -22,6,-38,-4,16,-26,-48,-36,22,56,-24,4,-48,60,11,-34,4,-56,30,56,
/*    J(7,12601)=    */    -100,-114,-22,-52,-20,-57,
/*    J(3,14851)=    */    -32,-68,20,-56,20,4,-12,9,6,-18,16,38,-14,-44,-2,26,7,0,
/*    J(5,14851)=    */    26,6,-86,9,-4,54,-20,-92,-15,12,24,6,-32,35,8,50,20,-36,-33,22,
/*    J(11,14851)=    */    -38,-74,68,10,24,22,0,24,-10,17,
/*    J(2,15401)=    */    -51,-80,0,-80,
/*    J(1,15401)=    */    -5,0,124,0,
/*    J(4,15401)=    */    -51,-80,0,-80,
/*    J(5,15401)=    */    10,-36,54,42,36,42,-10,4,-15,38,62,-54,42,14,40,60,-66,38,-4,2,
/*    J(7,15401)=    */    -12,118,34,84,76,63,
/*    J(11,15401)=    */    32,-8,0,13,108,16,-8,32,-42,10,
/*    J(2,15601)=    */    0,40,-50,-50,-49,50,-50,-40,
/*    J(1,15601)=    */    16,25,72,-5,-31,85,10,35,
/*    J(4,15601)=    */    83,0,-66,0,0,0,-66,0,
/*    J(3,15601)=    */    -79,-144,
/*    J(5,15601)=    */    -32,-26,-24,-42,22,-16,10,-12,-14,-22,-24,-10,6,-36,42,6,-78,22,5,52,
/*    J(13,15601)=    */    68,22,-38,-38,-14,2,25,-10,26,-30,66,-2,
/*    J(3,17551)=    */    26,-40,-20,-70,12,-22,-31,58,-34,-28,-46,34,-54,2,-50,-45,14,-16,
/*    J(5,17551)=    */    30,5,-46,-34,-22,-44,-6,16,-46,-62,36,-20,10,-90,-44,-18,2,12,-52,-48,
/*    J(13,17551)=    */    42,-16,-12,-63,50,10,-72,-14,-54,-34,-24,4,
/*    J(3,17851)=    */    -154,-69,
/*    J(5,17851)=    */    -28,4,-46,43,72,22,2,-70,67,-8,16,36,-36,49,62,-6,22,0,53,10,
/*    J(7,17851)=    */    96,110,160,46,78,97,
/*    J(17,17851)=    */    -16,-18,62,-38,-22,54,8,22,-32,16,12,72,40,34,6,3,
/*    J(2,18701)=    */    115,-74,
/*    J(5,18701)=    */    -34,-10,-12,-33,24,-28,-10,-82,65,68,-22,20,-40,-15,38,-32,22,-12,39,28,
/*    J(11,18701)=    */    78,100,24,78,46,136,64,14,16,59,
/*    J(17,18701)=    */    -8,14,92,50,62,56,40,52,-16,106,62,32,38,28,12,59,
/*    J(2,19801)=    */    37,96,0,96,
/*    J(1,19801)=    */    99,0,100,0,
/*    J(4,19801)=    */    37,96,0,96,
/*    J(3,19801)=    */    20,-90,45,-30,-144,0,
/*    J(5,19801)=    */    42,24,-48,-70,-26,-56,61,-32,-42,-30,28,40,-46,-20,12,0,0,16,-28,4,
/*    J(11,19801)=    */    104,10,60,27,-4,-42,32,4,28,-44,
/*    J(2,21601)=    */    0,44,-50,8,-16,10,-22,66,47,-66,-22,-10,-16,-8,-50,-44,
/*    J(1,21601)=    */    -26,21,-71,-58,-32,16,53,1,-27,37,-41,34,2,8,-31,45,
/*    J(4,21601)=    */    143,0,0,0,-24,0,0,0,0,0,0,0,-24,0,0,0,
/*    J(3,21601)=    */    4,0,-24,68,82,36,8,11,-76,60,-26,-4,72,28,12,12,40,-52,
/*    J(5,21601)=    */    -30,12,-2,6,-44,-29,44,2,-22,-108,-42,0,16,-32,44,-20,-4,38,-6,-24,
/*    J(2,23801)=    */    -153,14,0,14,
/*    J(1,23801)=    */    149,0,40,0,
/*    J(4,23801)=    */    153,-14,0,-14,
/*    J(5,23801)=    */    18,57,-38,34,-22,48,32,28,52,34,-22,-4,74,24,-2,50,36,58,68,14,
/*    J(7,23801)=    */    76,42,91,70,168,0,
/*    J(17,23801)=    */    -84,-64,-42,-2,-26,12,-28,-9,52,12,-94,10,-46,4,-24,-12,
/*    J(3,28051)=    */    -130,-189,
/*    J(5,28051)=    */    -68,-80,-30,-47,-94,24,-28,-38,-67,-18,-36,-4,-82,-65,-6,-36,8,-38,-33,-38,
/*    J(11,28051)=    */    0,-76,-166,-50,-88,-56,-78,-98,-2,-91,
/*    J(17,28051)=    */    -10,-14,-110,-12,24,-4,-4,-70,18,-42,-62,-48,-88,-80,14,-57,
/*    J(3,33151)=    */    185,6,
/*    J(5,33151)=    */    -18,38,-102,58,25,-52,44,-16,62,1,-26,30,-108,-20,-41,-60,42,-6,2,-69,
/*    J(13,33151)=    */    30,60,-14,76,150,24,108,14,81,104,106,14,
/*    J(17,33151)=    */    34,44,14,42,-42,-46,-6,-52,-6,13,-88,42,44,36,-80,16,
/*    J(3,34651)=    */    80,-131,-92,84,-157,-64,
/*    J(5,34651)=    */    8,68,-56,41,64,-32,28,-26,-77,30,-84,20,-66,37,86,-28,-12,-64,51,56,
/*    J(7,34651)=    */    -28,-40,-38,162,-6,5,
/*    J(11,34651)=    */    -66,-142,42,-58,28,28,-8,-60,-82,-35,
/*    J(2,40801)=    */    102,74,38,8,0,8,-38,74,-102,4,28,-40,-35,40,28,-4,
/*    J(1,40801)=    */    -23,64,42,-13,-41,133,37,50,7,-35,30,-88,-14,20,21,-7,
/*    J(4,40801)=    */    187,0,0,0,-54,0,0,0,0,0,0,0,-54,0,0,0,
/*    J(3,40801)=    */    -40,-219,
/*    J(5,40801)=    */    124,-40,-73,-122,-14,60,-50,11,-46,16,100,-86,19,-30,4,90,-54,-17,8,64,
/*    J(17,40801)=    */    -48,120,-40,-36,-76,-64,18,32,10,25,-26,44,14,-24,44,40,
/*    J(2,42901)=    */    -201,-50,
/*    J(3,42901)=    */    -235,-156,
/*    J(5,42901)=    */    30,21,48,22,38,62,-76,90,88,-26,40,-4,84,90,0,62,-48,154,104,50,
/*    J(11,42901)=    */    56,-110,-10,22,38,-67,-66,-126,-14,-98,
/*    J(13,42901)=    */    -36,-6,-90,-52,-96,-112,-124,-50,-86,48,-20,-157,
/*    J(2,44201)=    */    207,-26,0,-26,
/*    J(1,44201)=    */    -115,0,-176,0,
/*    J(4,44201)=    */    -207,26,0,26,
/*    J(5,44201)=    */    -98,-20,-12,126,-36,-113,78,-28,86,54,-62,50,-42,86,58,-68,-2,-36,150,48,
/*    J(13,44201)=    */    20,88,92,76,150,-70,68,92,80,4,-12,35,
/*    J(17,44201)=    */    2,-194,-68,-54,-66,-108,-88,-108,-68,-101,-104,-164,-104,-140,-42,-90,
/*    J(5,50051)=    */    46,-6,114,15,-30,58,22,-48,-83,-60,48,14,40,-93,40,-48,-20,74,-19,-10,
/*    J(7,50051)=    */    84,156,182,90,-46,149,
/*    J(11,50051)=    */    70,58,-150,44,-42,-48,-52,-56,-98,-57,
/*    J(13,50051)=    */    -68,70,4,-44,-4,30,-58,-62,-120,34,-22,109,
/*    J(3,53551)=    */    20,178,97,-54,224,38,
/*    J(5,53551)=    */    -28,-46,14,18,112,-42,-28,14,-80,50,-90,-4,-133,-42,146,-46,-62,-20,-46,122,
/*    J(7,53551)=    */    -93,117,141,-9,119,-3,
/*    J(17,53551)=    */    -148,-80,-146,-58,-5,-136,-112,-114,-114,-206,-22,-32,-74,-64,-58,-60,
/*    J(2,54601)=    */    77,-156,0,-156,
/*    J(1,54601)=    */    -149,0,-180,0,
/*    J(4,54601)=    */    77,-156,0,-156,
/*    J(3,54601)=    */    -64,195,
/*    J(5,54601)=    */    -48,2,-110,-44,-66,-16,78,-76,65,18,-154,36,-12,20,-36,-58,76,-18,80,-38,
/*    J(7,54601)=    */    30,146,10,-110,60,-95,
/*    J(13,54601)=    */    -106,-100,-109,-28,-102,-124,-96,42,-32,16,-58,-188,
/*    J(2,56101)=    */    -225,74,
/*    J(3,56101)=    */    -229,15,
/*    J(5,56101)=    */    45,38,-52,36,6,64,-30,-102,46,42,22,2,-96,100,-80,118,112,-6,94,20,
/*    J(11,56101)=    */    26,-111,-54,-176,-2,-66,-4,68,-26,-118,
/*    J(17,56101)=    */    -8,52,-74,22,56,-10,145,94,72,-66,70,72,96,20,32,72,
/*    J(2,66301)=    */    -149,-210,
/*    J(3,66301)=    */    -289,-84,
/*    J(5,66301)=    */    142,-10,116,-5,-44,-12,-40,0,44,-12,6,26,18,-72,-6,28,-10,-102,-4,46,
/*    J(13,66301)=    */    -176,-18,-109,-210,-184,-124,-142,-8,-116,-184,-64,-70,
/*    J(17,66301)=    */    -36,-144,-44,42,-24,-108,-76,-146,-74,41,-106,42,-40,44,-40,-46,
/*    J(2,70201)=    */    -251,60,0,60,
/*    J(1,70201)=    */    51,0,-260,0,
/*    J(4,70201)=    */    -251,60,0,60,
/*    J(3,70201)=    */    18,160,-28,164,-28,-46,60,31,-60,14,178,80,64,-30,-4,60,31,4,
/*    J(5,70201)=    */    -66,28,52,62,8,104,67,38,-2,-16,94,60,-4,-22,50,-138,48,50,36,30,
/*    J(13,70201)=    */    -217,-63,-99,-45,95,-57,-93,-25,39,-47,-35,-65,
/*    J(5,77351)=    */    158,-22,-166,-142,50,40,-8,-192,-56,-76,82,16,-190,-4,-12,49,8,-124,-66,-16,
/*    J(7,77351)=    */    -238,-54,104,-40,14,-81,
/*    J(13,77351)=    */    22,228,178,194,158,90,68,139,30,120,60,220,
/*    J(17,77351)=    */    -56,46,18,48,-36,102,142,36,88,182,-54,57,0,108,110,58,
/*    J(2,79201)=    */    -162,54,0,54,0,54,0,54,162,54,-18,-60,-37,60,-18,-54,
/*    J(1,79201)=    */    39,-45,12,189,9,93,-27,51,-55,-78,-36,60,18,-108,-21,36,
/*    J(4,79201)=    */    251,0,0,0,-90,0,0,0,0,0,0,0,-90,0,0,0,
/*    J(3,79201)=    */    -52,-116,289,36,-40,161,
/*    J(5,79201)=    */    -46,-64,-48,-198,-12,14,-108,50,-146,-10,-42,28,46,-4,-78,-22,50,14,-139,-56,
/*    J(11,79201)=    */    85,72,150,38,92,-128,28,128,-78,-14,
/*    J(2,81901)=    */    -285,-26,
/*    J(3,81901)=    */    -256,-21,-66,-114,-153,60,
/*    J(5,81901)=    */    70,-62,48,87,38,-20,-124,116,65,78,110,-52,126,1,-28,114,-150,-26,-17,80,
/*    J(7,81901)=    */    64,-192,4,-140,-220,-63,
/*    J(13,81901)=    */    46,-60,-118,54,-138,-82,-94,14,92,6,48,-133,
/*    J(2,91801)=    */    -251,120,0,120,
/*    J(1,91801)=    */    -125,0,276,0,
/*    J(4,91801)=    */    -251,120,0,120,
/*    J(3,91801)=    */    -18,124,-31,140,24,54,-28,104,108,-32,-36,-111,108,-70,54,72,-32,10,
/*    J(5,91801)=    */    -12,170,84,-102,42,-118,94,52,10,-98,-119,-42,62,0,-78,-42,106,38,-84,-94,
/*    J(17,91801)=    */    -49,171,11,89,57,115,111,49,65,139,-69,-11,-51,35,83,155,
/*    J(2,92401)=    */    0,184,-82,-50,79,50,-82,-184,
/*    J(1,92401)=    */    -72,-119,-8,-101,65,-91,-182,131,
/*    J(4,92401)=    */    -301,0,30,0,0,0,30,0,
/*    J(3,92401)=    */    176,351,
/*    J(5,92401)=    */    -46,-10,-91,-118,24,54,-36,40,-28,98,-76,14,-100,-52,210,12,-20,-28,-78,210,
/*    J(7,92401)=    */    -266,-236,-323,-236,-350,-284,
/*    J(11,92401)=    */    26,-90,-20,-50,14,128,-158,62,124,117,
/*    J(2,93601)=    */    0,-38,-58,132,32,34,126,80,-15,-80,126,-34,32,-132,-58,38,
/*    J(1,93601)=    */    90,-10,57,67,158,9,-97,-8,-33,82,-125,-75,4,-53,-21,104,
/*    J(4,93601)=    */    -17,0,0,0,-216,0,0,0,0,0,0,0,-216,0,0,0,
/*    J(3,93601)=    */    -34,-336,42,-66,-225,-12,
/*    J(5,93601)=    */    54,6,57,46,88,-2,-194,-77,108,-68,16,-84,-57,118,62,36,-130,53,42,90,
/*    J(13,93601)=    */    50,196,224,24,128,18,40,86,56,-41,176,-22,
/*    J(3,103951)=    */    0,58,-52,-49,-20,50,-10,220,136,58,-48,24,-78,64,64,-126,200,96,
/*    J(5,103951)=    */    22,20,6,-28,-133,-64,86,-4,38,50,-84,-32,-76,-52,30,-140,-76,-60,56,-160,
/*    J(7,103951)=    */    -24,248,-99,-114,-30,-66,
/*    J(11,103951)=    */    -258,-252,-228,-132,-258,-290,-378,-180,-174,-249,
/*    J(2,107101)=    */    251,210,
/*    J(3,107101)=    */    -223,-152,-194,-99,26,-214,
/*    J(5,107101)=    */    34,58,-130,-110,-72,-30,-118,106,26,-116,18,-38,98,14,4,-8,60,-40,-4,-53,
/*    J(7,107101)=    */    -213,-60,-156,-132,-376,-240,
/*    J(17,107101)=    */    -74,-72,-8,-56,-116,6,-32,-222,-168,38,-44,117,-96,30,-36,-50,
/*    J(2,109201)=    */    0,-40,-2,-106,-289,106,-2,40,
/*    J(1,109201)=    */    -192,77,-16,-73,-239,25,-46,31,
/*    J(4,109201)=    */    -317,0,-66,0,0,0,-66,0,
/*    J(3,109201)=    */    29,-315,
/*    J(5,109201)=    */    -60,-88,180,16,-103,12,76,138,136,15,38,-22,-34,82,-85,124,-26,76,36,-27,
/*    J(7,109201)=    */    -232,56,16,120,-15,152,
/*    J(13,109201)=    */    104,-92,-10,-150,-6,126,-62,106,144,24,-84,29,
/*    J(2,118801)=    */    73,156,-144,108,0,108,144,156,
/*    J(1,118801)=    */    -33,204,-172,-168,-84,24,-60,-84,
/*    J(4,118801)=    */    343,0,24,0,0,0,24,0,
/*    J(3,118801)=    */    24,182,8,36,113,84,-112,-8,-100,4,26,24,104,-144,24,-72,50,20,
/*    J(5,118801)=    */    102,64,18,56,-51,-204,164,82,86,-48,52,94,162,92,-48,34,60,82,62,-80,
/*    J(11,118801)=    */    -147,29,13,207,-43,5,211,41,109,69,
/*    J(2,122401)=    */    78,-62,-74,50,0,50,74,-62,-78,-116,136,50,133,-50,136,116,
/*    J(1,122401)=    */    -69,17,-2,58,-125,-4,-101,-71,9,-2,-82,-19,-66,239,103,-62,
/*    J(4,122401)=    */    -37,0,0,0,-246,0,0,0,0,0,0,0,-246,0,0,0,
/*    J(3,122401)=    */    194,29,44,240,133,262,
/*    J(5,122401)=    */    10,64,-38,106,159,-84,-10,-20,152,-84,22,-8,76,262,34,-14,52,8,244,78,
/*    J(17,122401)=    */    86,152,82,102,4,110,-14,-115,54,86,104,-50,54,-152,94,116,
/*    J(2,140401)=    */    121,220,112,44,0,44,-112,220,
/*    J(1,140401)=    */    -209,-20,-132,-8,44,248,44,-116,
/*    J(4,140401)=    */    -217,0,-216,0,0,0,-216,0,
/*    J(3,140401)=    */    8,40,-84,38,40,71,0,116,-42,-100,-104,158,132,0,191,-40,154,24,
/*    J(5,140401)=    */    -190,86,-44,104,130,-74,50,152,112,38,-58,-38,214,72,38,-84,60,-36,61,166,
/*    J(13,140401)=    */    -184,-51,-320,-96,-76,10,-64,-40,56,-108,-226,-20,
/*    J(3,150151)=    */    -286,-441,
/*    J(5,150151)=    */    0,186,74,46,-170,74,22,-106,60,-244,28,198,-38,-66,-290,82,148,-72,42,-205,
/*    J(7,150151)=    */    -28,-26,-296,-30,-290,-297,
/*    J(11,150151)=    */    52,112,128,232,125,102,50,86,286,-162,
/*    J(13,150151)=    */    48,102,-168,62,133,-84,142,52,98,-58,-90,-160,
/*    J(2,151201)=    */    234,-60,18,-66,35,66,18,60,234,84,-36,-60,0,-60,36,84,
/*    J(1,151201)=    */    -75,-132,-105,144,26,18,60,24,57,201,21,57,-27,-69,84,-177,
/*    J(4,151201)=    */    251,0,0,0,210,0,0,0,0,0,0,0,210,0,0,0,
/*    J(3,151201)=    */    -14,-54,-226,2,176,101,36,-8,146,-8,-2,30,-8,20,117,-108,54,-42,
/*    J(5,151201)=    */    124,70,50,-294,142,146,6,139,-78,124,92,-22,86,-246,14,-18,-60,108,-80,46,
/*    J(7,151201)=    */    174,399,-30,132,36,44,
/*    J(3,160651)=    */    -274,26,-194,-142,38,2,-70,-77,-168,-94,66,-38,-144,-52,56,-84,-115,-102,
/*    J(5,160651)=    */    136,148,148,71,12,122,198,-30,-115,-38,208,128,6,-45,-50,38,118,140,-155,124,
/*    J(7,160651)=    */    -44,-30,112,358,-94,75,
/*    J(17,160651)=    */    104,-90,130,70,-100,-58,-22,-144,0,18,-4,-36,-24,-166,-86,203,
/*    J(3,193051)=    */    -28,172,122,-6,60,28,196,-261,-52,98,152,250,56,52,-66,160,-187,36,
/*    J(5,193051)=    */    228,60,-176,133,-136,-60,-28,-70,41,-88,-46,-178,4,35,102,-8,12,-36,59,-84,
/*    J(11,193051)=    */    -80,18,-54,2,-136,-8,-392,-154,126,-71,
/*    J(13,193051)=    */    54,320,270,296,38,380,110,48,226,164,142,109,
/*    J(2,198901)=    */    -249,-370,
/*    J(3,198901)=    */    -232,19,-176,-144,-217,-376,
/*    J(5,198901)=    */    28,-92,-108,-273,2,184,10,-146,-121,-92,56,-144,-92,-189,-186,186,-248,-82,-69,-20,
/*    J(13,198901)=    */    72,-148,-156,94,-60,-160,54,-144,-76,-364,-124,-133,
/*    J(17,198901)=    */    -98,-132,-20,-112,90,268,-30,-110,24,-78,64,-212,32,-66,-64,-67,
/*    J(2,200201)=    */    447,-14,0,-14,
/*    J(1,200201)=    */    349,0,-280,0,
/*    J(4,200201)=    */    -447,14,0,14,
/*    J(5,200201)=    */    36,-130,192,-28,-198,-169,-110,154,234,-134,4,60,82,214,-124,8,-92,188,54,-132,
/*    J(7,200201)=    */    -82,-120,380,26,8,-87,
/*    J(11,200201)=    */    219,316,80,342,86,98,154,98,222,-120,
/*    J(13,200201)=    */    42,118,358,-3,26,102,-18,-164,116,-16,180,12,
/*    J(2,218401)=    */    207,-128,-42,116,-32,-54,34,-226,0,-226,-34,-54,32,116,42,-128,
/*    J(1,218401)=    */    111,121,36,-124,-202,-108,122,-247,-24,-37,-80,30,58,86,-60,151,
/*    J(4,218401)=    */    -257,0,0,0,276,0,0,0,0,0,0,0,276,0,0,0,
/*    J(3,218401)=    */    419,504,
/*    J(5,218401)=    */    -6,-130,-306,64,140,192,-80,-40,-8,134,242,-150,-160,96,64,296,-144,-92,81,116,
/*    J(7,218401)=    */    -120,21,-288,-424,60,-132,
/*    J(13,218401)=    */    -234,64,-154,-28,-224,-126,-104,-138,-84,-334,-393,-66,
/*    J(2,224401)=    */    0,-116,-34,-302,-113,302,-34,116,
/*    J(1,224401)=    */    -192,-79,-196,-121,-95,193,182,-221,
/*    J(4,224401)=    */    227,0,294,0,0,0,294,0,
/*    J(3,224401)=    */    -385,144,
/*    J(5,224401)=    */    132,-92,-104,47,100,-4,-170,-156,-155,16,-52,-56,-32,-127,128,158,-8,-140,-293,232,
/*    J(11,224401)=    */    57,-28,306,102,198,30,126,390,-18,-42,
/*    J(17,224401)=    */    -38,-250,-32,8,240,16,-26,62,-212,65,-56,-94,-22,20,-192,-68,
/*    J(3,232051)=    */    365,546,
/*    J(5,232051)=    */    -102,-352,-18,60,-22,-38,-4,-40,260,-30,56,-200,-22,168,-128,48,-6,-29,86,-188,
/*    J(7,232051)=    */    130,318,-8,397,110,-108,
/*    J(13,232051)=    */    -214,-122,-276,22,-130,-350,-81,-100,36,96,-78,-260,
/*    J(17,232051)=    */    245,-17,179,107,99,307,291,65,151,77,297,101,257,229,161,-85,
/*    J(2,243101)=    */    -149,470,
/*    J(5,243101)=    */    186,-222,-110,13,-180,244,-86,-36,-127,-40,34,-104,-92,133,-208,-100,-146,-168,-23,-194,
/*    J(11,243101)=    */    -134,-2,-40,-50,-174,32,-466,-190,56,-23,
/*    J(13,243101)=    */    384,-42,-58,56,88,40,88,-186,-74,84,116,153,
/*    J(17,243101)=    */    -124,96,152,200,24,112,258,170,84,212,-124,308,220,120,80,81,
/*    J(2,257401)=    */    199,330,0,330,
/*    J(1,257401)=    */    501,0,-80,0,
/*    J(4,257401)=    */    -199,-330,0,-330,
/*    J(3,257401)=    */    254,-106,-241,-126,112,-377,
/*    J(5,257401)=    */    -450,20,94,166,87,-196,90,8,172,136,-218,128,148,6,-96,-162,94,94,112,-104,
/*    J(11,257401)=    */    -104,174,-96,131,332,-126,-104,-164,38,-38,
/*    J(13,257401)=    */    -120,60,-40,272,-58,-184,250,137,190,108,146,-8,
/*    J(2,300301)=    */    275,-474,
/*    J(3,300301)=    */    -484,111,
/*    J(5,300301)=    */    -190,22,38,171,-66,-106,-90,32,309,-398,-170,-8,6,37,-204,-200,-108,-34,291,2,
/*    J(7,300301)=    */    586,208,416,426,300,79,
/*    J(11,300301)=    */    214,16,-30,28,168,-130,8,-146,380,261,
/*    J(13,300301)=    */    76,110,-154,-248,-102,102,-168,-258,-34,-360,72,-77,
/*    J(2,321301)=    */    -201,-530,
/*    J(3,321301)=    */    108,-76,-226,294,104,96,14,191,46,76,-170,-340,290,-30,60,-222,-13,22,
/*    J(5,321301)=    */    240,168,-20,165,34,-234,-78,90,93,-36,76,-184,128,-87,34,-58,-50,14,-63,-68,
/*    J(7,321301)=    */    -166,-288,-256,-166,-500,171,
/*    J(17,321301)=    */    50,-32,-66,132,-152,188,84,132,264,268,-48,-206,220,34,124,-7,
/*    J(2,367201)=    */    -182,66,-38,200,0,200,38,66,182,124,116,-160,-317,160,116,-124,
/*    J(1,367201)=    */    163,66,76,199,-125,233,-139,4,-11,-73,-98,228,244,-164,-213,-53,
/*    J(4,367201)=    */    251,0,0,0,-390,0,0,0,0,0,0,0,-390,0,0,0,
/*    J(3,367201)=    */    -194,-232,-170,-112,-174,196,-42,-88,-190,88,-124,172,-35,-278,244,-6,-178,-24,
/*    J(5,367201)=    */    6,112,58,92,31,-100,398,-52,350,284,-16,-32,148,192,14,-126,52,156,206,146,
/*    J(17,367201)=    */    -300,9,-80,64,78,40,-122,-248,114,66,166,-238,-20,10,-272,-186,
/*    J(2,415801)=    */    -251,-420,0,-420,
/*    J(1,415801)=    */    435,0,476,0,
/*    J(4,415801)=    */    -251,-420,0,-420,
/*    J(3,415801)=    */    -54,44,-242,274,308,58,-92,-49,272,124,-116,56,-86,104,-114,-152,-153,168,
/*    J(5,415801)=    */    -92,-136,58,22,54,-90,-300,168,-134,-184,214,118,282,28,-90,142,-226,324,-143,-106,
/*    J(7,415801)=    */    -131,195,255,359,-353,73,
/*    J(11,415801)=    */    -124,-88,-368,-444,-408,-356,-404,-145,-116,-628,
/*    J(2,428401)=    */    119,200,328,-244,0,-244,-328,200,
/*    J(1,428401)=    */    -43,-266,-170,-346,128,326,-258,134,
/*    J(4,428401)=    */    -649,0,60,0,0,0,60,0,
/*    J(3,428401)=    */    -616,318,264,-267,276,150,
/*    J(5,428401)=    */    -193,314,410,78,-158,124,16,274,-76,110,26,96,434,44,114,22,-44,106,60,-18,
/*    J(7,428401)=    */    238,618,216,660,525,528,
/*    J(17,428401)=    */    -134,-546,-328,-438,-320,-184,-403,-436,-480,-194,-592,-402,-550,-276,-354,-382,
/*    J(2,448801)=    */    -226,28,-144,-110,251,110,-144,-28,-226,314,-98,-2,0,-2,98,314,
/*    J(1,448801)=    */    -177,68,-181,-143,238,-185,36,-20,295,-61,-153,-178,37,248,56,235,
/*    J(4,448801)=    */    -293,0,0,0,-426,0,0,0,0,0,0,0,-426,0,0,0,
/*    J(3,448801)=    */    440,771,
/*    J(5,448801)=    */    294,68,70,-226,-90,-206,259,36,-226,106,88,-84,-88,-52,-160,-62,-192,-6,-18,-82,
/*    J(11,448801)=    */    -16,-180,328,-86,244,384,-103,-152,-148,102,
/*    J(17,448801)=    */    -138,-206,-208,-194,-24,-320,-300,58,-100,-223,-368,-58,-284,276,4,-126,
/*    J(3,450451)=    */    -46,353,56,84,-401,-62,
/*    J(5,450451)=    */    -88,-20,394,-21,108,-314,78,206,-5,102,-266,-2,-122,-135,46,-358,194,152,-179,294,
/*    J(7,450451)=    */    72,440,30,194,302,-353,
/*    J(11,450451)=    */    240,-14,32,414,604,142,212,406,110,405,
/*    J(13,450451)=    */    178,168,236,368,64,-82,122,-82,74,-128,530,189,
/*    J(2,504901)=    */    -225,-674,
/*    J(3,504901)=    */    106,358,-168,100,126,254,320,-133,-282,252,-54,-192,28,192,6,258,-9,-248,
/*    J(5,504901)=    */    82,-258,-260,71,-16,-66,-2,-276,397,-58,4,174,-122,45,60,84,12,-240,15,-392,
/*    J(11,504901)=    */    152,-14,-182,-232,-400,-126,-466,156,-212,-393,
/*    J(17,504901)=    */    -198,-122,252,-164,192,-78,196,344,210,176,-16,-20,250,10,100,329,
/*    J(2,530401)=    */    66,190,50,236,0,236,-50,190,-66,64,-52,-328,323,328,-52,-64,
/*    J(1,530401)=    */    -143,296,-102,191,65,-83,255,382,-113,59,-90,86,-58,-38,275,203,
/*    J(4,530401)=    */    -101,0,0,0,-510,0,0,0,0,0,0,0,-510,0,0,0,
/*    J(3,530401)=    */    -331,504,
/*    J(5,530401)=    */    -204,144,132,178,342,-154,240,136,168,4,36,-75,140,576,204,-224,40,90,350,-14,
/*    J(13,530401)=    */    -351,-265,-359,-243,31,-87,-199,99,-119,-3,-357,305,
/*    J(17,530401)=    */    12,46,-417,-74,-40,236,-146,-268,-60,-130,-158,282,-94,156,156,-46,
/*    J(2,600601)=    */    -763,-96,0,-96,
/*    J(1,600601)=    */    -765,0,-124,0,
/*    J(4,600601)=    */    -763,-96,0,-96,
/*    J(3,600601)=    */    -805,-741,
/*    J(5,600601)=    */    54,540,186,358,-88,150,498,-182,418,172,14,113,168,196,118,36,198,36,300,94,
/*    J(7,600601)=    */    172,-172,-304,60,-715,-204,
/*    J(11,600601)=    */    -103,400,-476,-254,24,92,136,216,102,-28,
/*    J(13,600601)=    */    284,500,460,426,812,316,742,476,376,644,648,581,
/*    J(2,673201)=    */    0,166,238,496,113,-496,238,-166,
/*    J(1,673201)=    */    -296,-277,-82,199,295,-467,324,229,
/*    J(4,673201)=    */    -557,0,-426,0,0,0,-426,0,
/*    J(3,673201)=    */    -115,-816,192,-264,-708,36,
/*    J(5,673201)=    */    -170,-104,-126,-428,-162,-358,-14,-182,-278,-6,-316,162,-458,-292,-194,-147,212,-102,-664,-24,
/*    J(11,673201)=    */    362,66,133,226,242,-52,370,290,862,162,
/*    J(17,673201)=    */    508,438,428,506,668,434,85,326,0,366,540,194,160,444,368,552,
/*    J(2,729301)=    */    -201,830,
/*    J(3,729301)=    */    -799,-900,
/*    J(5,729301)=    */    132,-338,-353,136,44,258,-304,-245,-338,284,164,-278,-257,-356,98,140,-696,-277,-150,40,
/*    J(11,729301)=    */    -171,-352,-408,-526,-296,-50,-230,-708,218,-140,
/*    J(13,729301)=    */    -196,58,424,-172,58,-44,80,390,520,-77,388,78,
/*    J(17,729301)=    */    -222,22,-402,-252,-141,-452,-242,-726,-490,-224,78,-458,-258,-150,-410,-366,
/*    J(2,795601)=    */    0,190,50,-200,799,200,50,-190,
/*    J(1,795601)=    */    176,-5,-780,-5,-215,145,170,245,
/*    J(4,795601)=    */    -701,0,-390,0,0,0,-390,0,
/*    J(3,795601)=    */    -412,116,869,-324,316,328,
/*    J(5,795601)=    */    -88,264,-110,214,-258,-154,264,136,-292,-300,-72,256,8,56,-76,-272,-225,-36,-456,-40,
/*    J(13,795601)=    */    214,46,184,-168,-190,124,-194,-13,-410,184,-394,-528,
/*    J(17,795601)=    */    -464,-236,-38,-44,24,-336,224,393,-152,-112,32,-236,182,4,140,-232,
/*    J(2,800801)=    */    359,-38,-16,-180,72,116,-308,-434,0,-434,308,116,-72,-180,16,-38,
/*    J(1,800801)=    */    393,-183,-268,47,-320,159,-32,19,-340,303,234,95,256,215,158,-43,
/*    J(4,800801)=    */    -849,0,0,0,-200,0,0,0,0,0,0,0,-200,0,0,0,
/*    J(5,800801)=    */    162,-118,-156,-232,204,128,-136,-150,-60,224,10,214,-634,-161,258,94,70,240,148,214,
/*    J(7,800801)=    */    -511,135,581,-127,171,-159,
/*    J(11,800801)=    */    332,-391,-52,272,346,528,-160,252,24,-74,
/*    J(13,800801)=    */    -213,108,54,500,-420,-338,18,-386,-158,-80,104,-22,
/*    J(2,982801)=    */    17,246,-252,-606,0,-606,252,246,
/*    J(1,982801)=    */    -333,-324,460,-558,-324,-6,-24,372,
/*    J(4,982801)=    */    -217,0,-684,0,0,0,-684,0,
/*    J(3,982801)=    */    122,112,-216,212,-176,84,328,-8,-206,296,-224,272,180,-27,340,568,296,-454,
/*    J(5,982801)=    */    -370,86,-324,-328,-236,-372,230,-305,-224,-2,282,-218,-76,-108,-12,234,218,-408,-342,-126,
/*    J(7,982801)=    */    968,384,438,13,434,-124,
/*    J(13,982801)=    */    626,-2,528,472,500,100,408,785,676,68,308,80,
/*    J(2,1029601)=    */    -202,156,322,-506,0,-506,-322,156,202,-130,80,54,357,-54,80,130,
/*    J(1,1029601)=    */    -217,-408,112,-79,3,53,-13,-36,385,79,392,150,54,360,-565,-103,
/*    J(4,1029601)=    */    -901,0,0,0,330,0,0,0,0,0,0,0,330,0,0,0,
/*    J(3,1029601)=    */    464,-528,-312,-105,-72,-816,
/*    J(5,1029601)=    */    330,43,-38,-38,-210,64,264,156,262,382,-306,80,-156,272,-362,-244,168,-122,-136,60,
/*    J(11,1029601)=    */    968,774,424,520,2,452,64,464,547,360,
/*    J(13,1029601)=    */    212,169,364,112,530,26,628,192,300,904,358,-130,
/*    J(3,1093951)=    */    905,-320,-476,150,-370,74,
/*    J(5,1093951)=    */    413,364,218,214,138,351,362,-300,100,-36,149,656,-14,-78,250,-69,130,-180,480,-154,
/*    J(11,1093951)=    */    892,614,50,360,224,270,514,958,553,668,
/*    J(13,1093951)=    */    -276,-420,-308,-254,264,-186,90,-826,-374,-172,-633,-442,
/*    J(17,1093951)=    */    388,247,318,238,374,706,320,390,-174,290,116,-108,456,574,-212,156,
/*    J(2,1178101)=    */    -649,-870,
/*    J(3,1178101)=    */    -505,-530,490,84,-640,830,
/*    J(5,1178101)=    */    -276,-370,272,-22,60,370,234,154,246,-228,-54,100,560,-242,-29,-316,62,306,-304,136,
/*    J(7,1178101)=    */    -357,93,17,501,-723,-477,
/*    J(11,1178101)=    */    952,776,768,410,-2,520,257,436,726,810,
/*    J(17,1178101)=    */    -200,484,514,-276,114,121,50,156,-30,80,388,160,752,180,432,100,
/*    J(2,1201201)=    */    0,688,226,-274,-47,274,226,-688,
/*    J(1,1201201)=    */    -240,-239,-202,-625,479,-335,-472,-301,
/*    J(4,1201201)=    */    499,0,690,0,0,0,690,0,
/*    J(3,1201201)=    */    -904,315,
/*    J(5,1201201)=    */    -564,-562,-318,-273,-386,190,-796,-108,-168,-104,-24,-616,-80,-8,-184,-188,-348,72,-232,-384,
/*    J(7,1201201)=    */    258,-280,828,78,-288,-345,
/*    J(11,1201201)=    */    -626,66,198,-146,668,46,0,-427,-52,74,
/*    J(13,1201201)=    */    -312,-670,-870,-558,-553,-578,-806,-666,-1188,-288,-908,-664,
/*    J(2,1458601)=    */    -1201,90,0,90,
/*    J(1,1458601)=    */    -275,0,1176,0,
/*    J(4,1458601)=    */    1201,-90,0,-90,
/*    J(3,1458601)=    */    -376,975,
/*    J(5,1458601)=    */    -442,-214,-615,138,-144,-214,-412,-580,-52,206,-284,-650,-474,116,284,-826,-124,-526,-302,294,
/*    J(11,1458601)=    */    -307,-1187,-715,-557,-491,-355,71,-615,-251,-819,
/*    J(13,1458601)=    */    -730,-734,-714,-606,-566,-66,-602,-370,-596,-646,-1037,-1238,
/*    J(17,1458601)=    */    429,544,-192,586,150,658,388,66,92,-216,-46,232,-2,426,386,-272,
/*    J(3,2088451)=    */    -34,608,-150,-189,50,-576,-178,-452,532,92,164,620,206,204,-492,240,-166,846,
/*    J(5,2088451)=    */    -502,-84,526,-710,-620,-88,-398,561,-724,-396,-604,120,512,46,-50,48,-10,2,-418,-232,
/*    J(7,2088451)=    */    -164,66,-417,-504,-942,798,
/*    J(13,2088451)=    */    832,360,-162,-30,222,1150,318,912,305,382,812,644,
/*    J(17,2088451)=    */    416,72,-46,-382,294,228,364,56,406,838,206,822,-376,-111,538,380,
/*    J(2,2187901)=    */    1099,990,
/*    J(3,2187901)=    */    1082,-399,456,1428,285,342,
/*    J(5,2187901)=    */    174,-166,-562,19,-478,-492,-478,184,-89,-98,-24,-558,-294,-469,-488,656,-590,-282,-257,86,
/*    J(11,2187901)=    */    -1268,-844,-812,-986,-618,-1408,-520,-1152,-768,-95,
/*    J(13,2187901)=    */    134,448,1180,1232,226,1018,420,652,788,550,168,931,
/*    J(17,2187901)=    */    -538,-632,-816,-2,-446,-734,-634,-298,-370,-550,-966,-392,-1288,-1154,-224,-715,
/*    J(2,2402401)=    */    -62,-262,530,-250,0,-250,-530,-262,62,264,-576,614,-117,-614,-576,-264,
/*    J(1,2402401)=    */    21,-33,350,404,-77,134,-543,431,563,-166,544,393,192,-829,-21,-302,
/*    J(4,2402401)=    */    763,0,0,0,954,0,0,0,0,0,0,0,954,0,0,0,
/*    J(3,2402401)=    */    491,-1245,
/*    J(5,2402401)=    */    358,-870,250,116,-142,18,-418,102,-286,389,-398,-682,50,-562,-50,116,-70,938,-158,-92,
/*    J(7,2402401)=    */    -816,-216,-116,-1686,-861,-786,
/*    J(11,2402401)=    */    72,796,12,-132,-768,6,348,429,-300,834,
/*    J(13,2402401)=    */    -602,28,520,-542,-40,-122,-308,-102,343,-138,992,230,
/*    J(2,2570401)=    */    863,466,62,560,504,-326,-130,-32,0,-32,130,-326,-504,560,-62,466,
/*    J(1,2570401)=    */    139,226,438,697,-478,-467,528,-196,584,-38,596,311,-250,-389,84,-208,
/*    J(4,2570401)=    */    -1601,0,0,0,-60,0,0,0,0,0,0,0,-60,0,0,0,
/*    J(3,2570401)=    */    -188,646,140,-500,622,290,-302,65,-454,-270,198,-704,-692,218,828,-308,48,-556,
/*    J(5,2570401)=    */    392,-136,-198,312,552,684,-296,-412,44,1360,249,-184,-396,-244,184,-116,-168,-312,-274,988,
/*    J(7,2570401)=    */    510,546,396,-1202,285,-18,
/*    J(17,2570401)=    */    -1384,-548,-728,-538,-678,-750,-1022,-762,-158,-1322,-1034,-624,-816,-584,-172,-237,
/*    J(2,2702701)=    */    1475,-726,
/*    J(3,2702701)=    */    233,-6,410,530,-26,594,-286,-208,1116,149,-50,-246,636,64,684,288,382,226,
/*    J(5,2702701)=    */    -296,226,-542,-444,1058,-358,30,-576,-92,456,233,48,-344,-20,106,-230,374,-212,830,602,
/*    J(7,2702701)=    */    -1650,-596,-1317,-1176,-1170,-168,
/*    J(11,2702701)=    */    966,984,438,-90,1286,666,1200,1446,744,873,
/*    J(13,2702701)=    */    500,854,518,42,-422,310,230,-82,474,1272,-32,755,
/*    J(2,3088801)=    */    50,764,-252,-196,0,-196,252,764,-50,224,-10,518,1037,-518,-10,-224,
/*    J(1,3088801)=    */    -419,-65,592,109,-793,-101,-645,-263,399,100,120,124,910,-446,353,268,
/*    J(4,3088801)=    */    251,0,0,0,1230,0,0,0,0,0,0,0,1230,0,0,0,
/*    J(3,3088801)=    */    -16,306,396,-904,-212,-374,128,1128,-216,336,498,618,-644,608,26,-110,264,39,
/*    J(5,3088801)=    */    238,390,-438,-154,-1698,182,288,-270,-270,-630,224,388,64,-188,-812,130,694,-296,-183,0,
/*    J(11,3088801)=    */    -736,-884,-906,-270,-572,40,-580,-442,-1077,-1834,
/*    J(13,3088801)=    */    523,1112,318,-466,404,-602,700,96,-302,-288,174,306,
/*    J(2,3141601)=    */    -346,-276,-734,558,0,558,734,-276,346,710,-80,-26,165,26,-80,-710,
/*    J(1,3141601)=    */    -1093,-436,-94,-551,63,-607,473,-20,-487,-57,490,-210,30,-396,-307,-267,
/*    J(4,3141601)=    */    1723,0,0,0,-294,0,0,0,0,0,0,0,-294,0,0,0,
/*    J(3,3141601)=    */    2024,1275,
/*    J(5,3141601)=    */    456,-22,-306,-1234,494,-48,-254,522,-478,658,-64,110,180,-68,1064,230,-598,156,-397,228,
/*    J(7,3141601)=    */    -722,-980,-320,-1238,-1352,-1997,
/*    J(11,3141601)=    */    -638,-410,-844,-356,329,76,-1044,-778,-466,-1524,
/*    J(17,3141601)=    */    -75,652,-514,96,-2,300,818,86,174,-610,2,-74,-16,-882,-68,554,
/*    J(2,3712801)=    */    338,-476,460,232,0,232,-460,-476,-338,-888,14,434,-739,-434,14,888,
/*    J(1,3712801)=    */    41,-775,640,891,-193,-191,635,-1,635,30,-56,-386,230,568,129,694,
/*    J(4,3712801)=    */    443,0,0,0,1326,0,0,0,0,0,0,0,1326,0,0,0,
/*    J(3,3712801)=    */    -2185,-729,
/*    J(5,3712801)=    */    48,1288,406,-787,110,232,1106,438,-969,376,710,878,444,-405,218,74,1524,214,-231,-530,
/*    J(7,3712801)=    */    1134,1570,660,2170,1675,1610,
/*    J(13,3712801)=    */    -1072,128,-452,-600,-448,-404,-344,1160,88,-652,-195,-460,
/*    J(17,3712801)=    */    -236,-728,888,-266,-432,-174,-18,242,-260,-495,124,-506,194,746,212,742,
/*    J(2,5105101)=    */    -765,-2126,
/*    J(3,5105101)=    */    -1651,924,
/*    J(5,5105101)=    */    531,-530,-342,-696,-1284,537,50,-874,-108,-340,17,74,146,328,-280,-261,834,-294,270,-1194,
/*    J(7,5105101)=    */    230,-490,640,-30,-75,-2026,
/*    J(11,5105101)=    */    564,1572,290,1516,965,664,-446,96,452,1564,
/*    J(13,5105101)=    */    1858,50,576,0,-248,-248,290,-238,840,826,787,-230,
/*    J(17,5105101)=    */    -540,200,-421,1052,280,936,820,-680,702,1098,540,-48,714,582,608,208,
/*    J(2,5834401)=    */    202,234,304,-778,0,-778,-304,234,-202,910,-430,-736,-1067,736,-430,-910,
/*    J(1,5834401)=    */    -557,247,-662,-277,299,499,505,-73,-383,1662,742,-206,534,-518,-49,-540,
/*    J(4,5834401)=    */    2363,0,0,0,-354,0,0,0,0,0,0,0,-354,0,0,0,
/*    J(3,5834401)=    */    -640,2031,
/*    J(5,5834401)=    */    690,412,916,-76,-1396,130,618,1354,-1175,-420,-142,6,518,-60,-978,496,130,-166,-296,-1202,
/*    J(11,5834401)=    */    -594,58,-1580,-1684,-1030,-362,-2199,-1162,-1014,-1588,
/*    J(13,5834401)=    */    -198,1834,580,937,304,-66,-246,532,-246,770,738,1534,
/*    J(17,5834401)=    */    255,1453,-589,839,201,171,639,413,-47,67,595,-11,827,1627,1281,387,
/*    J(2,6806801)=    */    0,-38,-34,-456,-2527,456,-34,38,
/*    J(1,6806801)=    */    2280,-151,-342,125,-1121,255,-152,-329,
/*    J(4,6806801)=    */    2499,0,-530,0,0,0,-530,0,
/*    J(5,6806801)=    */    656,-704,-244,52,-656,-76,141,1482,36,-1168,370,-1152,310,-448,-872,-546,-56,-530,152,-588,
/*    J(7,6806801)=    */    -1886,-270,-215,-2090,-2030,-1980,
/*    J(11,6806801)=    */    -138,-968,1730,282,800,1394,592,308,1203,1176,
/*    J(13,6806801)=    */    542,-968,-574,190,700,1338,-32,772,-750,995,610,-536,
/*    J(17,6806801)=    */    952,-456,-296,-320,812,-124,168,1498,-822,-536,-687,-504,-82,-82,-732,20,
/*    J(2,7068601)=    */    1349,-1620,0,-1620,
/*    J(1,7068601)=    */    -1101,0,2420,0,
/*    J(4,7068601)=    */    1349,-1620,0,-1620,
/*    J(3,7068601)=    */    -504,1556,539,1404,266,-126,-1072,328,530,-290,596,435,1340,96,-684,420,-280,-160,
/*    J(5,7068601)=    */    -508,-488,588,660,-245,552,352,1630,-866,-593,-1126,140,1052,-670,-265,-534,-486,1076,-220,-965,
/*    J(7,7068601)=    */    -2879,-968,-1684,-1940,-820,-292,
/*    J(11,7068601)=    */    -1066,-280,-966,1018,-402,50,340,-1742,-1364,-319,
/*    J(17,7068601)=    */    1868,1714,2758,1644,686,1586,934,1296,1561,1616,818,1402,798,496,1358,1870,
/*    J(2,8353801)=    */    -2051,1440,0,1440,
/*    J(1,8353801)=    */    -2485,0,1476,0,
/*    J(4,8353801)=    */    -2051,1440,0,1440,
/*    J(3,8353801)=    */    192,884,-480,-958,870,-452,180,-800,1445,656,378,-1142,-830,-112,-918,666,-1244,-283,
/*    J(5,8353801)=    */    756,-296,872,1082,382,700,394,938,-236,498,398,412,1560,340,1628,1240,-1196,988,556,1263,
/*    J(7,8353801)=    */    -1392,-640,2028,-262,77,-890,
/*    J(13,8353801)=    */    488,2612,2528,1308,2212,1632,1292,1952,820,2086,857,1842,
/*    J(17,8353801)=    */    -760,-262,-916,-364,300,-1158,-568,-892,-939,-2940,-456,-334,-548,-340,-1120,-1114,
/*    J(2,17503201)=    */    0,-1580,1586,-338,312,308,194,-1754,-799,1754,194,-308,312,338,1586,1580,
/*    J(1,17503201)=    */    234,1562,1571,17,270,-161,-1393,-982,695,-2012,1511,71,-492,-1175,-679,796,
/*    J(4,17503201)=    */    3727,0,0,0,1344,0,0,0,0,0,0,0,1344,0,0,0,
/*    J(3,17503201)=    */    566,1794,768,-144,4668,447,
/*    J(5,17503201)=    */    524,1300,972,2268,-2300,1412,1366,-766,-802,-1250,832,1484,357,-288,-1304,146,-114,854,170,-1052,
/*    J(11,17503201)=    */    638,750,3104,-998,-876,876,-311,496,-1660,-106,
/*    J(13,17503201)=    */    2148,-654,1300,2940,3382,1664,1740,2314,2294,605,1946,2706,
/*    J(17,17503201)=    */    222,1116,946,2082,629,170,1050,1596,1390,856,-124,3988,1202,2090,2292,1268,
/*    J(3,22972951)=    */    1170,688,-2552,-382,1125,2212,870,-1192,2554,-46,520,-1494,-860,-169,1206,-912,-1080,-114,
/*    J(5,22972951)=    */    54,48,1226,-820,-468,268,608,-2996,847,-96,-220,-280,-1664,-114,142,452,1770,1016,710,-724,
/*    J(7,22972951)=    */    -1457,3822,2056,412,2318,-222,
/*    J(11,22972951)=    */    1928,2570,-114,2444,16,-1400,-719,2642,810,1942,
/*    J(13,22972951)=    */    2076,1846,1122,-760,2744,278,2742,1504,-372,567,622,-1554,
/*    J(17,22972951)=    */    295,534,-2326,884,332,-1412,-234,-728,2012,-846,-982,-844,642,526,2046,1052,
/*    J(2,52509601)=    */    -1470,710,-352,2860,0,2860,352,710,1470,746,1838,-1274,-4411,1274,1838,-746,
/*    J(1,52509601)=    */    1535,-1072,-1710,-2170,-329,-1478,-345,1090,3365,-373,2334,-277,-2978,3193,961,893,
/*    J(4,52509601)=    */    -2053,0,0,0,-4914,0,0,0,0,0,0,0,-4914,0,0,0,
/*    J(3,52509601)=    */    2612,1782,4376,-1564,200,1412,1440,1464,-968,1064,1406,-1558,96,-74,72,-247,3364,-526,
/*    J(5,52509601)=    */    -711,-1666,-2424,2076,30,-1280,-3436,-2560,974,1458,-224,-2336,-3900,848,-2882,-176,-3096,-2154,4860,-1662,
/*    J(11,52509601)=    */    -1438,-2016,-4992,-2846,-3020,-3690,-5872,-1603,-7640,-3646,
/*    J(13,52509601)=    */    1146,1442,5524,2142,5074,5974,2168,5039,1404,2042,4302,4926,
/*    J(17,52509601)=    */    -3276,-1848,-4932,-5066,-3658,-6302,-3182,-4500,-4152,-2048,-2540,-3906,-480,-2867,-5220,-5694,
/*    J(2,183783601)=    */    -1303,124,8264,-4768,0,-4768,-8264,124,
/*    J(1,183783601)=    */    3703,9434,-2480,1126,1876,7478,-1032,3626,
/*    J(4,183783601)=    */    7687,0,-7896,0,0,0,-7896,0,
/*    J(3,183783601)=    */    -1486,1452,1874,5960,-1926,712,-3798,-4570,2018,-2332,1178,-6988,-435,1626,-1118,-3718,3074,322,
/*    J(5,183783601)=    */    6714,-8030,1772,-3764,-918,6210,-1200,3505,-5504,-1212,9460,-336,3396,-3298,212,8030,-4204,-2168,-2404,-3582,
/*    J(7,183783601)=    */    3573,1611,7809,789,12183,11533,
/*    J(11,183783601)=    */    71,-4001,5887,4957,-4575,5117,3249,8627,3311,2491,
/*    J(13,183783601)=    */    530,9570,-40,1715,3276,2254,4898,6276,-1302,2026,2730,-5882,
/*    J(17,183783601)=    */    -9670,466,-3356,-3778,-5882,-30,-3810,-4962,-2342,-2927,3586,2366,-1782,-3918,-6442,-2162,
/*    J(2,367567201)=    */    -953,-9866,-6352,-580,-664,4116,-1300,5122,0,5122,1300,4116,664,-580,6352,-9866,
/*    J(1,367567201)=    */    5129,35,3194,145,5260,-11731,416,-1007,-5368,-1011,-3958,-8419,1272,4697,3796,-3177,
/*    J(4,367567201)=    */    -4913,0,0,0,13104,0,0,0,0,0,0,0,13104,0,0,0,
/*    J(3,367567201)=    */    -4466,-1810,8050,-1446,-672,-7348,-2710,-3864,7042,-1098,2522,11668,-952,-298,7066,-5412,-1917,4914,
/*    J(5,367567201)=    */    5086,6152,-5704,11483,-1530,5506,6968,414,2784,-850,4846,3070,-5850,7960,8172,7446,-76,-5662,-1120,2364,
/*    J(7,367567201)=    */    -1924,10014,11993,12578,-4496,9648,
/*    J(11,367567201)=    */    -2540,-13404,-9445,-15568,-9820,-9564,-9518,-154,-16862,-5658,
/*    J(13,367567201)=    */    -9300,-826,-4078,-9094,-8242,3798,-3891,-7126,-15834,-6086,-10350,-11626,
/*    J(17,367567201)=    */    6372,8122,15018,8170,8182,4596,5200,1102,2536,7414,10490,-5939,7882,7702,3612,6678,
/*a is 7 t is 1396755360    */
/*    J(5,191)=    */    10,2,-7,4,
/*    J(19,191)=    */    -2,2,6,2,0,0,-2,1,4,-2,-2,4,0,-4,0,-2,2,-8,
/*    J(2,229)=    */    15,-2,
/*    J(3,229)=    */    5,-12,
/*    J(19,229)=    */    -4,0,-8,0,2,-6,-6,-6,-12,-4,-4,0,-8,-4,-2,-5,-6,-4,
/*    J(11,419)=    */    6,2,-8,-6,-8,10,-4,4,6,-3,
/*    J(19,419)=    */    12,12,6,12,10,14,12,10,8,18,8,4,14,16,10,14,2,7,
/*    J(2,457)=    */    13,12,0,12,
/*    J(1,457)=    */    -21,0,4,0,
/*    J(4,457)=    */    13,12,0,12,
/*    J(3,457)=    */    17,24,
/*    J(19,457)=    */    -9,-9,-13,-5,1,-1,-3,-9,-3,-11,-9,-15,-1,-7,-13,-5,-9,-13,
/*    J(3,571)=    */    5,-21,
/*    J(5,571)=    */    -23,-14,-26,-8,
/*    J(19,571)=    */    4,6,8,10,4,-6,6,0,12,4,10,8,3,18,12,2,2,10,
/*    J(17,647)=    */    8,-2,10,2,12,14,14,5,4,4,2,8,4,-8,10,14,
/*    J(19,647)=    */    -4,-4,6,4,-6,6,-4,1,10,8,0,-6,-4,2,-14,2,-2,4,
/*    J(2,761)=    */    -27,4,0,4,
/*    J(1,761)=    */    19,0,-20,0,
/*    J(4,761)=    */    -27,4,0,4,
/*    J(5,761)=    */    -20,12,8,-1,
/*    J(19,761)=    */    -8,-2,-2,-12,0,0,-10,2,2,-2,-8,7,-4,2,-18,-14,-6,-4,
/*    J(3,1483)=    */    38,39,
/*    J(13,1483)=    */    10,-16,-20,-8,4,-6,10,-10,-6,16,-6,5,
/*    J(19,1483)=    */    0,-10,8,4,10,0,12,-12,-14,-10,10,0,-14,2,-12,2,-2,-13,
/*    J(2,1597)=    */    -21,-34,
/*    J(3,1597)=    */    -7,36,
/*    J(7,1597)=    */    5,19,-17,3,-5,-27,
/*    J(19,1597)=    */    -16,-24,-16,6,-14,-6,-18,-19,-14,-10,-6,-6,-20,-8,-28,-16,-26,-26,
/*    J(2,2053)=    */    -17,-42,
/*    J(3,2053)=    */    0,10,4,-16,6,-16,12,1,8,-2,4,6,-18,-18,-30,20,-3,-8,
/*    J(19,2053)=    */    -10,2,10,-4,-14,12,-24,-6,-14,-16,-4,-2,0,-10,-14,-26,0,5,
/*    J(2,2129)=    */    -23,-12,16,-20,0,-20,-16,-12,
/*    J(1,2129)=    */    -17,0,-8,4,-12,4,-24,-32,
/*    J(4,2129)=    */    -9,0,32,0,0,0,32,0,
/*    J(7,2129)=    */    22,-24,4,-14,-24,-21,
/*    J(19,2129)=    */    -20,7,0,16,12,-16,-12,14,-8,-10,-10,-6,-14,4,-6,-4,-8,-16,
/*    J(2,2281)=    */    47,6,0,6,
/*    J(1,2281)=    */    45,0,-16,0,
/*    J(4,2281)=    */    -47,-6,0,-6,
/*    J(3,2281)=    */    -55,-24,
/*    J(5,2281)=    */    -14,39,6,-12,
/*    J(19,2281)=    */    32,16,24,46,26,8,19,12,38,22,20,16,28,16,6,18,14,18,
/*    J(7,2927)=    */    -51,-65,-41,-51,-35,-45,
/*    J(11,2927)=    */    12,-30,-6,28,-4,4,7,18,10,26,
/*    J(19,2927)=    */    -10,-6,-14,12,22,-16,8,-12,6,-10,10,14,-2,-1,4,-24,-20,0,
/*    J(2,3041)=    */    2,-10,-6,24,0,24,6,-10,-2,0,-28,-4,3,4,-28,0,
/*    J(1,3041)=    */    1,-6,-16,-7,-3,-5,-7,-24,3,-15,36,12,-10,8,9,-11,
/*    J(4,3041)=    */    27,0,0,0,34,0,0,0,0,0,0,0,34,0,0,0,
/*    J(5,3041)=    */    -58,-14,4,-13,
/*    J(19,3041)=    */    22,24,12,30,4,4,8,26,34,-8,28,28,34,24,32,10,20,9,
/*    J(2,3877)=    */    31,54,
/*    J(3,3877)=    */    -52,-69,
/*    J(17,3877)=    */    14,-22,-10,-2,-6,-30,-8,-40,-18,-26,-22,2,-26,-10,-36,-33,
/*    J(19,3877)=    */    -8,-38,-12,-12,-24,-18,18,-10,-10,-6,-26,0,-12,24,-10,-12,0,3,
/*    J(3,4447)=    */    -4,58,-8,6,71,2,
/*    J(13,4447)=    */    -18,-20,38,1,24,18,-12,10,-14,-12,-10,20,
/*    J(19,4447)=    */    10,32,30,46,-16,30,36,16,26,24,38,20,30,34,23,16,24,-2,
/*    J(7,4523)=    */    2,20,-18,2,-3,-60,
/*    J(17,4523)=    */    5,38,32,2,-4,24,4,-6,-4,10,-20,30,-6,2,8,-14,
/*    J(19,4523)=    */    -20,40,14,20,-6,16,14,12,22,22,2,16,46,22,11,28,4,2,
/*    J(2,4561)=    */    0,-24,18,-30,-31,30,18,24,
/*    J(1,4561)=    */    24,-21,-6,-27,-41,27,-12,-15,
/*    J(4,4561)=    */    67,0,-6,0,0,0,-6,0,
/*    J(3,4561)=    */    56,75,
/*    J(5,4561)=    */    48,-27,-8,-24,
/*    J(19,4561)=    */    -10,20,18,22,33,30,2,0,-2,16,20,0,-10,14,10,22,-24,28,
/*    J(2,4789)=    */    55,-42,
/*    J(3,4789)=    */    -52,-11,52,-36,17,20,
/*    J(7,4789)=    */    -36,-28,-32,40,-12,-17,
/*    J(19,4789)=    */    6,6,-6,-30,12,16,0,-6,-10,-24,4,-36,-6,14,6,6,-16,25,
/*    J(3,6271)=    */    59,90,
/*    J(5,6271)=    */    -6,35,-60,-30,
/*    J(11,6271)=    */    -29,21,-17,5,-13,-63,-37,1,-17,-39,
/*    J(19,6271)=    */    2,-22,18,0,34,22,16,-18,42,23,-8,-8,-10,8,0,14,20,-20,
/*    J(2,6689)=    */    41,-10,8,4,24,8,-28,-30,0,-30,28,8,-24,4,-8,-10,
/*    J(1,6689)=    */    19,9,2,-11,-6,-9,24,5,44,-39,-14,23,-10,-19,-20,-19,
/*    J(4,6689)=    */    -81,0,0,0,-8,0,0,0,0,0,0,0,-8,0,0,0,
/*    J(11,6689)=    */    52,24,42,26,15,22,10,16,64,80,
/*    J(19,6689)=    */    -28,-10,-44,-38,10,-58,-58,-40,-34,-28,-20,-34,-46,-62,-25,-38,-22,-34,
/*    J(2,6841)=    */    71,-30,0,-30,
/*    J(1,6841)=    */    21,0,-80,0,
/*    J(4,6841)=    */    -71,30,0,30,
/*    J(3,6841)=    */    -76,13,34,-66,32,26,
/*    J(5,6841)=    */    24,-60,-25,-60,
/*    J(19,6841)=    */    48,-38,-28,-20,-12,-3,4,-20,4,6,8,8,16,-24,4,4,4,0,
/*    J(2,6917)=    */    79,-26,
/*    J(7,6917)=    */    64,60,16,76,12,9,
/*    J(13,6917)=    */    58,38,30,20,26,6,12,2,46,26,80,19,
/*    J(19,6917)=    */    18,32,26,-24,-6,4,-22,-14,2,-10,20,-22,-20,22,0,10,-32,15,
/*    J(3,7411)=    */    -94,-75,
/*    J(5,7411)=    */    -78,-84,-86,-93,
/*    J(13,7411)=    */    -36,24,-8,-16,18,-24,-22,-18,18,-22,-50,-47,
/*    J(19,7411)=    */    -16,12,-26,-8,-8,2,-4,-14,-40,-30,-32,-14,-6,-16,-46,2,-30,-69,
/*    J(2,7753)=    */    -65,-42,0,-42,
/*    J(1,7753)=    */    -3,0,-88,0,
/*    J(4,7753)=    */    65,42,0,42,
/*    J(3,7753)=    */    -19,-96,
/*    J(17,7753)=    */    -2,30,36,10,-12,0,18,30,-4,24,30,-48,10,-14,4,-11,
/*    J(19,7753)=    */    22,-26,-19,-18,-10,-16,38,-4,20,-4,-8,38,10,4,-28,12,28,-2,
/*    J(2,8209)=    */    49,-26,-28,-38,0,-38,28,-26,
/*    J(1,8209)=    */    -29,32,8,38,60,-34,4,8,
/*    J(4,8209)=    */    -89,0,12,0,0,0,12,0,
/*    J(3,8209)=    */    16,46,6,16,24,-4,-2,-10,-28,54,20,-18,22,-5,2,4,30,-42,
/*    J(19,8209)=    */    -64,-12,-46,-38,-38,-8,0,-60,-64,-28,-52,-34,-50,-58,-28,-24,-24,-19,
/*    J(3,8779)=    */    -7,90,
/*    J(7,8779)=    */    10,78,-12,-3,18,-36,
/*    J(11,8779)=    */    2,38,34,-40,52,24,22,-14,-19,-12,
/*    J(19,8779)=    */    -14,18,-6,6,-22,-20,13,-14,-8,-16,12,-12,-68,-26,2,-24,34,-8,
/*    J(2,8893)=    */    -53,-78,
/*    J(3,8893)=    */    20,66,48,-21,84,18,
/*    J(13,8893)=    */    -8,4,66,-26,22,10,-15,-38,-22,14,14,4,
/*    J(19,8893)=    */    -4,-4,-10,4,-32,2,-43,-22,8,-8,-10,-8,-56,-36,-44,12,-40,-52,
/*    J(2,10337)=    */    -31,26,38,30,-16,16,-30,16,0,16,30,16,16,30,-38,26,
/*    J(1,10337)=    */    33,19,10,36,28,-12,16,-3,40,-1,-54,14,8,26,-2,29,
/*    J(4,10337)=    */    -33,0,0,0,-68,0,0,0,0,0,0,0,-68,0,0,0,
/*    J(17,10337)=    */    -8,-16,-16,-6,-4,-70,6,-44,-58,-36,0,8,-4,-2,-55,-2,
/*    J(19,10337)=    */    34,6,4,-7,24,58,10,68,20,48,20,-14,14,66,22,12,20,12,
/*    J(3,11287)=    */    9,24,-20,-24,-2,-10,52,-18,8,-5,0,-10,-76,-36,-4,52,-46,12,
/*    J(11,11287)=    */    -8,-58,-72,-86,-47,-64,-74,-76,-30,-102,
/*    J(19,11287)=    */    16,-13,-22,-34,-4,-58,-30,-20,6,-30,22,-20,-68,0,-14,24,0,-22,
/*    J(3,11971)=    */    110,24,42,51,42,36,
/*    J(5,11971)=    */    48,33,122,6,
/*    J(7,11971)=    */    58,-22,74,69,56,-12,
/*    J(19,11971)=    */    -28,-30,30,10,44,8,-46,12,-46,-4,-6,-10,-44,4,0,-22,-30,5,
/*    J(2,12541)=    */    -21,-110,
/*    J(3,12541)=    */    -100,21,
/*    J(5,12541)=    */    3,-12,-108,16,
/*    J(11,12541)=    */    -16,29,-32,-60,50,10,34,22,18,-34,
/*    J(19,12541)=    */    36,-6,6,16,36,-36,-6,12,-14,-30,-2,9,18,-34,2,-52,-30,36,
/*    J(3,13339)=    */    -8,-6,2,22,-2,-44,46,9,-30,-12,22,10,-8,-16,-104,22,-25,-2,
/*    J(13,13339)=    */    -68,-22,-122,-70,-34,-76,-84,-72,-52,-82,-70,-29,
/*    J(19,13339)=    */    -2,-6,10,10,4,10,-28,36,54,-14,-48,-14,-20,-40,24,-32,30,-13,
/*    J(3,13567)=    */    77,-57,
/*    J(7,13567)=    */    -12,-78,-63,-24,-110,6,
/*    J(17,13567)=    */    -28,-18,18,2,-34,-4,-12,-4,-6,7,-64,30,-6,-22,-60,-72,
/*    J(19,13567)=    */    -64,26,-48,-8,-10,4,42,26,8,-10,-10,-4,0,4,-11,-46,-6,30,
/*    J(2,13681)=    */    0,-4,-82,10,-1,-10,-82,4,
/*    J(1,13681)=    */    -8,-59,0,91,-7,29,2,31,
/*    J(4,13681)=    */    -109,0,30,0,0,0,30,0,
/*    J(3,13681)=    */    62,3,-6,126,27,30,
/*    J(5,13681)=    */    -34,74,-4,-67,
/*    J(19,13681)=    */    -12,2,32,8,82,-12,-32,40,24,30,8,52,30,52,24,44,25,20,
/*    J(2,14821)=    */    111,-50,
/*    J(3,14821)=    */    140,81,
/*    J(5,14821)=    */    144,60,60,25,
/*    J(13,14821)=    */    26,-18,-8,-28,32,-32,-40,-40,-18,2,76,-31,
/*    J(19,14821)=    */    0,52,-38,20,-4,-2,6,14,14,10,14,56,-12,18,44,4,-56,-27,
/*    J(2,16417)=    */    49,-22,22,-14,48,-32,-46,20,0,20,46,-32,-48,-14,-22,-22,
/*    J(1,16417)=    */    -23,23,18,2,44,-50,68,53,8,-29,6,4,-8,20,30,29,
/*    J(4,16417)=    */    127,0,0,0,12,0,0,0,0,0,0,0,12,0,0,0,
/*    J(3,16417)=    */    42,-38,24,-18,6,8,-64,-10,16,22,-70,25,-40,-16,40,30,20,16,
/*    J(19,16417)=    */    14,20,-10,48,86,56,78,-10,66,38,60,18,16,62,40,2,44,55,
/*    J(5,17291)=    */    7,-34,-106,52,
/*    J(7,17291)=    */    128,62,44,-30,50,39,
/*    J(13,17291)=    */    -2,-54,-24,34,68,12,-6,16,32,-45,50,-30,
/*    J(19,17291)=    */    6,-66,-28,-10,18,-16,64,5,-18,22,-12,-44,2,-16,12,-62,-20,10,
/*    J(3,17443)=    */    36,-2,-38,-54,-34,24,10,55,6,58,44,-42,-28,30,-30,34,33,26,
/*    J(17,17443)=    */    8,20,28,-18,34,8,38,88,42,28,80,-26,18,26,82,19,
/*    J(19,17443)=    */    40,28,56,74,38,-4,62,28,6,40,102,48,80,34,54,106,30,51,
/*    J(2,18089)=    */    -51,-88,0,-88,
/*    J(1,18089)=    */    -133,0,-20,0,
/*    J(4,18089)=    */    -51,-88,0,-88,
/*    J(7,18089)=    */    103,159,49,85,39,75,
/*    J(17,18089)=    */    56,18,14,56,28,-28,32,26,6,-2,27,30,78,102,-12,44,
/*    J(19,18089)=    */    46,14,50,8,-4,-12,-40,16,-4,6,10,86,-40,-22,26,23,38,26,
/*    J(2,19381)=    */    135,34,
/*    J(3,19381)=    */    59,159,
/*    J(5,19381)=    */    15,-48,118,54,
/*    J(17,19381)=    */    -26,-32,-2,-58,-42,-62,-28,-50,-40,-16,-38,-64,-123,32,-14,-16,
/*    J(19,19381)=    */    60,0,28,70,20,52,32,70,-14,-4,33,48,-30,-2,24,36,68,78,
/*    J(2,20521)=    */    143,-6,0,-6,
/*    J(1,20521)=    */    45,0,136,0,
/*    J(4,20521)=    */    -143,6,0,6,
/*    J(3,20521)=    */    14,68,-6,0,56,-6,20,-2,-71,26,20,-22,-64,72,30,24,-46,-48,
/*    J(5,20521)=    */    -6,-90,-60,-155,
/*    J(19,20521)=    */    42,50,55,66,-22,-8,52,44,-30,32,-4,46,-22,20,20,-26,20,-32,
/*    J(2,20749)=    */    -93,110,
/*    J(3,20749)=    */    20,153,
/*    J(7,20749)=    */    -80,-74,-130,-84,-144,-161,
/*    J(13,20749)=    */    -32,-16,84,-18,56,60,54,16,0,92,36,31,
/*    J(19,20749)=    */    54,76,-8,28,-60,26,36,-28,0,-16,10,-16,20,0,-32,-34,28,-9,
/*    J(3,21319)=    */    2,147,
/*    J(11,21319)=    */    12,-16,-32,-2,42,34,-94,10,-79,-52,
/*    J(17,21319)=    */    -118,-68,-46,-48,-50,-52,-70,-78,-164,-82,-45,-68,-90,-60,-40,-78,
/*    J(19,21319)=    */    -44,-22,-40,-98,-78,-24,-50,-22,-60,-70,50,-27,-8,-64,-10,-48,-56,-14,
/*    J(2,21737)=    */    -147,-8,0,-8,
/*    J(1,21737)=    */    91,0,-116,0,
/*    J(4,21737)=    */    -147,-8,0,-8,
/*    J(11,21737)=    */    6,8,92,-58,18,32,16,66,92,57,
/*    J(13,21737)=    */    94,10,-20,34,76,14,100,13,32,18,-28,-6,
/*    J(19,21737)=    */    50,10,92,8,11,32,-28,34,70,28,-10,-14,50,20,-36,10,-30,6,
/*    J(2,22573)=    */    -93,118,
/*    J(3,22573)=    */    -12,32,-2,66,-84,-25,-48,-6,-4,-50,34,48,14,-28,19,-16,-2,66,
/*    J(11,22573)=    */    -16,26,-51,-54,40,-68,-40,-104,-4,28,
/*    J(19,22573)=    */    -9,-24,-58,-4,-42,-10,20,-46,18,-70,-58,-74,-14,-16,-86,-70,28,-18,
/*    J(2,25841)=    */    0,78,14,-44,97,44,14,-78,
/*    J(1,25841)=    */    8,-59,-38,-67,39,-9,-120,19,
/*    J(4,25841)=    */    147,0,46,0,0,0,46,0,
/*    J(5,25841)=    */    -89,-1,-49,103,
/*    J(17,25841)=    */    80,66,26,12,72,74,-6,56,42,58,65,90,130,88,-14,78,
/*    J(19,25841)=    */    34,-6,72,-6,42,-20,4,34,-1,94,56,-32,-26,50,-8,-18,-22,-20,
/*    J(2,27361)=    */    -22,8,54,-18,0,-18,-54,8,22,-86,-40,-26,-21,26,-40,86,
/*    J(1,27361)=    */    -57,-12,-36,49,-23,-19,-19,-40,-39,-41,44,-26,82,56,-39,-15,
/*    J(4,27361)=    */    -37,0,0,0,114,0,0,0,0,0,0,0,114,0,0,0,
/*    J(3,27361)=    */    35,86,-70,120,76,-110,
/*    J(5,27361)=    */    -7,-12,-18,-174,
/*    J(19,27361)=    */    -46,-40,16,-36,-4,22,-36,14,18,-18,20,-32,-25,-48,-36,-126,30,22,
/*    J(2,28729)=    */    -89,-102,0,-102,
/*    J(1,28729)=    */    -75,0,-152,0,
/*    J(4,28729)=    */    89,102,0,102,
/*    J(3,28729)=    */    64,68,-82,50,-40,-34,18,24,-92,-18,18,-66,36,-19,-34,56,18,-58,
/*    J(7,28729)=    */    -182,-40,-73,-48,-34,12,
/*    J(19,28729)=    */    138,52,96,54,64,60,106,36,22,22,-22,57,110,54,54,56,94,48,
/*    J(2,29641)=    */    29,120,0,120,
/*    J(1,29641)=    */    171,0,-20,0,
/*    J(4,29641)=    */    29,120,0,120,
/*    J(3,29641)=    */    131,195,
/*    J(5,29641)=    */    -45,67,3,-141,
/*    J(13,29641)=    */    52,-2,64,-16,42,-32,-4,-30,-46,33,126,-6,
/*    J(19,29641)=    */    58,38,6,-27,-26,110,70,20,38,42,20,72,46,46,64,72,26,122,
/*    J(2,30097)=    */    0,-22,-2,-40,161,40,-2,22,
/*    J(1,30097)=    */    16,-31,150,29,-25,-1,-68,-17,
/*    J(4,30097)=    */    163,0,-42,0,0,0,-42,0,
/*    J(3,30097)=    */    8,-120,-120,72,-75,0,
/*    J(11,30097)=    */    -34,-114,-87,-2,-74,-28,-130,22,18,-34,
/*    J(19,30097)=    */    -56,-22,-54,-50,24,-70,-46,-80,14,-34,-6,-48,-80,-132,-68,-11,-104,-52,
/*    J(3,31123)=    */    -172,76,25,-60,32,71,
/*    J(7,31123)=    */    137,150,14,128,22,24,
/*    J(13,31123)=    */    -24,-2,88,46,10,-10,82,50,87,112,-16,96,
/*    J(19,31123)=    */    78,-40,18,64,38,68,-2,44,30,44,10,34,64,88,9,58,126,104,
/*    J(5,35531)=    */    193,158,212,166,
/*    J(11,35531)=    */    -10,4,85,-18,156,86,-16,2,0,62,
/*    J(17,35531)=    */    -34,20,66,-32,-114,-8,-70,8,-114,-46,-18,-54,-19,-40,-72,-52,
/*    J(19,35531)=    */    34,140,38,46,4,24,-18,36,-18,58,74,-12,0,27,-50,70,56,60,
/*    J(2,35569)=    */    0,-66,-18,-12,161,12,-18,66,
/*    J(1,35569)=    */    24,-27,6,-27,15,-33,172,51,
/*    J(4,35569)=    */    179,0,-42,0,0,0,-42,0,
/*    J(3,35569)=    */    -88,86,-22,-99,-62,-110,
/*    J(13,35569)=    */    58,32,-1,-24,-10,-88,32,32,84,-52,-8,100,
/*    J(19,35569)=    */    0,32,4,52,80,92,12,24,52,76,16,132,16,-2,114,53,70,-26,
/*    J(3,35911)=    */    -60,-18,55,-16,40,-64,90,-34,-60,-64,-46,42,-14,-68,-106,60,-6,-14,
/*    J(5,35911)=    */    -42,-32,-78,141,
/*    J(7,35911)=    */    -114,12,14,-153,-108,-114,
/*    J(19,35911)=    */    38,76,34,48,72,76,60,60,102,35,60,38,78,-54,22,-18,84,138,
/*    J(7,38039)=    */    138,159,224,50,144,110,
/*    J(11,38039)=    */    37,-70,48,-46,4,-108,-114,14,48,-34,
/*    J(13,38039)=    */    -74,-140,-122,-62,-82,51,-80,-68,-60,-10,-16,-118,
/*    J(19,38039)=    */    31,-67,-53,43,-119,-25,-59,-83,-95,-45,-65,-37,-23,19,-23,-59,-105,-15,
/*    J(2,39521)=    */    0,30,-18,-2,16,84,-70,76,33,-76,-70,-84,16,2,-18,-30,
/*    J(1,39521)=    */    -34,-29,-21,-28,-10,-6,-101,-7,-81,61,85,12,-24,18,51,-69,
/*    J(4,39521)=    */    -177,0,0,0,-64,0,0,0,0,0,0,0,-64,0,0,0,
/*    J(5,39521)=    */    55,-103,53,139,
/*    J(13,39521)=    */    144,230,138,154,136,189,168,198,174,182,116,172,
/*    J(19,39521)=    */    164,58,120,34,22,116,66,32,68,22,40,62,94,32,-14,52,16,3,
/*    J(3,40699)=    */    -46,47,-40,156,73,-62,
/*    J(7,40699)=    */    60,108,90,-114,54,-17,
/*    J(17,40699)=    */    0,52,36,-26,60,-68,74,64,-30,-46,8,58,14,102,70,73,
/*    J(19,40699)=    */    32,4,12,-22,14,68,-104,-10,-84,-18,2,-36,-36,42,-42,-98,-50,21,
/*    J(3,43891)=    */    -1,-210,
/*    J(5,43891)=    */    43,-182,-8,46,
/*    J(7,43891)=    */    30,6,135,12,148,186,
/*    J(11,43891)=    */    -14,-40,-66,-8,16,-7,-98,14,-192,-90,
/*    J(19,43891)=    */    -102,36,0,64,-56,-24,-42,-66,-26,26,-62,-88,-34,-24,-28,-37,-138,-46,
/*    J(2,46817)=    */    0,-52,30,-30,40,-28,46,-78,-129,78,46,28,40,30,30,52,
/*    J(1,46817)=    */    -6,-2,55,-89,0,21,-21,-26,-163,-20,3,-3,38,39,-55,36,
/*    J(4,46817)=    */    -177,0,0,0,88,0,0,0,0,0,0,0,88,0,0,0,
/*    J(7,46817)=    */    -220,-102,-166,-109,-154,-6,
/*    J(11,46817)=    */    51,83,3,121,85,25,133,95,219,31,
/*    J(19,46817)=    */    64,-30,-54,12,52,-12,-79,-12,-4,-4,38,-66,-64,-24,-144,20,-40,4,
/*    J(2,47881)=    */    127,-126,0,-126,
/*    J(1,47881)=    */    -35,0,-216,0,
/*    J(4,47881)=    */    -127,126,0,126,
/*    J(3,47881)=    */    86,-194,-104,60,-88,23,
/*    J(5,47881)=    */    159,-75,-15,-85,
/*    J(7,47881)=    */    -78,0,-180,-146,48,-51,
/*    J(19,47881)=    */    14,-42,28,28,-46,-26,-4,-44,14,-82,-8,32,-46,-82,-56,-60,-79,116,
/*    J(3,48907)=    */    122,45,-90,-42,99,-162,
/*    J(11,48907)=    */    136,78,56,60,66,-122,56,62,106,73,
/*    J(13,48907)=    */    -92,-8,-60,-30,-30,-56,68,82,-90,-102,-10,-115,
/*    J(19,48907)=    */    -40,60,58,14,-22,-46,-8,44,-58,-50,-36,28,48,-44,146,10,6,3,
/*    J(3,51871)=    */    -259,-90,
/*    J(5,51871)=    */    -167,3,-63,111,
/*    J(7,51871)=    */    -23,-65,65,-201,-123,-53,
/*    J(13,51871)=    */    46,180,7,34,68,62,66,176,64,186,102,48,
/*    J(19,51871)=    */    95,-5,107,147,45,17,97,105,95,9,131,59,83,103,157,117,157,133,
/*    J(2,53353)=    */    109,144,0,144,
/*    J(1,53353)=    */    -37,0,228,0,
/*    J(4,53353)=    */    109,144,0,144,
/*    J(3,53353)=    */    -12,36,48,44,-68,-12,-2,70,36,-2,-72,63,120,-90,100,46,106,20,
/*    J(13,53353)=    */    -2,20,2,-34,56,-194,-40,34,52,15,0,38,
/*    J(19,53353)=    */    114,-2,72,38,10,32,70,46,44,88,-120,-46,38,57,30,18,64,92,
/*    J(3,56431)=    */    -138,44,86,-16,66,28,-48,30,56,-62,54,-19,66,0,88,-22,28,124,
/*    J(5,56431)=    */    143,-142,-58,-64,
/*    J(11,56431)=    */    84,32,124,-30,2,50,-86,98,145,108,
/*    J(19,56431)=    */    -82,-50,32,-58,32,-12,-70,-18,-26,34,-52,104,77,18,-114,8,-16,-36,
/*    J(2,57457)=    */    -193,62,28,74,0,74,-28,62,
/*    J(1,57457)=    */    -31,-8,-38,-58,-208,86,30,8,
/*    J(4,57457)=    */    -217,0,72,0,0,0,72,0,
/*    J(3,57457)=    */    88,-36,-95,-110,132,-32,68,76,-8,108,34,-56,10,56,-4,6,38,-36,
/*    J(7,57457)=    */    147,184,296,140,156,224,
/*    J(19,57457)=    */    -142,-105,-174,-52,-134,-116,-88,-142,-162,-92,-38,-122,-4,-34,-78,-92,-188,-138,
/*    J(7,58787)=    */    130,10,24,-56,150,189,
/*    J(13,58787)=    */    60,-18,90,34,88,-142,-32,56,-36,-6,-46,-75,
/*    J(17,58787)=    */    120,76,72,100,154,16,106,76,102,60,30,82,130,26,98,-93,
/*    J(19,58787)=    */    -108,16,-72,64,16,-42,-60,-92,6,66,-16,-42,94,38,-22,42,10,-51,
/*    J(2,59281)=    */    -217,-64,8,-44,0,-44,-8,-64,
/*    J(1,59281)=    */    101,22,-10,-34,72,62,-194,-26,
/*    J(4,59281)=    */    103,0,156,0,0,0,156,0,
/*    J(3,59281)=    */    221,261,
/*    J(5,59281)=    */    -48,-204,89,12,
/*    J(13,59281)=    */    8,44,178,-9,50,-56,-92,80,-2,-10,-6,74,
/*    J(19,59281)=    */    -36,-58,-52,-54,70,2,-98,-72,-58,-124,-64,22,-124,-10,-109,-50,-152,-98,
/*    J(2,63841)=    */    79,50,62,-4,0,82,70,-104,0,-104,-70,82,0,-4,-62,50,
/*    J(1,63841)=    */    -9,74,74,11,138,13,-116,-26,32,32,52,-73,78,-35,-4,-44,
/*    J(4,63841)=    */    223,0,0,0,84,0,0,0,0,0,0,0,84,0,0,0,
/*    J(3,63841)=    */    -211,69,
/*    J(5,63841)=    */    -31,80,-220,-100,
/*    J(7,63841)=    */    73,-69,213,113,7,-23,
/*    J(19,63841)=    */    -92,32,-66,-144,-52,-32,-166,-138,-86,-62,-24,-44,-152,-94,-100,-100,36,-85,
/*    J(2,71821)=    */    -189,190,
/*    J(3,71821)=    */    -4,-168,28,60,52,12,-48,-113,76,8,-162,-24,116,-26,36,20,3,112,
/*    J(5,71821)=    */    -196,-120,-300,-45,
/*    J(7,71821)=    */    232,214,182,12,48,193,
/*    J(19,71821)=    */    -42,-90,-84,46,-54,4,0,156,-52,-52,-60,-98,30,-12,-82,-2,-84,-57,
/*    J(2,72353)=    */    110,-36,90,22,0,22,-90,-36,-110,-34,8,106,-59,-106,8,34,
/*    J(1,72353)=    */    -31,-110,28,-55,-43,-95,15,-54,91,159,-74,0,-24,38,17,-11,
/*    J(4,72353)=    */    -261,0,0,0,-46,0,0,0,0,0,0,0,-46,0,0,0,
/*    J(7,72353)=    */    -298,-174,-214,-118,-196,-51,
/*    J(17,72353)=    */    22,146,24,-50,80,147,144,138,80,26,50,96,-28,104,-12,120,
/*    J(19,72353)=    */    40,86,-6,46,48,-64,154,74,14,-132,-4,-14,-2,8,51,64,2,90,
/*    J(3,75583)=    */    -184,-42,228,-114,69,78,
/*    J(13,75583)=    */    192,30,72,4,-83,-50,146,12,10,-10,84,86,
/*    J(17,75583)=    */    -124,-122,-60,-58,-74,-158,17,60,-60,-28,-54,-132,82,-38,16,-50,
/*    J(19,75583)=    */    4,76,-14,4,-94,10,-41,-44,-12,178,68,-34,34,34,-8,118,-44,30,
/*    J(2,77521)=    */    0,-94,82,-136,-97,136,82,94,
/*    J(1,77521)=    */    -80,-43,-96,29,-31,167,142,-101,
/*    J(4,77521)=    */    227,0,114,0,0,0,114,0,
/*    J(3,77521)=    */    -145,-321,
/*    J(5,77521)=    */    154,55,310,220,
/*    J(17,77521)=    */    -160,-188,-226,-173,-122,-144,-140,-272,-102,-162,-258,-134,-98,-210,-78,-118,
/*    J(19,77521)=    */    -124,40,-18,12,-74,-95,-54,48,-14,-112,76,-44,48,100,28,-52,-68,-40,
/*    J(3,87211)=    */    38,-44,-26,94,151,88,94,-68,40,-54,8,20,98,-10,110,152,-136,38,
/*    J(5,87211)=    */    -103,66,-6,252,
/*    J(17,87211)=    */    -93,-142,-222,-146,-42,-188,-144,-116,36,-126,18,-114,-166,-52,-100,-104,
/*    J(19,87211)=    */    -7,6,18,164,18,10,-32,102,-30,48,-2,12,-8,54,184,28,98,172,
/*    J(2,90289)=    */    0,68,98,-134,161,134,98,-68,
/*    J(1,90289)=    */    -128,-187,10,31,-129,73,-124,23,
/*    J(4,90289)=    */    -109,0,-198,0,0,0,-198,0,
/*    J(3,90289)=    */    56,24,30,90,22,88,32,98,32,59,12,118,-4,-152,38,114,2,132,
/*    J(11,90289)=    */    147,156,88,170,24,206,94,208,302,232,
/*    J(19,90289)=    */    172,18,32,76,62,112,-24,68,-20,48,140,62,12,-16,-138,76,16,-13,
/*    J(2,97813)=    */    -297,-98,
/*    J(3,97813)=    */    -244,-45,-90,54,87,-96,
/*    J(11,97813)=    */    80,-124,-20,-202,-26,24,-104,-164,56,39,
/*    J(13,97813)=    */    34,-150,-18,28,-168,-40,-52,170,18,34,-36,-81,
/*    J(19,97813)=    */    -78,-4,96,-58,-82,0,-60,-158,-98,-96,-32,38,-58,-136,-24,-190,48,-21,
/*    J(2,105337)=    */    37,-228,0,-228,
/*    J(1,105337)=    */    19,0,-324,0,
/*    J(4,105337)=    */    37,-228,0,-228,
/*    J(3,105337)=    */    -214,-133,-232,-90,-32,-242,
/*    J(7,105337)=    */    12,-93,120,96,126,-206,
/*    J(11,105337)=    */    -37,161,-49,45,55,-9,-27,115,-91,-197,
/*    J(19,105337)=    */    128,-42,-96,-32,8,-8,-54,-162,8,-97,64,-110,-70,-76,-192,-44,0,-62,
/*    J(3,106591)=    */    221,375,
/*    J(5,106591)=    */    182,116,374,47,
/*    J(11,106591)=    */    -141,-50,-136,126,-32,100,98,-98,-88,66,
/*    J(17,106591)=    */    -126,-54,-138,1,124,16,80,-146,48,18,-98,-98,28,-84,-68,-14,
/*    J(19,106591)=    */    163,242,102,120,196,164,310,82,86,110,200,84,216,188,74,118,46,120,
/*    J(2,108529)=    */    161,-134,-92,-122,0,-122,92,-134,
/*    J(1,108529)=    */    -13,-8,-36,-158,-236,154,-40,32,
/*    J(4,108529)=    */    -329,0,-12,0,0,0,-12,0,
/*    J(3,108529)=    */    -280,-363,
/*    J(7,108529)=    */    -102,190,152,-124,27,-88,
/*    J(17,108529)=    */    144,174,107,190,108,-8,48,40,-14,108,190,74,-52,118,52,216,
/*    J(19,108529)=    */    -184,-236,-212,-66,-42,-98,6,-162,-26,-118,-161,-18,-50,-126,6,-138,-130,-146,
/*    J(2,114913)=    */    0,84,18,-22,-40,64,34,-74,273,74,34,-64,-40,22,18,-84,
/*    J(1,114913)=    */    18,48,39,87,6,-43,295,16,-1,-74,-33,-7,-4,-13,85,-2,
/*    J(4,114913)=    */    271,0,0,0,-144,0,0,0,0,0,0,0,-144,0,0,0,
/*    J(3,114913)=    */    68,-86,88,-110,-88,160,74,-2,-50,74,60,-11,-38,-102,122,-42,-34,-204,
/*    J(7,114913)=    */    -352,-100,-182,-97,-122,40,
/*    J(19,114913)=    */    -28,-59,8,-80,18,-8,-56,-28,10,48,-42,-110,136,70,-8,-108,-24,-234,
/*    J(2,117041)=    */    7,88,-64,-216,0,-216,64,88,
/*    J(1,117041)=    */    -51,144,-78,200,160,40,-142,-16,
/*    J(4,117041)=    */    279,0,140,0,0,0,140,0,
/*    J(5,117041)=    */    -176,215,80,160,
/*    J(7,117041)=    */    -220,-228,-100,-412,-264,-219,
/*    J(11,117041)=    */    100,98,-22,62,131,186,106,-154,112,-92,
/*    J(19,117041)=    */    24,12,-3,-140,-56,-12,24,-36,-144,52,-198,0,0,-146,-180,-32,-4,40,
/*    J(2,124489)=    */    -161,-222,0,-222,
/*    J(1,124489)=    */    285,0,208,0,
/*    J(4,124489)=    */    161,222,0,222,
/*    J(3,124489)=    */    -70,-217,20,0,175,-50,
/*    J(7,124489)=    */    130,3,-66,116,-260,-92,
/*    J(13,124489)=    */    256,34,-74,48,72,152,-47,40,8,100,178,194,
/*    J(19,124489)=    */    -18,-2,-120,-48,-232,-88,-64,32,-140,-128,124,-32,10,2,-84,-30,-85,-162,
/*    J(3,131671)=    */    206,-98,-140,-51,-136,146,
/*    J(5,131671)=    */    -233,-134,-386,-368,
/*    J(7,131671)=    */    60,213,-150,18,-22,-204,
/*    J(11,131671)=    */    250,68,-46,102,125,12,-80,-2,64,254,
/*    J(19,131671)=    */    -232,-24,-16,43,-54,54,-94,38,-142,-198,-20,-78,-118,-120,-84,10,-52,60,
/*    J(2,134369)=    */    -114,38,46,-164,0,-164,-46,38,114,-24,100,112,-35,-112,100,24,
/*    J(1,134369)=    */    -43,10,-122,-215,195,3,53,-16,19,-51,-62,4,2,120,69,65,
/*    J(4,134369)=    */    -261,0,0,0,-182,0,0,0,0,0,0,0,-182,0,0,0,
/*    J(13,134369)=    */    113,68,150,280,182,2,254,266,162,286,172,274,
/*    J(17,134369)=    */    -56,54,133,-10,-12,208,206,54,168,-132,74,26,72,48,26,58,
/*    J(19,134369)=    */    -130,-214,-78,-76,-99,0,-122,46,-194,-92,-110,22,60,-110,-90,64,46,-102,
/*    J(2,135661)=    */    -205,306,
/*    J(3,135661)=    */    -349,36,
/*    J(5,135661)=    */    50,402,168,279,
/*    J(7,135661)=    */    254,62,242,230,392,65,
/*    J(17,135661)=    */    -132,-38,-146,-44,-38,-48,-122,-107,30,-80,-6,-200,-30,-242,-292,-36,
/*    J(19,135661)=    */    82,30,100,-142,-44,-144,-52,72,-104,28,-90,-104,22,90,-140,-72,35,52,
/*    J(2,139537)=    */    -17,82,-188,-166,0,-166,188,82,
/*    J(1,139537)=    */    -79,166,-34,80,96,-256,-134,-74,
/*    J(4,139537)=    */    215,0,-216,0,0,0,-216,0,
/*    J(3,139537)=    */    -166,-2,30,-24,-234,-32,78,-164,154,-160,32,22,-40,-226,-61,-28,-36,148,
/*    J(17,139537)=    */    20,-78,-8,-162,14,-32,106,-18,79,50,42,156,-58,-190,96,50,
/*    J(19,139537)=    */    -56,-166,-72,-126,-142,-130,58,-146,-10,90,-138,44,-18,36,-136,-160,-105,36,
/*    J(2,140449)=    */    0,-68,8,-44,48,56,184,152,-49,-152,184,-56,48,44,8,68,
/*    J(1,140449)=    */    60,-142,17,-28,-18,64,-53,-194,113,-22,37,56,204,100,-37,-2,
/*    J(4,140449)=    */    191,0,0,0,228,0,0,0,0,0,0,0,228,0,0,0,
/*    J(3,140449)=    */    -400,-57,
/*    J(7,140449)=    */    236,-170,-59,66,184,120,
/*    J(11,140449)=    */    -282,-52,-252,-202,-20,-118,-238,-99,46,-38,
/*    J(19,140449)=    */    -136,-52,-148,70,-2,-148,-106,36,-16,-102,136,-38,42,-92,-200,-112,-72,-87,
/*    J(3,146719)=    */    48,-18,-50,10,86,-168,-106,179,-96,176,-40,-18,32,186,4,76,129,-122,
/*    J(11,146719)=    */    65,79,107,-245,-107,-37,-43,-111,133,-117,
/*    J(13,146719)=    */    -82,154,-64,106,38,224,180,154,56,-36,30,227,
/*    J(19,146719)=    */    52,104,-36,160,202,173,52,254,14,146,142,74,0,-80,180,150,78,82,
/*    J(2,163021)=    */    -285,286,
/*    J(3,163021)=    */    -361,75,
/*    J(5,163021)=    */    258,203,-198,96,
/*    J(11,163021)=    */    202,38,320,216,230,118,-44,264,147,4,
/*    J(13,163021)=    */    -74,-190,-20,-84,-77,24,152,-38,166,-184,-116,76,
/*    J(19,163021)=    */    1,-224,10,86,-18,-86,112,104,22,114,14,162,-76,56,-30,26,154,66,
/*    J(2,177841)=    */    0,274,34,-112,-17,112,34,-274,
/*    J(1,177841)=    */    72,-55,-200,-215,193,-181,-74,-89,
/*    J(4,177841)=    */    179,0,-270,0,0,0,-270,0,
/*    J(3,177841)=    */    35,-50,-110,-24,-40,350,
/*    J(5,177841)=    */    -350,-18,-132,159,
/*    J(13,177841)=    */    -12,-106,-154,-20,-84,14,-138,-216,-147,-100,-398,34,
/*    J(19,177841)=    */    64,-106,-94,16,160,88,-16,-40,14,-66,-16,-191,-44,74,-238,60,8,60,
/*    J(2,186733)=    */    -93,422,
/*    J(3,186733)=    */    -77,-8,128,-118,-106,-50,54,-10,14,103,-282,-78,-104,-88,-8,36,-46,162,
/*    J(7,186733)=    */    -72,-240,-372,-332,-132,-435,
/*    J(13,186733)=    */    -213,4,-108,-102,-66,18,-136,40,22,62,-306,-256,
/*    J(19,186733)=    */    -216,-154,-54,-248,-64,-62,-140,-212,-178,-36,-44,-54,-322,-152,-284,-168,-323,-140,
/*    J(2,207481)=    */    -313,234,0,234,
/*    J(1,207481)=    */    245,0,384,0,
/*    J(4,207481)=    */    313,-234,0,-234,
/*    J(3,207481)=    */    455,456,
/*    J(5,207481)=    */    51,-356,-174,-402,
/*    J(7,207481)=    */    -241,-88,-32,140,-116,280,
/*    J(13,207481)=    */    -218,-398,-235,-86,-250,-140,-258,-314,-294,-450,-266,-394,
/*    J(19,207481)=    */    28,52,186,116,-30,-10,-112,78,48,91,176,-86,224,222,130,-44,214,8,
/*    J(2,213181)=    */    459,-50,
/*    J(3,213181)=    */    -76,-495,
/*    J(5,213181)=    */    125,-228,168,324,
/*    J(11,213181)=    */    -254,-52,28,-94,-254,-140,-96,-126,-256,-451,
/*    J(17,213181)=    */    -58,54,-82,-172,40,-72,92,78,-166,-50,92,-80,126,135,198,-136,
/*    J(19,213181)=    */    -242,98,118,-32,-113,-20,-194,-142,-54,4,-40,80,20,80,-164,116,28,0,
/*    J(2,217057)=    */    -190,-154,88,56,0,56,-88,-154,190,-174,6,26,117,-26,6,174,
/*    J(1,217057)=    */    69,-152,92,28,109,-60,33,-38,-117,191,-114,-279,-128,-37,-19,53,
/*    J(4,217057)=    */    443,0,0,0,102,0,0,0,0,0,0,0,102,0,0,0,
/*    J(3,217057)=    */    512,399,
/*    J(7,217057)=    */    303,83,259,235,555,223,
/*    J(17,217057)=    */    76,128,-166,72,-210,-8,118,-110,-37,86,190,128,14,28,30,-136,
/*    J(19,217057)=    */    210,94,148,139,-28,290,-50,76,192,352,180,152,116,60,56,140,124,-48,
/*    J(2,217361)=    */    0,64,-322,-30,-1,30,-322,-64,
/*    J(1,217361)=    */    40,-149,-48,409,33,151,2,9,
/*    J(4,217361)=    */    -381,0,190,0,0,0,190,0,
/*    J(5,217361)=    */    272,216,424,547,
/*    J(11,217361)=    */    169,296,328,130,-78,304,-52,110,122,56,
/*    J(13,217361)=    */    73,-53,-129,-75,101,-267,-55,-3,-231,-181,-225,121,
/*    J(19,217361)=    */    42,260,68,110,131,-34,0,148,46,168,0,48,44,-142,6,-88,280,128,
/*    J(2,225721)=    */    -89,330,0,330,
/*    J(1,225721)=    */    -395,0,264,0,
/*    J(4,225721)=    */    89,-330,0,-330,
/*    J(3,225721)=    */    -140,-56,-40,124,-12,-91,344,158,-34,-116,-168,-18,218,-8,93,124,188,-60,
/*    J(5,225721)=    */    -108,366,159,362,
/*    J(11,225721)=    */    -253,-292,-230,-138,-318,-134,-278,-382,-410,-514,
/*    J(19,225721)=    */    32,238,45,54,92,140,-158,114,156,334,118,86,206,54,234,60,-22,78,
/*    J(2,251941)=    */    479,-150,
/*    J(3,251941)=    */    380,-189,
/*    J(5,251941)=    */    -541,-340,-520,-220,
/*    J(13,251941)=    */    172,-78,146,161,326,198,-80,336,300,210,226,240,
/*    J(17,251941)=    */    -58,116,138,87,206,-76,158,-140,62,76,36,142,-10,-260,-60,126,
/*    J(19,251941)=    */    342,254,318,320,270,266,388,494,141,158,148,334,142,216,386,226,154,268,
/*    J(2,279073)=    */    249,-78,128,68,-40,-148,36,238,0,238,-36,-148,40,68,-128,-78,
/*    J(1,279073)=    */    19,-115,-66,-101,-42,-113,288,-49,140,107,14,-33,-150,27,308,25,
/*    J(4,279073)=    */    431,0,0,0,216,0,0,0,0,0,0,0,216,0,0,0,
/*    J(3,279073)=    */    -236,292,154,-36,147,-10,44,-112,-150,-294,68,30,74,104,94,-126,-196,-178,
/*    J(17,279073)=    */    121,148,58,38,-178,146,8,94,-46,202,142,284,262,-122,22,214,
/*    J(19,279073)=    */    417,40,116,162,280,106,128,250,352,212,224,48,-26,12,234,80,198,92,
/*    J(2,287281)=    */    0,-94,-322,-176,17,176,-322,94,
/*    J(1,287281)=    */    -120,233,-146,-419,-41,-97,52,-89,
/*    J(4,287281)=    */    467,0,-186,0,0,0,-186,0,
/*    J(3,287281)=    */    14,-158,-54,-68,210,128,-54,80,-254,152,-224,-44,124,276,75,194,152,-190,
/*    J(5,287281)=    */    -378,186,204,-113,
/*    J(7,287281)=    */    -50,-224,211,88,10,412,
/*    J(19,287281)=    */    -66,-2,276,-84,-218,-70,-236,60,-166,30,70,78,-41,-78,14,142,-78,64,
/*    J(2,300961)=    */    0,-182,6,-184,-112,26,110,228,-111,-228,110,-26,-112,184,6,182,
/*    J(1,300961)=    */    -130,-88,-93,-329,18,-131,75,-58,127,-4,-101,-111,-24,231,-107,-170,
/*    J(4,300961)=    */    431,0,0,0,240,0,0,0,0,0,0,0,240,0,0,0,
/*    J(3,300961)=    */    206,370,-293,-78,314,-376,
/*    J(5,300961)=    */    162,621,54,172,
/*    J(11,300961)=    */    171,0,240,162,-96,152,-156,-144,102,360,
/*    J(19,300961)=    */    -182,258,230,29,226,-70,-10,-52,126,-22,-40,60,116,-168,-68,36,50,-102,
/*    J(2,302329)=    */    -523,120,0,120,
/*    J(1,302329)=    */    -45,0,548,0,
/*    J(4,302329)=    */    -523,120,0,120,
/*    J(3,302329)=    */    -382,-343,-46,-192,88,-212,
/*    J(13,302329)=    */    80,-3,96,-192,378,286,32,-28,40,-132,82,-42,
/*    J(17,302329)=    */    0,20,30,-224,-184,-92,106,66,-361,70,-206,-132,-12,76,10,84,
/*    J(19,302329)=    */    -138,-309,-70,-84,-182,-104,90,-48,-158,-248,0,-108,0,182,-220,82,-48,70,
/*    J(3,342343)=    */    236,-224,-143,-144,-148,350,
/*    J(7,342343)=    */    -324,138,-168,-258,-542,-135,
/*    J(11,342343)=    */    360,92,123,206,70,190,-212,-42,260,382,
/*    J(13,342343)=    */    -453,70,20,16,-204,-252,-360,-170,-218,-102,-262,-10,
/*    J(19,342343)=    */    -228,-290,-144,-150,-134,-210,-114,-246,194,-174,8,-105,-342,-48,120,-20,-92,-40,
/*    J(2,351121)=    */    0,122,-34,344,-287,-344,-34,-122,
/*    J(1,351121)=    */    72,-199,370,29,239,127,-128,-281,
/*    J(4,351121)=    */    -317,0,-354,0,0,0,-354,0,
/*    J(3,351121)=    */    -475,189,
/*    J(5,351121)=    */    196,-476,-104,133,
/*    J(7,351121)=    */    200,10,70,-264,-20,465,
/*    J(11,351121)=    */    -48,-548,-22,-216,-328,-146,-248,-256,-152,93,
/*    J(19,351121)=    */    -281,32,-32,-230,-156,-346,100,-62,-6,-256,-164,-138,-170,-272,-186,-308,-366,-48,
/*    J(3,377911)=    */    446,-335,-374,168,-274,-28,
/*    J(5,377911)=    */    -53,91,-611,-143,
/*    J(13,377911)=    */    54,-34,108,111,-34,-116,-192,-86,270,230,-156,364,
/*    J(17,377911)=    */    -208,232,-130,-65,-12,298,2,-82,-90,86,202,184,162,208,-88,116,
/*    J(19,377911)=    */    200,196,68,-126,180,170,30,114,218,110,352,-6,170,343,-12,-12,-186,128,
/*    J(2,391249)=    */    0,98,158,-284,401,284,158,-98,
/*    J(1,391249)=    */    440,7,-118,-217,-201,-187,-56,241,
/*    J(4,391249)=    */    -541,0,222,0,0,0,222,0,
/*    J(3,391249)=    */    2,0,-621,-156,138,-552,
/*    J(11,391249)=    */    -394,53,46,-84,-226,130,34,58,-306,-346,
/*    J(13,391249)=    */    28,26,-128,-464,232,12,142,92,-36,2,211,38,
/*    J(19,391249)=    */    334,42,238,66,-66,114,-20,48,12,302,104,230,95,86,-76,174,468,204,
/*    J(2,406981)=    */    -609,190,
/*    J(3,406981)=    */    -301,-110,34,87,404,-232,
/*    J(5,406981)=    */    -708,-624,-621,-548,
/*    J(7,406981)=    */    -113,-252,-212,-8,428,-264,
/*    J(17,406981)=    */    26,-320,-20,-332,-438,-404,-21,-296,-358,-320,-220,-216,-214,-436,-140,-304,
/*    J(19,406981)=    */    -372,-197,-356,-348,-456,-100,-272,-502,-446,-250,-198,-454,-466,-386,-606,-320,-280,-376,
/*    J(2,451441)=    */    -89,360,-288,96,0,96,288,360,
/*    J(1,451441)=    */    -307,204,-210,-156,-48,-444,78,204,
/*    J(4,451441)=    */    -521,0,-300,0,0,0,-300,0,
/*    J(3,451441)=    */    -120,213,-442,-248,-204,-172,186,-34,-138,-146,205,-152,-254,-234,138,50,-90,-146,
/*    J(5,451441)=    */    366,-441,-174,-212,
/*    J(11,451441)=    */    282,483,462,180,618,0,228,384,232,78,
/*    J(19,451441)=    */    -22,-102,-35,-358,42,12,-78,-102,194,-30,-138,14,-242,126,-286,-386,-214,-30,
/*    J(2,456457)=    */    -673,-42,0,-42,
/*    J(1,456457)=    */    -259,0,-624,0,
/*    J(4,456457)=    */    673,42,0,42,
/*    J(3,456457)=    */    -328,-777,
/*    J(7,456457)=    */    264,740,499,604,360,640,
/*    J(11,456457)=    */    100,190,230,-96,188,434,334,-12,352,-225,
/*    J(13,456457)=    */    -40,96,-300,-264,-301,-360,-308,-128,-296,0,-4,-540,
/*    J(19,456457)=    */    -271,-348,-414,-190,-338,-150,-222,-378,-456,-90,-282,-452,-306,-222,-358,-42,-386,78,
/*    J(5,461891)=    */    -122,638,77,-44,
/*    J(11,461891)=    */    -37,2,60,306,-140,-172,-298,-74,152,-438,
/*    J(13,461891)=    */    652,294,152,162,116,170,-174,276,148,111,94,78,
/*    J(17,461891)=    */    306,-124,190,300,458,410,112,180,504,66,81,152,138,118,238,304,
/*    J(19,461891)=    */    70,240,86,-12,76,28,62,-302,40,-47,-186,-56,-24,198,98,-26,-368,198,
/*    J(2,489061)=    */    495,-494,
/*    J(3,489061)=    */    -61,574,-272,-297,-46,-178,
/*    J(5,489061)=    */    600,402,-187,234,
/*    J(11,489061)=    */    268,108,-36,212,392,-255,-44,-296,176,112,
/*    J(13,489061)=    */    -152,-182,-262,64,-52,262,-78,-512,12,-276,-233,-256,
/*    J(19,489061)=    */    16,-100,210,206,-52,576,101,76,96,346,290,160,56,220,254,348,160,114,
/*    J(2,511633)=    */    0,-164,-334,298,-239,-298,-334,164,
/*    J(1,511633)=    */    48,553,-218,43,263,173,232,35,
/*    J(4,511633)=    */    -125,0,498,0,0,0,498,0,
/*    J(3,511633)=    */    32,388,595,-228,464,296,
/*    J(11,511633)=    */    216,-24,-130,-212,20,-38,-212,-456,8,-515,
/*    J(17,511633)=    */    54,-304,-92,-310,365,26,-200,-76,-6,-116,94,194,86,86,102,-142,
/*    J(19,511633)=    */    86,238,-14,-102,96,208,296,-312,254,134,186,263,282,118,118,4,-184,76,
/*    J(2,526681)=    */    -233,486,0,486,
/*    J(1,526681)=    */    -91,0,720,0,
/*    J(4,526681)=    */    233,-486,0,-486,
/*    J(3,526681)=    */    -598,-274,-319,24,-278,-119,
/*    J(5,526681)=    */    824,305,80,200,
/*    J(7,526681)=    */    441,51,-57,-369,175,-291,
/*    J(11,526681)=    */    4,-500,228,34,15,-250,-424,-164,-292,-192,
/*    J(19,526681)=    */    168,98,262,-69,-54,-142,184,300,-142,246,-174,-220,48,44,-196,-94,-6,202,
/*    J(2,554269)=    */    363,-650,
/*    J(3,554269)=    */    77,780,
/*    J(11,554269)=    */    415,536,518,446,840,358,506,816,534,574,
/*    J(13,554269)=    */    -28,-176,-56,-790,-166,-126,-158,-180,-40,32,-232,-213,
/*    J(17,554269)=    */    298,264,-76,-22,-136,-124,76,38,-264,30,333,134,-128,186,-160,264,
/*    J(19,554269)=    */    -272,208,-322,-326,-130,-390,-368,-102,-260,-195,-360,-6,52,-34,-254,-374,-86,-88,
/*    J(2,568481)=    */    126,-38,102,362,0,362,-102,-38,-126,-8,328,130,-43,-130,328,8,
/*    J(1,568481)=    */    173,-265,-152,-128,-371,138,133,351,-145,42,14,201,-8,-117,211,98,
/*    J(4,568481)=    */    123,0,0,0,-526,0,0,0,0,0,0,0,-526,0,0,0,
/*    J(5,568481)=    */    -269,404,206,-422,
/*    J(11,568481)=    */    -412,-522,-138,-254,-270,-328,-376,116,-355,184,
/*    J(17,568481)=    */    110,144,310,544,120,166,528,402,184,128,408,96,617,176,192,124,
/*    J(19,568481)=    */    -85,-51,-157,-37,47,-149,109,175,183,191,55,-63,-261,-133,455,-245,-15,-77,
/*    J(2,608609)=    */    -126,212,322,-230,115,230,322,-212,-126,-256,-116,-36,0,-36,116,-256,
/*    J(1,608609)=    */    237,-330,177,-224,62,78,-14,-218,-139,223,43,1,-247,-117,-110,-383,
/*    J(4,608609)=    */    603,0,0,0,-350,0,0,0,0,0,0,0,-350,0,0,0,
/*    J(7,608609)=    */    -188,-654,-146,-719,-614,-606,
/*    J(11,608609)=    */    507,305,619,81,385,829,449,487,429,231,
/*    J(13,608609)=    */    492,234,-18,190,24,-76,-129,236,152,-130,-254,370,
/*    J(19,608609)=    */    66,46,-196,-122,-540,-3,-312,-350,-192,78,-326,-272,-18,-134,42,-270,-342,-6,
/*    J(2,651169)=    */    -306,200,-2,-314,-67,314,-2,-200,-306,-188,-236,-8,0,-8,236,-188,
/*    J(1,651169)=    */    29,32,39,-118,14,-32,154,-292,29,-587,-291,283,-25,-47,6,103,
/*    J(4,651169)=    */    -37,0,0,0,570,0,0,0,0,0,0,0,570,0,0,0,
/*    J(3,651169)=    */    11,-382,-88,99,-914,-110,
/*    J(7,651169)=    */    -664,256,172,126,67,42,
/*    J(17,651169)=    */    -83,-81,-375,-309,-125,-251,-59,-397,35,-519,-611,-81,19,-199,19,-231,
/*    J(19,651169)=    */    12,4,60,284,268,46,-374,-48,-64,54,-366,308,48,-68,216,-12,128,187,
/*    J(2,652081)=    */    0,70,-2,-44,-799,44,-2,-70,
/*    J(1,652081)=    */    608,49,-242,49,-465,19,-28,-41,
/*    J(4,652081)=    */    659,0,-330,0,0,0,-330,0,
/*    J(3,652081)=    */    176,-705,
/*    J(5,652081)=    */    864,135,160,0,
/*    J(11,652081)=    */    4,-164,-294,403,-18,-208,-316,-464,20,-372,
/*    J(13,652081)=    */    478,146,316,456,445,400,540,-186,-6,420,196,304,
/*    J(19,652081)=    */    -410,-64,120,-92,-8,-205,-244,-404,24,-256,-114,-48,206,-48,-226,142,208,-254,
/*    J(2,697681)=    */    0,164,-514,238,47,-238,-514,-164,
/*    J(1,697681)=    */    40,655,232,-271,169,-329,42,-35,
/*    J(4,697681)=    */    -541,0,-450,0,0,0,-450,0,
/*    J(3,697681)=    */    234,-262,62,-265,4,210,-22,116,430,382,138,-24,-441,42,382,-30,-62,398,
/*    J(5,697681)=    */    433,-512,-188,-404,
/*    J(17,697681)=    */    -14,-406,364,-30,112,236,-20,-36,-48,36,-58,328,86,-234,-320,-99,
/*    J(19,697681)=    */    300,460,-28,0,-82,178,224,188,-92,292,412,256,474,150,357,-154,266,256,
/*    J(3,733591)=    */    -294,243,136,-162,174,-70,-170,-362,94,-154,26,-210,-426,200,-88,338,-278,216,
/*    J(5,733591)=    */    -162,-667,-468,-954,
/*    J(11,733591)=    */    -366,-112,-144,262,-362,-268,386,-8,275,94,
/*    J(13,733591)=    */    304,94,-106,92,640,234,342,88,376,-113,-198,326,
/*    J(19,733591)=    */    216,-80,88,124,-30,-240,262,350,303,248,232,-78,172,190,550,-86,242,386,
/*    J(2,782497)=    */    -194,28,-328,-74,315,74,-328,-28,-194,-326,-282,66,0,66,282,-326,
/*    J(1,782497)=    */    461,-182,-99,-411,-264,3,-126,-74,-23,-331,227,242,-189,88,32,49,
/*    J(4,782497)=    */    827,0,0,0,222,0,0,0,0,0,0,0,222,0,0,0,
/*    J(3,782497)=    */    -184,210,120,3,12,-774,
/*    J(11,782497)=    */    92,170,-86,704,364,494,114,172,688,147,
/*    J(13,782497)=    */    563,412,286,-24,60,210,10,422,-354,96,164,-78,
/*    J(19,782497)=    */    197,-91,223,45,319,37,429,315,203,427,247,165,329,-181,413,631,253,9,
/*    J(2,790021)=    */    415,-786,
/*    J(3,790021)=    */    286,-118,-180,97,-122,-372,-162,-34,-272,330,-52,306,-419,-280,-286,-202,90,-174,
/*    J(5,790021)=    */    978,218,72,141,
/*    J(7,790021)=    */    280,86,242,981,50,138,
/*    J(11,790021)=    */    356,482,34,8,256,-318,382,280,644,273,
/*    J(19,790021)=    */    372,44,152,-208,-12,-24,-180,-90,22,368,126,-388,91,-24,-82,436,82,150,
/*    J(2,813961)=    */    863,186,0,186,
/*    J(1,813961)=    */    285,0,-856,0,
/*    J(4,813961)=    */    -863,-186,0,-186,
/*    J(3,813961)=    */    530,621,216,486,144,558,
/*    J(5,813961)=    */    -483,-169,-861,-903,
/*    J(7,813961)=    */    938,338,277,-90,330,68,
/*    J(17,813961)=    */    -178,-628,14,-426,-76,78,188,-94,42,-122,-192,121,-4,-470,-66,10,
/*    J(19,813961)=    */    377,257,443,613,347,293,401,99,689,581,295,297,353,839,149,323,241,185,
/*    J(2,895357)=    */    -21,-946,
/*    J(3,895357)=    */    116,-8,298,345,518,-532,
/*    J(7,895357)=    */    147,639,361,225,771,-261,
/*    J(11,895357)=    */    -254,-83,-510,-856,108,-392,-268,-496,-148,16,
/*    J(17,895357)=    */    250,471,678,712,392,306,346,336,546,106,54,766,52,370,282,384,
/*    J(19,895357)=    */    214,-457,-274,-296,124,-226,90,96,62,-204,-132,-20,-548,-34,-132,-490,-258,-176,
/*    J(3,1027027)=    */    128,38,595,-316,-104,-56,-82,490,320,108,430,374,-338,-64,-100,-188,-106,364,
/*    J(7,1027027)=    */    598,568,-496,105,-90,238,
/*    J(11,1027027)=    */    -222,-116,-510,386,20,-268,-366,-224,-802,-143,
/*    J(13,1027027)=    */    509,322,-404,-46,-158,-266,-130,-144,396,-224,208,-194,
/*    J(19,1027027)=    */    -64,-160,-78,-256,-420,-194,-488,-242,-340,-2,-210,-68,-178,-410,-486,-88,513,-402,
/*    J(2,1053361)=    */    353,454,476,-178,0,-178,-476,454,
/*    J(1,1053361)=    */    -221,134,-60,716,-268,-44,488,398,
/*    J(4,1053361)=    */    343,0,684,0,0,0,684,0,
/*    J(3,1053361)=    */    449,542,86,-528,406,-260,
/*    J(5,1053361)=    */    -273,-624,624,32,
/*    J(7,1053361)=    */    867,1029,771,1159,969,1035,
/*    J(11,1053361)=    */    934,1030,980,822,362,894,585,518,578,754,
/*    J(19,1053361)=    */    338,382,658,80,540,78,-10,196,-204,414,426,-64,-132,242,294,134,-32,231,
/*    J(2,1058149)=    */    1007,-210,
/*    J(3,1058149)=    */    302,-287,472,-156,-955,146,
/*    J(7,1058149)=    */    -348,-302,-334,-184,-552,641,
/*    J(13,1058149)=    */    -360,-80,82,242,-588,96,-186,194,576,28,20,131,
/*    J(17,1058149)=    */    48,-534,-86,-480,-114,132,-212,-324,-672,-302,-624,-350,-382,60,-332,-521,
/*    J(19,1058149)=    */    -14,-12,160,-64,198,332,222,74,218,-58,828,70,-84,336,364,490,328,449,
/*    J(2,1108537)=    */    -697,-558,0,-558,
/*    J(1,1108537)=    */    949,0,-456,0,
/*    J(4,1108537)=    */    697,558,0,558,
/*    J(3,1108537)=    */    968,-153,
/*    J(11,1108537)=    */    -390,-206,-526,-646,-102,-334,-950,-874,-308,-791,
/*    J(13,1108537)=    */    18,-605,418,-204,-516,-504,-298,102,-22,130,-88,-96,
/*    J(17,1108537)=    */    -430,-206,42,-500,-526,-403,-634,-144,-794,-282,-418,-12,52,-276,-640,-304,
/*    J(19,1108537)=    */    16,96,35,8,80,-8,142,476,-282,-100,184,144,136,108,104,-62,76,-774,
/*    J(3,1133731)=    */    -268,-84,98,166,120,-48,436,71,18,-132,90,250,128,-54,-480,950,217,204,
/*    J(5,1133731)=    */    552,-254,54,-753,
/*    J(13,1133731)=    */    -474,156,-24,-414,206,108,-516,-606,-390,-396,126,-351,
/*    J(17,1133731)=    */    708,550,72,176,138,326,430,440,196,272,338,-46,240,-266,426,-193,
/*    J(19,1133731)=    */    260,402,-38,392,288,1002,206,-14,116,144,-68,346,222,-56,208,290,86,317,
/*    J(2,1264033)=    */    -98,-6,-334,-484,-187,484,-334,6,-98,230,416,182,0,182,-416,230,
/*    J(1,1264033)=    */    355,476,15,152,338,-52,-26,-282,167,-5,617,29,71,-255,-242,391,
/*    J(4,1264033)=    */    1115,0,0,0,-102,0,0,0,0,0,0,0,-102,0,0,0,
/*    J(3,1264033)=    */    326,-208,830,-16,-46,134,326,-34,-160,360,229,586,-56,70,202,178,454,220,
/*    J(7,1264033)=    */    432,-72,918,327,-406,60,
/*    J(11,1264033)=    */    168,-10,-852,232,28,335,-76,-32,392,298,
/*    J(19,1264033)=    */    -264,-24,-316,34,167,98,192,240,-120,-144,-52,728,120,-178,466,-16,400,36,
/*    J(2,1279081)=    */    557,-696,0,-696,
/*    J(1,1279081)=    */    -1125,0,-116,0,
/*    J(4,1279081)=    */    557,-696,0,-696,
/*    J(3,1279081)=    */    107,-164,220,-357,824,584,
/*    J(5,1279081)=    */    443,908,782,1346,
/*    J(11,1279081)=    */    813,728,472,238,-246,736,92,70,206,344,
/*    J(17,1279081)=    */    -400,-48,-380,-26,-490,-148,-334,-246,-300,-483,-172,-546,-930,104,190,-178,
/*    J(19,1279081)=    */    286,256,-300,244,-114,-78,190,-390,235,-34,110,-142,-130,-348,-104,320,622,-16,
/*    J(2,1369369)=    */    901,528,0,528,
/*    J(1,1369369)=    */    1155,0,188,0,
/*    J(4,1369369)=    */    901,528,0,528,
/*    J(3,1369369)=    */    68,72,-1116,48,33,-1212,
/*    J(7,1369369)=    */    544,796,1366,229,826,592,
/*    J(11,1369369)=    */    -114,-410,-76,216,377,-24,-748,158,-662,-236,
/*    J(13,1369369)=    */    176,49,276,104,390,74,488,-82,662,1060,-26,130,
/*    J(19,1369369)=    */    -364,164,-108,-124,-752,-100,248,-516,-62,-160,-412,-326,-451,-258,136,64,214,-196,
/*    J(2,1492261)=    */    1119,-490,
/*    J(3,1492261)=    */    1385,924,
/*    J(5,1492261)=    */    168,263,1212,-204,
/*    J(7,1492261)=    */    -180,62,-435,-1218,132,-252,
/*    J(11,1492261)=    */    362,-248,636,758,392,242,418,-285,-134,630,
/*    J(17,1492261)=    */    424,306,386,798,-190,520,92,54,330,940,528,329,560,760,542,318,
/*    J(19,1492261)=    */    -708,-8,-416,-356,-524,-418,-194,-125,-490,-374,368,-12,-108,76,-624,-596,-92,-264,
/*    J(2,1580041)=    */    -1153,-354,0,-354,
/*    J(1,1580041)=    */    -1155,0,-496,0,
/*    J(4,1580041)=    */    1153,354,0,354,
/*    J(3,1580041)=    */    -106,-536,104,624,324,264,318,236,-14,-24,-778,328,165,672,66,210,250,-412,
/*    J(5,1580041)=    */    -351,-1385,-45,-315,
/*    J(7,1580041)=    */    911,-248,-164,0,760,588,
/*    J(11,1580041)=    */    -482,444,-454,-580,-460,-198,568,-166,-8,81,
/*    J(19,1580041)=    */    60,56,-166,14,378,246,-344,-140,-248,-76,-852,284,78,135,190,-328,40,292,
/*    J(2,1790713)=    */    -1241,-354,0,-354,
/*    J(1,1790713)=    */    -363,0,-1288,0,
/*    J(4,1790713)=    */    1241,354,0,354,
/*    J(3,1790713)=    */    -745,-4,1214,-657,430,646,
/*    J(7,1790713)=    */    -418,-988,49,116,394,-764,
/*    J(11,1790713)=    */    -844,166,54,-687,-350,-674,284,-28,156,140,
/*    J(17,1790713)=    */    -48,66,188,244,263,272,772,-378,32,196,-126,-108,-266,-22,-4,890,
/*    J(19,1790713)=    */    948,8,280,360,160,-120,-256,136,-32,452,130,452,-22,547,518,400,318,-252,
/*    J(2,1813969)=    */    0,-228,126,-762,719,762,126,228,
/*    J(1,1813969)=    */    -1000,-513,-72,489,-39,111,378,-387,
/*    J(4,1813969)=    */    1187,0,-450,0,0,0,-450,0,
/*    J(3,1813969)=    */    48,-126,456,-602,-43,276,648,586,366,-62,-202,796,-776,-107,374,368,282,216,
/*    J(13,1813969)=    */    430,20,254,-739,-86,-216,-2,544,-350,-312,364,430,
/*    J(17,1813969)=    */    -503,68,-362,28,-216,14,-356,-340,-646,-426,-232,260,194,-20,-620,538,
/*    J(19,1813969)=    */    -194,-366,92,-224,-250,-241,-294,-372,116,-282,-594,-32,-478,-240,-868,400,372,-42,
/*    J(2,1867321)=    */    -217,-954,0,-954,
/*    J(1,1867321)=    */    1365,0,-64,0,
/*    J(4,1867321)=    */    217,954,0,954,
/*    J(3,1867321)=    */    106,100,336,646,-742,496,38,-222,-280,-62,390,764,476,-442,-155,-68,-218,-396,
/*    J(5,1867321)=    */    1049,1020,-330,450,
/*    J(7,1867321)=    */    -850,-606,-393,-1682,-730,-556,
/*    J(13,1867321)=    */    -189,-39,-595,-799,-483,-367,193,-715,465,77,-523,-393,
/*    J(19,1867321)=    */    -202,364,-318,-338,-168,106,-100,40,246,-202,234,658,85,138,-398,-566,414,-260,
/*    J(3,1939939)=    */    1310,-153,
/*    J(7,1939939)=    */    -492,1132,354,654,702,519,
/*    J(11,1939939)=    */    -422,-32,322,18,-16,-224,-646,-276,-846,647,
/*    J(13,1939939)=    */    338,306,138,426,398,168,792,524,648,-800,142,325,
/*    J(17,1939939)=    */    -96,418,616,712,316,6,-106,280,252,290,-146,58,620,-654,306,119,
/*    J(19,1939939)=    */    -382,-1088,148,-530,-408,-528,-544,-178,-188,-602,-214,-290,-176,-770,-262,88,86,-15,
/*    J(2,2217073)=    */    511,-366,-36,918,0,918,36,-366,
/*    J(1,2217073)=    */    -543,-528,402,-30,632,498,-618,672,
/*    J(4,2217073)=    */    295,0,-1032,0,0,0,-1032,0,
/*    J(3,2217073)=    */    -781,936,
/*    J(11,2217073)=    */    456,592,408,1168,128,496,109,552,728,-568,
/*    J(13,2217073)=    */    950,680,1094,-26,626,1350,982,822,1028,354,460,623,
/*    J(17,2217073)=    */    -166,-568,37,524,-114,-304,468,-184,666,-232,50,158,416,670,172,-200,
/*    J(19,2217073)=    */    -1362,-218,-18,-46,12,-258,-166,304,-284,-40,-130,257,-234,-108,144,64,-34,-126,
/*    J(3,2238391)=    */    1397,-494,304,999,-106,-490,
/*    J(5,2238391)=    */    -87,1333,627,1251,
/*    J(7,2238391)=    */    -261,-891,867,165,183,545,
/*    J(11,2238391)=    */    34,-604,588,-291,-410,150,328,986,-52,-70,
/*    J(17,2238391)=    */    -244,-486,334,-48,-385,786,-268,-176,-386,146,454,512,154,268,22,404,
/*    J(19,2238391)=    */    -576,284,100,310,-542,-148,-4,-1034,-134,-150,-82,218,270,-216,-121,-150,268,148,
/*    J(2,2282281)=    */    -1283,564,0,564,
/*    J(1,2282281)=    */    891,0,1220,0,
/*    J(4,2282281)=    */    -1283,564,0,564,
/*    J(3,2282281)=    */    -1729,-1065,
/*    J(5,2282281)=    */    756,1204,1176,1833,
/*    J(7,2282281)=    */    -747,-435,123,-897,637,345,
/*    J(11,2282281)=    */    -342,-380,-403,-160,-1130,-752,-1104,202,6,-844,
/*    J(13,2282281)=    */    -488,-696,148,-390,344,-456,-94,-253,366,700,-376,-522,
/*    J(19,2282281)=    */    895,269,321,-525,445,-7,-375,-7,-139,-387,-131,305,-311,213,31,-135,-95,-349,
/*    J(2,2351441)=    */    0,-12,-514,-834,-657,834,-514,12,
/*    J(1,2351441)=    */    -456,783,-228,-679,-927,111,-90,-371,
/*    J(4,2351441)=    */    -1533,0,-26,0,0,0,-26,0,
/*    J(5,2351441)=    */    -424,904,-224,-967,
/*    J(7,2351441)=    */    296,1056,584,296,1173,-480,
/*    J(13,2351441)=    */    -216,-428,-698,-168,-604,-786,-1005,298,-136,32,66,-984,
/*    J(17,2351441)=    */    -20,234,-252,-150,-334,296,-204,784,235,216,-196,-204,518,-338,-804,82,
/*    J(19,2351441)=    */    214,-92,-500,348,-28,-272,16,78,744,-262,622,376,340,1,-156,-24,770,446,
/*    J(2,2489761)=    */    278,704,-464,-46,-693,46,-464,-704,278,62,-450,90,0,90,450,62,
/*    J(1,2489761)=    */    25,-352,287,311,72,133,-40,736,985,-427,57,-246,99,-396,486,29,
/*    J(4,2489761)=    */    1531,0,0,0,270,0,0,0,0,0,0,0,270,0,0,0,
/*    J(3,2489761)=    */    -1012,1089,630,102,321,-24,
/*    J(5,2489761)=    */    -723,51,1119,-463,
/*    J(7,2489761)=    */    -440,-696,-1616,-1013,-40,-1152,
/*    J(13,2489761)=    */    -532,-407,-216,-296,-1294,-866,-558,386,-678,-102,-6,-710,
/*    J(19,2489761)=    */    -666,-464,84,-874,-209,-258,64,-108,626,-300,-326,-570,-670,190,-310,-590,-562,-226,
/*    J(3,2645371)=    */    488,-519,-456,-264,-369,1128,
/*    J(5,2645371)=    */    -169,494,1106,-832,
/*    J(7,2645371)=    */    1353,1931,889,1579,933,1063,
/*    J(13,2645371)=    */    513,1050,250,-66,60,728,-38,164,314,-666,400,-396,
/*    J(17,2645371)=    */    284,-94,-686,-68,-200,30,-208,1146,-112,-78,110,-280,72,-126,-622,-291,
/*    J(19,2645371)=    */    1368,1100,660,863,1046,1174,986,906,1336,240,1204,932,996,874,1250,730,606,334,
/*    J(2,2771341)=    */    -221,-1650,
/*    J(3,2771341)=    */    -1021,900,
/*    J(5,2771341)=    */    -329,619,631,-1157,
/*    J(11,2771341)=    */    618,-32,284,-545,964,1220,186,716,238,90,
/*    J(13,2771341)=    */    -466,-78,-1534,-182,-354,-966,-848,-558,-954,-1147,-918,-290,
/*    J(17,2771341)=    */    224,330,208,-252,750,-646,88,106,-125,-270,-192,702,990,128,120,218,
/*    J(19,2771341)=    */    14,514,498,-294,-246,734,216,-236,352,-570,598,82,271,-146,-358,-502,-248,4,
/*    J(2,2934361)=    */    -937,1014,0,1014,
/*    J(1,2934361)=    */    -1275,0,1144,0,
/*    J(4,2934361)=    */    937,-1014,0,-1014,
/*    J(3,2934361)=    */    638,-95,-370,672,662,238,186,302,-916,584,-184,644,314,342,668,74,68,-486,
/*    J(5,2934361)=    */    -228,66,1579,-258,
/*    J(11,2934361)=    */    -648,147,408,-144,-788,-660,-744,264,-732,564,
/*    J(13,2934361)=    */    1110,388,-386,26,-366,835,366,834,78,400,890,452,
/*    J(19,2934361)=    */    -84,-132,-404,-698,490,-288,708,4,-376,-226,72,464,-180,231,784,208,-312,194,
/*    J(2,2984521)=    */    1357,756,0,756,
/*    J(1,2984521)=    */    555,0,-1636,0,
/*    J(4,2984521)=    */    1357,756,0,756,
/*    J(3,2984521)=    */    119,-1665,
/*    J(5,2984521)=    */    -1046,-1970,-1220,-1775,
/*    J(7,2984521)=    */    -1324,-364,-980,-1092,-2016,-1099,
/*    J(11,2984521)=    */    602,1360,738,666,-376,318,404,526,397,1414,
/*    J(17,2984521)=    */    -1408,-912,-648,-802,-1164,-676,-854,-520,-384,-792,-1532,-376,-284,-231,-292,-856,
/*    J(19,2984521)=    */    128,124,314,636,636,466,712,503,1372,438,484,744,1070,-132,568,1112,610,1082,
/*    J(5,3233231)=    */    -1176,-770,-1870,-1925,
/*    J(7,3233231)=    */    525,-1202,-860,-942,328,-678,
/*    J(11,3233231)=    */    -264,-6,200,-54,-1402,-902,238,-831,244,-392,
/*    J(13,3233231)=    */    831,775,723,1019,1579,651,709,1049,1853,1305,321,1261,
/*    J(17,3233231)=    */    270,922,-38,-328,257,678,708,756,492,766,-138,-136,694,-392,642,-88,
/*    J(19,3233231)=    */    18,234,-904,-530,-422,-762,-1078,-433,-732,-768,-612,-712,-1010,-156,-912,-630,-598,-1432,
/*    J(2,3627937)=    */    -81,758,-198,-314,448,108,334,880,0,880,-334,108,-448,-314,198,758,
/*    J(1,3627937)=    */    655,-349,-12,-394,-310,354,134,-823,-72,-161,-648,-712,318,568,140,777,
/*    J(4,3627937)=    */    575,0,0,0,1284,0,0,0,0,0,0,0,1284,0,0,0,
/*    J(3,3627937)=    */    -228,-558,446,142,-154,168,-26,859,-526,28,-1044,-536,-122,-346,-164,478,-341,-714,
/*    J(13,3627937)=    */    16,33,-1744,-160,-94,-102,-284,372,72,-56,-474,366,
/*    J(17,3627937)=    */    -524,-426,-248,-750,-792,700,-640,-675,-872,-1124,-440,-402,-276,-950,268,-92,
/*    J(19,3627937)=    */    -348,-962,-624,-1652,-1044,-1108,-1558,-1069,-1006,-1360,-852,-956,-768,-1310,-352,-1064,-1242,-1574,
/*    J(2,3837241)=    */    917,1224,0,1224,
/*    J(1,3837241)=    */    1779,0,820,0,
/*    J(4,3837241)=    */    917,1224,0,1224,
/*    J(3,3837241)=    */    18,-84,546,516,-245,38,-686,478,-48,500,1310,-194,102,-29,404,-742,64,-446,
/*    J(5,3837241)=    */    -312,483,1512,-784,
/*    J(11,3837241)=    */    828,-642,-374,-1408,72,298,-560,185,172,-178,
/*    J(17,3837241)=    */    480,-300,968,1360,204,190,536,478,718,75,410,-18,976,70,340,-504,
/*    J(19,3837241)=    */    1348,1158,1506,668,564,1426,784,596,1406,1848,1156,1365,992,896,1378,1028,400,1050,
/*    J(2,3912481)=    */    967,556,44,14,376,-670,-656,-396,0,-396,656,-670,-376,14,-44,556,
/*    J(1,3912481)=    */    889,239,562,-733,488,673,-392,-595,-136,-803,286,157,220,-393,12,-201,
/*    J(4,3912481)=    */    -1297,0,0,0,-1056,0,0,0,0,0,0,0,-1056,0,0,0,
/*    J(3,3912481)=    */    1067,-866,826,1779,224,452,
/*    J(5,3912481)=    */    -863,16,1444,-548,
/*    J(11,3912481)=    */    738,626,-525,506,1138,340,1438,210,494,-434,
/*    J(13,3912481)=    */    910,726,318,1248,756,-112,76,262,-94,1266,-400,-69,
/*    J(19,3912481)=    */    -735,167,-9,63,-103,651,-103,-35,143,-61,-383,-1019,787,65,31,-365,-917,-363,
/*    J(2,3979361)=    */    -529,368,712,292,-304,720,-696,164,0,164,696,720,304,292,-712,368,
/*    J(1,3979361)=    */    -715,344,306,-402,634,-234,502,-216,144,-440,-10,566,946,862,-406,-40,
/*    J(4,3979361)=    */    1167,0,0,0,-1144,0,0,0,0,0,0,0,-1144,0,0,0,
/*    J(5,3979361)=    */    1396,2364,1376,1943,
/*    J(7,3979361)=    */    827,683,-297,567,963,2051,
/*    J(11,3979361)=    */    109,-428,-488,934,-304,1026,802,-72,-750,248,
/*    J(17,3979361)=    */    924,894,-188,194,504,9,616,430,92,494,-660,520,68,1168,1116,516,
/*    J(19,3979361)=    */    832,828,920,1646,838,616,602,996,953,1748,1374,880,790,1258,792,372,316,1604,
/*    J(3,4157011)=    */    1106,-866,-917,-114,-592,797,
/*    J(5,4157011)=    */    -141,630,390,2140,
/*    J(11,4157011)=    */    -1212,807,30,66,900,228,312,486,1082,402,
/*    J(13,4157011)=    */    -1130,-896,0,-1684,-760,-1092,-381,-1394,-1606,-1262,-1062,-330,
/*    J(17,4157011)=    */    -1272,-860,-798,-536,-1354,-500,-509,-922,-552,-410,-992,34,-1062,-1626,-1524,-1330,
/*    J(19,4157011)=    */    928,-324,-160,-298,-52,54,722,24,-705,694,-268,-556,-446,-236,-154,-590,-534,418,
/*    J(2,4232593)=    */    0,-790,-914,-656,673,656,-914,790,
/*    J(1,4232593)=    */    80,-1043,962,713,239,-413,720,979,
/*    J(4,4232593)=    */    2051,0,-114,0,0,0,-114,0,
/*    J(3,4232593)=    */    1175,-716,124,1527,-1552,-640,
/*    J(7,4232593)=    */    -1671,-1807,-65,-1229,-207,-419,
/*    J(13,4232593)=    */    -306,527,-1618,-252,360,426,-122,-10,-38,438,310,492,
/*    J(17,4232593)=    */    857,-333,-537,467,267,-79,921,311,783,-419,413,311,-229,-717,-297,405,
/*    J(19,4232593)=    */    484,642,770,-56,1436,872,785,888,540,1148,174,526,876,1248,1250,168,-228,826,
/*    J(2,4476781)=    */    -2109,-170,
/*    J(3,4476781)=    */    1970,-759,-552,54,-252,-426,
/*    J(5,4476781)=    */    -1789,164,716,-472,
/*    J(7,4476781)=    */    2506,1752,1536,690,1188,1035,
/*    J(11,4476781)=    */    1014,1254,-342,1278,666,1230,1162,606,792,-423,
/*    J(17,4476781)=    */    -220,-149,-972,-1036,-620,-296,-546,624,-1476,-498,70,-180,-388,-586,220,136,
/*    J(19,4476781)=    */    376,418,-6,87,478,86,676,-1008,-60,38,-766,-704,374,550,-196,246,-762,20,
/*    J(3,5135131)=    */    -254,-238,-160,-484,314,514,662,-731,318,504,1152,-372,576,38,-702,534,-469,294,
/*    J(5,5135131)=    */    -1780,762,258,549,
/*    J(7,5135131)=    */    -468,-570,232,-1398,1038,885,
/*    J(11,5135131)=    */    148,1240,788,-456,866,-982,-134,114,922,837,
/*    J(13,5135131)=    */    -1330,-1358,0,-978,-1128,-434,-476,-1450,-1476,-2160,-730,-1221,
/*    J(19,5135131)=    */    -152,362,62,-698,888,-22,-430,-292,1162,276,476,-262,-716,-324,-696,334,-462,-191,
/*    J(2,5372137)=    */    -595,1584,0,1584,
/*    J(1,5372137)=    */    91,0,2316,0,
/*    J(4,5372137)=    */    -595,1584,0,1584,
/*    J(3,5372137)=    */    -509,636,-44,-478,-300,524,390,-508,-328,72,8,-972,-1308,-246,-390,1310,-592,-176,
/*    J(7,5372137)=    */    -712,-1044,-2556,63,-540,-756,
/*    J(11,5372137)=    */    1060,617,-1288,520,-320,246,-774,-850,-222,-244,
/*    J(17,5372137)=    */    328,1427,-72,584,-258,358,510,-134,-530,640,36,-298,-14,486,-1038,-394,
/*    J(19,5372137)=    */    -426,82,-44,-152,198,-327,-218,-788,1376,-202,-994,-608,-830,-646,-126,-614,330,-2,
/*    J(2,5868721)=    */    0,820,1022,1006,-641,-1006,1022,-820,
/*    J(1,5868721)=    */    496,415,44,137,-239,1487,-1338,-1171,
/*    J(4,5868721)=    */    2323,0,486,0,0,0,486,0,
/*    J(3,5868721)=    */    -166,114,-476,-636,-532,-442,-778,792,958,236,-1478,172,-832,-570,-56,-708,633,954,
/*    J(5,5868721)=    */    -1892,853,652,-104,
/*    J(11,5868721)=    */    -828,-58,-366,-1264,-1928,-866,-756,427,-1672,-214,
/*    J(13,5868721)=    */    1522,710,-288,-108,-720,-60,938,78,-448,1172,-17,652,
/*    J(19,5868721)=    */    478,600,776,-276,-770,862,-446,1242,1011,598,290,814,966,276,-276,752,102,-8,
/*    J(2,6046561)=    */    -114,-298,-1136,-88,0,-88,1136,-298,114,370,818,902,-85,-902,818,-370,
/*    J(1,6046561)=    */    -335,568,660,-1064,395,-68,-1017,-526,-97,-575,206,-661,68,-323,1107,647,
/*    J(4,6046561)=    */    1819,0,0,0,-1170,0,0,0,0,0,0,0,-1170,0,0,0,
/*    J(3,6046561)=    */    -862,-1073,-1364,1224,-337,-10,
/*    J(5,6046561)=    */    -171,1320,780,2540,
/*    J(13,6046561)=    */    -2052,-1556,-759,-792,-1244,80,-54,-408,-1446,-898,-100,-1250,
/*    J(17,6046561)=    */    -504,81,-284,120,-1512,28,-460,-1260,-706,-522,-470,1138,-68,-460,-740,76,
/*    J(19,6046561)=    */    -544,210,-92,-232,280,-600,-560,-768,-206,-504,992,-914,-1060,-1156,-347,-348,-1348,90,
/*    J(2,6348889)=    */    2519,42,0,42,
/*    J(1,6348889)=    */    -1755,0,1808,0,
/*    J(4,6348889)=    */    -2519,-42,0,-42,
/*    J(3,6348889)=    */    -730,-836,-10,-116,458,541,50,-756,-748,-1132,-630,-482,-452,-68,-56,1272,-1540,-106,
/*    J(7,6348889)=    */    -628,-1436,1612,180,400,501,
/*    J(13,6348889)=    */    807,907,605,489,455,1089,1249,531,677,1881,-1029,-5,
/*    J(17,6348889)=    */    612,976,1140,206,509,1662,-24,384,1196,172,570,272,1262,34,368,-874,
/*    J(19,6348889)=    */    -254,1092,-142,-40,1192,782,1034,1132,444,1214,956,1026,152,750,-516,1083,1128,1050,
/*    J(2,6651217)=    */    791,-316,-784,1516,0,1516,784,-316,
/*    J(1,6651217)=    */    -1107,352,722,596,-1024,-860,-478,-1552,
/*    J(4,6651217)=    */    1927,0,1212,0,0,0,1212,0,
/*    J(3,6651217)=    */    -280,-520,-1291,1740,460,397,
/*    J(11,6651217)=    */    -2098,-616,-608,717,-92,-72,-198,328,-1046,362,
/*    J(13,6651217)=    */    1782,558,-26,542,568,-1308,262,126,1062,930,152,551,
/*    J(17,6651217)=    */    -344,-1376,-402,328,540,220,-80,322,156,-856,-232,-156,-370,-1559,786,-480,
/*    J(19,6651217)=    */    -412,-338,-964,-588,402,-760,-402,-398,-1058,-54,484,-158,-220,-1268,-1178,-828,-260,-1919,
/*    J(3,6715171)=    */    -316,-588,550,427,-324,-1038,394,-416,-142,-178,-1442,-448,301,-906,812,616,370,-502,
/*    J(5,6715171)=    */    472,1726,1354,2977,
/*    J(7,6715171)=    */    807,-1114,924,2090,464,874,
/*    J(11,6715171)=    */    -594,1254,48,417,810,756,72,1794,894,-832,
/*    J(17,6715171)=    */    -1018,-392,246,214,-614,-610,41,80,-988,-584,-1320,-1530,632,-942,242,48,
/*    J(19,6715171)=    */    -174,216,-940,-12,-1236,-272,-582,52,366,-426,942,14,-1362,-108,734,266,241,-228,
/*    J(2,6846841)=    */    -2267,924,0,924,
/*    J(1,6846841)=    */    1171,0,2340,0,
/*    J(4,6846841)=    */    -2267,924,0,924,
/*    J(3,6846841)=    */    -538,-306,549,-2694,-696,-123,
/*    J(5,6846841)=    */    96,-1956,1011,788,
/*    J(7,6846841)=    */    -1368,-480,1725,-336,-88,-1008,
/*    J(11,6846841)=    */    -1450,-576,-100,-416,-2272,-742,-1332,-1430,427,-426,
/*    J(13,6846841)=    */    1229,-791,293,-167,-413,1007,363,583,1409,953,-735,-79,
/*    J(19,6846841)=    */    1040,30,76,-292,1776,516,754,1254,-20,577,-68,-6,298,992,168,656,-28,-694,
/*    J(2,7162849)=    */    686,258,-1076,380,-1037,-380,-1076,-258,686,-316,-758,728,0,728,758,-316,
/*    J(1,7162849)=    */    749,-626,391,-201,698,-721,212,-164,-631,477,319,-730,797,-118,-1236,-1225,
/*    J(4,7162849)=    */    -901,0,0,0,1782,0,0,0,0,0,0,0,1782,0,0,0,
/*    J(3,7162849)=    */    -1120,1505,2072,-840,1120,-8,
/*    J(7,7162849)=    */    -594,-654,1578,-942,-1028,-1539,
/*    J(11,7162849)=    */    1673,1521,-605,1121,311,1295,-143,-417,25,157,
/*    J(17,7162849)=    */    1460,218,1170,1428,948,1988,1096,1742,698,404,1512,2588,950,719,866,912,
/*    J(19,7162849)=    */    -978,-16,324,86,616,-794,234,-956,-222,60,404,-194,1272,766,-662,399,874,78,
/*    J(2,7674481)=    */    0,116,-238,1154,-2207,-1154,-238,-116,
/*    J(1,7674481)=    */    -512,281,410,827,-81,-691,2404,467,
/*    J(4,7674481)=    */    -2573,0,-726,0,0,0,-726,0,
/*    J(3,7674481)=    */    -600,-84,-496,190,1138,1092,350,-148,1240,44,-886,452,-676,1564,652,-614,-696,987,
/*    J(5,7674481)=    */    87,591,2559,-713,
/*    J(11,7674481)=    */    -2148,-1011,-1956,-504,-1476,-3048,-1896,-1200,-1440,-964,
/*    J(17,7674481)=    */    -556,328,92,36,290,-270,-1049,-102,-1134,-124,-488,-256,-916,1368,910,-918,
/*    J(19,7674481)=    */    -430,-572,-178,240,-530,-212,-555,-564,-70,-88,-634,-948,-994,652,-1748,264,904,560,
/*    J(2,9767521)=    */    0,102,12,1156,-104,-1216,1256,-298,873,298,1256,1216,-104,-1156,12,-102,
/*    J(1,9767521)=    */    292,-1195,1029,-227,634,1189,611,-289,-369,-627,1215,-789,-1224,-793,1,-341,
/*    J(4,9767521)=    */    863,0,0,0,2124,0,0,0,0,0,0,0,2124,0,0,0,
/*    J(3,9767521)=    */    -840,-116,-682,384,1336,83,-656,102,-26,1154,-578,1082,690,-44,-205,-220,-832,-858,
/*    J(5,9767521)=    */    777,-1239,-391,2457,
/*    J(7,9767521)=    */    -2292,-1054,-276,809,-652,-2402,
/*    J(17,9767521)=    */    198,136,166,574,-956,1096,513,1996,-156,-1206,234,-204,-938,312,54,-154,
/*    J(19,9767521)=    */    -968,-200,4,108,1288,-220,-1077,232,-136,1092,-324,1016,-344,-176,492,582,-1182,-948,
/*    J(2,11737441)=    */    0,1096,-40,1008,800,-116,-288,-1576,927,1576,-288,116,800,-1008,-40,-1096,
/*    J(1,11737441)=    */    -868,-558,-877,-464,234,1678,109,-492,889,1672,-441,1190,-576,508,-339,-814,
/*    J(4,11737441)=    */    3167,0,0,0,-924,0,0,0,0,0,0,0,-924,0,0,0,
/*    J(3,11737441)=    */    1248,780,1176,-940,1784,580,-718,-606,980,442,-280,119,-1056,998,1380,-510,246,-860,
/*    J(5,11737441)=    */    -1897,-1312,-3028,-3964,
/*    J(11,11737441)=    */    1570,1000,174,1923,1526,3756,1118,448,1154,1850,
/*    J(13,11737441)=    */    -1310,-2838,-432,-2088,-748,-1908,-1154,-1786,-2472,-2356,-3245,-1972,
/*    J(19,11737441)=    */    366,570,-1386,-70,-2306,-166,-564,-1632,-84,334,-1282,-1788,-1081,-296,-510,-466,-456,-508,
/*    J(3,12471031)=    */    -850,254,-230,-940,-234,864,-952,-266,176,-483,-1532,1058,-1184,-696,-606,-2132,206,1042,
/*    J(5,12471031)=    */    2529,1490,4260,2730,
/*    J(11,12471031)=    */    1758,2854,2051,480,3200,2166,3536,1766,1348,1894,
/*    J(13,12471031)=    */    -2337,-2242,-1358,-3124,-2692,-3914,-1408,-2578,-2272,-2690,-3214,-2124,
/*    J(17,12471031)=    */    33,1454,266,536,726,3356,186,664,890,264,1974,1604,918,780,486,176,
/*    J(19,12471031)=    */    273,785,-5,-1325,1215,-183,-181,805,1521,903,1279,1053,-797,661,-71,1901,735,75,
/*    J(2,12697777)=    */    -425,2140,824,1000,0,1000,-824,2140,
/*    J(1,12697777)=    */    -1675,982,-890,-1078,-40,2138,1470,-490,
/*    J(4,12697777)=    */    343,0,2508,0,0,0,2508,0,
/*    J(3,12697777)=    */    286,-804,-1546,-1440,486,934,-898,-40,-230,456,742,-6,-322,590,1292,-2148,1189,-270,
/*    J(7,12697777)=    */    1098,1806,204,3630,759,-274,
/*    J(13,12697777)=    */    -610,1729,2130,1484,1102,1348,2728,-228,2304,1512,1164,962,
/*    J(17,12697777)=    */    1370,-482,2066,2226,1714,-217,1318,1330,266,550,-674,1044,658,888,1196,74,
/*    J(19,12697777)=    */    1736,342,1348,122,1602,76,-2003,320,638,974,792,642,252,1496,644,578,1106,772,
/*    J(2,17907121)=    */    -2263,-2372,-184,-856,0,-856,184,-2372,
/*    J(1,17907121)=    */    1943,742,1160,-1774,2036,1474,-1488,-746,
/*    J(4,17907121)=    */    1927,0,-2664,0,0,0,-2664,0,
/*    J(3,17907121)=    */    3098,-878,1135,3726,1376,1631,
/*    J(5,17907121)=    */    792,-2184,2384,2337,
/*    J(7,17907121)=    */    -534,1720,-1281,2584,2902,404,
/*    J(11,17907121)=    */    -890,-292,1412,1754,1308,172,3218,-201,1950,2128,
/*    J(17,17907121)=    */    -460,-1770,1988,1230,222,450,-534,1682,-854,-504,-1230,-496,-276,986,-481,-838,
/*    J(19,17907121)=    */    -2560,-1820,1218,-216,-1688,-312,-1801,-896,-1368,-468,14,404,368,-584,146,-1136,400,-1102,
/*    J(2,24942061)=    */    -45,4994,
/*    J(3,24942061)=    */    -1518,1826,2496,1412,-56,22,836,2340,-1052,-752,1526,608,-828,538,-340,-1134,2925,-630,
/*    J(5,24942061)=    */    267,681,849,5387,
/*    J(11,24942061)=    */    2817,-968,2436,3610,1586,-910,414,-46,622,1240,
/*    J(13,24942061)=    */    -1060,1116,-1080,2878,1518,1520,2364,1531,1080,-1196,1934,2238,
/*    J(17,24942061)=    */    2509,324,-32,716,1358,-1856,-1252,-572,248,1472,1760,-1256,1212,-314,1800,308,
/*    J(19,24942061)=    */    -478,-92,-340,-1598,2424,-1764,-304,1176,1306,-1000,-2020,-1690,-1604,-398,-514,-93,-1122,-554,
/*    J(2,27387361)=    */    -134,-1430,124,-40,845,40,124,1430,-134,-2816,-1458,-1096,0,-1096,1458,-2816,
/*    J(1,27387361)=    */    -181,2578,1049,395,-464,55,-436,-400,1871,-1811,-141,-1042,1413,1550,-1506,-2081,
/*    J(4,27387361)=    */    827,0,0,0,3654,0,0,0,0,0,0,0,3654,0,0,0,
/*    J(3,27387361)=    */    -3736,3729,696,-2232,3537,2136,
/*    J(5,27387361)=    */    5544,5725,4320,4680,
/*    J(7,27387361)=    */    -4258,1489,406,1044,634,1230,
/*    J(11,27387361)=    */    -2697,316,-1774,280,-2248,-4702,-2898,-236,-2056,-1938,
/*    J(13,27387361)=    */    1750,682,4200,1152,456,2536,1634,2881,2490,-492,-880,2024,
/*    J(19,27387361)=    */    -570,-2992,545,-188,-566,-2160,378,490,476,154,-1164,-1946,-3298,288,-1882,-812,-862,-1662,
/*    J(2,31744441)=    */    5543,-714,0,-714,
/*    J(1,31744441)=    */    5525,0,-1104,0,
/*    J(4,31744441)=    */    -5543,714,0,714,
/*    J(3,31744441)=    */    556,-480,-652,1544,-850,-412,1500,2560,-28,1312,-470,-516,2188,-2758,708,1242,4551,1242,
/*    J(5,31744441)=    */    -814,-6226,-2524,-4147,
/*    J(7,31744441)=    */    1320,-1152,-2640,-3768,-16,-4371,
/*    J(13,31744441)=    */    -111,764,1162,-1854,-1374,-2858,2214,-1170,-50,2546,794,-714,
/*    J(17,31744441)=    */    -3094,-2100,-2496,-1218,-756,-190,-3610,620,-674,-1780,-412,-1378,620,-1602,-7,1382,
/*    J(19,31744441)=    */    1108,2700,-660,-1376,-654,844,444,-1714,320,-396,1956,972,1988,-848,1146,-423,342,2724,
/*    J(2,35814241)=    */    0,1452,70,-638,1216,3352,-1234,1034,-401,-1034,-1234,-3352,1216,638,70,-1452,
/*    J(1,35814241)=    */    806,2104,1157,427,520,-349,2273,-1262,-1307,8,-1351,1907,1406,831,-3489,234,
/*    J(4,35814241)=    */    5903,0,0,0,696,0,0,0,0,0,0,0,696,0,0,0,
/*    J(3,35814241)=    */    -4,-4788,-2385,-1968,108,-3249,
/*    J(5,35814241)=    */    1960,-5048,-1232,679,
/*    J(7,35814241)=    */    -874,-1207,542,-3532,3158,2402,
/*    J(11,35814241)=    */    1944,2240,1596,2531,-272,3856,5448,3736,470,3794,
/*    J(17,35814241)=    */    -1690,-1782,-2416,-3330,978,-1610,152,-2007,-1312,-1350,-3806,-2538,-1916,-2254,-4798,-412,
/*    J(19,35814241)=    */    -2232,1238,-213,-1874,-564,1500,616,-1176,-1046,42,-1114,-990,558,290,-1238,-610,-1252,-4552,
/*    J(2,41081041)=    */    0,-1246,98,-2716,-4817,2716,98,1246,
/*    J(1,41081041)=    */    2680,1127,4724,-2149,-1599,1673,-278,-511,
/*    J(4,41081041)=    */    2627,0,-4134,0,0,0,-4134,0,
/*    J(3,41081041)=    */    -2246,1618,932,-476,416,713,-790,432,-3284,-1128,-1854,-1220,-1424,-674,3009,780,-1504,-2988,
/*    J(5,41081041)=    */    4740,4062,6763,7194,
/*    J(7,41081041)=    */    320,-688,544,-3024,4464,2849,
/*    J(11,41081041)=    */    1006,149,6986,2204,1960,3942,3100,2764,3042,2324,
/*    J(13,41081041)=    */    184,3270,2980,2248,-2938,2584,325,544,1186,-1188,-784,818,
/*    J(19,41081041)=    */    3824,726,995,4898,1348,692,578,2870,1640,4602,2602,1924,704,1226,2134,1184,-466,1046,
/*    J(2,46558513)=    */    1513,-168,-3528,-3108,0,-3108,3528,-168,
/*    J(1,46558513)=    */    279,1470,-1296,4662,3196,-2394,72,-2226,
/*    J(4,46558513)=    */    -5689,0,-2664,0,0,0,-2664,0,
/*    J(3,46558513)=    */    -43,5176,-1898,2781,5306,1970,
/*    J(7,46558513)=    */    4214,170,3344,7544,2447,2300,
/*    J(11,46558513)=    */    2114,-2106,1705,3898,2426,-2024,420,-2184,1208,-904,
/*    J(13,46558513)=    */    580,-155,-2156,664,-2850,2086,-1442,-2368,-180,-1906,-5038,-2498,
/*    J(17,46558513)=    */    -3842,-2846,-1312,-2971,-288,-2666,-1978,-860,1020,-2524,-1198,-972,-4252,-5530,-1128,-2484,
/*    J(19,46558513)=    */    1534,1772,2228,-2394,-1008,-76,3024,1046,1708,3670,1152,349,3872,-694,1920,1238,1360,844,
/*    J(2,53721361)=    */    -2231,3180,-3600,-1140,0,-1140,3600,3180,
/*    J(1,53721361)=    */    -1585,1860,-744,-4320,-3228,0,1704,-3900,
/*    J(4,53721361)=    */    4247,0,-4224,0,0,0,-4224,0,
/*    J(3,53721361)=    */    1484,834,2750,2916,-986,-2162,-396,-3340,3339,-1164,84,3502,3030,-2152,-2686,-426,-2768,624,
/*    J(5,53721361)=    */    -2244,1359,-2604,-7232,
/*    J(7,53721361)=    */    4740,4967,2208,8928,5370,4278,
/*    J(11,53721361)=    */    4550,7221,6018,2260,6404,3956,6664,3160,2508,4228,
/*    J(17,53721361)=    */    -87,2284,-888,4148,1006,-380,-936,3718,1166,-626,2170,1576,4356,3426,2184,2144,
/*    J(19,53721361)=    */    1762,108,-1224,-1268,-1492,-1886,-3188,62,-2470,-1012,-1522,-2444,-3096,-2100,-1940,-2626,-636,-6303,
/*    J(2,107442721)=    */    0,-2798,-3548,-1238,-1288,-406,-604,5070,-2785,-5070,-604,406,-1288,1238,-3548,2798,
/*    J(1,107442721)=    */    2624,298,4231,6052,-2766,-2240,-235,760,-143,-988,-41,-3480,3064,-1740,1837,-1934,
/*    J(4,107442721)=    */    -1153,0,0,0,7284,0,0,0,0,0,0,0,7284,0,0,0,
/*    J(3,107442721)=    */    564,-86,592,-4076,-3242,-3428,620,-423,-2532,-58,-308,3120,-3256,214,-3714,6604,1649,-4796,
/*    J(5,107442721)=    */    -3036,3750,-2365,-8910,
/*    J(7,107442721)=    */    -672,-5346,7098,3128,1284,3621,
/*    J(11,107442721)=    */    1914,10938,4331,7342,6866,5504,2452,8148,4036,4040,
/*    J(17,107442721)=    */    -1138,-408,1389,-1400,4538,-4880,-3308,1586,-210,-188,-1908,-212,-4784,-4022,-3438,1008,
/*    J(19,107442721)=    */    -846,1992,1360,-300,5412,3108,7264,3496,-1152,3160,3908,988,5012,1196,1792,-130,5528,1721,
/*    J(2,174594421)=    */    -12889,2910,
/*    J(3,174594421)=    */    672,-2204,-3862,2238,6496,-992,-2450,-787,236,346,-610,-488,2474,-2118,5466,-548,5481,3966,
/*    J(5,174594421)=    */    -10420,-1528,5828,-3811,
/*    J(7,174594421)=    */    -2382,4002,-6706,-510,1320,9189,
/*    J(11,174594421)=    */    -7844,-3054,-3050,-3082,266,7638,-5002,-248,-2024,1395,
/*    J(13,174594421)=    */    -4312,-1138,-1682,-5472,-64,-4568,-392,-6250,-362,-2956,-10596,-10231,
/*    J(17,174594421)=    */    322,-50,10050,5808,3160,4510,2500,3164,10894,1472,5652,5582,6370,4130,900,4521,
/*    J(19,174594421)=    */    1190,6970,890,1704,3664,-808,-634,2290,762,626,956,2752,-3820,86,318,3998,-7464,-1815,
/*    J(2,232792561)=    */    0,4736,-2162,6586,9583,-6586,-2162,-4736,
/*    J(1,232792561)=    */    2960,5191,10508,-6283,-1591,-1301,-5994,2653,
/*    J(4,232792561)=    */    -14093,0,4134,0,0,0,4134,0,
/*    J(3,232792561)=    */    6614,11081,3692,-2706,12352,-2042,
/*    J(5,232792561)=    */    13371,12489,-1339,6183,
/*    J(7,232792561)=    */    5214,15274,1259,10454,5814,-76,
/*    J(11,232792561)=    */    9096,242,-4100,656,473,-408,7492,9438,-1712,5068,
/*    J(13,232792561)=    */    8628,5484,6656,9348,1904,-1786,8512,9964,7017,10540,8216,12330,
/*    J(17,232792561)=    */    -3690,-690,-164,-6084,4619,-6124,-3596,-1214,-390,-10504,-8204,-2868,-400,940,-212,-3240,
/*    J(19,232792561)=    */    132,3984,-1176,-4122,5148,5002,6500,3022,3122,-114,9921,1466,5298,5026,192,6074,5808,7530,
/*a is 8 t is 6983776800    */
/*    J(2,1901)=    */    35,-26,
/*    J(5,1901)=    */    26,2,10,-3,0,-8,0,-4,-9,8,4,-2,-8,7,8,-18,-6,0,-3,10,
/*    J(19,1901)=    */    2,-10,8,-2,6,-26,-4,-8,-24,-8,2,2,10,-16,-10,2,4,-5,
/*    J(3,2851)=    */    14,-45,
/*    J(5,2851)=    */    14,-6,-20,1,20,-8,10,-20,13,12,4,6,4,-1,4,24,2,-20,-17,-8,
/*    J(19,2851)=    */    -4,-28,-2,-14,0,-22,6,8,-26,-8,-28,-34,-16,-4,-14,-14,-22,-7,
/*    J(2,5701)=    */    15,-74,
/*    J(3,5701)=    */    -76,-75,
/*    J(5,5701)=    */    -16,-36,22,-17,-34,-2,-26,2,5,-22,2,-30,6,-43,-26,-10,14,14,-9,-10,
/*    J(19,5701)=    */    4,-40,-12,20,-32,4,-2,4,-32,-26,-48,-12,-8,-16,-8,-16,-12,3,
/*    J(2,39901)=    */    155,126,
/*    J(3,39901)=    */    20,-189,
/*    J(5,39901)=    */    70,2,126,127,-46,10,20,92,105,-62,2,48,22,113,-84,62,-20,50,95,-38,
/*    J(7,39901)=    */    142,90,78,246,96,75,
/*    J(19,39901)=    */    -14,-98,32,-20,-20,-42,22,-102,-62,-82,-18,68,-4,-40,10,10,32,-53,
/*    J(2,41801)=    */    -51,-140,0,-140,
/*    J(1,41801)=    */    -149,0,140,0,
/*    J(4,41801)=    */    -51,-140,0,-140,
/*    J(5,41801)=    */    -106,-18,13,-56,-22,-108,22,-8,2,0,-92,70,8,42,30,-50,-16,8,-124,44,
/*    J(11,41801)=    */    128,212,128,98,181,174,140,204,72,136,
/*    J(19,41801)=    */    -116,-79,-88,96,-16,-8,20,-66,-16,4,-14,-34,-4,-44,2,-20,-72,-40,
/*    J(2,53201)=    */    -49,70,-68,-126,0,-126,68,70,
/*    J(1,53201)=    */    -63,-96,-42,-74,112,-106,-70,64,
/*    J(4,53201)=    */    -201,0,-80,0,0,0,-80,0,
/*    J(5,53201)=    */    14,42,-14,30,-16,24,57,-90,38,152,68,86,-112,58,-34,18,36,-82,-18,52,
/*    J(7,53201)=    */    50,-3,134,-124,-96,24,
/*    J(19,53201)=    */    -84,-28,-20,26,-38,-2,14,12,104,28,48,-102,32,72,-35,20,88,-22,
/*    J(2,62701)=    */    99,230,
/*    J(3,62701)=    */    284,189,
/*    J(5,62701)=    */    -174,90,2,-41,-134,-130,88,-26,-1,-76,-122,72,44,-93,-94,-130,24,58,27,-10,
/*    J(11,62701)=    */    -64,-116,86,-140,-88,-58,60,60,4,-53,
/*    J(19,62701)=    */    -54,-18,-68,-40,-158,-70,-58,-96,-196,-170,30,-106,-24,-52,-110,-70,-76,-109,
/*    J(2,64601)=    */    -201,-110,0,-110,
/*    J(1,64601)=    */    -251,0,-40,0,
/*    J(4,64601)=    */    201,110,0,110,
/*    J(5,64601)=    */    122,44,-8,62,-10,28,-20,50,92,-8,4,40,28,188,-68,130,55,38,180,-8,
/*    J(17,64601)=    */    -124,-42,-38,-56,-45,-44,-42,42,-104,32,-24,-120,-214,-58,-24,-92,
/*    J(19,64601)=    */    62,191,70,12,60,46,-92,74,-26,72,78,34,84,22,102,80,98,20,
/*    J(2,74101)=    */    -249,110,
/*    J(3,74101)=    */    116,-195,
/*    J(5,74101)=    */    92,-20,-4,-103,48,68,-20,6,-19,68,-46,-126,-56,17,94,0,60,12,-43,176,
/*    J(13,74101)=    */    148,170,168,78,142,106,-58,116,-18,136,160,73,
/*    J(19,74101)=    */    -84,14,-14,12,-88,-108,-28,32,-4,-116,-8,-48,-200,-86,-66,-88,52,-85,
/*    J(2,79801)=    */    103,186,0,186,
/*    J(1,79801)=    */    149,0,240,0,
/*    J(4,79801)=    */    -103,-186,0,-186,
/*    J(3,79801)=    */    245,309,
/*    J(5,79801)=    */    -24,-60,-24,-56,-195,-72,-66,100,-70,-95,-118,-12,-40,-102,9,-82,-88,42,-120,-43,
/*    J(7,79801)=    */    308,268,196,90,115,198,
/*    J(19,79801)=    */    148,84,172,92,20,52,32,144,184,128,186,86,113,242,90,48,92,176,
/*    J(2,98801)=    */    0,-136,142,-66,-113,66,142,136,
/*    J(1,98801)=    */    24,-47,48,-221,-175,93,-46,59,
/*    J(4,98801)=    */    -237,0,-146,0,0,0,-146,0,
/*    J(5,98801)=    */    -24,-118,-41,-122,216,-20,-144,4,32,100,36,-12,-18,32,60,32,-128,-54,-68,-24,
/*    J(13,98801)=    */    -12,-100,-105,-24,-84,100,-104,-126,-226,-94,-50,-216,
/*    J(19,98801)=    */    -68,-104,-217,-104,-84,-200,-204,-98,-146,-200,36,-148,-16,-92,-84,-150,-182,-68,
/*    J(5,113051)=    */    79,84,-132,-72,-116,69,44,-22,-2,-54,-13,-26,0,20,-310,69,-22,-76,-36,-150,
/*    J(7,113051)=    */    -65,54,-38,178,100,-216,
/*    J(17,113051)=    */    -35,99,131,35,-33,-31,-137,29,-61,-7,-9,117,-79,13,173,-53,
/*    J(19,113051)=    */    56,-24,-22,88,-82,8,-58,-102,5,-188,-72,-70,112,-46,-152,-114,-6,-94,
/*    J(2,119701)=    */    -345,-26,
/*    J(3,119701)=    */    104,179,254,-66,247,148,
/*    J(5,119701)=    */    -76,-76,88,85,116,-68,40,-96,-21,32,18,-134,-172,75,28,-100,8,-144,85,76,
/*    J(7,119701)=    */    -52,-228,-288,-240,12,-255,
/*    J(19,119701)=    */    -94,-38,-248,-152,-158,-204,-180,-114,-90,16,-188,-186,-124,-72,-150,34,-96,-199,
/*    J(5,135851)=    */    -102,-166,-72,57,-240,-76,-68,-176,-23,-106,-72,-58,-234,83,-92,14,-166,-184,-41,-84,
/*    J(11,135851)=    */    164,294,160,244,222,210,316,142,374,51,
/*    J(13,135851)=    */    -66,84,18,286,130,176,28,164,210,60,62,225,
/*    J(19,135851)=    */    8,-124,122,-122,-124,140,20,-4,-146,-134,-94,-26,-52,-20,-38,26,-10,83,
/*    J(2,148201)=    */    -49,-270,0,-270,
/*    J(1,148201)=    */    301,0,-240,0,
/*    J(4,148201)=    */    49,270,0,270,
/*    J(3,148201)=    */    -280,159,
/*    J(5,148201)=    */    -170,-84,-8,-76,112,-92,-20,-90,-4,24,40,-112,-88,-14,10,-144,-195,-76,-54,-190,
/*    J(13,148201)=    */    70,-111,-150,-36,-204,-106,-26,-20,44,126,-238,0,
/*    J(19,148201)=    */    172,176,87,4,-52,80,198,-114,172,112,120,134,214,140,104,132,10,58,
/*    J(2,205201)=    */    0,284,98,-106,-47,106,98,-284,
/*    J(1,205201)=    */    -240,-173,134,-191,7,-169,-136,-127,
/*    J(4,205201)=    */    451,0,-30,0,0,0,-30,0,
/*    J(3,205201)=    */    -74,-174,94,46,68,-173,-6,60,122,-18,-2,164,178,106,-176,-166,182,-88,
/*    J(5,205201)=    */    212,130,94,-48,10,54,230,12,-112,174,72,364,108,-77,-54,186,290,44,-136,66,
/*    J(19,205201)=    */    49,47,91,9,-103,-145,-1,91,75,-89,-121,-49,-25,-187,-59,155,-173,149,
/*    J(3,219451)=    */    230,-309,
/*    J(5,219451)=    */    164,48,66,57,-2,140,184,50,177,-22,106,18,94,-21,318,264,-44,82,55,70,
/*    J(7,219451)=    */    -110,-498,-296,-284,-46,-321,
/*    J(11,219451)=    */    -62,68,42,-254,124,68,322,56,-42,117,
/*    J(19,219451)=    */    -8,-208,-194,-8,-292,-34,-228,-216,-86,-74,-80,-46,-290,-212,-86,86,-118,-35,
/*    J(2,290701)=    */    99,-530,
/*    J(3,290701)=    */    -34,-69,-420,168,87,114,
/*    J(5,290701)=    */    -14,234,102,101,-116,-122,158,-194,57,112,-70,144,-88,213,134,-40,352,-184,233,162,
/*    J(17,290701)=    */    -348,-204,-328,-198,-348,-334,-120,-316,-506,-144,-334,-364,-372,-456,-204,-151,
/*    J(19,290701)=    */    124,54,-32,80,-38,-164,28,84,-16,240,316,-74,108,-60,270,16,-28,-35,
/*    J(2,292601)=    */    -537,-46,0,-46,
/*    J(1,292601)=    */    149,0,520,0,
/*    J(4,292601)=    */    537,46,0,46,
/*    J(5,292601)=    */    124,108,126,-6,84,98,73,72,150,68,272,-260,114,234,70,210,136,102,-92,-14,
/*    J(7,292601)=    */    224,72,44,-406,135,126,
/*    J(11,292601)=    */    449,284,182,50,220,162,218,-144,198,-14,
/*    J(19,292601)=    */    -72,318,240,174,72,-40,-20,70,-98,52,144,36,-70,208,-150,112,84,155,
/*    J(2,319201)=    */    -311,-50,152,20,152,-224,84,-70,0,-70,-84,-224,-152,20,-152,-50,
/*    J(1,319201)=    */    -53,281,10,109,270,-1,-138,161,24,85,-280,-73,-58,109,28,5,
/*    J(4,319201)=    */    527,0,0,0,144,0,0,0,0,0,0,0,144,0,0,0,
/*    J(3,319201)=    */    -496,-615,
/*    J(5,319201)=    */    -278,202,172,-103,12,-228,102,142,137,-212,-198,-48,200,-95,-330,-332,34,176,41,-60,
/*    J(7,319201)=    */    -306,-372,-112,-654,-432,-351,
/*    J(19,319201)=    */    28,132,-66,128,56,78,-145,-70,60,108,-158,-12,108,96,158,404,-116,46,
/*    J(3,333451)=    */    -126,310,-10,148,-96,-100,-250,-136,-76,-139,52,20,278,-88,-340,-32,-78,-58,
/*    J(5,333451)=    */    234,68,122,102,404,118,-28,-204,152,254,99,-62,-244,124,56,88,112,-154,122,276,
/*    J(13,333451)=    */    148,-200,42,196,-98,-114,134,-62,-104,115,134,380,
/*    J(19,333451)=    */    -259,-263,-249,-425,-135,109,-323,-247,-227,-221,-369,-365,-301,-329,-239,-403,-133,-277,
/*    J(3,339151)=    */    419,-246,
/*    J(5,339151)=    */    364,-219,66,-70,14,66,-322,-14,6,-38,124,-350,-2,-146,-36,200,-168,-138,8,-316,
/*    J(7,339151)=    */    -58,-408,8,-457,-434,-402,
/*    J(17,339151)=    */    38,28,390,152,20,22,-150,-116,200,-126,98,136,2,44,-2,283,
/*    J(19,339151)=    */    112,-64,-18,188,176,172,330,38,211,32,228,110,146,470,156,-10,62,282,
/*    J(2,359101)=    */    555,-226,
/*    J(3,359101)=    */    76,-94,-104,-188,146,188,-116,-85,124,150,-350,50,-290,92,14,-200,-177,-152,
/*    J(5,359101)=    */    -276,-184,258,61,224,-180,-282,164,91,22,-48,-404,290,187,232,-142,-192,384,45,134,
/*    J(7,359101)=    */    -390,-414,78,-342,40,33,
/*    J(19,359101)=    */    -206,6,-108,46,180,28,-208,-272,-120,124,-288,-28,-10,-268,-82,-190,-132,-221,
/*    J(2,410401)=    */    447,230,-14,62,-88,196,30,36,0,36,-30,196,88,62,14,230,
/*    J(1,410401)=    */    -485,-17,92,46,198,-100,110,-13,232,49,36,-138,42,156,0,13,
/*    J(4,410401)=    */    127,0,0,0,-444,0,0,0,0,0,0,0,-444,0,0,0,
/*    J(3,410401)=    */    40,-164,-172,-176,-112,28,108,54,-36,-78,-236,-160,-293,-32,-296,374,124,-198,
/*    J(5,410401)=    */    8,252,-268,236,-20,-84,-72,-50,394,-118,-206,-52,-378,185,-42,-144,54,10,206,98,
/*    J(19,410401)=    */    -22,-156,-160,298,-148,146,14,-246,-222,-18,-104,-109,-272,-118,26,162,-94,34,
/*    J(2,452201)=    */    -147,464,0,464,
/*    J(1,452201)=    */    475,0,-476,0,
/*    J(4,452201)=    */    -147,464,0,464,
/*    J(5,452201)=    */    -164,236,-242,50,182,-310,224,198,92,76,-343,72,-248,-146,-4,-114,170,-294,-10,-36,
/*    J(7,452201)=    */    516,40,232,348,-224,-45,
/*    J(17,452201)=    */    -307,151,-55,173,21,9,-13,9,121,127,65,-139,161,293,-197,-267,
/*    J(19,452201)=    */    465,-77,1,-91,-131,-243,167,35,-35,109,15,3,-19,-103,241,149,63,1,
/*    J(2,478801)=    */    -113,334,284,202,0,202,-284,334,
/*    J(1,478801)=    */    161,184,310,-314,168,166,122,-392,
/*    J(4,478801)=    */    503,0,-336,0,0,0,-336,0,
/*    J(3,478801)=    */    440,-170,-200,570,-430,-301,
/*    J(5,478801)=    */    -20,162,-4,-219,-84,72,118,12,-56,-332,-314,314,-198,-56,-292,66,368,164,-128,-14,
/*    J(7,478801)=    */    261,-36,18,-330,406,366,
/*    J(19,478801)=    */    438,470,189,266,22,328,212,186,248,2,320,56,284,168,22,-16,258,-72,
/*    J(2,501601)=    */    -289,-134,266,158,288,-8,110,16,0,16,-110,-8,-288,158,-266,-134,
/*    J(1,501601)=    */    235,103,-38,-388,102,4,-260,205,-108,79,62,190,66,242,194,73,
/*    J(4,501601)=    */    -673,0,0,0,-156,0,0,0,0,0,0,0,-156,0,0,0,
/*    J(3,501601)=    */    455,816,
/*    J(5,501601)=    */    -38,-390,268,-156,132,10,-313,-30,-26,114,40,-34,342,-10,30,192,-4,-172,-146,-20,
/*    J(11,501601)=    */    758,618,344,372,222,166,220,446,389,490,
/*    J(19,501601)=    */    -52,284,18,394,161,-58,266,-88,-120,160,260,84,-146,-52,148,-46,52,-164,
/*    J(3,532951)=    */    221,-594,
/*    J(5,532951)=    */    -82,22,296,254,427,-28,-98,204,280,-58,174,206,-26,238,96,250,-30,96,28,210,
/*    J(11,532951)=    */    74,-142,-155,-80,416,-114,358,114,-284,-144,
/*    J(17,532951)=    */    150,60,165,266,256,134,432,370,610,66,-34,496,148,328,276,322,
/*    J(19,532951)=    */    -290,-232,70,-2,-30,-204,-256,-332,106,-200,-158,82,138,-40,-87,6,228,-320,
/*    J(2,564301)=    */    -701,270,
/*    J(3,564301)=    */    54,82,-56,44,-46,-26,-56,-7,232,364,206,-106,-42,248,114,-236,285,-260,
/*    J(5,564301)=    */    -246,142,-372,-207,-150,18,212,124,-379,-86,-114,-8,70,-263,-180,126,232,34,-187,188,
/*    J(11,564301)=    */    402,-174,112,338,478,136,290,258,632,343,
/*    J(19,564301)=    */    -172,-226,-486,-94,-326,-504,-300,-62,-330,-172,-152,-130,-38,74,-410,-144,-512,-235,
/*    J(3,658351)=    */    -754,155,-244,-576,-191,118,
/*    J(5,658351)=    */    540,-242,-14,-262,-32,252,-116,154,-416,132,544,-168,-163,-478,-164,428,-258,-180,-186,-112,
/*    J(7,658351)=    */    -30,236,-630,-48,123,306,
/*    J(11,658351)=    */    336,570,46,188,132,-22,-26,628,317,514,
/*    J(19,658351)=    */    346,32,498,606,378,364,276,444,72,156,375,518,174,330,202,-30,216,476,
/*    J(2,666901)=    */    -665,474,
/*    J(3,666901)=    */    371,-72,180,-44,-50,-210,-292,120,-252,164,178,-112,-320,88,-18,60,-300,-218,
/*    J(5,666901)=    */    338,122,-195,124,-154,200,50,-120,398,-280,454,154,202,136,-420,472,270,-150,240,-462,
/*    J(13,666901)=    */    398,292,-349,-156,182,220,126,200,126,318,24,490,
/*    J(19,666901)=    */    -23,-156,-106,-172,-368,-6,234,-134,-20,-156,-140,296,-402,-238,-338,-136,-400,-130,
/*    J(3,778051)=    */    392,-686,-392,168,-211,224,
/*    J(5,778051)=    */    -75,-504,326,-498,-160,-82,-354,-4,-160,202,108,-482,-218,-260,-6,-2,-222,142,42,-94,
/*    J(7,778051)=    */    -810,-981,-900,-810,-710,-900,
/*    J(13,778051)=    */    307,106,206,68,350,346,48,482,-52,612,758,356,
/*    J(19,778051)=    */    282,-165,128,358,454,-208,172,-44,88,330,298,400,20,398,482,170,198,96,
/*    J(2,839801)=    */    693,-424,0,-424,
/*    J(1,839801)=    */    851,0,340,0,
/*    J(4,839801)=    */    693,-424,0,-424,
/*    J(5,839801)=    */    -242,-26,-316,216,192,-78,394,-442,170,264,408,142,-200,188,179,-354,134,-24,98,256,
/*    J(13,839801)=    */    482,-140,-288,288,-226,106,40,516,324,214,-12,-31,
/*    J(17,839801)=    */    -180,-34,-34,-22,22,34,174,-352,-368,22,32,-100,-123,280,-112,590,
/*    J(19,839801)=    */    316,638,310,30,118,-214,170,-22,502,10,-50,348,180,264,122,149,218,444,
/*    J(2,957601)=    */    202,-198,16,268,0,268,-16,-198,-202,106,510,146,261,-146,510,-106,
/*    J(1,957601)=    */    127,-138,232,202,-429,14,141,28,-183,-511,-332,87,-350,281,3,95,
/*    J(4,957601)=    */    -901,0,0,0,270,0,0,0,0,0,0,0,270,0,0,0,
/*    J(3,957601)=    */    944,-177,-456,648,-369,-240,
/*    J(5,957601)=    */    818,56,-184,-20,-8,505,130,116,-172,-118,598,-246,188,92,-198,628,-380,160,-120,-306,
/*    J(7,957601)=    */    36,-654,-582,-125,-858,-30,
/*    J(19,957601)=    */    -316,26,-582,-326,206,-157,-234,-254,-478,-118,-392,-292,54,130,90,-370,74,-102,
/*    J(2,1037401)=    */    887,354,0,354,
/*    J(1,1037401)=    */    805,0,624,0,
/*    J(4,1037401)=    */    -887,-354,0,-354,
/*    J(3,1037401)=    */    -601,-1176,
/*    J(5,1037401)=    */    22,-246,-572,108,-140,520,-100,-538,-234,302,-14,-342,-382,-195,-154,-154,-182,-398,-374,82,
/*    J(7,1037401)=    */    240,-911,-460,-520,-90,-430,
/*    J(13,1037401)=    */    -575,-608,-722,-388,134,-368,-380,28,-204,-562,36,-422,
/*    J(19,1037401)=    */    416,216,102,-110,576,404,62,302,12,-292,38,-62,152,16,220,382,-305,150,
/*    J(2,1065901)=    */    115,1026,
/*    J(3,1065901)=    */    -805,-1164,
/*    J(5,1065901)=    */    234,-68,218,-164,232,402,98,-46,-274,-18,-118,-108,696,-464,5,86,74,94,-400,-330,
/*    J(11,1065901)=    */    -242,279,-502,-176,-404,-656,-382,354,-124,-304,
/*    J(17,1065901)=    */    184,290,-208,-60,130,38,280,284,-551,112,-396,-254,-158,216,120,-198,
/*    J(19,1065901)=    */    -86,-324,-10,58,-520,322,164,-64,-421,-164,-240,-242,-164,-422,-222,-388,274,16,
/*    J(2,1128601)=    */    -937,354,0,354,
/*    J(1,1128601)=    */    -699,0,-800,0,
/*    J(4,1128601)=    */    937,-354,0,-354,
/*    J(3,1128601)=    */    -198,-370,434,130,-24,166,-90,126,502,94,-4,298,-68,-543,418,-184,14,-72,
/*    J(5,1128601)=    */    -56,-112,-176,72,196,-400,-12,-468,592,180,-392,-130,-48,136,224,-253,-444,56,656,398,
/*    J(11,1128601)=    */    563,0,-130,314,964,300,512,424,22,418,
/*    J(19,1128601)=    */    -2,1,-350,-292,-22,-236,-514,2,410,-390,64,-572,-114,42,74,126,84,-22,
/*    J(3,1222651)=    */    -454,328,-116,-147,44,920,
/*    J(5,1222651)=    */    162,-580,-84,-460,-476,178,278,-54,-466,10,119,-222,98,-550,114,100,-152,308,-272,188,
/*    J(11,1222651)=    */    654,-158,-174,-536,-214,128,-580,72,-122,-193,
/*    J(13,1222651)=    */    -80,-26,-219,100,-566,58,-426,-194,-40,-622,-892,-32,
/*    J(19,1222651)=    */    -378,120,572,26,-12,197,330,16,430,-254,92,-170,324,346,-76,22,-276,-94,
/*    J(2,1259701)=    */    455,1026,
/*    J(3,1259701)=    */    -325,924,
/*    J(5,1259701)=    */    -162,68,-30,416,406,266,-106,-106,-298,748,22,-498,-30,-101,36,-92,-228,118,-154,424,
/*    J(13,1259701)=    */    -230,-638,-1144,-286,-396,-700,22,-525,-150,-420,-336,-398,
/*    J(17,1259701)=    */    606,266,210,48,847,492,890,558,90,592,578,754,540,404,738,478,
/*    J(19,1259701)=    */    665,35,345,-197,-155,-293,-151,105,-289,-89,-127,177,-77,113,-201,213,147,-431,
/*    J(2,1504801)=    */    0,140,-32,458,456,146,-452,-212,-295,212,-452,-146,456,-458,-32,-140,
/*    J(1,1504801)=    */    112,517,-171,-371,-254,269,-313,-383,-5,83,-441,167,-128,155,33,-637,
/*    J(4,1504801)=    */    1151,0,0,0,300,0,0,0,0,0,0,0,300,0,0,0,
/*    J(3,1504801)=    */    374,238,904,204,272,1301,
/*    J(5,1504801)=    */    -316,24,424,-50,-572,-252,-264,-60,208,-232,-308,25,-144,56,-816,0,-448,-336,12,58,
/*    J(11,1504801)=    */    518,295,-278,-732,288,-548,12,124,-148,-16,
/*    J(19,1504801)=    */    -68,-208,-336,182,-114,-402,-72,-44,-92,-394,-263,274,-736,-180,-608,-570,-334,278,
/*    J(2,1808801)=    */    -230,-516,638,-132,0,-132,-638,-516,230,-38,148,-292,323,292,148,38,
/*    J(1,1808801)=    */    -241,139,550,266,315,262,-39,-171,-147,20,-648,607,-296,-457,-109,202,
/*    J(4,1808801)=    */    987,0,0,0,-646,0,0,0,0,0,0,0,-646,0,0,0,
/*    J(5,1808801)=    */    18,129,-74,-192,-1014,156,271,392,-294,-494,436,307,372,-22,-384,614,-307,450,8,-768,
/*    J(7,1808801)=    */    120,1264,168,544,685,-248,
/*    J(17,1808801)=    */    -154,-462,-516,-562,-226,-490,-578,18,-706,-842,-1074,-280,-694,122,-348,-723,
/*    J(19,1808801)=    */    364,-34,12,-484,-308,192,-572,12,-60,124,-422,-600,-232,-282,-248,591,-208,-126,
/*    J(3,1889551)=    */    29,-1328,-284,-378,-394,-670,
/*    J(5,1889551)=    */    240,144,412,-140,-161,-44,-526,336,-86,-299,214,-886,-126,358,-41,254,-288,766,-200,-103,
/*    J(13,1889551)=    */    -16,566,436,394,264,1074,-274,28,-110,-185,608,334,
/*    J(17,1889551)=    */    -30,-240,-526,-1016,-94,-340,-584,-360,-406,-722,-609,-924,124,-790,-550,-142,
/*    J(19,1889551)=    */    298,-110,242,-152,-242,596,372,156,170,774,-226,246,38,473,896,380,92,252,
/*    J(2,2074801)=    */    0,-256,238,554,1103,-554,238,256,
/*    J(1,2074801)=    */    584,-457,460,-187,201,491,998,29,
/*    J(4,2074801)=    */    -749,0,870,0,0,0,870,0,
/*    J(3,2074801)=    */    -1651,-651,
/*    J(5,2074801)=    */    -994,282,-46,-497,180,-30,490,170,-564,298,-812,564,306,-302,554,-460,308,68,190,324,
/*    J(7,2074801)=    */    -928,680,484,98,589,70,
/*    J(13,2074801)=    */    -548,-81,156,-336,-12,-428,602,144,632,-154,-716,64,
/*    J(19,2074801)=    */    -640,-584,-176,-6,-44,-260,-354,-312,-880,600,-8,-52,-152,-50,1,-22,-444,-608,
/*    J(2,2173601)=    */    -481,-154,510,184,-168,-226,-418,-632,0,-632,418,-226,168,184,-510,-154,
/*    J(1,2173601)=    */    -185,228,686,-335,-242,-591,-38,-466,332,-324,-482,-45,98,275,-68,-582,
/*    J(4,2173601)=    */    -1473,0,0,0,44,0,0,0,0,0,0,0,44,0,0,0,
/*    J(5,2173601)=    */    -130,412,-502,540,-350,-128,802,-520,-360,-644,-374,239,-256,-48,-668,-94,734,-198,-506,-720,
/*    J(11,2173601)=    */    -683,156,420,-94,-1086,-206,-662,-578,-14,-532,
/*    J(13,2173601)=    */    378,550,1036,394,212,508,1174,349,1062,1304,524,958,
/*    J(19,2173601)=    */    322,284,428,1218,626,223,1118,186,212,-128,694,344,308,112,296,460,504,620,
/*    J(2,2445301)=    */    -1385,-726,
/*    J(3,2445301)=    */    1190,-422,607,1026,314,-412,
/*    J(5,2445301)=    */    462,502,-106,-800,-120,454,-40,888,-806,-74,586,506,48,-1074,-343,332,-156,94,-644,-130,
/*    J(11,2445301)=    */    -276,-635,-692,236,-710,-274,-162,-1246,-874,-1154,
/*    J(13,2445301)=    */    500,1270,444,42,232,432,-30,64,170,680,-564,399,
/*    J(19,2445301)=    */    -1149,-1076,-678,-900,-678,-1096,-128,-660,-612,-1296,-1164,-1260,-776,-1020,-1132,-786,-1216,-790,
/*    J(2,2667601)=    */    0,1032,-306,-366,287,366,-306,-1032,
/*    J(1,2667601)=    */    -552,135,-316,765,633,579,-690,669,
/*    J(4,2667601)=    */    451,0,1110,0,0,0,1110,0,
/*    J(3,2667601)=    */    -80,-748,-342,58,20,-303,-382,124,-1156,292,-28,-94,230,398,176,94,-486,-366,
/*    J(5,2667601)=    */    -318,-140,150,-594,-665,96,-112,1058,-640,-68,570,454,106,-334,-406,-144,454,548,-234,-242,
/*    J(13,2667601)=    */    788,948,-216,1152,796,1104,320,106,-40,309,632,574,
/*    J(19,2667601)=    */    284,-212,-144,-164,52,-332,-116,388,-992,430,576,652,226,-101,-194,208,-88,-94,
/*    J(3,3052351)=    */    432,62,440,552,-60,394,418,48,284,-64,645,836,-276,56,306,554,92,1112,
/*    J(5,3052351)=    */    -608,298,-692,-114,48,-226,-904,-298,-360,-556,-156,271,612,-280,-244,-106,-154,110,-188,-354,
/*    J(7,3052351)=    */    132,42,-1026,-1080,-75,-1438,
/*    J(17,3052351)=    */    -18,-312,-74,-188,268,-718,66,432,898,-240,-130,-210,-796,345,-418,-504,
/*    J(19,3052351)=    */    -846,32,-322,460,-276,-276,-426,-20,-196,-82,-61,176,-458,-638,-636,-578,-1006,478,
/*    J(2,3511201)=    */    -287,-562,70,274,-912,160,-286,-616,0,-616,286,160,912,274,-70,-562,
/*    J(1,3511201)=    */    -483,-155,412,824,-60,-460,-298,-857,116,-739,248,-14,-264,130,-104,-805,
/*    J(4,3511201)=    */    1343,0,0,0,924,0,0,0,0,0,0,0,924,0,0,0,
/*    J(3,3511201)=    */    -64,-1905,
/*    J(5,3511201)=    */    -84,-777,-232,-640,744,996,-536,180,60,680,-62,216,20,-256,328,-176,48,154,-896,472,
/*    J(7,3511201)=    */    1963,1191,1599,1919,2023,1363,
/*    J(11,3511201)=    */    -1064,-420,-970,122,-1012,-1544,-1188,-334,-1055,-38,
/*    J(19,3511201)=    */    -780,-786,-384,-1082,-1190,-694,-660,-438,-1183,-578,-912,-1594,-1322,-1458,-452,-614,-752,-1500,
/*    J(3,3730651)=    */    134,1995,
/*    J(5,3730651)=    */    -903,1060,576,118,-10,-257,668,754,-24,-150,-279,520,-100,-122,-698,283,1018,482,-514,-878,
/*    J(7,3730651)=    */    -224,-439,-1234,-114,-1984,-1074,
/*    J(11,3730651)=    */    -1124,-618,-1146,90,-786,-1686,-759,-1632,-1248,-552,
/*    J(17,3730651)=    */    0,-746,-1362,-528,-1418,-32,-448,-706,-550,-684,-848,-468,-575,450,-554,-746,
/*    J(19,3730651)=    */    1450,612,-424,444,406,136,622,422,678,32,-18,-42,-500,330,203,632,-22,472,
/*    J(2,3779101)=    */    795,1774,
/*    J(3,3779101)=    */    515,490,-434,1935,-40,-820,
/*    J(5,3779101)=    */    -360,196,-790,-664,656,954,-298,184,-360,584,121,-168,88,-448,538,714,8,744,-334,304,
/*    J(13,3779101)=    */    -542,-492,-1286,-523,-38,-804,166,-188,-490,-548,-932,-1786,
/*    J(17,3779101)=    */    374,232,672,-28,252,1058,-304,648,454,886,396,489,-744,-474,534,280,
/*    J(19,3779101)=    */    -785,404,348,-118,-20,-84,-776,372,668,244,-168,616,-240,-152,-792,-136,122,-568,
/*    J(2,3950101)=    */    -1785,-874,
/*    J(3,3950101)=    */    1232,-334,-128,739,542,722,-1128,-82,-804,632,-186,138,175,388,114,-808,146,-354,
/*    J(5,3950101)=    */    -795,-402,-930,-836,656,-281,-400,-1218,-68,534,-81,-426,-106,-508,792,-619,-222,-26,-1058,638,
/*    J(7,3950101)=    */    -796,1308,1008,-24,828,-141,
/*    J(11,3950101)=    */    -182,-1336,-703,-1100,-1862,-24,-966,-820,-1224,-298,
/*    J(19,3950101)=    */    -454,192,-536,622,300,25,964,974,-512,-320,-66,340,-62,318,-442,178,-298,258,
/*    J(2,4069801)=    */    1853,-564,0,-564,
/*    J(1,4069801)=    */    -1349,0,1500,0,
/*    J(4,4069801)=    */    1853,-564,0,-564,
/*    J(3,4069801)=    */    164,593,-670,1818,937,-176,
/*    J(5,4069801)=    */    1441,614,500,838,-402,589,-8,494,94,172,413,546,-94,-16,198,761,76,-424,48,664,
/*    J(7,4069801)=    */    -1820,-1832,-2275,-792,-1572,-1328,
/*    J(17,4069801)=    */    502,844,772,415,524,180,1190,-624,226,472,-346,100,756,1262,548,590,
/*    J(19,4069801)=    */    1335,1300,1216,424,170,648,20,1134,72,506,534,156,626,828,-244,558,704,424,
/*    J(2,4149601)=    */    0,-766,698,704,48,50,626,268,271,-268,626,-50,48,-704,698,766,
/*    J(1,4149601)=    */    -538,530,-1061,25,356,-193,165,476,-103,-574,-915,-157,178,-487,-603,-448,
/*    J(4,4149601)=    */    -49,0,0,0,-1440,0,0,0,0,0,0,0,-1440,0,0,0,
/*    J(3,4149601)=    */    2009,2064,
/*    J(5,4149601)=    */    -346,380,-38,-544,286,-758,-126,-84,-1552,520,-1060,65,-170,-452,464,78,150,-800,-1210,616,
/*    J(7,4149601)=    */    494,717,-738,60,342,1770,
/*    J(13,4149601)=    */    -120,124,-264,168,-64,-366,184,666,-1681,-570,-676,-2,
/*    J(19,4149601)=    */    1000,1174,990,824,1534,904,812,554,1324,930,1858,832,780,1488,382,1387,930,156,
/*    J(3,4408951)=    */    -1021,-2415,
/*    J(5,4408951)=    */    1376,-680,-444,-6,759,824,-254,18,566,910,910,-954,-1086,-262,840,384,-288,-730,-106,762,
/*    J(7,4408951)=    */    797,-21,1781,1139,-551,375,
/*    J(13,4408951)=    */    1172,1146,552,162,-96,-538,960,754,-236,1166,454,873,
/*    J(17,4408951)=    */    1594,1116,1456,1142,176,1242,432,1038,190,376,886,930,142,919,780,92,
/*    J(19,4408951)=    */    310,-328,122,-212,1234,288,-384,969,-214,-382,-568,54,-52,470,-52,-254,-588,-262,
/*    J(2,5038801)=    */    1841,-534,-252,-690,0,-690,252,-534,
/*    J(1,5038801)=    */    1683,-396,608,-594,-972,246,60,-564,
/*    J(4,5038801)=    */    -1849,0,-900,0,0,0,-900,0,
/*    J(3,5038801)=    */    -2320,-159,
/*    J(5,5038801)=    */    -342,-108,1100,-20,-26,-28,-652,914,-218,-490,-1020,-338,175,1002,-422,-556,-346,158,-616,-978,
/*    J(13,5038801)=    */    1235,1104,1694,1750,686,-150,1760,452,802,446,902,706,
/*    J(17,5038801)=    */    -30,-474,-176,-555,-528,-1662,-738,-268,-790,444,-1496,-1108,-452,-60,-824,-226,
/*    J(19,5038801)=    */    -1042,-868,-872,-1088,-426,-812,-1218,-864,-1111,-936,-198,64,-1026,552,-704,-788,-1602,-552,
/*    J(2,6104701)=    */    -1349,2070,
/*    J(3,6104701)=    */    420,234,-886,-742,1030,-348,-476,-691,-1456,208,224,-1254,168,642,580,-764,-23,-740,
/*    J(5,6104701)=    */    -86,266,296,-307,88,-870,440,-218,293,-84,-116,-74,-448,-255,158,-1254,-1548,688,1,324,
/*    J(7,6104701)=    */    -350,2326,746,274,1016,-247,
/*    J(17,6104701)=    */    638,330,526,804,-258,-400,-174,1902,76,714,1052,666,994,902,-272,387,
/*    J(19,6104701)=    */    -474,-1120,-302,-466,-368,-382,-828,112,896,-580,-366,-444,-478,334,12,1174,-884,-131,
/*    J(2,6224401)=    */    721,1070,1252,-374,0,-374,-1252,1070,
/*    J(1,6224401)=    */    691,-44,-944,-1678,-204,458,-628,1180,
/*    J(4,6224401)=    */    1063,0,1596,0,0,0,1596,0,
/*    J(3,6224401)=    */    -10,255,-1446,-1980,960,-840,
/*    J(5,6224401)=    */    -999,496,-884,436,340,1056,258,-1186,720,-114,-220,144,-1590,-450,480,12,-226,-1116,-190,202,
/*    J(7,6224401)=    */    -1968,-420,-1775,-1836,-2448,-528,
/*    J(13,6224401)=    */    30,-576,-248,-1302,-2,-70,430,132,108,1146,-1444,-727,
/*    J(19,6224401)=    */    -617,-1800,-892,694,-822,34,-222,288,40,-360,468,-692,-172,-518,-402,-730,210,-284,
/*    J(3,8558551)=    */    1982,915,-462,3066,801,402,
/*    J(5,8558551)=    */    -514,-826,-266,918,-1006,234,894,-668,-368,-1294,-694,-6,-1479,-36,-1420,-152,1158,-1082,356,-1320,
/*    J(7,8558551)=    */    2485,870,580,-950,1064,-270,
/*    J(11,8558551)=    */    -1528,-1226,1096,-286,773,-792,710,852,-158,426,
/*    J(13,8558551)=    */    -1370,-1424,612,-1314,-1042,-929,-16,-528,-726,-2142,-1648,-1954,
/*    J(19,8558551)=    */    -1057,-871,-1075,-455,-1833,361,231,-63,-419,-677,-41,-879,-905,101,-153,431,-917,-1831,
/*    J(2,9781201)=    */    0,846,-18,2040,161,-2040,-18,-846,
/*    J(1,9781201)=    */    -768,-1317,-1574,327,-1329,861,-264,-1515,
/*    J(4,9781201)=    */    1187,0,2046,0,0,0,2046,0,
/*    J(3,9781201)=    */    -748,-345,-2244,-252,-729,-3396,
/*    J(5,9781201)=    */    -282,398,774,1090,282,1060,254,1092,-230,-517,1082,1140,-498,-424,246,-986,802,310,866,20,
/*    J(11,9781201)=    */    -968,1166,1412,190,310,1668,778,552,1892,-819,
/*    J(13,9781201)=    */    -823,-1571,-1485,-541,-1579,507,-2437,-1767,-137,-81,-347,-387,
/*    J(19,9781201)=    */    114,-1164,-4,206,472,260,-756,1004,1872,-194,1030,833,-150,142,-316,712,52,1244,
/*    J(3,11191951)=    */    -3280,949,460,-660,-1,1280,
/*    J(5,11191951)=    */    1286,218,1790,-98,225,-196,200,1482,1140,37,362,476,998,1604,1187,-798,1272,1742,502,-245,
/*    J(7,11191951)=    */    -3162,-1107,-408,822,-1096,-1518,
/*    J(11,11191951)=    */    286,-953,788,1546,434,-690,2328,846,1864,392,
/*    J(17,11191951)=    */    3015,954,546,1102,1448,602,80,996,406,772,842,2162,2468,1170,688,1380,
/*    J(19,11191951)=    */    1308,1110,290,-772,-896,-460,454,-122,320,90,-191,-828,1662,-1128,8,962,358,418,
/*    J(2,11411401)=    */    1951,-1950,0,-1950,
/*    J(1,11411401)=    */    349,0,-3360,0,
/*    J(4,11411401)=    */    -1951,1950,0,1950,
/*    J(3,11411401)=    */    -3760,-981,
/*    J(5,11411401)=    */    -390,-732,-934,2100,392,-918,-988,452,3,426,-1968,-782,142,76,904,-460,-794,-342,808,1774,
/*    J(7,11411401)=    */    -784,-1664,-2180,-2274,-1931,-4062,
/*    J(11,11411401)=    */    -372,-1822,-165,1502,504,-704,1256,-208,1480,-416,
/*    J(13,11411401)=    */    666,-224,-480,-504,948,-1514,792,2216,596,731,-660,448,
/*    J(19,11411401)=    */    -222,594,-1000,-1592,-1658,-434,-1184,-356,698,-2226,-4,-6,206,-566,-845,-534,190,-410,
/*    J(2,14922601)=    */    527,2706,0,2706,
/*    J(1,14922601)=    */    -2835,0,-2624,0,
/*    J(4,14922601)=    */    -527,-2706,0,-2706,
/*    J(3,14922601)=    */    -799,-4200,
/*    J(5,14922601)=    */    176,1636,2252,368,200,-474,1214,38,2738,88,-662,1006,1033,1386,286,-636,1370,1474,-94,-490,
/*    J(7,14922601)=    */    -1412,930,330,-3429,-1122,-1542,
/*    J(11,14922601)=    */    -876,876,-1694,-538,-132,-196,-2048,-320,-217,2284,
/*    J(17,14922601)=    */    822,1064,1789,1352,-566,940,662,-20,1232,1454,380,16,-1630,1744,-444,-330,
/*    J(19,14922601)=    */    -780,-1888,-930,-1692,-1320,-839,-2336,-2020,-1210,-1080,524,-444,-1164,-1142,508,-2892,-770,-476,
/*    J(2,16279201)=    */    0,-190,-1148,-382,776,-1146,-1244,-1718,-671,1718,-1244,1146,776,382,-1148,190,
/*    J(1,16279201)=    */    656,644,919,-688,-384,1076,-247,662,-689,-162,-2437,76,-1114,-1504,1181,984,
/*    J(4,16279201)=    */    -4033,0,0,0,-84,0,0,0,0,0,0,0,-84,0,0,0,
/*    J(3,16279201)=    */    584,2858,-2218,-861,-674,-434,
/*    J(5,16279201)=    */    -1430,1739,-932,-600,-1452,-1800,2067,-622,-1300,-346,-1162,1851,-2060,-686,-1166,-2704,1059,206,174,-102,
/*    J(7,16279201)=    */    334,1832,-847,3480,2594,1020,
/*    J(17,16279201)=    */    749,197,-1657,1743,605,745,471,1581,-1279,-1333,-475,-179,251,127,-1115,605,
/*    J(19,16279201)=    */    -684,476,-1044,480,284,-412,-1220,-926,88,-2306,-1388,714,-557,-482,-1448,-2678,-112,-1098,
/*    J(2,17117101)=    */    -2925,-2926,
/*    J(3,17117101)=    */    2114,343,-578,-2526,731,-814,
/*    J(5,17117101)=    */    484,666,1896,-478,656,-1774,460,1586,-1078,1274,-568,-106,2718,-984,-78,-1778,34,208,-1973,1094,
/*    J(7,17117101)=    */    -2887,-4064,-4472,-3522,-2978,-4548,
/*    J(11,17117101)=    */    -135,983,1657,383,2693,-527,939,2089,1199,-1659,
/*    J(13,17117101)=    */    482,-386,930,2426,-1711,-1170,-754,-570,-218,1032,-914,1242,
/*    J(19,17117101)=    */    1381,2591,-247,-455,-533,-299,1267,685,187,1943,1321,259,425,2015,113,1909,367,1453,
/*    J(2,17635801)=    */    4151,-450,0,-450,
/*    J(1,17635801)=    */    -3899,0,-1560,0,
/*    J(4,17635801)=    */    -4151,450,0,450,
/*    J(3,17635801)=    */    -1,-4200,
/*    J(5,17635801)=    */    -956,1168,-642,-216,-1394,-3128,1756,436,204,216,608,374,-264,897,-2,-530,560,-614,1244,-558,
/*    J(7,17635801)=    */    -462,240,-132,878,-3984,-1035,
/*    J(13,17635801)=    */    -464,1972,1013,684,-2368,-320,-428,16,464,656,-896,1932,
/*    J(17,17635801)=    */    -2,-936,1931,724,82,-1584,-700,410,-112,-268,1386,-570,1976,1488,-378,904,
/*    J(19,17635801)=    */    3550,2498,554,2910,1312,2032,434,2298,1650,1772,1490,1617,794,1472,1810,766,310,204,
/*    J(2,19186201)=    */    2053,-2736,0,-2736,
/*    J(1,19186201)=    */    3651,0,2420,0,
/*    J(4,19186201)=    */    2053,-2736,0,-2736,
/*    J(3,19186201)=    */    1178,-2450,-1482,-970,247,110,-864,1790,-1686,818,154,428,330,783,-994,302,718,-618,
/*    J(5,19186201)=    */    -1062,-396,-1866,-310,-2076,274,-172,-198,-946,-270,844,-1776,580,1934,-460,-252,-1167,-1008,434,-188,
/*    J(11,19186201)=    */    1212,-1658,-1682,-430,-1128,-79,-4072,-1154,-978,-1010,
/*    J(17,19186201)=    */    230,-254,-191,-494,894,-3220,102,-470,-1794,-1414,-570,262,1162,594,-854,-1090,
/*    J(19,19186201)=    */    1006,38,-448,-1424,852,-2642,-1512,406,-716,-566,-2099,-598,-1312,-982,-432,-1374,-2208,-88,
/*    J(2,19562401)=    */    0,-90,276,1906,2216,770,-668,-286,255,286,-668,-770,2216,-1906,276,90,
/*    J(1,19562401)=    */    -360,-776,-977,-2050,626,1898,-45,-758,1409,-1550,349,-1114,732,1458,345,756,
/*    J(4,19562401)=    */    -4033,0,0,0,-1284,0,0,0,0,0,0,0,-1284,0,0,0,
/*    J(3,19562401)=    */    -2554,-2756,3133,-2304,-754,1976,
/*    J(5,19562401)=    */    634,640,-2116,-102,1644,942,949,-2188,86,1548,-66,1656,-1336,826,-1190,-946,-328,-1366,-1114,1316,
/*    J(11,19562401)=    */    -372,226,-3352,-2416,-1203,-3616,-272,-826,-1340,-2252,
/*    J(13,19562401)=    */    857,3641,3963,1837,3057,1759,2935,3449,2173,2647,721,3133,
/*    J(19,19562401)=    */    1600,2051,2492,-184,2418,1692,1372,1264,-1598,250,1266,926,1574,1350,756,568,764,2338,
/*    J(2,22383901)=    */    -2485,4026,
/*    J(3,22383901)=    */    -2755,-80,1474,-15,-1690,3974,
/*    J(5,22383901)=    */    -566,284,-1112,-148,-2166,204,-42,-3406,382,-116,-1206,986,-1581,470,-2656,-2158,170,-1270,1530,-1030,
/*    J(7,22383901)=    */    -2094,-1904,-4479,-444,906,-2304,
/*    J(11,22383901)=    */    -2560,-2808,832,-524,1052,-2020,136,1046,37,-1506,
/*    J(17,22383901)=    */    -942,308,340,2240,2068,-1668,-372,680,-1176,-650,-508,-538,632,-1415,1548,-378,
/*    J(19,22383901)=    */    -716,-1554,-1082,-436,190,-120,-1003,-708,-4938,-508,-1134,-1898,-1536,-596,-846,-1192,-1448,-1338,
/*    J(2,22822801)=    */    0,-410,-3262,-776,31,776,-3262,410,
/*    J(1,22822801)=    */    552,911,272,1079,-143,4069,-610,-1871,
/*    J(4,22822801)=    */    -3901,0,-1950,0,0,0,-1950,0,
/*    J(3,22822801)=    */    4745,4809,
/*    J(5,22822801)=    */    348,-1166,-930,-2660,-1086,-782,644,-860,-1330,-2078,-953,-330,-150,-960,-2536,-674,1222,-2940,-890,-550,
/*    J(7,22822801)=    */    -8,-1365,-588,-1848,-3486,2310,
/*    J(11,22822801)=    */    176,1540,148,810,36,3145,2436,-1946,788,-952,
/*    J(13,22822801)=    */    -238,-2080,-2158,144,-539,-1968,-250,-3568,754,820,368,-672,
/*    J(19,22822801)=    */    -294,-454,1476,396,644,-36,-250,-1648,-937,-904,-542,-128,1448,-784,1100,-1658,-930,2512,
/*    J(2,23514401)=    */    0,-72,578,-282,-488,116,-1150,2602,-2447,-2602,-1150,-116,-488,282,578,72,
/*    J(1,23514401)=    */    -290,-652,749,1075,-1814,-955,-2223,796,-357,90,621,1865,816,-1577,1999,602,
/*    J(4,23514401)=    */    2799,0,0,0,2800,0,0,0,0,0,0,0,2800,0,0,0,
/*    J(5,23514401)=    */    -910,172,-247,1198,2756,-1810,2226,1000,226,-1358,-1124,206,868,470,-504,-2162,-240,972,368,-378,
/*    J(7,23514401)=    */    -160,-2964,-114,599,2786,-1912,
/*    J(13,23514401)=    */    120,-446,730,3056,1828,758,-1325,3034,32,796,894,-950,
/*    J(17,23514401)=    */    -2872,-2782,-2970,-3324,-2330,-750,-266,-3754,-1138,-1834,-264,-2182,-2406,-520,-1092,-2083,
/*    J(19,23514401)=    */    -2026,-1000,-1932,-3307,-1248,608,-1834,-2504,-1920,-1884,-1882,-2022,-3866,-1046,-3146,-1054,-2120,-308,
/*    J(2,25581601)=    */    0,-704,-88,216,32,620,224,-1088,4609,1088,224,-620,32,-216,-88,704,
/*    J(1,25581601)=    */    -380,-782,-817,188,-1840,30,-1027,904,-1833,-44,319,-858,742,128,-3599,-1070,
/*    J(4,25581601)=    */    4607,0,0,0,-1476,0,0,0,0,0,0,0,-1476,0,0,0,
/*    J(3,25581601)=    */    -2470,-2136,-426,-3426,-1989,-3648,
/*    J(5,25581601)=    */    -996,-422,1418,-248,522,1048,-612,-234,886,-290,926,144,372,2100,-974,-354,288,3845,936,1004,
/*    J(11,25581601)=    */    -882,1520,622,2834,149,-434,4038,1036,398,-768,
/*    J(17,25581601)=    */    -886,-1720,1878,-2326,-610,-1762,-1392,-546,280,-552,-744,576,1781,388,-1648,-1796,
/*    J(19,25581601)=    */    -669,-2023,-2741,-2221,-1241,-3183,-1663,-3247,-1565,-2597,-227,-23,-2347,-805,-9,-3737,-1739,-1637,
/*    J(3,25675651)=    */    -1252,142,-1272,944,-1666,-268,1850,-1497,-1880,-1116,2276,270,3046,-1010,-140,1876,-639,-1210,
/*    J(5,25675651)=    */    1192,-1516,334,-407,-3188,-744,-1096,1228,449,-2246,416,-3664,1866,-589,-3028,-1220,-1336,508,591,-2226,
/*    J(7,25675651)=    */    -3798,-3618,1248,-1452,186,-1177,
/*    J(11,25675651)=    */    2456,-1004,1256,3558,3432,-298,1590,-266,550,1045,
/*    J(13,25675651)=    */    1390,1256,1068,3614,1042,2960,840,-266,230,1036,3166,-1283,
/*    J(19,25675651)=    */    -144,-1844,-2210,416,-1006,-1130,-2294,-2022,-2060,-764,-1684,176,966,-724,-1046,494,-3306,93,
/*    J(2,31600801)=    */    378,1534,1154,-140,0,-140,-1154,1534,-378,932,3244,-740,163,740,3244,-932,
/*    J(1,31600801)=    */    735,-620,-928,1183,1791,-619,233,-1826,1841,359,1670,2470,384,-22,1715,-2437,
/*    J(4,31600801)=    */    -2213,0,0,0,-3654,0,0,0,0,0,0,0,-3654,0,0,0,
/*    J(3,31600801)=    */    -1035,184,-766,876,-1742,-1914,-32,344,1892,-1771,-1244,866,-1116,-2672,-2180,1364,-1110,3464,
/*    J(5,31600801)=    */    1187,146,-2636,984,-1650,2260,-2332,-2276,1590,1008,4002,-76,-1026,842,-886,2710,-342,-1830,2392,162,
/*    J(7,31600801)=    */    583,2176,-3436,-524,236,3260,
/*    J(11,31600801)=    */    -780,-4871,-136,-872,1772,-736,-1302,1174,32,-948,
/*    J(19,31600801)=    */    -2312,-1210,-4612,-2340,-1390,-1870,-438,-1455,-3626,90,-874,-728,-3004,-778,-100,-2000,-672,-232,
/*    J(2,35271601)=    */    4801,2374,668,-170,0,-170,-668,2374,
/*    J(1,35271601)=    */    2067,-1054,412,536,-4900,-1664,-512,1546,
/*    J(4,35271601)=    */    -5801,0,-900,0,0,0,-900,0,
/*    J(3,35271601)=    */    -2779,-6819,
/*    J(5,35271601)=    */    2055,-2050,-1696,1756,78,216,-484,1022,3984,-134,712,-2036,-1260,4040,38,2176,-652,1100,2094,1730,
/*    J(7,35271601)=    */    2140,3246,936,-2247,2280,-2478,
/*    J(13,35271601)=    */    3838,1996,4882,1496,2652,2534,4300,5006,1276,3274,3223,102,
/*    J(17,35271601)=    */    -2638,328,-1878,346,2060,1210,-3203,-1642,-640,-1578,-2258,-484,-2502,88,-1328,-1012,
/*    J(19,35271601)=    */    708,3976,984,2014,1058,4344,3232,2212,4280,2824,284,3260,812,1975,2504,3660,2092,1200,
/*    J(2,37346401)=    */    -257,3112,1202,482,2624,520,138,-322,0,-322,-138,520,-2624,482,-1202,3112,
/*    J(1,37346401)=    */    883,566,848,-2825,-314,299,-1934,298,-2220,-2692,-30,1265,1410,961,-1642,2036,
/*    J(4,37346401)=    */    4447,0,0,0,2964,0,0,0,0,0,0,0,2964,0,0,0,
/*    J(3,37346401)=    */    -1336,-3020,614,-765,4758,124,-260,2154,-196,-1732,-1864,994,-109,2758,1608,1312,-82,270,
/*    J(5,37346401)=    */    -3314,1926,365,-1600,1122,-270,2090,349,-1110,-2262,222,118,1981,550,930,-254,1808,429,286,2158,
/*    J(7,37346401)=    */    -1011,-3777,3489,805,675,2037,
/*    J(13,37346401)=    */    1765,2515,1901,3321,2469,-481,1155,1389,-1171,1765,2079,5457,
/*    J(19,37346401)=    */    684,454,-2360,652,222,-1324,500,2086,2886,1230,-546,1578,-2229,1342,950,246,-1738,610,
/*    J(2,38372401)=    */    -479,-1774,3748,-1370,0,-1370,-3748,-1774,
/*    J(1,38372401)=    */    -701,628,-512,-4714,1236,-1090,1724,3052,
/*    J(4,38372401)=    */    -2537,0,3996,0,0,0,3996,0,
/*    J(3,38372401)=    */    -1946,-1166,712,-616,30,-218,2122,1729,-3380,-2766,-1168,164,2952,-566,888,1828,2713,-1946,
/*    J(5,38372401)=    */    -1849,2810,1320,-1772,-1230,232,3358,1958,6,3042,-1912,3284,464,-720,242,-2032,940,1582,822,4,
/*    J(11,38372401)=    */    98,-426,2120,-604,-1352,-936,875,-2060,4312,2284,
/*    J(17,38372401)=    */    -1286,-720,3800,-224,-1088,-734,1332,484,2118,2542,-294,2115,-1442,118,-494,-176,
/*    J(19,38372401)=    */    3700,1212,3194,2678,22,1894,3256,179,3292,4410,4442,1262,1926,1824,2860,1964,174,1534,
/*    J(2,45349201)=    */    -1207,1688,544,-4336,0,-4336,-544,1688,
/*    J(1,45349201)=    */    -1601,-3556,3488,-1516,-2412,308,1600,2684,
/*    J(4,45349201)=    */    -5257,0,-2976,0,0,0,-2976,0,
/*    J(3,45349201)=    */    500,-1548,-144,-3855,998,1094,2398,-1714,-478,-610,-2190,-1562,-2575,2340,-56,1984,2852,-840,
/*    J(5,45349201)=    */    -3724,-142,168,1998,-2288,-2802,810,-2850,-810,-2676,-1144,2270,1538,-392,-407,-1066,724,1188,-150,-2796,
/*    J(13,45349201)=    */    -903,-951,-2375,-1645,-3263,-2453,-4049,-1939,-4169,-1559,-4039,-7067,
/*    J(17,45349201)=    */    1322,-2684,-2024,-2642,-1530,-1716,-748,1492,1576,-2078,-474,-52,428,-2722,320,-3667,
/*    J(19,45349201)=    */    5144,1184,-498,2746,1192,1028,-538,300,-432,1263,-1144,924,3106,2964,1044,722,102,1944,
/*    J(2,59690401)=    */    5343,78,-892,3042,856,1618,1348,-590,0,-590,-1348,1618,-856,3042,892,78,
/*    J(1,59690401)=    */    -4711,-206,1224,-230,-950,1618,-812,1704,-3016,1296,2340,-2094,26,-30,1756,2258,
/*    J(4,59690401)=    */    7663,0,0,0,-696,0,0,0,0,0,0,0,-696,0,0,0,
/*    J(3,59690401)=    */    -8821,-5565,
/*    J(5,59690401)=    */    -1810,-3018,-908,-1102,-108,-5084,-1105,2514,558,-1208,-162,-4238,-796,278,-1194,630,-1332,1874,-1448,438,
/*    J(7,59690401)=    */    4062,3996,7647,4132,7746,8004,
/*    J(11,59690401)=    */    -1838,-2664,-7430,-3571,-4238,-4364,-4050,-640,-6802,-1914,
/*    J(17,59690401)=    */    -2240,-1233,-2916,-888,-1642,598,-4510,-4982,2106,-1294,938,450,410,-1342,-1822,-2034,
/*    J(19,59690401)=    */    -4113,-2688,-2498,-5610,-3764,-3612,-3402,-4590,-4412,-6496,-1752,-92,-2198,-3258,-1752,-2928,-3510,-6254,
/*    J(2,67151701)=    */    -3465,7426,
/*    J(3,67151701)=    */    -2528,-1516,-437,-2526,342,-3318,520,1116,382,-1934,-4090,-2537,60,-3188,-496,-64,-1340,-3140,
/*    J(5,67151701)=    */    -743,-2384,-5376,524,1226,2471,-756,-4986,1820,4132,495,1026,-4370,2046,2196,801,800,-6846,2418,1260,
/*    J(7,67151701)=    */    -4785,-1329,-399,1473,-3285,-7433,
/*    J(11,67151701)=    */    -2260,1488,1648,-4362,732,2747,2632,1054,3284,-2608,
/*    J(17,67151701)=    */    298,1184,4316,-242,3459,-1070,-1244,-196,-766,56,1626,2084,-4582,-610,1104,362,
/*    J(19,67151701)=    */    353,925,477,121,-33,2847,-703,3479,379,4007,3879,-3071,-1905,391,893,-803,1929,-607,
/*    J(2,83140201)=    */    5903,-4914,0,-4914,
/*    J(1,83140201)=    */    4301,0,8040,0,
/*    J(4,83140201)=    */    -5903,4914,0,4914,
/*    J(3,83140201)=    */    -4444,1592,-3052,-3204,-6761,-2696,
/*    J(5,83140201)=    */    -3442,-28,-1000,650,4716,-2434,-4212,-3340,2214,-2740,-2926,-2740,-2116,2046,1152,-724,-568,-2020,2556,-1385,
/*    J(11,83140201)=    */    -2278,-222,-1232,-566,-2764,-5278,5719,1010,1072,6,
/*    J(13,83140201)=    */    5081,2619,1727,609,4509,7735,3263,6981,-491,1987,3855,1813,
/*    J(17,83140201)=    */    3260,642,1526,-2336,2586,225,-798,1388,-454,1238,1580,4956,1040,5330,1482,-3136,
/*    J(19,83140201)=    */    -596,-452,-2854,-1286,-6838,-135,302,-3494,-2826,-4176,-620,-3716,-1920,-2136,-4832,-3640,1456,-656,
/*    J(2,129329201)=    */    -5337,-3084,-784,6348,0,6348,784,-3084,
/*    J(1,129329201)=    */    861,888,7458,-1924,-3888,-5364,2610,-4216,
/*    J(4,129329201)=    */    8727,0,-5156,0,0,0,-5156,0,
/*    J(5,129329201)=    */    3954,1022,3416,1354,-902,246,4280,2712,6234,-3326,-154,2348,3447,4322,1284,1108,412,7744,386,-3948,
/*    J(7,129329201)=    */    7840,6204,10420,6385,13980,8580,
/*    J(11,129329201)=    */    -5940,1338,-1726,-5636,-2208,-5618,-7742,-2524,3592,-3039,
/*    J(13,129329201)=    */    -8767,-6583,-1793,-2605,-1409,-1477,-6373,-2201,3095,-2841,-751,-4995,
/*    J(17,129329201)=    */    3203,2989,-1063,-4737,659,1671,3735,2389,5241,-2185,4939,2203,-557,5701,2513,1807,
/*    J(19,129329201)=    */    -5418,-2178,-4132,-5708,-5250,-4234,-7118,-8876,-8613,580,-3074,-4970,-5618,-3392,-7592,-2910,-2074,-6064,
/*    J(2,134303401)=    */    -10801,-2970,0,-2970,
/*    J(1,134303401)=    */    -4499,0,-10680,0,
/*    J(4,134303401)=    */    10801,2970,0,2970,
/*    J(3,134303401)=    */    3020,-1228,-4454,-3032,4434,2060,3058,2360,3336,3296,4118,-7942,-650,4038,726,2230,-864,1819,
/*    J(5,134303401)=    */    408,-10,10,2122,-2442,-768,346,1746,-1522,-8126,-4505,242,-374,2010,-1354,-1656,-1702,4790,-4482,-1294,
/*    J(7,134303401)=    */    357,-6060,-9346,3138,-3486,570,
/*    J(11,134303401)=    */    -1716,-7060,-3384,-7274,-10182,136,-6932,-5524,-7420,-1421,
/*    J(17,134303401)=    */    4350,2112,-322,7056,3376,-26,784,-582,5196,-2864,130,108,4593,4020,6510,2380,
/*    J(19,134303401)=    */    366,-1996,648,-1778,3860,8326,246,-334,3362,-480,1242,-339,3062,1076,3706,-3698,810,-258,
/*    J(2,193993801)=    */    -3073,9606,0,9606,
/*    J(1,193993801)=    */    -13155,0,4576,0,
/*    J(4,193993801)=    */    3073,-9606,0,-9606,
/*    J(3,193993801)=    */    15896,10065,
/*    J(5,193993801)=    */    -562,-2440,1532,-3062,-6,-5214,-4746,1430,-2477,-7810,-3346,-5378,982,5886,-3380,-2224,-6954,3080,-3396,-4316,
/*    J(7,193993801)=    */    1247,1537,5389,11565,-5555,747,
/*    J(11,193993801)=    */    28,-6752,-829,2812,-3244,9388,2502,-934,-3754,1530,
/*    J(13,193993801)=    */    5064,2212,11971,11476,-32,6756,4062,6838,936,5264,4782,2290,
/*    J(17,193993801)=    */    -3780,-6806,954,-4254,-6506,-362,-4886,-2240,-5028,470,-2746,6094,315,-4762,-5490,-4426,
/*    J(19,193993801)=    */    -6974,-4477,370,-7084,1598,-1160,-8284,-10574,-5102,-3898,-3282,-5534,-2334,-2726,-502,-4656,-864,-4362,
/*    J(2,249420601)=    */    -4987,10596,0,10596,
/*    J(1,249420601)=    */    12851,0,-9180,0,
/*    J(4,249420601)=    */    -4987,10596,0,10596,
/*    J(3,249420601)=    */    -2986,-3892,-637,596,-5244,8436,3902,-1294,7748,1986,-4580,232,-824,2898,4356,2092,-7574,3784,
/*    J(5,249420601)=    */    -2724,996,-3696,927,10700,8548,-4382,3622,-593,4488,2540,-872,3024,-145,2974,3170,-3124,4396,-1225,3240,
/*    J(11,249420601)=    */    -10529,-12773,-8879,-7619,-2897,-4445,-3719,-7247,-9653,-16577,
/*    J(13,249420601)=    */    6179,68,3854,-900,6176,-404,3198,4770,6692,8372,10804,-4766,
/*    J(17,249420601)=    */    2765,-308,-290,7106,5290,-6402,-742,-6052,-2422,3530,-1904,-542,-578,6234,-2378,2234,
/*    J(19,249420601)=    */    -1232,3594,4816,4586,1966,-3044,5272,3447,-556,540,-158,-5258,-384,7602,-852,7324,2402,7174,
/*    J(3,436486051)=    */    9708,-11360,2644,-2334,2506,6980,396,-5092,4687,12154,-5474,-1524,3458,4496,2064,-6562,-6382,-2815,
/*    J(5,436486051)=    */    10448,-9482,-1874,-1818,7656,4608,-5524,-964,-72,7108,10128,-3586,-4538,2378,1732,4680,568,2353,11810,8628,
/*    J(7,436486051)=    */    -3086,-19470,148,-68,4874,-6885,
/*    J(11,436486051)=    */    -7581,1326,-10746,-1674,1590,-3966,-28,564,-12480,-16308,
/*    J(13,436486051)=    */    -58,5340,6668,4756,3568,-1528,1791,18570,834,12234,10870,7986,
/*    J(17,436486051)=    */    -118,-7102,3352,-10869,-4594,-9876,-7924,-5066,-7306,-9332,-7104,-12906,-1908,-7870,-15714,-12556,
/*    J(19,436486051)=    */    -7418,343,-5540,-13298,-4312,-7204,-15798,-1004,-3380,1886,-9502,-8724,-9088,-342,-2466,-5088,-1166,-3166,
/*    J(2,634888801)=    */    -1526,4020,-5756,-8632,0,-8632,5756,4020,1526,-12052,-2442,-1778,-8595,1778,-2442,12052,
/*    J(1,634888801)=    */    -2169,-181,1958,3477,-259,6131,69,4577,21961,-5388,-2540,-4900,2268,-2006,1605,1448,
/*    J(4,634888801)=    */    24443,0,0,0,4326,0,0,0,0,0,0,0,4326,0,0,0,
/*    J(3,634888801)=    */    1008,-1508,-15698,1908,-4420,7394,-7814,6093,17244,2030,316,-11976,-3150,396,10196,-3490,5456,8170,
/*    J(5,634888801)=    */    1120,7194,2310,2634,-5116,9907,14904,6660,-3012,-7282,3816,5892,550,3706,-18496,12936,12302,714,-1932,-7058,
/*    J(7,634888801)=    */    -356,9510,21216,-8469,3048,-1374,
/*    J(13,634888801)=    */    2532,12156,-1020,-1074,3660,10548,5148,13941,-4788,-2868,6468,-10150,
/*    J(17,634888801)=    */    -10084,6668,-6218,-3486,-11786,-6340,5376,1814,4675,-1678,1364,-980,6418,-7662,-11110,592,
/*    J(19,634888801)=    */    -5666,-6384,-4102,594,-594,4204,8018,-1010,-10282,15370,240,4202,-2106,-662,6012,5386,3764,419,
/*    J(2,1163962801)=    */    10367,7626,-16668,13866,0,13866,16668,7626,
/*    J(1,1163962801)=    */    -8415,-27606,14922,5388,5616,1836,5734,3402,
/*    J(4,1163962801)=    */    -14777,0,-21744,0,0,0,-21744,0,
/*    J(3,1163962801)=    */    16031,-13348,-25870,6687,-11582,4666,
/*    J(5,1163962801)=    */    7644,-3028,12970,-9098,-6464,16262,1646,7654,-22380,3690,7654,-4568,16548,-3540,-5190,6314,-15278,-2173,-11046,-1808,
/*    J(7,1163962801)=    */    14477,-2488,26308,11948,-12644,1108,
/*    J(11,1163962801)=    */    11791,3136,8432,4320,-854,-7360,20840,-12778,-12108,-5432,
/*    J(13,1163962801)=    */    -680,-1036,16192,13522,10968,13848,30326,-3936,11576,13568,11552,16907,
/*    J(17,1163962801)=    */    5968,6003,-14844,11336,-3468,11550,13498,-7160,-7982,-5786,-5810,-7206,-2600,4062,1146,-9792,
/*    J(19,1163962801)=    */    -6620,-9366,-18424,-6544,-1464,4596,686,-4364,1580,2646,-7524,16240,-16286,-11509,-2426,1196,-5108,-5406,
/*    J(2,1745944201)=    */    8351,28950,0,28950,
/*    J(1,1745944201)=    */    33949,0,24360,0,
/*    J(4,1745944201)=    */    -8351,-28950,0,-28950,
/*    J(3,1745944201)=    */    3980,-1104,-2494,5280,-3906,10844,-15732,-12110,12528,-12328,7352,5402,8399,-13572,11100,8636,14136,7854,
/*    J(5,1745944201)=    */    6726,22708,9102,9408,-22574,13518,-3518,-9688,8256,-14540,2916,986,-3998,8821,-13816,14284,-3174,-8302,15136,798,
/*    J(7,1745944201)=    */    -7602,26588,19392,-16030,7520,-9947,
/*    J(11,1745944201)=    */    -12922,19240,-19298,-8090,2615,17658,17326,1932,-1474,5936,
/*    J(13,1745944201)=    */    5898,9805,14222,-15548,4272,-6548,2946,-15384,-4440,9302,-22684,-13692,
/*    J(17,1745944201)=    */    -15920,-37256,-30082,-18909,-27582,-23244,-20440,-32368,-29648,-21294,-44056,-18204,-35980,-20512,-29234,-19116,
/*    J(19,1745944201)=    */    5506,-10704,-11734,4548,2449,-2328,-20074,-19352,-17230,-1676,-15844,-1222,-15192,-4842,-20606,-16764,14,-23252,
0};


static const struct jpq_t jpqs[JPQSMAX+1]={
{5,0,2},
{7,2,3},
{13,4,2},
{13,6,3},
{11,8,5},
{31,12,3},
{31,14,5},
{61,18,2},
{61,20,3},
{61,22,5},
{19,26,3},
{37,32,2},
{37,34,3},
{181,40,2},
{181,42,3},
{181,48,5},
{29,52,2},
{29,54,7},
{41,60,2},
{41,64,1},
{41,68,4},
{41,72,5},
{43,76,3},
{43,78,7},
{71,84,5},
{71,88,7},
{211,94,3},
{211,96,5},
{211,100,7},
{281,106,2},
{281,110,1},
{281,114,4},
{281,118,5},
{281,122,7},
{421,128,2},
{421,130,3},
{421,132,5},
{421,136,7},
{127,142,3},
{127,148,7},
{631,154,3},
{631,160,5},
{631,164,7},
{17,170,2},
{17,178,1},
{17,186,4},
{113,194,2},
{113,202,1},
{113,210,4},
{113,218,7},
{241,224,2},
{241,232,1},
{241,240,4},
{241,248,3},
{241,250,5},
{337,254,2},
{337,262,1},
{337,270,4},
{337,278,3},
{337,280,7},
{73,286,2},
{73,290,1},
{73,294,4},
{73,298,3},
{2521,304,2},
{2521,308,1},
{2521,312,4},
{2521,316,3},
{2521,322,5},
{2521,326,7},
{1009,332,2},
{1009,340,1},
{1009,348,4},
{1009,356,3},
{1009,362,7},
{109,368,2},
{109,370,3},
{271,388,3},
{271,406,5},
{379,410,3},
{379,428,7},
{433,434,2},
{433,442,1},
{433,450,4},
{433,458,3},
{541,476,2},
{541,478,3},
{541,496,5},
{757,500,2},
{757,502,3},
{757,520,7},
{2161,526,2},
{2161,534,1},
{2161,542,4},
{2161,550,3},
{2161,568,5},
{7561,572,2},
{7561,576,1},
{7561,580,4},
{7561,584,3},
{7561,602,5},
{7561,606,7},
{15121,612,2},
{15121,620,1},
{15121,628,4},
{15121,636,3},
{15121,654,5},
{15121,658,7},
{23,664,11},
{67,674,3},
{67,676,11},
{89,686,2},
{89,690,1},
{89,694,4},
{89,698,11},
{199,708,3},
{199,714,11},
{331,724,3},
{331,726,5},
{331,730,11},
{397,740,2},
{397,742,3},
{397,748,11},
{463,758,3},
{463,760,7},
{463,766,11},
{617,776,2},
{617,780,1},
{617,784,4},
{617,788,7},
{617,794,11},
{661,804,2},
{661,806,3},
{661,808,5},
{661,812,11},
{881,822,2},
{881,830,1},
{881,838,4},
{881,846,5},
{881,850,11},
{991,860,3},
{991,866,5},
{991,870,11},
{1321,880,2},
{1321,884,1},
{1321,888,4},
{1321,892,3},
{1321,894,5},
{1321,898,11},
{2311,908,3},
{2311,910,5},
{2311,914,7},
{2311,920,11},
{3697,930,2},
{3697,938,1},
{3697,946,4},
{3697,954,3},
{3697,956,7},
{3697,962,11},
{4621,972,2},
{4621,974,3},
{4621,976,5},
{4621,980,7},
{4621,986,11},
{9241,996,2},
{9241,1000,1},
{9241,1004,4},
{9241,1008,3},
{9241,1010,5},
{9241,1014,7},
{9241,1020,11},
{18481,1030,2},
{18481,1038,1},
{18481,1046,4},
{18481,1054,3},
{18481,1056,5},
{18481,1060,7},
{18481,1066,11},
{55441,1076,2},
{55441,1084,1},
{55441,1092,4},
{55441,1100,3},
{55441,1106,5},
{55441,1110,7},
{55441,1116,11},
{97,1126,2},
{97,1142,1},
{97,1158,4},
{97,1174,3},
{353,1176,2},
{353,1192,1},
{353,1208,4},
{353,1224,11},
{673,1234,2},
{673,1250,1},
{673,1266,4},
{673,1282,3},
{673,1284,7},
{2017,1290,2},
{2017,1306,1},
{2017,1322,4},
{2017,1338,3},
{2017,1344,7},
{3169,1350,2},
{3169,1366,1},
{3169,1382,4},
{3169,1398,3},
{3169,1404,11},
{3361,1414,2},
{3361,1430,1},
{3361,1446,4},
{3361,1462,3},
{3361,1464,5},
{3361,1468,7},
{5281,1474,2},
{5281,1490,1},
{5281,1506,4},
{5281,1522,3},
{5281,1524,5},
{5281,1528,11},
{7393,1538,2},
{7393,1554,1},
{7393,1570,4},
{7393,1586,3},
{7393,1588,7},
{7393,1594,11},
{110881,1604,2},
{110881,1620,1},
{110881,1636,4},
{110881,1652,3},
{110881,1658,5},
{110881,1662,7},
{110881,1668,11},
{53,1678,2},
{53,1680,13},
{79,1692,3},
{79,1694,13},
{131,1706,5},
{131,1710,13},
{157,1722,2},
{157,1724,3},
{157,1726,13},
{313,1738,2},
{313,1742,1},
{313,1746,4},
{313,1750,3},
{313,1752,13},
{521,1764,2},
{521,1768,1},
{521,1772,4},
{521,1776,5},
{521,1780,13},
{547,1792,3},
{547,1794,7},
{547,1800,13},
{859,1812,3},
{859,1814,11},
{859,1824,13},
{911,1836,5},
{911,1840,7},
{911,1846,13},
{937,1858,2},
{937,1862,1},
{937,1866,4},
{937,1870,3},
{937,1876,13},
{1093,1888,2},
{1093,1890,3},
{1093,1892,7},
{1093,1898,13},
{1171,1910,3},
{1171,1916,5},
{1171,1920,13},
{1873,1932,2},
{1873,1940,1},
{1873,1948,4},
{1873,1956,3},
{1873,1962,13},
{2003,1974,7},
{2003,1980,11},
{2003,1990,13},
{2341,2002,2},
{2341,2004,3},
{2341,2010,5},
{2341,2014,13},
{2731,2026,3},
{2731,2028,5},
{2731,2032,7},
{2731,2038,13},
{2861,2050,2},
{2861,2052,5},
{2861,2056,11},
{2861,2066,13},
{3121,2078,2},
{3121,2086,1},
{3121,2094,4},
{3121,2102,3},
{3121,2104,5},
{3121,2108,13},
{3433,2120,2},
{3433,2124,1},
{3433,2128,4},
{3433,2132,3},
{3433,2134,11},
{3433,2144,13},
{6007,2156,3},
{6007,2158,7},
{6007,2164,11},
{6007,2174,13},
{6553,2186,2},
{6553,2190,1},
{6553,2194,4},
{6553,2198,3},
{6553,2204,7},
{6553,2210,13},
{8009,2222,2},
{8009,2226,1},
{8009,2230,4},
{8009,2234,7},
{8009,2240,11},
{8009,2250,13},
{8191,2262,3},
{8191,2268,5},
{8191,2272,7},
{8191,2278,13},
{8581,2290,2},
{8581,2292,3},
{8581,2294,5},
{8581,2298,11},
{8581,2308,13},
{16381,2320,2},
{16381,2322,3},
{16381,2328,5},
{16381,2332,7},
{16381,2338,13},
{20021,2350,2},
{20021,2352,5},
{20021,2356,7},
{20021,2362,11},
{20021,2372,13},
{20593,2384,2},
{20593,2392,1},
{20593,2400,4},
{20593,2408,3},
{20593,2414,11},
{20593,2424,13},
{21841,2436,2},
{21841,2444,1},
{21841,2452,4},
{21841,2460,3},
{21841,2462,5},
{21841,2466,7},
{21841,2472,13},
{25741,2484,2},
{25741,2486,3},
{25741,2492,5},
{25741,2496,11},
{25741,2506,13},
{36037,2518,2},
{36037,2520,3},
{36037,2526,7},
{36037,2532,11},
{36037,2542,13},
{48049,2554,2},
{48049,2562,1},
{48049,2570,4},
{48049,2578,3},
{48049,2580,7},
{48049,2586,11},
{48049,2596,13},
{51481,2608,2},
{51481,2612,1},
{51481,2616,4},
{51481,2620,3},
{51481,2626,5},
{51481,2630,11},
{51481,2640,13},
{65521,2652,2},
{65521,2660,1},
{65521,2668,4},
{65521,2676,3},
{65521,2682,5},
{65521,2686,7},
{65521,2692,13},
{72073,2704,2},
{72073,2708,1},
{72073,2712,4},
{72073,2716,3},
{72073,2722,7},
{72073,2728,11},
{72073,2738,13},
{120121,2750,2},
{120121,2754,1},
{120121,2758,4},
{120121,2762,3},
{120121,2764,5},
{120121,2768,7},
{120121,2774,11},
{120121,2784,13},
{180181,2796,2},
{180181,2798,3},
{180181,2804,5},
{180181,2808,7},
{180181,2814,11},
{180181,2824,13},
{1249,2836,2},
{1249,2852,1},
{1249,2868,4},
{1249,2884,3},
{1249,2886,13},
{2081,2898,2},
{2081,2914,1},
{2081,2930,4},
{2081,2946,5},
{2081,2950,13},
{8737,2962,2},
{8737,2978,1},
{8737,2994,4},
{8737,3010,3},
{8737,3012,7},
{8737,3018,13},
{13729,3030,2},
{13729,3046,1},
{13729,3062,4},
{13729,3078,3},
{13729,3080,11},
{13729,3090,13},
{14561,3102,2},
{14561,3118,1},
{14561,3134,4},
{14561,3150,5},
{14561,3154,7},
{14561,3160,13},
{26209,3172,2},
{26209,3188,1},
{26209,3204,4},
{26209,3220,3},
{26209,3226,7},
{26209,3232,13},
{96097,3244,2},
{96097,3260,1},
{96097,3276,4},
{96097,3292,3},
{96097,3294,7},
{96097,3300,11},
{96097,3310,13},
{131041,3322,2},
{131041,3338,1},
{131041,3354,4},
{131041,3370,3},
{131041,3376,5},
{131041,3380,7},
{131041,3386,13},
{2377,3398,2},
{2377,3402,1},
{2377,3406,4},
{2377,3410,3},
{2377,3428,11},
{2971,3438,3},
{2971,3456,5},
{2971,3460,11},
{3511,3470,3},
{3511,3488,5},
{3511,3492,13},
{4159,3504,3},
{4159,3522,7},
{4159,3528,11},
{7723,3538,3},
{7723,3556,11},
{7723,3566,13},
{8317,3578,2},
{8317,3580,3},
{8317,3598,7},
{8317,3604,11},
{9829,3614,2},
{9829,3616,3},
{9829,3634,7},
{9829,3640,13},
{16633,3652,2},
{16633,3656,1},
{16633,3660,4},
{16633,3664,3},
{16633,3682,7},
{16633,3688,11},
{23761,3698,2},
{23761,3706,1},
{23761,3714,4},
{23761,3722,3},
{23761,3740,5},
{23761,3744,11},
{24571,3754,3},
{24571,3772,5},
{24571,3776,7},
{24571,3782,13},
{28081,3794,2},
{28081,3802,1},
{28081,3810,4},
{28081,3818,3},
{28081,3836,5},
{28081,3840,13},
{30241,3852,2},
{30241,3868,1},
{30241,3884,4},
{30241,3900,3},
{30241,3918,5},
{30241,3922,7},
{38611,3928,3},
{38611,3946,5},
{38611,3950,11},
{38611,3960,13},
{39313,3972,2},
{39313,3980,1},
{39313,3988,4},
{39313,3996,3},
{39313,4014,7},
{39313,4020,13},
{47521,4032,2},
{47521,4048,1},
{47521,4064,4},
{47521,4080,3},
{47521,4098,5},
{47521,4102,11},
{66529,4112,2},
{66529,4128,1},
{66529,4144,4},
{66529,4160,3},
{66529,4178,7},
{66529,4184,11},
{108109,4194,2},
{108109,4196,3},
{108109,4214,7},
{108109,4220,11},
{108109,4230,13},
{123553,4242,2},
{123553,4258,1},
{123553,4274,4},
{123553,4290,3},
{123553,4308,11},
{123553,4318,13},
{196561,4330,2},
{196561,4338,1},
{196561,4346,4},
{196561,4354,3},
{196561,4372,5},
{196561,4376,7},
{196561,4382,13},
{216217,4394,2},
{216217,4398,1},
{216217,4402,4},
{216217,4406,3},
{216217,4424,7},
{216217,4430,11},
{216217,4440,13},
{270271,4452,3},
{270271,4470,5},
{270271,4474,7},
{270271,4480,11},
{270271,4490,13},
{332641,4502,2},
{332641,4518,1},
{332641,4534,4},
{332641,4550,3},
{332641,4568,5},
{332641,4572,7},
{332641,4578,11},
{393121,4588,2},
{393121,4604,1},
{393121,4620,4},
{393121,4636,3},
{393121,4654,5},
{393121,4658,7},
{393121,4664,13},
{432433,4676,2},
{432433,4684,1},
{432433,4692,4},
{432433,4700,3},
{432433,4718,7},
{432433,4724,11},
{432433,4734,13},
{540541,4746,2},
{540541,4748,3},
{540541,4766,5},
{540541,4770,7},
{540541,4776,11},
{540541,4786,13},
{617761,4798,2},
{617761,4814,1},
{617761,4830,4},
{617761,4846,3},
{617761,4864,5},
{617761,4868,11},
{617761,4878,13},
{4324321,4890,2},
{4324321,4906,1},
{4324321,4922,4},
{4324321,4938,3},
{4324321,4956,5},
{4324321,4960,7},
{4324321,4966,11},
{4324321,4976,13},
{103,4988,3},
{103,4990,17},
{137,5006,2},
{137,5010,1},
{137,5014,4},
{137,5018,17},
{239,5034,7},
{239,5040,17},
{307,5056,3},
{307,5062,17},
{409,5078,2},
{409,5082,1},
{409,5086,4},
{409,5090,3},
{409,5092,17},
{443,5108,13},
{443,5120,17},
{613,5136,2},
{613,5138,3},
{613,5144,17},
{953,5160,2},
{953,5164,1},
{953,5168,4},
{953,5172,7},
{953,5178,17},
{1021,5194,2},
{1021,5196,3},
{1021,5198,5},
{1021,5202,17},
{1123,5218,3},
{1123,5220,11},
{1123,5230,17},
{1327,5246,3},
{1327,5248,13},
{1327,5260,17},
{1361,5276,2},
{1361,5284,1},
{1361,5292,4},
{1361,5300,5},
{1361,5304,17},
{1429,5320,2},
{1429,5322,3},
{1429,5324,7},
{1429,5330,17},
{1531,5346,3},
{1531,5352,5},
{1531,5356,17},
{1871,5372,5},
{1871,5376,11},
{1871,5386,17},
{2143,5402,3},
{2143,5408,7},
{2143,5414,17},
{2381,5430,2},
{2381,5432,5},
{2381,5436,7},
{2381,5442,17},
{2857,5458,2},
{2857,5462,1},
{2857,5466,4},
{2857,5470,3},
{2857,5472,7},
{2857,5478,17},
{3061,5494,2},
{3061,5496,3},
{3061,5502,5},
{3061,5506,17},
{3571,5522,3},
{3571,5524,5},
{3571,5528,7},
{3571,5534,17},
{4421,5550,2},
{4421,5552,5},
{4421,5556,13},
{4421,5568,17},
{5237,5584,2},
{5237,5586,7},
{5237,5592,11},
{5237,5602,17},
{6121,5618,2},
{6121,5622,1},
{6121,5626,4},
{6121,5630,3},
{6121,5636,5},
{6121,5640,17},
{6733,5656,2},
{6733,5658,3},
{6733,5664,11},
{6733,5674,17},
{7481,5690,2},
{7481,5694,1},
{7481,5698,4},
{7481,5702,5},
{7481,5706,11},
{7481,5716,17},
{8161,5732,2},
{8161,5748,1},
{8161,5764,4},
{8161,5780,3},
{8161,5782,5},
{8161,5786,17},
{9283,5802,3},
{9283,5804,7},
{9283,5810,13},
{9283,5822,17},
{9521,5838,2},
{9521,5846,1},
{9521,5854,4},
{9521,5862,5},
{9521,5866,7},
{9521,5872,17},
{10711,5888,3},
{10711,5894,5},
{10711,5898,7},
{10711,5904,17},
{12241,5920,2},
{12241,5928,1},
{12241,5936,4},
{12241,5944,3},
{12241,5950,5},
{12241,5954,17},
{12377,5970,2},
{12377,5974,1},
{12377,5978,4},
{12377,5982,7},
{12377,5988,13},
{12377,6000,17},
{14281,6016,2},
{14281,6020,1},
{14281,6024,4},
{14281,6028,3},
{14281,6030,5},
{14281,6034,7},
{14281,6040,17},
{15913,6056,2},
{15913,6060,1},
{15913,6064,4},
{15913,6068,3},
{15913,6074,13},
{15913,6086,17},
{16831,6102,3},
{16831,6108,5},
{16831,6112,11},
{16831,6122,17},
{17137,6138,2},
{17137,6146,1},
{17137,6154,4},
{17137,6162,3},
{17137,6168,7},
{17137,6174,17},
{17681,6190,2},
{17681,6198,1},
{17681,6206,4},
{17681,6214,5},
{17681,6218,13},
{17681,6230,17},
{19891,6246,3},
{19891,6252,5},
{19891,6256,13},
{19891,6268,17},
{22441,6284,2},
{22441,6288,1},
{22441,6292,4},
{22441,6296,3},
{22441,6298,5},
{22441,6302,11},
{22441,6312,17},
{23563,6328,3},
{23563,6334,7},
{23563,6340,11},
{23563,6350,17},
{24481,6366,2},
{24481,6382,1},
{24481,6398,4},
{24481,6414,3},
{24481,6420,5},
{24481,6424,17},
{27847,6440,3},
{27847,6446,7},
{27847,6452,13},
{27847,6464,17},
{29173,6480,2},
{29173,6482,3},
{29173,6484,11},
{29173,6494,13},
{29173,6506,17},
{29921,6522,2},
{29921,6538,1},
{29921,6554,4},
{29921,6570,5},
{29921,6574,11},
{29921,6584,17},
{30941,6600,2},
{30941,6602,5},
{30941,6606,7},
{30941,6612,13},
{30941,6624,17},
{34273,6640,2},
{34273,6656,1},
{34273,6672,4},
{34273,6688,3},
{34273,6694,7},
{34273,6700,17},
{42841,6716,2},
{42841,6720,1},
{42841,6724,4},
{42841,6728,3},
{42841,6734,5},
{42841,6738,7},
{42841,6744,17},
{43759,6760,3},
{43759,6766,11},
{43759,6776,13},
{43759,6788,17},
{46411,6804,3},
{46411,6806,5},
{46411,6810,7},
{46411,6816,13},
{46411,6828,17},
{52361,6844,2},
{52361,6848,1},
{52361,6852,4},
{52361,6856,5},
{52361,6860,7},
{52361,6866,11},
{52361,6876,17},
{53857,6892,2},
{53857,6908,1},
{53857,6924,4},
{53857,6940,3},
{53857,6946,11},
{53857,6956,17},
{63649,6972,2},
{63649,6988,1},
{63649,7004,4},
{63649,7020,3},
{63649,7026,13},
{63649,7038,17},
{72931,7054,3},
{72931,7056,5},
{72931,7060,11},
{72931,7070,13},
{72931,7082,17},
{74257,7098,2},
{74257,7106,1},
{74257,7114,4},
{74257,7122,3},
{74257,7124,7},
{74257,7130,13},
{74257,7142,17},
{78541,7158,2},
{78541,7160,3},
{78541,7162,5},
{78541,7166,7},
{78541,7172,11},
{78541,7182,17},
{79561,7198,2},
{79561,7202,1},
{79561,7206,4},
{79561,7210,3},
{79561,7216,5},
{79561,7220,13},
{79561,7232,17},
{87517,7248,2},
{87517,7250,3},
{87517,7256,11},
{87517,7266,13},
{87517,7278,17},
{92821,7294,2},
{92821,7296,3},
{92821,7298,5},
{92821,7302,7},
{92821,7308,13},
{92821,7320,17},
{97241,7336,2},
{97241,7340,1},
{97241,7344,4},
{97241,7348,5},
{97241,7352,11},
{97241,7362,13},
{97241,7374,17},
{102103,7390,3},
{102103,7392,7},
{102103,7398,11},
{102103,7408,13},
{102103,7420,17},
{116689,7436,2},
{116689,7444,1},
{116689,7452,4},
{116689,7460,3},
{116689,7462,11},
{116689,7472,13},
{116689,7484,17},
{117811,7500,3},
{117811,7506,5},
{117811,7510,7},
{117811,7516,11},
{117811,7526,17},
{145861,7542,2},
{145861,7544,3},
{145861,7546,5},
{145861,7550,11},
{145861,7560,13},
{145861,7572,17},
{148513,7588,2},
{148513,7604,1},
{148513,7620,4},
{148513,7636,3},
{148513,7638,7},
{148513,7644,13},
{148513,7656,17},
{157081,7672,2},
{157081,7676,1},
{157081,7680,4},
{157081,7684,3},
{157081,7686,5},
{157081,7690,7},
{157081,7696,11},
{157081,7706,17},
{185641,7722,2},
{185641,7726,1},
{185641,7730,4},
{185641,7734,3},
{185641,7736,5},
{185641,7740,7},
{185641,7746,13},
{185641,7758,17},
{209441,7774,2},
{209441,7790,1},
{209441,7806,4},
{209441,7822,5},
{209441,7826,7},
{209441,7832,11},
{209441,7842,17},
{235621,7858,2},
{235621,7860,3},
{235621,7866,5},
{235621,7870,7},
{235621,7876,11},
{235621,7886,17},
{269281,7902,2},
{269281,7918,1},
{269281,7934,4},
{269281,7950,3},
{269281,7956,5},
{269281,7960,11},
{269281,7970,17},
{291721,7986,2},
{291721,7990,1},
{291721,7994,4},
{291721,7998,3},
{291721,8000,5},
{291721,8004,11},
{291721,8014,13},
{291721,8026,17},
{314161,8042,2},
{314161,8050,1},
{314161,8058,4},
{314161,8066,3},
{314161,8068,5},
{314161,8072,7},
{314161,8078,11},
{314161,8088,17},
{371281,8104,2},
{371281,8112,1},
{371281,8120,4},
{371281,8128,3},
{371281,8130,5},
{371281,8134,7},
{371281,8140,13},
{371281,8152,17},
{388961,8168,2},
{388961,8184,1},
{388961,8200,4},
{388961,8216,5},
{388961,8220,11},
{388961,8230,13},
{388961,8242,17},
{445537,8258,2},
{445537,8274,1},
{445537,8290,4},
{445537,8306,3},
{445537,8312,7},
{445537,8318,13},
{445537,8330,17},
{471241,8346,2},
{471241,8350,1},
{471241,8354,4},
{471241,8358,3},
{471241,8364,5},
{471241,8368,7},
{471241,8374,11},
{471241,8384,17},
{612613,8400,2},
{612613,8402,3},
{612613,8408,7},
{612613,8414,11},
{612613,8424,13},
{612613,8436,17},
{680681,8452,2},
{680681,8456,1},
{680681,8460,4},
{680681,8464,5},
{680681,8468,7},
{680681,8474,11},
{680681,8484,13},
{680681,8496,17},
{700129,8512,2},
{700129,8528,1},
{700129,8544,4},
{700129,8560,3},
{700129,8566,11},
{700129,8576,13},
{700129,8588,17},
{816817,8604,2},
{816817,8612,1},
{816817,8620,4},
{816817,8628,3},
{816817,8630,7},
{816817,8636,11},
{816817,8646,13},
{816817,8658,17},
{1633633,8674,2},
{1633633,8690,1},
{1633633,8706,4},
{1633633,8722,3},
{1633633,8724,7},
{1633633,8730,11},
{1633633,8740,13},
{1633633,8752,17},
{4084081,8768,2},
{4084081,8776,1},
{4084081,8784,4},
{4084081,8792,3},
{4084081,8794,5},
{4084081,8798,7},
{4084081,8804,11},
{4084081,8814,13},
{4084081,8826,17},
{8168161,8842,2},
{8168161,8858,1},
{8168161,8874,4},
{8168161,8890,3},
{8168161,8892,5},
{8168161,8896,7},
{8168161,8902,11},
{8168161,8912,13},
{8168161,8924,17},
{919,8940,3},
{919,8958,17},
{3673,8974,2},
{3673,8978,1},
{3673,8982,4},
{3673,8986,3},
{3673,9004,17},
{4591,9020,3},
{4591,9038,5},
{4591,9042,17},
{6427,9058,3},
{6427,9076,7},
{6427,9082,17},
{9181,9098,2},
{9181,9100,3},
{9181,9118,5},
{9181,9122,17},
{10099,9138,3},
{10099,9156,11},
{10099,9166,17},
{12853,9182,2},
{12853,9184,3},
{12853,9202,7},
{12853,9208,17},
{23869,9224,2},
{23869,9226,3},
{23869,9244,13},
{23869,9256,17},
{36721,9272,2},
{36721,9280,1},
{36721,9288,4},
{36721,9296,3},
{36721,9314,5},
{36721,9318,17},
{47737,9334,2},
{47737,9338,1},
{47737,9342,4},
{47737,9346,3},
{47737,9364,13},
{47737,9376,17},
{59671,9392,3},
{59671,9410,5},
{59671,9414,13},
{59671,9426,17},
{70687,9442,3},
{70687,9460,7},
{70687,9466,11},
{70687,9476,17},
{100981,9492,2},
{100981,9494,3},
{100981,9512,5},
{100981,9516,11},
{100981,9526,17},
{128521,9542,2},
{128521,9546,1},
{128521,9550,4},
{128521,9554,3},
{128521,9572,5},
{128521,9576,7},
{128521,9582,17},
{161569,9598,2},
{161569,9614,1},
{161569,9630,4},
{161569,9646,3},
{161569,9664,11},
{161569,9674,17},
{167077,9690,2},
{167077,9692,3},
{167077,9710,7},
{167077,9716,13},
{167077,9728,17},
{201961,9744,2},
{201961,9748,1},
{201961,9752,4},
{201961,9756,3},
{201961,9774,5},
{201961,9778,11},
{201961,9788,17},
{238681,9804,2},
{238681,9808,1},
{238681,9812,4},
{238681,9816,3},
{238681,9834,5},
{238681,9838,13},
{238681,9850,17},
{417691,9866,3},
{417691,9884,5},
{417691,9888,7},
{417691,9894,13},
{417691,9906,17},
{477361,9922,2},
{477361,9930,1},
{477361,9938,4},
{477361,9946,3},
{477361,9964,5},
{477361,9968,13},
{477361,9980,17},
{514081,9996,2},
{514081,10012,1},
{514081,10028,4},
{514081,10044,3},
{514081,10062,5},
{514081,10066,7},
{514081,10072,17},
{565489,10088,2},
{565489,10096,1},
{565489,10104,4},
{565489,10112,3},
{565489,10130,7},
{565489,10136,11},
{565489,10146,17},
{656371,10162,3},
{656371,10180,5},
{656371,10184,11},
{656371,10194,13},
{656371,10206,17},
{1670761,10222,2},
{1670761,10226,1},
{1670761,10230,4},
{1670761,10234,3},
{1670761,10252,5},
{1670761,10256,7},
{1670761,10262,13},
{1670761,10274,17},
{1837837,10290,2},
{1837837,10292,3},
{1837837,10310,7},
{1837837,10316,11},
{1837837,10326,13},
{1837837,10338,17},
{2625481,10354,2},
{2625481,10358,1},
{2625481,10362,4},
{2625481,10366,3},
{2625481,10384,5},
{2625481,10388,11},
{2625481,10398,13},
{2625481,10410,17},
{5250961,10426,2},
{5250961,10434,1},
{5250961,10442,4},
{5250961,10450,3},
{5250961,10468,5},
{5250961,10472,11},
{5250961,10482,13},
{5250961,10494,17},
{5654881,10510,2},
{5654881,10526,1},
{5654881,10542,4},
{5654881,10558,3},
{5654881,10576,5},
{5654881,10580,7},
{5654881,10586,11},
{5654881,10596,17},
{9189181,10612,2},
{9189181,10614,3},
{9189181,10632,5},
{9189181,10636,7},
{9189181,10642,11},
{9189181,10652,13},
{9189181,10664,17},
{10501921,10680,2},
{10501921,10696,1},
{10501921,10712,4},
{10501921,10728,3},
{10501921,10746,5},
{10501921,10750,11},
{10501921,10760,13},
{10501921,10772,17},
{101,10788,2},
{101,10790,5},
{151,10810,3},
{151,10812,5},
{401,10832,2},
{401,10840,1},
{401,10848,4},
{401,10856,5},
{601,10876,2},
{601,10880,1},
{601,10884,4},
{601,10888,3},
{601,10890,5},
{701,10910,2},
{701,10912,5},
{701,10932,7},
{1051,10938,3},
{1051,10940,5},
{1051,10960,7},
{1201,10966,2},
{1201,10974,1},
{1201,10982,4},
{1201,10990,3},
{1201,10992,5},
{1301,11012,2},
{1301,11014,5},
{1301,11034,13},
{1801,11046,2},
{1801,11050,1},
{1801,11054,4},
{1801,11058,3},
{1801,11064,5},
{1951,11084,3},
{1951,11086,5},
{1951,11106,13},
{2551,11118,3},
{2551,11120,5},
{2551,11140,17},
{2801,11156,2},
{2801,11164,1},
{2801,11172,4},
{2801,11180,5},
{2801,11200,7},
{3301,11206,2},
{3301,11208,3},
{3301,11210,5},
{3301,11230,11},
{3851,11240,5},
{3851,11260,7},
{3851,11266,11},
{4201,11276,2},
{4201,11280,1},
{4201,11284,4},
{4201,11288,3},
{4201,11290,5},
{4201,11310,7},
{4951,11316,3},
{4951,11322,5},
{4951,11342,11},
{5101,11352,2},
{5101,11354,3},
{5101,11356,5},
{5101,11376,17},
{5851,11392,3},
{5851,11398,5},
{5851,11418,13},
{6301,11430,2},
{6301,11432,3},
{6301,11438,5},
{6301,11458,7},
{7151,11464,5},
{7151,11484,11},
{7151,11494,13},
{9901,11506,2},
{9901,11508,3},
{9901,11514,5},
{9901,11534,11},
{11551,11544,3},
{11551,11546,5},
{11551,11566,7},
{11551,11572,11},
{11701,11582,2},
{11701,11584,3},
{11701,11590,5},
{11701,11610,13},
{12601,11622,2},
{12601,11626,1},
{12601,11630,4},
{12601,11634,3},
{12601,11640,5},
{12601,11660,7},
{14851,11666,3},
{14851,11684,5},
{14851,11704,11},
{15401,11714,2},
{15401,11718,1},
{15401,11722,4},
{15401,11726,5},
{15401,11746,7},
{15401,11752,11},
{15601,11762,2},
{15601,11770,1},
{15601,11778,4},
{15601,11786,3},
{15601,11788,5},
{15601,11808,13},
{17551,11820,3},
{17551,11838,5},
{17551,11858,13},
{17851,11870,3},
{17851,11872,5},
{17851,11892,7},
{17851,11898,17},
{18701,11914,2},
{18701,11916,5},
{18701,11936,11},
{18701,11946,17},
{19801,11962,2},
{19801,11966,1},
{19801,11970,4},
{19801,11974,3},
{19801,11980,5},
{19801,12000,11},
{21601,12010,2},
{21601,12026,1},
{21601,12042,4},
{21601,12058,3},
{21601,12076,5},
{23801,12096,2},
{23801,12100,1},
{23801,12104,4},
{23801,12108,5},
{23801,12128,7},
{23801,12134,17},
{28051,12150,3},
{28051,12152,5},
{28051,12172,11},
{28051,12182,17},
{33151,12198,3},
{33151,12200,5},
{33151,12220,13},
{33151,12232,17},
{34651,12248,3},
{34651,12254,5},
{34651,12274,7},
{34651,12280,11},
{40801,12290,2},
{40801,12306,1},
{40801,12322,4},
{40801,12338,3},
{40801,12340,5},
{40801,12360,17},
{42901,12376,2},
{42901,12378,3},
{42901,12380,5},
{42901,12400,11},
{42901,12410,13},
{44201,12422,2},
{44201,12426,1},
{44201,12430,4},
{44201,12434,5},
{44201,12454,13},
{44201,12466,17},
{50051,12482,5},
{50051,12502,7},
{50051,12508,11},
{50051,12518,13},
{53551,12530,3},
{53551,12536,5},
{53551,12556,7},
{53551,12562,17},
{54601,12578,2},
{54601,12582,1},
{54601,12586,4},
{54601,12590,3},
{54601,12592,5},
{54601,12612,7},
{54601,12618,13},
{56101,12630,2},
{56101,12632,3},
{56101,12634,5},
{56101,12654,11},
{56101,12664,17},
{66301,12680,2},
{66301,12682,3},
{66301,12684,5},
{66301,12704,13},
{66301,12716,17},
{70201,12732,2},
{70201,12736,1},
{70201,12740,4},
{70201,12744,3},
{70201,12762,5},
{70201,12782,13},
{77351,12794,5},
{77351,12814,7},
{77351,12820,13},
{77351,12832,17},
{79201,12848,2},
{79201,12864,1},
{79201,12880,4},
{79201,12896,3},
{79201,12902,5},
{79201,12922,11},
{81901,12932,2},
{81901,12934,3},
{81901,12940,5},
{81901,12960,7},
{81901,12966,13},
{91801,12978,2},
{91801,12982,1},
{91801,12986,4},
{91801,12990,3},
{91801,13008,5},
{91801,13028,17},
{92401,13044,2},
{92401,13052,1},
{92401,13060,4},
{92401,13068,3},
{92401,13070,5},
{92401,13090,7},
{92401,13096,11},
{93601,13106,2},
{93601,13122,1},
{93601,13138,4},
{93601,13154,3},
{93601,13160,5},
{93601,13180,13},
{103951,13192,3},
{103951,13210,5},
{103951,13230,7},
{103951,13236,11},
{107101,13246,2},
{107101,13248,3},
{107101,13254,5},
{107101,13274,7},
{107101,13280,17},
{109201,13296,2},
{109201,13304,1},
{109201,13312,4},
{109201,13320,3},
{109201,13322,5},
{109201,13342,7},
{109201,13348,13},
{118801,13360,2},
{118801,13368,1},
{118801,13376,4},
{118801,13384,3},
{118801,13402,5},
{118801,13422,11},
{122401,13432,2},
{122401,13448,1},
{122401,13464,4},
{122401,13480,3},
{122401,13486,5},
{122401,13506,17},
{140401,13522,2},
{140401,13530,1},
{140401,13538,4},
{140401,13546,3},
{140401,13564,5},
{140401,13584,13},
{150151,13596,3},
{150151,13598,5},
{150151,13618,7},
{150151,13624,11},
{150151,13634,13},
{151201,13646,2},
{151201,13662,1},
{151201,13678,4},
{151201,13694,3},
{151201,13712,5},
{151201,13732,7},
{160651,13738,3},
{160651,13756,5},
{160651,13776,7},
{160651,13782,17},
{193051,13798,3},
{193051,13816,5},
{193051,13836,11},
{193051,13846,13},
{198901,13858,2},
{198901,13860,3},
{198901,13866,5},
{198901,13886,13},
{198901,13898,17},
{200201,13914,2},
{200201,13918,1},
{200201,13922,4},
{200201,13926,5},
{200201,13946,7},
{200201,13952,11},
{200201,13962,13},
{218401,13974,2},
{218401,13990,1},
{218401,14006,4},
{218401,14022,3},
{218401,14024,5},
{218401,14044,7},
{218401,14050,13},
{224401,14062,2},
{224401,14070,1},
{224401,14078,4},
{224401,14086,3},
{224401,14088,5},
{224401,14108,11},
{224401,14118,17},
{232051,14134,3},
{232051,14136,5},
{232051,14156,7},
{232051,14162,13},
{232051,14174,17},
{243101,14190,2},
{243101,14192,5},
{243101,14212,11},
{243101,14222,13},
{243101,14234,17},
{257401,14250,2},
{257401,14254,1},
{257401,14258,4},
{257401,14262,3},
{257401,14268,5},
{257401,14288,11},
{257401,14298,13},
{300301,14310,2},
{300301,14312,3},
{300301,14314,5},
{300301,14334,7},
{300301,14340,11},
{300301,14350,13},
{321301,14362,2},
{321301,14364,3},
{321301,14382,5},
{321301,14402,7},
{321301,14408,17},
{367201,14424,2},
{367201,14440,1},
{367201,14456,4},
{367201,14472,3},
{367201,14490,5},
{367201,14510,17},
{415801,14526,2},
{415801,14530,1},
{415801,14534,4},
{415801,14538,3},
{415801,14556,5},
{415801,14576,7},
{415801,14582,11},
{428401,14592,2},
{428401,14600,1},
{428401,14608,4},
{428401,14616,3},
{428401,14622,5},
{428401,14642,7},
{428401,14648,17},
{448801,14664,2},
{448801,14680,1},
{448801,14696,4},
{448801,14712,3},
{448801,14714,5},
{448801,14734,11},
{448801,14744,17},
{450451,14760,3},
{450451,14766,5},
{450451,14786,7},
{450451,14792,11},
{450451,14802,13},
{504901,14814,2},
{504901,14816,3},
{504901,14834,5},
{504901,14854,11},
{504901,14864,17},
{530401,14880,2},
{530401,14896,1},
{530401,14912,4},
{530401,14928,3},
{530401,14930,5},
{530401,14950,13},
{530401,14962,17},
{600601,14978,2},
{600601,14982,1},
{600601,14986,4},
{600601,14990,3},
{600601,14992,5},
{600601,15012,7},
{600601,15018,11},
{600601,15028,13},
{673201,15040,2},
{673201,15048,1},
{673201,15056,4},
{673201,15064,3},
{673201,15070,5},
{673201,15090,11},
{673201,15100,17},
{729301,15116,2},
{729301,15118,3},
{729301,15120,5},
{729301,15140,11},
{729301,15150,13},
{729301,15162,17},
{795601,15178,2},
{795601,15186,1},
{795601,15194,4},
{795601,15202,3},
{795601,15208,5},
{795601,15228,13},
{795601,15240,17},
{800801,15256,2},
{800801,15272,1},
{800801,15288,4},
{800801,15304,5},
{800801,15324,7},
{800801,15330,11},
{800801,15340,13},
{982801,15352,2},
{982801,15360,1},
{982801,15368,4},
{982801,15376,3},
{982801,15394,5},
{982801,15414,7},
{982801,15420,13},
{1029601,15432,2},
{1029601,15448,1},
{1029601,15464,4},
{1029601,15480,3},
{1029601,15486,5},
{1029601,15506,11},
{1029601,15516,13},
{1093951,15528,3},
{1093951,15534,5},
{1093951,15554,11},
{1093951,15564,13},
{1093951,15576,17},
{1178101,15592,2},
{1178101,15594,3},
{1178101,15600,5},
{1178101,15620,7},
{1178101,15626,11},
{1178101,15636,17},
{1201201,15652,2},
{1201201,15660,1},
{1201201,15668,4},
{1201201,15676,3},
{1201201,15678,5},
{1201201,15698,7},
{1201201,15704,11},
{1201201,15714,13},
{1458601,15726,2},
{1458601,15730,1},
{1458601,15734,4},
{1458601,15738,3},
{1458601,15740,5},
{1458601,15760,11},
{1458601,15770,13},
{1458601,15782,17},
{2088451,15798,3},
{2088451,15816,5},
{2088451,15836,7},
{2088451,15842,13},
{2088451,15854,17},
{2187901,15870,2},
{2187901,15872,3},
{2187901,15878,5},
{2187901,15898,11},
{2187901,15908,13},
{2187901,15920,17},
{2402401,15936,2},
{2402401,15952,1},
{2402401,15968,4},
{2402401,15984,3},
{2402401,15986,5},
{2402401,16006,7},
{2402401,16012,11},
{2402401,16022,13},
{2570401,16034,2},
{2570401,16050,1},
{2570401,16066,4},
{2570401,16082,3},
{2570401,16100,5},
{2570401,16120,7},
{2570401,16126,17},
{2702701,16142,2},
{2702701,16144,3},
{2702701,16162,5},
{2702701,16182,7},
{2702701,16188,11},
{2702701,16198,13},
{3088801,16210,2},
{3088801,16226,1},
{3088801,16242,4},
{3088801,16258,3},
{3088801,16276,5},
{3088801,16296,11},
{3088801,16306,13},
{3141601,16318,2},
{3141601,16334,1},
{3141601,16350,4},
{3141601,16366,3},
{3141601,16368,5},
{3141601,16388,7},
{3141601,16394,11},
{3141601,16404,17},
{3712801,16420,2},
{3712801,16436,1},
{3712801,16452,4},
{3712801,16468,3},
{3712801,16470,5},
{3712801,16490,7},
{3712801,16496,13},
{3712801,16508,17},
{5105101,16524,2},
{5105101,16526,3},
{5105101,16528,5},
{5105101,16548,7},
{5105101,16554,11},
{5105101,16564,13},
{5105101,16576,17},
{5834401,16592,2},
{5834401,16608,1},
{5834401,16624,4},
{5834401,16640,3},
{5834401,16642,5},
{5834401,16662,11},
{5834401,16672,13},
{5834401,16684,17},
{6806801,16700,2},
{6806801,16708,1},
{6806801,16716,4},
{6806801,16724,5},
{6806801,16744,7},
{6806801,16750,11},
{6806801,16760,13},
{6806801,16772,17},
{7068601,16788,2},
{7068601,16792,1},
{7068601,16796,4},
{7068601,16800,3},
{7068601,16818,5},
{7068601,16838,7},
{7068601,16844,11},
{7068601,16854,17},
{8353801,16870,2},
{8353801,16874,1},
{8353801,16878,4},
{8353801,16882,3},
{8353801,16900,5},
{8353801,16920,7},
{8353801,16926,13},
{8353801,16938,17},
{17503201,16954,2},
{17503201,16970,1},
{17503201,16986,4},
{17503201,17002,3},
{17503201,17008,5},
{17503201,17028,11},
{17503201,17038,13},
{17503201,17050,17},
{22972951,17066,3},
{22972951,17084,5},
{22972951,17104,7},
{22972951,17110,11},
{22972951,17120,13},
{22972951,17132,17},
{52509601,17148,2},
{52509601,17164,1},
{52509601,17180,4},
{52509601,17196,3},
{52509601,17214,5},
{52509601,17234,11},
{52509601,17244,13},
{52509601,17256,17},
{183783601,17272,2},
{183783601,17280,1},
{183783601,17288,4},
{183783601,17296,3},
{183783601,17314,5},
{183783601,17334,7},
{183783601,17340,11},
{183783601,17350,13},
{183783601,17362,17},
{367567201,17378,2},
{367567201,17394,1},
{367567201,17410,4},
{367567201,17426,3},
{367567201,17444,5},
{367567201,17464,7},
{367567201,17470,11},
{367567201,17480,13},
{367567201,17492,17},
{191,17508,5},
{191,17512,19},
{229,17530,2},
{229,17532,3},
{229,17534,19},
{419,17552,11},
{419,17562,19},
{457,17580,2},
{457,17584,1},
{457,17588,4},
{457,17592,3},
{457,17594,19},
{571,17612,3},
{571,17614,5},
{571,17618,19},
{647,17636,17},
{647,17652,19},
{761,17670,2},
{761,17674,1},
{761,17678,4},
{761,17682,5},
{761,17686,19},
{1483,17704,3},
{1483,17706,13},
{1483,17718,19},
{1597,17736,2},
{1597,17738,3},
{1597,17740,7},
{1597,17746,19},
{2053,17764,2},
{2053,17766,3},
{2053,17784,19},
{2129,17802,2},
{2129,17810,1},
{2129,17818,4},
{2129,17826,7},
{2129,17832,19},
{2281,17850,2},
{2281,17854,1},
{2281,17858,4},
{2281,17862,3},
{2281,17864,5},
{2281,17868,19},
{2927,17886,7},
{2927,17892,11},
{2927,17902,19},
{3041,17920,2},
{3041,17936,1},
{3041,17952,4},
{3041,17968,5},
{3041,17972,19},
{3877,17990,2},
{3877,17992,3},
{3877,17994,17},
{3877,18010,19},
{4447,18028,3},
{4447,18034,13},
{4447,18046,19},
{4523,18064,7},
{4523,18070,17},
{4523,18086,19},
{4561,18104,2},
{4561,18112,1},
{4561,18120,4},
{4561,18128,3},
{4561,18130,5},
{4561,18134,19},
{4789,18152,2},
{4789,18154,3},
{4789,18160,7},
{4789,18166,19},
{6271,18184,3},
{6271,18186,5},
{6271,18190,11},
{6271,18200,19},
{6689,18218,2},
{6689,18234,1},
{6689,18250,4},
{6689,18266,11},
{6689,18276,19},
{6841,18294,2},
{6841,18298,1},
{6841,18302,4},
{6841,18306,3},
{6841,18312,5},
{6841,18316,19},
{6917,18334,2},
{6917,18336,7},
{6917,18342,13},
{6917,18354,19},
{7411,18372,3},
{7411,18374,5},
{7411,18378,13},
{7411,18390,19},
{7753,18408,2},
{7753,18412,1},
{7753,18416,4},
{7753,18420,3},
{7753,18422,17},
{7753,18438,19},
{8209,18456,2},
{8209,18464,1},
{8209,18472,4},
{8209,18480,3},
{8209,18498,19},
{8779,18516,3},
{8779,18518,7},
{8779,18524,11},
{8779,18534,19},
{8893,18552,2},
{8893,18554,3},
{8893,18560,13},
{8893,18572,19},
{10337,18590,2},
{10337,18606,1},
{10337,18622,4},
{10337,18638,17},
{10337,18654,19},
{11287,18672,3},
{11287,18690,11},
{11287,18700,19},
{11971,18718,3},
{11971,18724,5},
{11971,18728,7},
{11971,18734,19},
{12541,18752,2},
{12541,18754,3},
{12541,18756,5},
{12541,18760,11},
{12541,18770,19},
{13339,18788,3},
{13339,18806,13},
{13339,18818,19},
{13567,18836,3},
{13567,18838,7},
{13567,18844,17},
{13567,18860,19},
{13681,18878,2},
{13681,18886,1},
{13681,18894,4},
{13681,18902,3},
{13681,18908,5},
{13681,18912,19},
{14821,18930,2},
{14821,18932,3},
{14821,18934,5},
{14821,18938,13},
{14821,18950,19},
{16417,18968,2},
{16417,18984,1},
{16417,19000,4},
{16417,19016,3},
{16417,19034,19},
{17291,19052,5},
{17291,19056,7},
{17291,19062,13},
{17291,19074,19},
{17443,19092,3},
{17443,19110,17},
{17443,19126,19},
{18089,19144,2},
{18089,19148,1},
{18089,19152,4},
{18089,19156,7},
{18089,19162,17},
{18089,19178,19},
{19381,19196,2},
{19381,19198,3},
{19381,19200,5},
{19381,19204,17},
{19381,19220,19},
{20521,19238,2},
{20521,19242,1},
{20521,19246,4},
{20521,19250,3},
{20521,19268,5},
{20521,19272,19},
{20749,19290,2},
{20749,19292,3},
{20749,19294,7},
{20749,19300,13},
{20749,19312,19},
{21319,19330,3},
{21319,19332,11},
{21319,19342,17},
{21319,19358,19},
{21737,19376,2},
{21737,19380,1},
{21737,19384,4},
{21737,19388,11},
{21737,19398,13},
{21737,19410,19},
{22573,19428,2},
{22573,19430,3},
{22573,19448,11},
{22573,19458,19},
{25841,19476,2},
{25841,19484,1},
{25841,19492,4},
{25841,19500,5},
{25841,19504,17},
{25841,19520,19},
{27361,19538,2},
{27361,19554,1},
{27361,19570,4},
{27361,19586,3},
{27361,19592,5},
{27361,19596,19},
{28729,19614,2},
{28729,19618,1},
{28729,19622,4},
{28729,19626,3},
{28729,19644,7},
{28729,19650,19},
{29641,19668,2},
{29641,19672,1},
{29641,19676,4},
{29641,19680,3},
{29641,19682,5},
{29641,19686,13},
{29641,19698,19},
{30097,19716,2},
{30097,19724,1},
{30097,19732,4},
{30097,19740,3},
{30097,19746,11},
{30097,19756,19},
{31123,19774,3},
{31123,19780,7},
{31123,19786,13},
{31123,19798,19},
{35531,19816,5},
{35531,19820,11},
{35531,19830,17},
{35531,19846,19},
{35569,19864,2},
{35569,19872,1},
{35569,19880,4},
{35569,19888,3},
{35569,19894,13},
{35569,19906,19},
{35911,19924,3},
{35911,19942,5},
{35911,19946,7},
{35911,19952,19},
{38039,19970,7},
{38039,19976,11},
{38039,19986,13},
{38039,19998,19},
{39521,20016,2},
{39521,20032,1},
{39521,20048,4},
{39521,20064,5},
{39521,20068,13},
{39521,20080,19},
{40699,20098,3},
{40699,20104,7},
{40699,20110,17},
{40699,20126,19},
{43891,20144,3},
{43891,20146,5},
{43891,20150,7},
{43891,20156,11},
{43891,20166,19},
{46817,20184,2},
{46817,20200,1},
{46817,20216,4},
{46817,20232,7},
{46817,20238,11},
{46817,20248,19},
{47881,20266,2},
{47881,20270,1},
{47881,20274,4},
{47881,20278,3},
{47881,20284,5},
{47881,20288,7},
{47881,20294,19},
{48907,20312,3},
{48907,20318,11},
{48907,20328,13},
{48907,20340,19},
{51871,20358,3},
{51871,20360,5},
{51871,20364,7},
{51871,20370,13},
{51871,20382,19},
{53353,20400,2},
{53353,20404,1},
{53353,20408,4},
{53353,20412,3},
{53353,20430,13},
{53353,20442,19},
{56431,20460,3},
{56431,20478,5},
{56431,20482,11},
{56431,20492,19},
{57457,20510,2},
{57457,20518,1},
{57457,20526,4},
{57457,20534,3},
{57457,20552,7},
{57457,20558,19},
{58787,20576,7},
{58787,20582,13},
{58787,20594,17},
{58787,20610,19},
{59281,20628,2},
{59281,20636,1},
{59281,20644,4},
{59281,20652,3},
{59281,20654,5},
{59281,20658,13},
{59281,20670,19},
{63841,20688,2},
{63841,20704,1},
{63841,20720,4},
{63841,20736,3},
{63841,20738,5},
{63841,20742,7},
{63841,20748,19},
{71821,20766,2},
{71821,20768,3},
{71821,20786,5},
{71821,20790,7},
{71821,20796,19},
{72353,20814,2},
{72353,20830,1},
{72353,20846,4},
{72353,20862,7},
{72353,20868,17},
{72353,20884,19},
{75583,20902,3},
{75583,20908,13},
{75583,20920,17},
{75583,20936,19},
{77521,20954,2},
{77521,20962,1},
{77521,20970,4},
{77521,20978,3},
{77521,20980,5},
{77521,20984,17},
{77521,21000,19},
{87211,21018,3},
{87211,21036,5},
{87211,21040,17},
{87211,21056,19},
{90289,21074,2},
{90289,21082,1},
{90289,21090,4},
{90289,21098,3},
{90289,21116,11},
{90289,21126,19},
{97813,21144,2},
{97813,21146,3},
{97813,21152,11},
{97813,21162,13},
{97813,21174,19},
{105337,21192,2},
{105337,21196,1},
{105337,21200,4},
{105337,21204,3},
{105337,21210,7},
{105337,21216,11},
{105337,21226,19},
{106591,21244,3},
{106591,21246,5},
{106591,21250,11},
{106591,21260,17},
{106591,21276,19},
{108529,21294,2},
{108529,21302,1},
{108529,21310,4},
{108529,21318,3},
{108529,21320,7},
{108529,21326,17},
{108529,21342,19},
{114913,21360,2},
{114913,21376,1},
{114913,21392,4},
{114913,21408,3},
{114913,21426,7},
{114913,21432,19},
{117041,21450,2},
{117041,21458,1},
{117041,21466,4},
{117041,21474,5},
{117041,21478,7},
{117041,21484,11},
{117041,21494,19},
{124489,21512,2},
{124489,21516,1},
{124489,21520,4},
{124489,21524,3},
{124489,21530,7},
{124489,21536,13},
{124489,21548,19},
{131671,21566,3},
{131671,21572,5},
{131671,21576,7},
{131671,21582,11},
{131671,21592,19},
{134369,21610,2},
{134369,21626,1},
{134369,21642,4},
{134369,21658,13},
{134369,21670,17},
{134369,21686,19},
{135661,21704,2},
{135661,21706,3},
{135661,21708,5},
{135661,21712,7},
{135661,21718,17},
{135661,21734,19},
{139537,21752,2},
{139537,21760,1},
{139537,21768,4},
{139537,21776,3},
{139537,21794,17},
{139537,21810,19},
{140449,21828,2},
{140449,21844,1},
{140449,21860,4},
{140449,21876,3},
{140449,21878,7},
{140449,21884,11},
{140449,21894,19},
{146719,21912,3},
{146719,21930,11},
{146719,21940,13},
{146719,21952,19},
{163021,21970,2},
{163021,21972,3},
{163021,21974,5},
{163021,21978,11},
{163021,21988,13},
{163021,22000,19},
{177841,22018,2},
{177841,22026,1},
{177841,22034,4},
{177841,22042,3},
{177841,22048,5},
{177841,22052,13},
{177841,22064,19},
{186733,22082,2},
{186733,22084,3},
{186733,22102,7},
{186733,22108,13},
{186733,22120,19},
{207481,22138,2},
{207481,22142,1},
{207481,22146,4},
{207481,22150,3},
{207481,22152,5},
{207481,22156,7},
{207481,22162,13},
{207481,22174,19},
{213181,22192,2},
{213181,22194,3},
{213181,22196,5},
{213181,22200,11},
{213181,22210,17},
{213181,22226,19},
{217057,22244,2},
{217057,22260,1},
{217057,22276,4},
{217057,22292,3},
{217057,22294,7},
{217057,22300,17},
{217057,22316,19},
{217361,22334,2},
{217361,22342,1},
{217361,22350,4},
{217361,22358,5},
{217361,22362,11},
{217361,22372,13},
{217361,22384,19},
{225721,22402,2},
{225721,22406,1},
{225721,22410,4},
{225721,22414,3},
{225721,22432,5},
{225721,22436,11},
{225721,22446,19},
{251941,22464,2},
{251941,22466,3},
{251941,22468,5},
{251941,22472,13},
{251941,22484,17},
{251941,22500,19},
{279073,22518,2},
{279073,22534,1},
{279073,22550,4},
{279073,22566,3},
{279073,22584,17},
{279073,22600,19},
{287281,22618,2},
{287281,22626,1},
{287281,22634,4},
{287281,22642,3},
{287281,22660,5},
{287281,22664,7},
{287281,22670,19},
{300961,22688,2},
{300961,22704,1},
{300961,22720,4},
{300961,22736,3},
{300961,22742,5},
{300961,22746,11},
{300961,22756,19},
{302329,22774,2},
{302329,22778,1},
{302329,22782,4},
{302329,22786,3},
{302329,22792,13},
{302329,22804,17},
{302329,22820,19},
{342343,22838,3},
{342343,22844,7},
{342343,22850,11},
{342343,22860,13},
{342343,22872,19},
{351121,22890,2},
{351121,22898,1},
{351121,22906,4},
{351121,22914,3},
{351121,22916,5},
{351121,22920,7},
{351121,22926,11},
{351121,22936,19},
{377911,22954,3},
{377911,22960,5},
{377911,22964,13},
{377911,22976,17},
{377911,22992,19},
{391249,23010,2},
{391249,23018,1},
{391249,23026,4},
{391249,23034,3},
{391249,23040,11},
{391249,23050,13},
{391249,23062,19},
{406981,23080,2},
{406981,23082,3},
{406981,23088,5},
{406981,23092,7},
{406981,23098,17},
{406981,23114,19},
{451441,23132,2},
{451441,23140,1},
{451441,23148,4},
{451441,23156,3},
{451441,23174,5},
{451441,23178,11},
{451441,23188,19},
{456457,23206,2},
{456457,23210,1},
{456457,23214,4},
{456457,23218,3},
{456457,23220,7},
{456457,23226,11},
{456457,23236,13},
{456457,23248,19},
{461891,23266,5},
{461891,23270,11},
{461891,23280,13},
{461891,23292,17},
{461891,23308,19},
{489061,23326,2},
{489061,23328,3},
{489061,23334,5},
{489061,23338,11},
{489061,23348,13},
{489061,23360,19},
{511633,23378,2},
{511633,23386,1},
{511633,23394,4},
{511633,23402,3},
{511633,23408,11},
{511633,23418,17},
{511633,23434,19},
{526681,23452,2},
{526681,23456,1},
{526681,23460,4},
{526681,23464,3},
{526681,23470,5},
{526681,23474,7},
{526681,23480,11},
{526681,23490,19},
{554269,23508,2},
{554269,23510,3},
{554269,23512,11},
{554269,23522,13},
{554269,23534,17},
{554269,23550,19},
{568481,23568,2},
{568481,23584,1},
{568481,23600,4},
{568481,23616,5},
{568481,23620,11},
{568481,23630,17},
{568481,23646,19},
{608609,23664,2},
{608609,23680,1},
{608609,23696,4},
{608609,23712,7},
{608609,23718,11},
{608609,23728,13},
{608609,23740,19},
{651169,23758,2},
{651169,23774,1},
{651169,23790,4},
{651169,23806,3},
{651169,23812,7},
{651169,23818,17},
{651169,23834,19},
{652081,23852,2},
{652081,23860,1},
{652081,23868,4},
{652081,23876,3},
{652081,23878,5},
{652081,23882,11},
{652081,23892,13},
{652081,23904,19},
{697681,23922,2},
{697681,23930,1},
{697681,23938,4},
{697681,23946,3},
{697681,23964,5},
{697681,23968,17},
{697681,23984,19},
{733591,24002,3},
{733591,24020,5},
{733591,24024,11},
{733591,24034,13},
{733591,24046,19},
{782497,24064,2},
{782497,24080,1},
{782497,24096,4},
{782497,24112,3},
{782497,24118,11},
{782497,24128,13},
{782497,24140,19},
{790021,24158,2},
{790021,24160,3},
{790021,24178,5},
{790021,24182,7},
{790021,24188,11},
{790021,24198,19},
{813961,24216,2},
{813961,24220,1},
{813961,24224,4},
{813961,24228,3},
{813961,24234,5},
{813961,24238,7},
{813961,24244,17},
{813961,24260,19},
{895357,24278,2},
{895357,24280,3},
{895357,24286,7},
{895357,24292,11},
{895357,24302,17},
{895357,24318,19},
{1027027,24336,3},
{1027027,24354,7},
{1027027,24360,11},
{1027027,24370,13},
{1027027,24382,19},
{1053361,24400,2},
{1053361,24408,1},
{1053361,24416,4},
{1053361,24424,3},
{1053361,24430,5},
{1053361,24434,7},
{1053361,24440,11},
{1053361,24450,19},
{1058149,24468,2},
{1058149,24470,3},
{1058149,24476,7},
{1058149,24482,13},
{1058149,24494,17},
{1058149,24510,19},
{1108537,24528,2},
{1108537,24532,1},
{1108537,24536,4},
{1108537,24540,3},
{1108537,24542,11},
{1108537,24552,13},
{1108537,24564,17},
{1108537,24580,19},
{1133731,24598,3},
{1133731,24616,5},
{1133731,24620,13},
{1133731,24632,17},
{1133731,24648,19},
{1264033,24666,2},
{1264033,24682,1},
{1264033,24698,4},
{1264033,24714,3},
{1264033,24732,7},
{1264033,24738,11},
{1264033,24748,19},
{1279081,24766,2},
{1279081,24770,1},
{1279081,24774,4},
{1279081,24778,3},
{1279081,24784,5},
{1279081,24788,11},
{1279081,24798,17},
{1279081,24814,19},
{1369369,24832,2},
{1369369,24836,1},
{1369369,24840,4},
{1369369,24844,3},
{1369369,24850,7},
{1369369,24856,11},
{1369369,24866,13},
{1369369,24878,19},
{1492261,24896,2},
{1492261,24898,3},
{1492261,24900,5},
{1492261,24904,7},
{1492261,24910,11},
{1492261,24920,17},
{1492261,24936,19},
{1580041,24954,2},
{1580041,24958,1},
{1580041,24962,4},
{1580041,24966,3},
{1580041,24984,5},
{1580041,24988,7},
{1580041,24994,11},
{1580041,25004,19},
{1790713,25022,2},
{1790713,25026,1},
{1790713,25030,4},
{1790713,25034,3},
{1790713,25040,7},
{1790713,25046,11},
{1790713,25056,17},
{1790713,25072,19},
{1813969,25090,2},
{1813969,25098,1},
{1813969,25106,4},
{1813969,25114,3},
{1813969,25132,13},
{1813969,25144,17},
{1813969,25160,19},
{1867321,25178,2},
{1867321,25182,1},
{1867321,25186,4},
{1867321,25190,3},
{1867321,25208,5},
{1867321,25212,7},
{1867321,25218,13},
{1867321,25230,19},
{1939939,25248,3},
{1939939,25250,7},
{1939939,25256,11},
{1939939,25266,13},
{1939939,25278,17},
{1939939,25294,19},
{2217073,25312,2},
{2217073,25320,1},
{2217073,25328,4},
{2217073,25336,3},
{2217073,25338,11},
{2217073,25348,13},
{2217073,25360,17},
{2217073,25376,19},
{2238391,25394,3},
{2238391,25400,5},
{2238391,25404,7},
{2238391,25410,11},
{2238391,25420,17},
{2238391,25436,19},
{2282281,25454,2},
{2282281,25458,1},
{2282281,25462,4},
{2282281,25466,3},
{2282281,25468,5},
{2282281,25472,7},
{2282281,25478,11},
{2282281,25488,13},
{2282281,25500,19},
{2351441,25518,2},
{2351441,25526,1},
{2351441,25534,4},
{2351441,25542,5},
{2351441,25546,7},
{2351441,25552,13},
{2351441,25564,17},
{2351441,25580,19},
{2489761,25598,2},
{2489761,25614,1},
{2489761,25630,4},
{2489761,25646,3},
{2489761,25652,5},
{2489761,25656,7},
{2489761,25662,13},
{2489761,25674,19},
{2645371,25692,3},
{2645371,25698,5},
{2645371,25702,7},
{2645371,25708,13},
{2645371,25720,17},
{2645371,25736,19},
{2771341,25754,2},
{2771341,25756,3},
{2771341,25758,5},
{2771341,25762,11},
{2771341,25772,13},
{2771341,25784,17},
{2771341,25800,19},
{2934361,25818,2},
{2934361,25822,1},
{2934361,25826,4},
{2934361,25830,3},
{2934361,25848,5},
{2934361,25852,11},
{2934361,25862,13},
{2934361,25874,19},
{2984521,25892,2},
{2984521,25896,1},
{2984521,25900,4},
{2984521,25904,3},
{2984521,25906,5},
{2984521,25910,7},
{2984521,25916,11},
{2984521,25926,17},
{2984521,25942,19},
{3233231,25960,5},
{3233231,25964,7},
{3233231,25970,11},
{3233231,25980,13},
{3233231,25992,17},
{3233231,26008,19},
{3627937,26026,2},
{3627937,26042,1},
{3627937,26058,4},
{3627937,26074,3},
{3627937,26092,13},
{3627937,26104,17},
{3627937,26120,19},
{3837241,26138,2},
{3837241,26142,1},
{3837241,26146,4},
{3837241,26150,3},
{3837241,26168,5},
{3837241,26172,11},
{3837241,26182,17},
{3837241,26198,19},
{3912481,26216,2},
{3912481,26232,1},
{3912481,26248,4},
{3912481,26264,3},
{3912481,26270,5},
{3912481,26274,11},
{3912481,26284,13},
{3912481,26296,19},
{3979361,26314,2},
{3979361,26330,1},
{3979361,26346,4},
{3979361,26362,5},
{3979361,26366,7},
{3979361,26372,11},
{3979361,26382,17},
{3979361,26398,19},
{4157011,26416,3},
{4157011,26422,5},
{4157011,26426,11},
{4157011,26436,13},
{4157011,26448,17},
{4157011,26464,19},
{4232593,26482,2},
{4232593,26490,1},
{4232593,26498,4},
{4232593,26506,3},
{4232593,26512,7},
{4232593,26518,13},
{4232593,26530,17},
{4232593,26546,19},
{4476781,26564,2},
{4476781,26566,3},
{4476781,26572,5},
{4476781,26576,7},
{4476781,26582,11},
{4476781,26592,17},
{4476781,26608,19},
{5135131,26626,3},
{5135131,26644,5},
{5135131,26648,7},
{5135131,26654,11},
{5135131,26664,13},
{5135131,26676,19},
{5372137,26694,2},
{5372137,26698,1},
{5372137,26702,4},
{5372137,26706,3},
{5372137,26724,7},
{5372137,26730,11},
{5372137,26740,17},
{5372137,26756,19},
{5868721,26774,2},
{5868721,26782,1},
{5868721,26790,4},
{5868721,26798,3},
{5868721,26816,5},
{5868721,26820,11},
{5868721,26830,13},
{5868721,26842,19},
{6046561,26860,2},
{6046561,26876,1},
{6046561,26892,4},
{6046561,26908,3},
{6046561,26914,5},
{6046561,26918,13},
{6046561,26930,17},
{6046561,26946,19},
{6348889,26964,2},
{6348889,26968,1},
{6348889,26972,4},
{6348889,26976,3},
{6348889,26994,7},
{6348889,27000,13},
{6348889,27012,17},
{6348889,27028,19},
{6651217,27046,2},
{6651217,27054,1},
{6651217,27062,4},
{6651217,27070,3},
{6651217,27076,11},
{6651217,27086,13},
{6651217,27098,17},
{6651217,27114,19},
{6715171,27132,3},
{6715171,27150,5},
{6715171,27154,7},
{6715171,27160,11},
{6715171,27170,17},
{6715171,27186,19},
{6846841,27204,2},
{6846841,27208,1},
{6846841,27212,4},
{6846841,27216,3},
{6846841,27222,5},
{6846841,27226,7},
{6846841,27232,11},
{6846841,27242,13},
{6846841,27254,19},
{7162849,27272,2},
{7162849,27288,1},
{7162849,27304,4},
{7162849,27320,3},
{7162849,27326,7},
{7162849,27332,11},
{7162849,27342,17},
{7162849,27358,19},
{7674481,27376,2},
{7674481,27384,1},
{7674481,27392,4},
{7674481,27400,3},
{7674481,27418,5},
{7674481,27422,11},
{7674481,27432,17},
{7674481,27448,19},
{9767521,27466,2},
{9767521,27482,1},
{9767521,27498,4},
{9767521,27514,3},
{9767521,27532,5},
{9767521,27536,7},
{9767521,27542,17},
{9767521,27558,19},
{11737441,27576,2},
{11737441,27592,1},
{11737441,27608,4},
{11737441,27624,3},
{11737441,27642,5},
{11737441,27646,11},
{11737441,27656,13},
{11737441,27668,19},
{12471031,27686,3},
{12471031,27704,5},
{12471031,27708,11},
{12471031,27718,13},
{12471031,27730,17},
{12471031,27746,19},
{12697777,27764,2},
{12697777,27772,1},
{12697777,27780,4},
{12697777,27788,3},
{12697777,27806,7},
{12697777,27812,13},
{12697777,27824,17},
{12697777,27840,19},
{17907121,27858,2},
{17907121,27866,1},
{17907121,27874,4},
{17907121,27882,3},
{17907121,27888,5},
{17907121,27892,7},
{17907121,27898,11},
{17907121,27908,17},
{17907121,27924,19},
{24942061,27942,2},
{24942061,27944,3},
{24942061,27962,5},
{24942061,27966,11},
{24942061,27976,13},
{24942061,27988,17},
{24942061,28004,19},
{27387361,28022,2},
{27387361,28038,1},
{27387361,28054,4},
{27387361,28070,3},
{27387361,28076,5},
{27387361,28080,7},
{27387361,28086,11},
{27387361,28096,13},
{27387361,28108,19},
{31744441,28126,2},
{31744441,28130,1},
{31744441,28134,4},
{31744441,28138,3},
{31744441,28156,5},
{31744441,28160,7},
{31744441,28166,13},
{31744441,28178,17},
{31744441,28194,19},
{35814241,28212,2},
{35814241,28228,1},
{35814241,28244,4},
{35814241,28260,3},
{35814241,28266,5},
{35814241,28270,7},
{35814241,28276,11},
{35814241,28286,17},
{35814241,28302,19},
{41081041,28320,2},
{41081041,28328,1},
{41081041,28336,4},
{41081041,28344,3},
{41081041,28362,5},
{41081041,28366,7},
{41081041,28372,11},
{41081041,28382,13},
{41081041,28394,19},
{46558513,28412,2},
{46558513,28420,1},
{46558513,28428,4},
{46558513,28436,3},
{46558513,28442,7},
{46558513,28448,11},
{46558513,28458,13},
{46558513,28470,17},
{46558513,28486,19},
{53721361,28504,2},
{53721361,28512,1},
{53721361,28520,4},
{53721361,28528,3},
{53721361,28546,5},
{53721361,28550,7},
{53721361,28556,11},
{53721361,28566,17},
{53721361,28582,19},
{107442721,28600,2},
{107442721,28616,1},
{107442721,28632,4},
{107442721,28648,3},
{107442721,28666,5},
{107442721,28670,7},
{107442721,28676,11},
{107442721,28686,17},
{107442721,28702,19},
{174594421,28720,2},
{174594421,28722,3},
{174594421,28740,5},
{174594421,28744,7},
{174594421,28750,11},
{174594421,28760,13},
{174594421,28772,17},
{174594421,28788,19},
{232792561,28806,2},
{232792561,28814,1},
{232792561,28822,4},
{232792561,28830,3},
{232792561,28836,5},
{232792561,28840,7},
{232792561,28846,11},
{232792561,28856,13},
{232792561,28868,17},
{232792561,28884,19},
{1901,28902,2},
{1901,28904,5},
{1901,28924,19},
{2851,28942,3},
{2851,28944,5},
{2851,28964,19},
{5701,28982,2},
{5701,28984,3},
{5701,28986,5},
{5701,29006,19},
{39901,29024,2},
{39901,29026,3},
{39901,29028,5},
{39901,29048,7},
{39901,29054,19},
{41801,29072,2},
{41801,29076,1},
{41801,29080,4},
{41801,29084,5},
{41801,29104,11},
{41801,29114,19},
{53201,29132,2},
{53201,29140,1},
{53201,29148,4},
{53201,29156,5},
{53201,29176,7},
{53201,29182,19},
{62701,29200,2},
{62701,29202,3},
{62701,29204,5},
{62701,29224,11},
{62701,29234,19},
{64601,29252,2},
{64601,29256,1},
{64601,29260,4},
{64601,29264,5},
{64601,29284,17},
{64601,29300,19},
{74101,29318,2},
{74101,29320,3},
{74101,29322,5},
{74101,29342,13},
{74101,29354,19},
{79801,29372,2},
{79801,29376,1},
{79801,29380,4},
{79801,29384,3},
{79801,29386,5},
{79801,29406,7},
{79801,29412,19},
{98801,29430,2},
{98801,29438,1},
{98801,29446,4},
{98801,29454,5},
{98801,29474,13},
{98801,29486,19},
{113051,29504,5},
{113051,29524,7},
{113051,29530,17},
{113051,29546,19},
{119701,29564,2},
{119701,29566,3},
{119701,29572,5},
{119701,29592,7},
{119701,29598,19},
{135851,29616,5},
{135851,29636,11},
{135851,29646,13},
{135851,29658,19},
{148201,29676,2},
{148201,29680,1},
{148201,29684,4},
{148201,29688,3},
{148201,29690,5},
{148201,29710,13},
{148201,29722,19},
{205201,29740,2},
{205201,29748,1},
{205201,29756,4},
{205201,29764,3},
{205201,29782,5},
{205201,29802,19},
{219451,29820,3},
{219451,29822,5},
{219451,29842,7},
{219451,29848,11},
{219451,29858,19},
{290701,29876,2},
{290701,29878,3},
{290701,29884,5},
{290701,29904,17},
{290701,29920,19},
{292601,29938,2},
{292601,29942,1},
{292601,29946,4},
{292601,29950,5},
{292601,29970,7},
{292601,29976,11},
{292601,29986,19},
{319201,30004,2},
{319201,30020,1},
{319201,30036,4},
{319201,30052,3},
{319201,30054,5},
{319201,30074,7},
{319201,30080,19},
{333451,30098,3},
{333451,30116,5},
{333451,30136,13},
{333451,30148,19},
{339151,30166,3},
{339151,30168,5},
{339151,30188,7},
{339151,30194,17},
{339151,30210,19},
{359101,30228,2},
{359101,30230,3},
{359101,30248,5},
{359101,30268,7},
{359101,30274,19},
{410401,30292,2},
{410401,30308,1},
{410401,30324,4},
{410401,30340,3},
{410401,30358,5},
{410401,30378,19},
{452201,30396,2},
{452201,30400,1},
{452201,30404,4},
{452201,30408,5},
{452201,30428,7},
{452201,30434,17},
{452201,30450,19},
{478801,30468,2},
{478801,30476,1},
{478801,30484,4},
{478801,30492,3},
{478801,30498,5},
{478801,30518,7},
{478801,30524,19},
{501601,30542,2},
{501601,30558,1},
{501601,30574,4},
{501601,30590,3},
{501601,30592,5},
{501601,30612,11},
{501601,30622,19},
{532951,30640,3},
{532951,30642,5},
{532951,30662,11},
{532951,30672,17},
{532951,30688,19},
{564301,30706,2},
{564301,30708,3},
{564301,30726,5},
{564301,30746,11},
{564301,30756,19},
{658351,30774,3},
{658351,30780,5},
{658351,30800,7},
{658351,30806,11},
{658351,30816,19},
{666901,30834,2},
{666901,30836,3},
{666901,30854,5},
{666901,30874,13},
{666901,30886,19},
{778051,30904,3},
{778051,30910,5},
{778051,30930,7},
{778051,30936,13},
{778051,30948,19},
{839801,30966,2},
{839801,30970,1},
{839801,30974,4},
{839801,30978,5},
{839801,30998,13},
{839801,31010,17},
{839801,31026,19},
{957601,31044,2},
{957601,31060,1},
{957601,31076,4},
{957601,31092,3},
{957601,31098,5},
{957601,31118,7},
{957601,31124,19},
{1037401,31142,2},
{1037401,31146,1},
{1037401,31150,4},
{1037401,31154,3},
{1037401,31156,5},
{1037401,31176,7},
{1037401,31182,13},
{1037401,31194,19},
{1065901,31212,2},
{1065901,31214,3},
{1065901,31216,5},
{1065901,31236,11},
{1065901,31246,17},
{1065901,31262,19},
{1128601,31280,2},
{1128601,31284,1},
{1128601,31288,4},
{1128601,31292,3},
{1128601,31310,5},
{1128601,31330,11},
{1128601,31340,19},
{1222651,31358,3},
{1222651,31364,5},
{1222651,31384,11},
{1222651,31394,13},
{1222651,31406,19},
{1259701,31424,2},
{1259701,31426,3},
{1259701,31428,5},
{1259701,31448,13},
{1259701,31460,17},
{1259701,31476,19},
{1504801,31494,2},
{1504801,31510,1},
{1504801,31526,4},
{1504801,31542,3},
{1504801,31548,5},
{1504801,31568,11},
{1504801,31578,19},
{1808801,31596,2},
{1808801,31612,1},
{1808801,31628,4},
{1808801,31644,5},
{1808801,31664,7},
{1808801,31670,17},
{1808801,31686,19},
{1889551,31704,3},
{1889551,31710,5},
{1889551,31730,13},
{1889551,31742,17},
{1889551,31758,19},
{2074801,31776,2},
{2074801,31784,1},
{2074801,31792,4},
{2074801,31800,3},
{2074801,31802,5},
{2074801,31822,7},
{2074801,31828,13},
{2074801,31840,19},
{2173601,31858,2},
{2173601,31874,1},
{2173601,31890,4},
{2173601,31906,5},
{2173601,31926,11},
{2173601,31936,13},
{2173601,31948,19},
{2445301,31966,2},
{2445301,31968,3},
{2445301,31974,5},
{2445301,31994,11},
{2445301,32004,13},
{2445301,32016,19},
{2667601,32034,2},
{2667601,32042,1},
{2667601,32050,4},
{2667601,32058,3},
{2667601,32076,5},
{2667601,32096,13},
{2667601,32108,19},
{3052351,32126,3},
{3052351,32144,5},
{3052351,32164,7},
{3052351,32170,17},
{3052351,32186,19},
{3511201,32204,2},
{3511201,32220,1},
{3511201,32236,4},
{3511201,32252,3},
{3511201,32254,5},
{3511201,32274,7},
{3511201,32280,11},
{3511201,32290,19},
{3730651,32308,3},
{3730651,32310,5},
{3730651,32330,7},
{3730651,32336,11},
{3730651,32346,17},
{3730651,32362,19},
{3779101,32380,2},
{3779101,32382,3},
{3779101,32388,5},
{3779101,32408,13},
{3779101,32420,17},
{3779101,32436,19},
{3950101,32454,2},
{3950101,32456,3},
{3950101,32474,5},
{3950101,32494,7},
{3950101,32500,11},
{3950101,32510,19},
{4069801,32528,2},
{4069801,32532,1},
{4069801,32536,4},
{4069801,32540,3},
{4069801,32546,5},
{4069801,32566,7},
{4069801,32572,17},
{4069801,32588,19},
{4149601,32606,2},
{4149601,32622,1},
{4149601,32638,4},
{4149601,32654,3},
{4149601,32656,5},
{4149601,32676,7},
{4149601,32682,13},
{4149601,32694,19},
{4408951,32712,3},
{4408951,32714,5},
{4408951,32734,7},
{4408951,32740,13},
{4408951,32752,17},
{4408951,32768,19},
{5038801,32786,2},
{5038801,32794,1},
{5038801,32802,4},
{5038801,32810,3},
{5038801,32812,5},
{5038801,32832,13},
{5038801,32844,17},
{5038801,32860,19},
{6104701,32878,2},
{6104701,32880,3},
{6104701,32898,5},
{6104701,32918,7},
{6104701,32924,17},
{6104701,32940,19},
{6224401,32958,2},
{6224401,32966,1},
{6224401,32974,4},
{6224401,32982,3},
{6224401,32988,5},
{6224401,33008,7},
{6224401,33014,13},
{6224401,33026,19},
{8558551,33044,3},
{8558551,33050,5},
{8558551,33070,7},
{8558551,33076,11},
{8558551,33086,13},
{8558551,33098,19},
{9781201,33116,2},
{9781201,33124,1},
{9781201,33132,4},
{9781201,33140,3},
{9781201,33146,5},
{9781201,33166,11},
{9781201,33176,13},
{9781201,33188,19},
{11191951,33206,3},
{11191951,33212,5},
{11191951,33232,7},
{11191951,33238,11},
{11191951,33248,17},
{11191951,33264,19},
{11411401,33282,2},
{11411401,33286,1},
{11411401,33290,4},
{11411401,33294,3},
{11411401,33296,5},
{11411401,33316,7},
{11411401,33322,11},
{11411401,33332,13},
{11411401,33344,19},
{14922601,33362,2},
{14922601,33366,1},
{14922601,33370,4},
{14922601,33374,3},
{14922601,33376,5},
{14922601,33396,7},
{14922601,33402,11},
{14922601,33412,17},
{14922601,33428,19},
{16279201,33446,2},
{16279201,33462,1},
{16279201,33478,4},
{16279201,33494,3},
{16279201,33500,5},
{16279201,33520,7},
{16279201,33526,17},
{16279201,33542,19},
{17117101,33560,2},
{17117101,33562,3},
{17117101,33568,5},
{17117101,33588,7},
{17117101,33594,11},
{17117101,33604,13},
{17117101,33616,19},
{17635801,33634,2},
{17635801,33638,1},
{17635801,33642,4},
{17635801,33646,3},
{17635801,33648,5},
{17635801,33668,7},
{17635801,33674,13},
{17635801,33686,17},
{17635801,33702,19},
{19186201,33720,2},
{19186201,33724,1},
{19186201,33728,4},
{19186201,33732,3},
{19186201,33750,5},
{19186201,33770,11},
{19186201,33780,17},
{19186201,33796,19},
{19562401,33814,2},
{19562401,33830,1},
{19562401,33846,4},
{19562401,33862,3},
{19562401,33868,5},
{19562401,33888,11},
{19562401,33898,13},
{19562401,33910,19},
{22383901,33928,2},
{22383901,33930,3},
{22383901,33936,5},
{22383901,33956,7},
{22383901,33962,11},
{22383901,33972,17},
{22383901,33988,19},
{22822801,34006,2},
{22822801,34014,1},
{22822801,34022,4},
{22822801,34030,3},
{22822801,34032,5},
{22822801,34052,7},
{22822801,34058,11},
{22822801,34068,13},
{22822801,34080,19},
{23514401,34098,2},
{23514401,34114,1},
{23514401,34130,4},
{23514401,34146,5},
{23514401,34166,7},
{23514401,34172,13},
{23514401,34184,17},
{23514401,34200,19},
{25581601,34218,2},
{25581601,34234,1},
{25581601,34250,4},
{25581601,34266,3},
{25581601,34272,5},
{25581601,34292,11},
{25581601,34302,17},
{25581601,34318,19},
{25675651,34336,3},
{25675651,34354,5},
{25675651,34374,7},
{25675651,34380,11},
{25675651,34390,13},
{25675651,34402,19},
{31600801,34420,2},
{31600801,34436,1},
{31600801,34452,4},
{31600801,34468,3},
{31600801,34486,5},
{31600801,34506,7},
{31600801,34512,11},
{31600801,34522,19},
{35271601,34540,2},
{35271601,34548,1},
{35271601,34556,4},
{35271601,34564,3},
{35271601,34566,5},
{35271601,34586,7},
{35271601,34592,13},
{35271601,34604,17},
{35271601,34620,19},
{37346401,34638,2},
{37346401,34654,1},
{37346401,34670,4},
{37346401,34686,3},
{37346401,34704,5},
{37346401,34724,7},
{37346401,34730,13},
{37346401,34742,19},
{38372401,34760,2},
{38372401,34768,1},
{38372401,34776,4},
{38372401,34784,3},
{38372401,34802,5},
{38372401,34822,11},
{38372401,34832,17},
{38372401,34848,19},
{45349201,34866,2},
{45349201,34874,1},
{45349201,34882,4},
{45349201,34890,3},
{45349201,34908,5},
{45349201,34928,13},
{45349201,34940,17},
{45349201,34956,19},
{59690401,34974,2},
{59690401,34990,1},
{59690401,35006,4},
{59690401,35022,3},
{59690401,35024,5},
{59690401,35044,7},
{59690401,35050,11},
{59690401,35060,17},
{59690401,35076,19},
{67151701,35094,2},
{67151701,35096,3},
{67151701,35114,5},
{67151701,35134,7},
{67151701,35140,11},
{67151701,35150,17},
{67151701,35166,19},
{83140201,35184,2},
{83140201,35188,1},
{83140201,35192,4},
{83140201,35196,3},
{83140201,35202,5},
{83140201,35222,11},
{83140201,35232,13},
{83140201,35244,17},
{83140201,35260,19},
{129329201,35278,2},
{129329201,35286,1},
{129329201,35294,4},
{129329201,35302,5},
{129329201,35322,7},
{129329201,35328,11},
{129329201,35338,13},
{129329201,35350,17},
{129329201,35366,19},
{134303401,35384,2},
{134303401,35388,1},
{134303401,35392,4},
{134303401,35396,3},
{134303401,35414,5},
{134303401,35434,7},
{134303401,35440,11},
{134303401,35450,17},
{134303401,35466,19},
{193993801,35484,2},
{193993801,35488,1},
{193993801,35492,4},
{193993801,35496,3},
{193993801,35498,5},
{193993801,35518,7},
{193993801,35524,11},
{193993801,35534,13},
{193993801,35546,17},
{193993801,35562,19},
{249420601,35580,2},
{249420601,35584,1},
{249420601,35588,4},
{249420601,35592,3},
{249420601,35610,5},
{249420601,35630,11},
{249420601,35640,13},
{249420601,35652,17},
{249420601,35668,19},
{436486051,35686,3},
{436486051,35704,5},
{436486051,35724,7},
{436486051,35730,11},
{436486051,35740,13},
{436486051,35752,17},
{436486051,35768,19},
{634888801,35786,2},
{634888801,35802,1},
{634888801,35818,4},
{634888801,35834,3},
{634888801,35852,5},
{634888801,35872,7},
{634888801,35878,13},
{634888801,35890,17},
{634888801,35906,19},
{1163962801,35924,2},
{1163962801,35932,1},
{1163962801,35940,4},
{1163962801,35948,3},
{1163962801,35954,5},
{1163962801,35974,7},
{1163962801,35980,11},
{1163962801,35990,13},
{1163962801,36002,17},
{1163962801,36018,19},
{1745944201,36036,2},
{1745944201,36040,1},
{1745944201,36044,4},
{1745944201,36048,3},
{1745944201,36066,5},
{1745944201,36086,7},
{1745944201,36092,11},
{1745944201,36102,13},
{1745944201,36114,17},
{1745944201,36130,19},
{0,0,0}};

#endif
