/* BEGIN software license
 *
 * msXpertSuite - mass spectrometry software suite
 * -----------------------------------------------
 * Copyright (C) 2009--2020 Filippo Rusconi
 *
 * http://www.msxpertsuite.org
 * * This file is part of the msXpertSuite project.  * * The msXpertSuite
 * project is the successor of the massXpert project. This project now includes
 * various independent modules:
 *
 * - massXpert, model polymer chemistries and simulate mass spectrometric data;
 * - mineXpert, a powerful TIC chromatogram/mass spectrum viewer/miner;
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see <http://www.gnu.org/licenses/>.
 *
 * END software license
 */


/////////////////////// StdLib includes
#include <vector>
#include <cmath>


/////////////////////// Qt includes
#include <QDebug>


/////////////////////// pappsomspp includes


/////////////////////// Local includes
#include "BrukerTimsTofMsRunProber.hpp"


int BrukerTimsTofMsRunProberMetaTypeId =
  qRegisterMetaType<msxps::minexpert::BrukerTimsTofMsRunProber>(
    "msxps::minexpert::BrukerTimsTofMsRunProber");

int BrukerTimsTofMsRunProberSPtrMetaTypeId =
  qRegisterMetaType<msxps::minexpert::BrukerTimsTofMsRunProberSPtr>(
    "msxps::minexpert::BrukerTimsTofMsRunProberSPtr");


namespace msxps
{
namespace minexpert
{

// Needed for qRegisterMetaType
BrukerTimsTofMsRunProber::BrukerTimsTofMsRunProber(
  pappso::MsFileAccessorSPtr ms_file_accessor_sp)
  : MsRunProber(ms_file_accessor_sp)
{
}


// Needed for qRegisterMetaType
BrukerTimsTofMsRunProber::BrukerTimsTofMsRunProber(
  const BrukerTimsTofMsRunProber &other)
  : MsRunProber(other)
{
}

BrukerTimsTofMsRunProber::~BrukerTimsTofMsRunProber()
{
}

MsRunProbeInfo
BrukerTimsTofMsRunProber::probe()
{
  // This object is for us to fill in here and return it.
  // Note that it has a probeErrors QString member.
  MsRunProbeInfo ms_run_probe_info;

  // This object serves to configure the readSpectrumCollection2 function for
  // our local needs.
  pappso::MsRunReadConfig ms_run_read_config;

  ms_run_read_config.setNeedPeakList(false);

  // Question for Olivier: how could we get in one step the number of MS1
  // and MS2 frames?

  ms_run_read_config.setMsLevels({1});
  
  // Configure the accessor to prefer pappso::FileReaderType::tims_frames.
  msp_msFileAccessor->setPreferredFileReaderType(
    pappso::MzFormat::brukerTims, pappso::FileReaderType::tims_frames);
  
  // Get a MsRunReader now to actually look at the MS run data. The caller has 
  // already set the selected MsRunId. 
  
  pappso::MsRunReaderSPtr ms_run_reader_sp =
  msp_msFileAccessor->msRunReaderSPtrForSelectedMsRunIdIndex();

  pappso::MsRunReaderQualifiedSpectrumList spectrum_list_reader;
  ms_run_reader_sp->readSpectrumCollection2(ms_run_read_config,
                                         spectrum_list_reader);

  // How many frames are there as MS1 spectra (not MS/MS)
  ms_run_probe_info.ms1SpectraCount =
    spectrum_list_reader.getQualifiedMassSpectrumList().size();

  QString data_exploration_results =
    QString("The number of spectra for MS level 1 is %1\n")
      .arg(ms_run_probe_info.ms1SpectraCount);

  double first_spectrum_rt =
    spectrum_list_reader.getQualifiedMassSpectrumList().at(0).getRtInMinutes();

  ms_run_probe_info.retentionTimeBeginInSeconds = first_spectrum_rt * 60;

  double last_spectrum_rt = spectrum_list_reader.getQualifiedMassSpectrumList()
                              .at(ms_run_probe_info.ms1SpectraCount - 1)
                              .getRtInMinutes();

  ms_run_probe_info.retentionTimeEndInSeconds = last_spectrum_rt * 60;

  double tic_chrom_duration_minutes      = last_spectrum_rt - last_spectrum_rt;
  double half_tic_chrom_duration_minutes = tic_chrom_duration_minutes / 2;

  data_exploration_results += QString(
                                "The acquisition was performed over the RT "
                                "range [%1-%2] for a duration of %3 min.\n")
                                .arg(first_spectrum_rt)
                                .arg(last_spectrum_rt)
                                .arg(tic_chrom_duration_minutes);

  // At this point look in detail at the m/z dimension of the spectrum at
  // the middle of the acquisition
  std::size_t middle_acquisition_spectrum_index =
    ms_run_probe_info.ms1SpectraCount / 2;

  pappso::QualifiedMassSpectrum qualified_mass_spectrum =
    spectrum_list_reader.getQualifiedMassSpectrumList().at(
      middle_acquisition_spectrum_index);

  half_tic_chrom_duration_minutes = qualified_mass_spectrum.getRtInMinutes();

  data_exploration_results +=
    QString(
      "The retention time of the spectrum at mid-acquisition is (min) %1\n")
      .arg(half_tic_chrom_duration_minutes);

  qDebug("The rt_minutes for the middle-acquisition spectrum is: %f",
         half_tic_chrom_duration_minutes);

  // Now try to actually configure the ms run reader to get that rt data.

  ms_run_read_config.setRetentionTimeStartInSeconds(
    (half_tic_chrom_duration_minutes * 60) - 1);
  ms_run_read_config.setRetentionTimeEndInSeconds(
    (half_tic_chrom_duration_minutes * 60) + 1);
  ms_run_read_config.setNeedPeakList(true);

  spectrum_list_reader.clear();
  ms_run_reader_sp->readSpectrumCollection2(ms_run_read_config,
                                         spectrum_list_reader);

  // How many frames are there as MS1 spectra (not MS/MS)
  std::size_t half_tic_chrom_ms1_spectra_count =
    spectrum_list_reader.getQualifiedMassSpectrumList().size();

  qDebug("The number of spectra for MS level 1 is: %lu",
         half_tic_chrom_ms1_spectra_count);

  if(half_tic_chrom_ms1_spectra_count < 1)
    {
      ms_run_probe_info.probeErrors += QString(
        "Error: there is not a single mass spectrum. Please, select a wider "
        "retention time range for extracting the frames from the data "
        "file.");

      return ms_run_probe_info;
    }

  // Get the first spectrum in the list.
  qualified_mass_spectrum =
    spectrum_list_reader.getQualifiedMassSpectrumList().at(0);

  std::size_t mass_spectrum_size = qualified_mass_spectrum.size();

  data_exploration_results +=
    QString("That spectrum has size %1\n").arg(mass_spectrum_size);

  quint32 mz_index_begin =
    qualified_mass_spectrum
      .getParameterValue(
        pappso::QualifiedMassSpectrumParameter::TimsFrameMzIndexBegin)
      .toUInt();
  qDebug("mz_index_begin: %u", mz_index_begin);

  quint32 mz_index_end =
    qualified_mass_spectrum
      .getParameterValue(
        pappso::QualifiedMassSpectrumParameter::TimsFrameMzIndexEnd)
      .toUInt();
  qDebug("mz_index_end: %u", mz_index_end);

  ms_run_probe_info.mzBinCount = mz_index_end - mz_index_begin;

  data_exploration_results += QString(
                                "The TOF indices range for that  spectrum  is "
                                "[%1-%2] = %3 total indices\n")
                                .arg(mz_index_begin)
                                .arg(mz_index_end)
                                .arg(ms_run_probe_info.mzBinCount);

  qDebug("There are %d TOF indices in the middle-acquisition mass spectrum",
         ms_run_probe_info.mzBinCount);

  double mz_begin = qualified_mass_spectrum.getMassSpectrumCstSPtr()->at(0).x;
  double mz_end   = qualified_mass_spectrum.getMassSpectrumCstSPtr()
                    ->at(mass_spectrum_size - 1)
                    .x;

  ms_run_probe_info.mzBegin = mz_begin;
  ms_run_probe_info.mzEnd   = mz_end;

  data_exploration_results +=
    QString(
      "The mass spectrum made out of all the mobility scans has m/z  range "
      "[%1-%2] = %3  m/z range \n")
      .arg(mz_begin)
      .arg(mz_end)
      .arg(mz_end - mz_begin);

  qDebug("Mass spectrum spans [ %f.5 - %f.5 ]", mz_begin, mz_end);
  qDebug("Rough m/z bin per TOF index: %f.5",
         (mz_end - mz_begin) / (mz_index_end - mz_index_begin));

  data_exploration_results +=
    QString("The digitizer thus sets one TOF index to a m/z delta of %1\n")
      .arg((mz_end - mz_begin) / (mz_index_end - mz_index_begin));

  std::size_t tims_frame_scans_count =
    qualified_mass_spectrum
      .getParameterValue(
        pappso::QualifiedMassSpectrumParameter::TimsFrameScansCount)
      .toUInt();

  // This is always true, since the scans numbers are actually indices
  ms_run_probe_info.ionMobilityScansBegin = 0;
  ms_run_probe_info.ionMobilityScansEnd   = tims_frame_scans_count - 1;

  double inv_ko_range_begin =
    qualified_mass_spectrum
      .getParameterValue(
        pappso::QualifiedMassSpectrumParameter::TimsFrameInvKoBegin)
      .toDouble();

  double inv_ko_range_end =
    qualified_mass_spectrum
      .getParameterValue(
        pappso::QualifiedMassSpectrumParameter::TimsFrameInvKoEnd)
      .toDouble();

  ms_run_probe_info.ionMobilityOneOverKoBegin = inv_ko_range_begin;
  ms_run_probe_info.ionMobilityOneOverKoEnd   = inv_ko_range_end;

  data_exploration_results +=
    QString(
      "The mass spectrum has ion mobility scans in the 1/K0 range [%1-%2] "
      "over %3 ion mobility scans\n")
      .arg(inv_ko_range_begin)
      .arg(inv_ko_range_end)
      .arg(tims_frame_scans_count);

  qDebug("The data exploration results: %s.\nNow opening the dialog.",
         data_exploration_results.toLatin1().data());

  // As a last step, we want to know how many MS2 scans there are in the
  // whole acquisition. We have to reninitialize the config!
  ms_run_read_config.reset();
  ms_run_read_config.setMsLevels({2});
  ms_run_read_config.setNeedPeakList(false);

  spectrum_list_reader.clear();

  ms_run_reader_sp->readSpectrumCollection2(ms_run_read_config,
                                         spectrum_list_reader);
  ms_run_probe_info.ms2SpectraCount =
    spectrum_list_reader.getQualifiedMassSpectrumList().size();

  qDebug("Number of MS2 spectra: %zu",
         spectrum_list_reader.getQualifiedMassSpectrumList().size());

  // Clear the list to free some memory.
  spectrum_list_reader.clear();

  return ms_run_probe_info;
}


} // namespace minexpert

} // namespace msxps
