/***************************************************************************
 *   Copyright (C) 2007 by Anistratov Oleg                                 *
 *   ower86@gmail.com                                                      *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation;                         *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 ***************************************************************************/
#include "messagefiltereditor.h"

#include <QGridLayout>

#include "messagefilter.h"
#include "filtrationrule.h"
#include "filtrationruleeditor.h"

MessageFilterEditor::MessageFilterEditor(QWidget *parent)
 : QWidget(parent),
  m_filter(NULL)
{
  QGridLayout* grid = new QGridLayout(this);

  m_rulesListWgt          = new QListWidget(this);
  m_addRuleBtn            = new QPushButton(this);
  m_delRuleBtn            = new QPushButton(this);
  m_enableDisableRuleBtn  = new QPushButton(this);

  grid->addWidget(m_rulesListWgt, 0, 0, 1, 3);
  grid->addWidget(m_addRuleBtn  , 1, 0);
  grid->addWidget(m_delRuleBtn  , 1, 1);
  grid->addWidget(m_enableDisableRuleBtn, 1, 2);

  connect(m_addRuleBtn  , SIGNAL(clicked()), this, SLOT(addRule()));
  connect(m_delRuleBtn  , SIGNAL(clicked()), this, SLOT(removeRule()));
  connect(m_enableDisableRuleBtn, SIGNAL(clicked()), this, SLOT(enableDisableRule()));
  connect(m_rulesListWgt, SIGNAL(itemDoubleClicked(QListWidgetItem*)), this, SLOT(editRule(QListWidgetItem*)));
  connect(m_rulesListWgt, SIGNAL(currentRowChanged(int)), this, SLOT(itemActivated(int)));

  m_enableDisableRuleBtn->setText(tr("Disable rule"));
  m_enableDisableRuleBtn->setEnabled(false);
  retranslate();
}

MessageFilterEditor::~MessageFilterEditor()
{
}

void MessageFilterEditor::retranslate()
{
  m_addRuleBtn->setText(tr("Add rule..."));
  m_delRuleBtn->setText(tr("Remove rule"));
}

void MessageFilterEditor::refresh()
{
  QListWidgetItem* item;

  retranslate();

  if(!m_filter)
    return;

  m_rulesListWgt->clear();

  foreach(FiltrationRule* fr, m_filter->blackRules())
  {
    item = new QListWidgetItem(fr->name() + (fr->activated() ? "" : tr(" (disabled)")));// + tr(" (black rule)"));
    item->setData(Qt::UserRole, (quint64)(fr));
    m_rulesListWgt->addItem(item);
  }

  foreach(FiltrationRule* fr, m_filter->whiteRules())
  {
    item = new QListWidgetItem(fr->name() + tr(" (white rule)"));
    item->setData(Qt::UserRole, (quint64)(fr));
    m_rulesListWgt->addItem(item);
  }
}

void MessageFilterEditor::addRule()
{
  FiltrationRule* rule = new FiltrationRule();

  rule->setName(tr("New Rule"));

  if(!editRule(rule))
    delete rule;
  else
    m_filter->addBlackRule(rule);

  refresh();
}

bool MessageFilterEditor::editRule(FiltrationRule* rule)
{
  int ret;

  FiltrationRuleEditor* fre = new FiltrationRuleEditor(this);

  fre->init(rule);

  ret = fre->exec();

  delete fre;

  refresh();

  return ret;
}

void MessageFilterEditor::editRule(QListWidgetItem* item)
{
  FiltrationRule* rule;

  rule = static_cast<FiltrationRule*>((void*)(item->data(Qt::UserRole).toInt()));

  if(rule)
    editRule(rule);
}

void MessageFilterEditor::removeRule()
{
  QListWidgetItem* item = m_rulesListWgt->currentItem();
  FiltrationRule* rule;

  if(item)
  {
    rule = static_cast<FiltrationRule*>((void*)(item->data(Qt::UserRole).toInt()));

    if(rule)
      m_filter->removeRule(rule);

    refresh();
  }
}

void MessageFilterEditor::enableDisableRule()
{
  QListWidgetItem* item = m_rulesListWgt->currentItem();
  FiltrationRule* rule;

  if(item)
  {
    rule = static_cast<FiltrationRule*>((void*)(item->data(Qt::UserRole).toInt()));

    if(rule)
    {
      rule->setActivated(!rule->activated());
      m_enableDisableRuleBtn->setText(rule->activated() ? tr("Disable rule") : tr("Enable rule"));
    }

    refresh();
  }
}

void MessageFilterEditor::itemActivated(int row)
{
  QListWidgetItem* item = m_rulesListWgt->currentItem();
  FiltrationRule* rule;

  if(item)
  {
    rule = static_cast<FiltrationRule*>((void*)(item->data(Qt::UserRole).toInt()));
    if(rule)
      m_enableDisableRuleBtn->setText(rule->activated() ? tr("Disable rule") : tr("Enable rule"));
  }

  m_enableDisableRuleBtn->setEnabled(row >= 0);
}
