#! /bin/bash

set -e

VERBOSE_SIGNOND=0
VERBOSE_LIBACCOUNTS=0

while [ $# -gt 0 ]
do
    case "$1" in
    (--verbose-signond) VERBOSE_SIGNOND=1;;
    (--verbose-libaccounts) VERBOSE_LIBACCOUNTS=1;;
    (--) shift; break;;
    (-*) echo "$0: Error: unrecognized option $1" 1>&2; exit 1;;
    (*) break;;
    esac
    shift
done

if [ $# -lt 2 ]; then
    echo "Usage: $0 [options] <TMP_ACCOUNT_DIR> <TEST_PROGRAM> [args...]"
    echo "Options:"
    echo "  --verbose-signond      Enable signond logging"
    echo "  --verbose-libaccounts  Enable libaccounts logging"
    echo
    echo "<TMP_ACCOUNT_DIR> is the location for the temporary account DB."
    echo "NOTE: this directory will be DELETED after the execution of the tests."
    echo
    exit 1
fi

export TMP_ACCOUNT_DIR="$1"
shift
PROGRAM="$1"
shift
ARGS=()
while [ $# -gt 0 ]
do
    ARGS+=("-p")
    ARGS+=("$1")
    shift
done

# location of the accounts DB
export ACCOUNTS="${TMP_ACCOUNT_DIR}"
if [ ${VERBOSE_LIBACCOUNTS} -eq 1 ]; then
    echo "Enabling verbose logging for libaccounts"
    export G_MESSAGES_DEBUG=all
    export AG_DEBUG=all
fi

# It's possible to instruct libaccounts to look for provider, application, and
# service files under different directories. We don't alter these directories
# here, since it's safe to let the tests run with the files currently installed
# on the machine (they are opened in read-only mode), but feel free to set
# these in the test environment, in case it's needed:
#   AG_PROVIDERS
#   AG_APPLICATIONS
#   AG_SERVICES
#   AG_SERVICE_TYPES

# location of the credentials DB
export SSO_STORAGE_PATH="${TMP_ACCOUNT_DIR}"
if [ ${VERBOSE_SIGNOND} -eq 1 ]; then
    echo "Enabling verbose logging for signond"
    export SSO_LOGGING_LEVEL=2
fi
# We set SSO_EXTENSIONS_DIR to an invalid value, so that signond will not find
# the GNOME keyring or the Apparmor extension.
export SSO_EXTENSIONS_DIR=/tmp/this/should/not/exist

# timeout values; one generally doesn't need to change these
export SSO_DAEMON_TIMEOUT=5
export SSO_IDENTITY_TIMEOUT=5
export SSO_AUTHSESSION_TIMEOUT=5

# signond creates its socket somewhere under XDG_RUNTIME_DIR; we change the
# value of XDG_RUNTIME_DIR to avoid conflicts with any other signond instance
# which might be running on the machine under test.
export XDG_RUNTIME_DIR="${TMP_ACCOUNT_DIR}/runtime-dir"
mkdir -p "$XDG_RUNTIME_DIR"

# start a local signond

# We use dbus-test-runner to run the tests in a separate D-Bus session
echo "Starting dbus-test-runner"
dbus-test-runner -m 180 --keep-env -t "$PROGRAM" "${ARGS[@]}"

rm -rf ${TMP_ACCOUNT_DIR}

