/** -*- C++ -*-
 * @file cache/entity/package.h
 * @author Peter Rockai <me@mornfall.net>
 */

#include <apt-pkg/pkgcache.h>

#include <apt-front/forward.h>
#include <apt-front/cache/entity/entity.h>
#include <apt-front/cache/entity/tag.h>
#include <tagcoll/OpSet.h>
#include <string>
#include <stdexcept>

#ifndef APTFRONT_CACHE_ENTITY_PACKAGE_H
#define APTFRONT_CACHE_ENTITY_PACKAGE_H

namespace aptFront {

namespace cache {
namespace entity {

class Version;

/**
   @brief The standard interface to interact with packages in cache.

   PackageIterator is one of the central classes to libapt-front. You will
   come into contact with it on every corner. It allows you to query cache
   for data about packages, it allows you to mark packages for
   (un)installation, gives you access to VersionIterator's associated
   with individual available package versions and so on. 

   The class is intended to be used as a regular by-value class, since its
   data part is fairly small and copy overhead is negligible this should be
   fairly ok in most cases. Use const references if you need to get few
   cycles more out of it.
*/
class Package : public Implementation<Package> {
public:
    struct State {
        enum Query {
            Install = 1 << 0,
            Upgrade = 1 << 1,
            Keep = 1 << 2,
            Remove = 1 << 3,
            Installed = 1 << 4,
            Upgradable = 1 << 5,
            NowBroken = 1 << 6,
            WillBreak = 1 << 7
        };

        typedef unsigned state;

        operator unsigned() { return m_state; };

        State &operator=( unsigned i ) {
            m_state = i;
            return *this;
        }

        State &operator|=( const State &s ) {
            m_state |= s.m_state;
            return *this;
        }

        State( unsigned a ) {
            m_state = a;
        }

        State() : m_state( 0 ) {}

    protected:
        unsigned m_state;
    };

    typedef actor::Actor< Package > Actor;
    typedef actor::Bound< Package > BoundActor;

    Package( Cache *c, pkgCache::Package *p )
        : m_package( p ), m_hashIndex( 0 )
        {
            setCache( c );
        }
    Package() : m_package( 0 ), m_hashIndex( 0 ) {}
    ~Package() {}

    /* PackageIterator( const PackageIterator &p )
       : IteratorImpl<PackageIterator>( p )
       {
       setCache( p.m_cache );
       m_package = p.m_package;
       m_hashIndex = p.m_hashIndex;
       } */

    Package( const Entity &i ) {
        initFromBase( i.impl() );
    }

    /* pkgCache::PkgIterator conversion */
    Package( pkgCache::PkgIterator P );
    operator pkgCache::PkgIterator() const;

    virtual Entity stable() const;

    virtual bool valid() const {
        return m_package;
    }

    /* Package functionality */
    bool operator ==( const Package &p ) const {
        if (!p.valid() || !valid())
            return p.valid() == valid();
        return ( p.m_package == m_package && ( &p.cache() ) == ( &cache() ) );
    }

    bool operator <( const Package &p ) const {
        if (valid()) {
            if (p.valid())
                return id() < p.id(); // name() < p.name();
            else
                return false;
        } else 
            return p.valid() != valid();
    }

    State state() const;
    State state( const State & ) const throw();

    inline pkgCache::Package *pkgPtr () const {
        checkValid();
        return m_package;
    }

    bool markedInstall() const { return state() & State::Install; }
    bool markedUpgrade() const { return markedInstall() && isUpgradable(); }
    bool markedNewInstall() const { return markedInstall() && !isUpgradable(); }
    bool markedKeep() const { return state() & State::Keep; }
    bool markedRemove() const { return state() & State::Remove; }
    
    bool isInstalled() const { return state() & State::Installed; }
    bool isUpgradable() const { return state() & State::Upgradable; }
    bool isBroken() const { return state() & State::NowBroken; }
    bool willBreak() const { return state() & State::WillBreak; }

    bool canInstall() const {
        return ((! isInstalled()) && ! markedInstall()) || markedRemove()
            || isBroken(); }
    bool canRemove() const { return isInstalled() && not markedRemove(); }
    bool canKeep() const {
        return markedUpgrade() || markedRemove() || markedInstall(); }
    bool canUpgrade() const { return isUpgradable () && ! markedUpgrade (); }
    bool canRevertInstall() const { return markedNewInstall(); }

    Package nextInCache() const;

    template <class I> void versions( I i ) const;

    Version firstVersionInCache() const;
    Version anyVersion() const;
    bool hasVersion() const;
    Version versionByString( const std::string& ver ) const;

    std::string shortDescription( const std::string& d ) const throw();
    std::string shortDescription() const;

    std::string longDescription( std::string d ) const throw();
    std::string longDescription() const;

    std::string statusString( const std::string& d ) const throw();
    std::string statusString() const {
        checkValid(); return statusString( "" ); }

    std::string actionString( const std::string& d ) const throw();
    std::string actionString() const {
        checkValid(); return actionString( "" ); }

    std::string name( const std::string& ) const throw();
    std::string name() const { checkValid(); return name(""); }

    std::string section( const std::string& ) const throw();
    std::string section() const { checkValid(); return section( "" ); }

    std::string maintainer( const std::string& ) const throw();
    std::string maintainer() const;

    std::string architecture( const std::string& ) const throw();
    std::string architecture() const;

    inline unsigned short id() const {
        checkValid();
        return m_package->ID;
    }

    /**
     * Return the Debtags tags attached to this package
     */
    Tagcoll::OpSet<entity::Tag> tags( const Tagcoll::OpSet<entity::Tag>& ) const throw ();
    Tagcoll::OpSet<entity::Tag> tags() const;

    Version installedVersion() const;
    Version candidateVersion() const;

    // marking intefrace
    void markInstall();
    void markRemove();
    void markKeep();

    BoundActor install() const;
    BoundActor remove() const;
    BoundActor keep() const;
    BoundActor upgrade() const;

    static Actor unboundInstall();
    static Actor unboundRemove();
    static Actor unboundKeep();
    static Actor unboundUpgrade();

protected:
    typedef actor::FromMember< Package > ActorFM;
    pkgCache::Package *m_package;
    long m_hashIndex;
};

}
}
}

#include <apt-front/cache/component/packages.h>
#include <apt-front/cache/component/state.h>
#include <apt-front/cache/component/packagetags.h>
#include <apt-front/cache/entity/version.h>
#include <apt-front/actor.h>

namespace aptFront {

namespace cache {
namespace entity {

/**
   @brief Stable variant of PackageIterator (invariant to cache reloads).

   This class acts exactly like PackageIterator, apart from the fact it
   will survive cache reloads and/or updates without harm (unless the package
   disappeared from archive in which case it will become invalid (past the
   end) iterator.
*/
class StablePackage : public Implementation<StablePackage, Package>, public Observer {
public:
    StablePackage( const Package &p ) {
        Package::initFromBase( &p );
        setCache( Package::m_cache );
        observeComponent<component::Packages>();
    }
    StablePackage( Cache *c = 0 ) : Observer (c) {
        setCache( c );
        observeComponent<component::Packages>();
    }
    StablePackage( const Entity &i ) {
        Package::initFromBase( i.impl() );
        setCache( Package::m_cache );
        observeComponent<component::Packages>();
    }
    virtual ~StablePackage() {}
    void notifyPreRebuild( component::Base *c );
    void notifyPostRebuild( component::Base *c );
    virtual void setCache( Cache *c ) {
        Package::setCache( c );
        Observer::setCache( c );
    }
protected:
    std::string m_pkgName;
};

inline Entity Package::stable() const {
    return StablePackage( *this );
}

inline Version Package::installedVersion() const
{
    checkValid ();
    return Version( m_cache,
                    packageCache().verPtr(pkgPtr()->CurrentVer) );
}

inline Version Package::candidateVersion() const // XXX - beautify
{
    checkValid();
    return cache().state().GetCandidateVer(*this);
}

inline Version Package::firstVersionInCache() const
{
    return Version( m_cache,
                    packageCache().verPtr(
                        pkgPtr()->VersionList ) );
}
inline bool Package::hasVersion() const
{
    return firstVersionInCache().valid();
}

inline Package::State Package::state( const State &s ) const throw() {
    if (!valid()) return s;
    return cache().state().packageState(*this);
}

inline Package::State Package::state() const {
    checkValid();
    return state( State() );
}

inline Tagcoll::OpSet<entity::Tag> Package::tags(const Tagcoll::OpSet<entity::Tag>& d) const throw () {
	if (!valid()) return d;
	return (debtagsCache().tagdb().getTags( *this ));
}

inline Tagcoll::OpSet<entity::Tag> Package::tags() const {
	checkValid();
	return (debtagsCache().tagdb().getTags( *this ));
}

inline std::string Package::name( const std::string& d ) const throw()
{
    if (!valid()) return d;
    return packageCache().packageName( *this );
}

inline std::string Package::shortDescription() const
{
    if (hasVersion())
        return shortDescription( "" );
    throw std::out_of_range( "No short description for this package" );
}

inline std::string Package::shortDescription( const std::string& d ) const throw()
{
    if (!hasVersion()) return d;
    return anyVersion().shortDescription( d );
}

inline std::string Package::longDescription() const
{
    if (hasVersion())
        return longDescription( "" );
    throw std::out_of_range( "No long description for this package" );
}

inline std::string Package::longDescription( std::string d ) const throw()
{
    if (!hasVersion()) return d;
    return anyVersion().longDescription( d );
}

inline std::string Package::maintainer() const
{
    anyVersion().checkValid();
    return maintainer( "" );
}

inline std::string Package::maintainer( const std::string& d ) const throw()
{
    if (!hasVersion()) return d;
    return anyVersion().maintainer( d );
}

inline std::string Package::architecture() const
{
    anyVersion().checkValid();
    return architecture( "" );
}

inline std::string Package::architecture( const std::string& d ) const throw()
{
    if (!hasVersion()) return d;
    return anyVersion().architecture( d );
}

inline std::string Package::section( const std::string& d ) const throw() {
    if (!valid()) return d;
    return (packageCache().strPtr( pkgPtr()->Section ));
}

inline Package::Actor Package::unboundInstall() {
    return ActorFM( "package-install", "Install Package",
                    &entity::Package::markInstall,
                    &entity::Package::canInstall );
}

inline Package::Actor Package::unboundRemove() {
    return ActorFM( "package-remove", "Remove Package",
                    &entity::Package::markRemove,
                    &entity::Package::canRemove );
}

inline Package::Actor Package::unboundUpgrade() {
    return ActorFM( "package-upgrade", "Upgrade Package",
                    &entity::Package::markInstall,
                    &entity::Package::canUpgrade );
}

inline Package::Actor Package::unboundKeep() {
    return ActorFM( "package-keep", "Keep Package",
                    &entity::Package::markKeep,
                    &entity::Package::canKeep);
}

inline Package::BoundActor Package::install() const {
    return BoundActor( unboundInstall(), *this );
}

inline Package::BoundActor Package::upgrade() const {
    return BoundActor( unboundUpgrade(), *this );
}

inline Package::BoundActor Package::keep() const {
    return BoundActor( unboundKeep(), *this );
}

inline Package::BoundActor Package::remove() const {
    return BoundActor( unboundRemove(), *this );
}

}
}
}

#include <apt-front/cache/entity/version.h>
namespace aptFront { namespace cache {
namespace entity {
template <class I> void Package::versions( I i ) const {
    Version v = firstVersionInCache();
    while (v.valid()) {
        *i++ = v;
        v = v.nextInCache();
    }
}
}}}
#endif
