/** -*- C++ -*-
    @file utils/range.h
    @author Peter Rockai <me@mornfall.net>
*/

#include <apt-front/forward.h>
#include <apt-front/utils/multitype.h>
#include <apt-front/utils/shared.h>
#include <iterator>
#include <vector>
#include <set>
#include <apt-front/error.h>
#include <algorithm>
#include <ext/algorithm>

#ifndef APTFRONT_UTILS_RANGE_H
#define APTFRONT_UTILS_RANGE_H

namespace aptFront {
namespace utils {

typedef void SortabilityTag;

template< typename T, typename I >
struct ExtIteratorTraits {
    typedef SortabilityTag Unsortable;
};

template< typename T >
struct ExtIteratorTraits< T, typename std::set< T >::iterator > {
    typedef SortabilityTag Sorted;
};

template< typename T >
struct ExtIteratorTraits< T, typename std::vector< T >::iterator > {
    typedef SortabilityTag Sortable;
};

template< typename T >
struct RangeBase : MultiTypeBase< RangeBase< T > >
{
    typedef T value_type;
    typedef ptrdiff_t difference_type;
    typedef T *pointer;
    typedef T &reference;
    typedef const T &const_reference;

    virtual T current() const = 0;
    virtual void next() = 0;
    virtual void ensureSorted() = 0;
    virtual Range< T > last_impl() const = 0;
    operator Range< T >() const;
    virtual bool empty() const = 0;
};

template< typename T, typename Self, typename Base = RangeBase< T > >
struct RangeImpl: MultiTypeImpl< Self, Base, equalityComparable >
{
    typedef T ElementType;
    Self &operator++() { this->self().next(); return this->self(); }
    T operator*() const { return this->self().current(); }
    virtual Range< T > last_impl() const {
        return this->self().last();
    }
    virtual bool empty() const {
        return this->self() == this->self().last();
    }
};

template< typename T >
struct Range : MultiType< Range< T >, RangeBase< T > >
{
    typedef MultiType< Range< T >, RangeBase< T > > Super;
    typedef T ElementType;

    Range( const RangeBase< T > *i ) : Super( i ) {}
    Range( const Range &i ) : Super( i ) {}
    Range() {}

    typedef std::forward_iterator_tag iterator_category;
    typedef T value_type;
    typedef ptrdiff_t difference_type;
    typedef T *pointer;
    typedef T &reference;
    typedef const T &const_reference;

    bool empty() const {
        if (this->impl())
            return this->impl()->empty();
        return true;
    }

    struct Proxy {
        Proxy( T _x ) : x( _x ) {}
        T x;
        const T *operator->() const { return &x; }
    };

    T current() const { return this->m_impl->current(); }
    Range last() const { return this->m_impl->last_impl(); }
    void next() { this->m_impl->next(); }
    void ensureSorted() { this->m_impl->ensureSorted(); }
    T operator*() const { return this->m_impl->current(); }
    Proxy operator->() const { return Proxy(this->m_impl->current()); }
    Range &operator++() { this->m_impl->next(); return *this; }
    Range operator++(int) {
        Range< T > tmp = *this;
        this->m_impl->next();
        return tmp;
    }
    void output( Consumer< T > t ) const {
        std::copy( *this, last(), t );
    }

    template< typename C >
    operator Range< C >();
};

template< typename T >
inline RangeBase< T >::operator Range< T >() const
{
    return Range< T >( this );
}

template< typename R >
Range< typename R::ElementType > range( R r ) {
    return r;
}

}
}

// ----- individual range implementations follow
#include <apt-front/utils/consumer.h>

namespace aptFront {
namespace utils {

// sfinae: substitution failure is not an error
template< typename T, typename I >
typename ExtIteratorTraits< T, I >::Unsortable ensureSortedT( I, I ) {
    throw exception::InternalError( "range not sortable" );
}

template< typename T, typename I >
typename ExtIteratorTraits< T, I >::Sorted ensureSortedT( I, I ) {
    return;
}

template< typename T, typename I >
typename ExtIteratorTraits< T, I >::Sortable ensureSortedT( I a, I b ) {
    if (__gnu_cxx::is_sorted( a, b ))
        return;
    std::sort( a, b );
}

template< typename In >
struct IteratorRange : public RangeImpl<
    typename std::iterator_traits< In >::value_type,
    IteratorRange< In > >
{
    typedef typename std::iterator_traits< In >::value_type Value;
    IteratorRange( In c, In e )
        : m_current( c ), m_end( e ) {}
    virtual Value current() const {
        return *m_current;
    }
    virtual void next() {
        ++m_current;
    }
    bool operator==( const IteratorRange &r ) const {
        return r.m_current == m_current && r.m_end == m_end;
    }
    IteratorRange last() const {
        return IteratorRange< In >( m_end, m_end );
    }

    void ensureSorted() {
        ensureSortedT< Value, In >( m_current, m_end );
    }

protected:
    In m_current, m_end;
};

template< typename T, typename Casted >
struct UpcastRange : public RangeImpl< T, UpcastRange< T, Casted > >
{
    UpcastRange( Range< Casted > r ) : m_casted( r ) {}
    virtual T current() const {
        return m_casted.current();
    }
    virtual void next() { m_casted.next(); }
    bool operator==( const UpcastRange &r ) const {
        return m_casted == r.m_casted; }
    UpcastRange last() const {
        return UpcastRange< T, Casted >( m_casted.last() );
    }
    void ensureSorted() {
        m_casted.ensureSorted();
    }
protected:
    Range< Casted > m_casted;
};

template< typename T, typename C >
Range< T > upcastRange( C r ) {
    return UpcastRange< T, typename C::ElementType >( r );
}

template< typename T> template< typename C >
Range< T >::operator Range< C >() {
    return upcastRange< C >( *this );
}

template< typename In >
Range< typename In::value_type > range( In b, In e ) {
    return IteratorRange< In >( b, e );
}


template< typename T>
struct SharedVector : std::vector< T >, SharedBase
{
};

template< typename T >
struct IntersectionRange : RangeImpl< T, IntersectionRange< T > >
{
    IntersectionRange( Range< T > r1, Range< T > r2 )
        : m_first( r1 ), m_second( r2 ),
        m_valid( false ) {
        r1.ensureSorted();
        r2.ensureSorted();
    }

    void find() const {
        if (!m_valid) {
            while (m_first != m_first.last()
                   && m_second != m_second.last()) {
                if (*m_first < *m_second)
                    ++m_first;
                else if (*m_second < *m_first)
                    ++m_second;
                else break;
            }
        }
        m_valid = true;
    }

    virtual void next() {
        find();
        ++m_first;
        ++m_second;
        m_valid = false;
    }

    virtual T current() const {
        find();
        return m_first.current();
    }

    IntersectionRange last() const {
        return IntersectionRange< T >( m_first.last(), m_second.last() );
    }

    bool operator==( const IntersectionRange &f ) const {
        find();
        f.find();
        return m_first == f.m_first;
        // return m_pred == f.m_pred && m_range == f.m_range;
    }

    void ensureSorted() {} // we are always sorted...

protected:
    mutable Range< T > m_first, m_second;
    mutable bool m_valid:1;
};

template< typename R >
IntersectionRange< typename R::ElementType > intersectionRange( R r1, R r2 ) {
    return IntersectionRange< typename R::ElementType >( r1, r2 );
}

template< typename T, typename Pred >
struct FilteredRange : RangeImpl< T, FilteredRange< T, Pred > >
{
    FilteredRange( Range< T > r, Pred p ) : m_range( r ), m_pred( p ),
        m_valid( false ) {}

    void find() const {
        if (!m_valid)
            m_range = std::find_if( m_range, m_range.last(), m_pred );
        m_valid = true;
    }

    virtual void next() {
        find();
        ++m_range;
        m_valid = false;
    }

    virtual T current() const {
        find();
        return m_range.current();
    }

    FilteredRange last() const {
        return FilteredRange< T, Pred >( m_range.last(), m_pred );
    }

    bool operator==( const FilteredRange &f ) const {
        find();
        f.find();
        return m_range == f.m_range;
        // return m_pred == f.m_pred && m_range == f.m_range;
    }

    void ensureSorted() {
        m_range.ensureSorted();
    }

protected:
    mutable Range< T > m_range;
    Pred m_pred;
    mutable bool m_valid:1;
};

template< typename Transform >
struct TransformedRange : RangeImpl< typename Transform::result_type,
                                     TransformedRange< Transform > >
{
    typedef typename Transform::argument_type Source;
    typedef typename Transform::result_type Result;
    TransformedRange( Range< Source > r, Transform t )
        : m_range( r ), m_transform( t ) {}

    bool operator==( const TransformedRange &o ) const {
        return m_range == o.m_range; // XXX buaaaahaaa!
    }

    Result current() const {
        return m_transform( m_range.current() );
    }

    void next() {
        ++m_range;
    }

    TransformedRange last() const {
        return TransformedRange( m_range.last(), m_transform );
    }

    void ensureSorted() { throw 0; } // hmm!

protected:
    Transform m_transform;
    Range< Source > m_range;
};

template< typename Trans >
TransformedRange< Trans > transformedRange(
    Range< typename Trans::argument_type > r, Trans t ) {
    return TransformedRange< Trans >( r, t );
}

template< typename Pred >
FilteredRange< typename Pred::argument_type, Pred > filteredRange(
    Range< typename Pred::argument_type > r, Pred p ) {
    return FilteredRange< typename Pred::argument_type, Pred >( r, p );
}

template< typename T >
struct VectorRange : RangeImpl< T, VectorRange< T > >,
    ConsumerImpl< T, VectorRange< T > >
{
    typedef std::random_access_iterator_tag iterator_category;
    VectorRange() : m_vector( new Vector ), m_position( 0 ) {}
    VectorRange( const Range< T > &i ) {
        RangeImpl< T, VectorRange< T > >::initFromBase( i.impl() );
    }

    virtual void consume( const T &a ) {
        m_vector->push_back( a );
    }

    virtual void next() {
        ++m_position;
    }

    virtual T current() const {
        return m_vector->operator[]( m_position );
    }

    VectorRange< T > last() const {
        VectorRange r = *this;
        r.m_position = std::distance( m_vector->begin(), m_vector->end() );
        return r;
    }

    Range< T > begin() const {
        return this;
    }

    bool operator==( const VectorRange &r ) const {
        return m_position == r.m_position;
    }

    VectorRange &operator+=( ptrdiff_t off ) {
        m_position += off;
        return *this;
    }
    VectorRange &operator--() {
        --m_position;
        return *this;
    }
    VectorRange operator--( int ) {
        VectorRange tmp( *this );
        --m_position;
        return tmp;
    }

    ptrdiff_t operator-( const VectorRange &r ) {
        return m_position - r.m_position;
    }
    VectorRange operator-( ptrdiff_t off ) {
        VectorRange< T > r( *this );
        r.m_position = m_position - off;
        return r;
    }

    VectorRange operator+( ptrdiff_t off ) {
        VectorRange< T > r( *this );
        r.m_position = m_position + off;
        return r;
    }

    bool operator<( const VectorRange &r ) {
        return m_position < r.m_position;
    }

    size_t size() const {
        return m_vector->size() - m_position;
    }

    T &operator*() {
        return m_vector->operator[]( m_position );
    }

    void clear() {
        m_vector->clear();
        m_position = 0;
    }

    void ensureSorted() { // reuse IteratorRange
        range( m_vector->begin(), m_vector->end() ).ensureSorted();
    }
protected:
    typedef SharedVector< T > Vector;
    typedef SharedPtr< Vector > VectorPointer;
    VectorPointer m_vector;
    ptrdiff_t m_position;
};

}
}

#endif
