/*
 * Merge different vocabularies together
 *
 * Copyright (C) 2003,2004,2005  Enrico Zini <enrico@debian.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <tagcoll/ParserBase.h>
#include <stdio.h>

#include <string>
#include <map>

#ifndef APTFRONT_UTILS_VOCABULARYMERGER_H
#define APTFRONT_UTILS_VOCABULARYMERGER_H

namespace aptFront {
namespace utils {

class VocabularyMerger
{
protected:
	class TagData : public std::map<std::string, std::string>
	{
	public:
		std::string name;
		// Offset in the last written file (used for indexing)
		long ofs;
		int len;

		TagData() : ofs(0), len(0) {}
	};
	class FacetData : public std::map<std::string, std::string>
	{
	public:
		std::string name;
		std::map<std::string, TagData> tags;
		// Offset in the last written file (used for indexing)
		long ofs;
		int len;

		FacetData() : ofs(0), len(0) {}

		TagData& obtainTag(const std::string& fullname);
	};
	std::map<std::string, FacetData> facets;
	
	FacetData& obtainFacet(const std::string& name);
	TagData& obtainTag(const std::string& fullname);
	
public:
	/**
	 * Parse and import the vocabulary from `input', merging the data with the
	 * previously imported ones
	 */
	void read(Tagcoll::ParserInput& input);

	/**
	 * Write the vocabulary data to the given file
	 */
	void write(const std::string& fname);

	/**
	 * Write the vocabulary data to the given output stream
	 */
	void write(FILE* out);

	/**
	 * Write the vocabulary index to the given file
	 *
	 * @param fname
	 *   Pathname of the index file to write
	 *
	 * @param base_ofs
	 *   The offset to the beginning of the indexed file.  It is usually 0, but
	 *   it can be set to something higher, for example, to index data in a
	 *   file with a fixed-length header.
	 */
	void writeIndex(const std::string& fname, long base_ofs = 0);


	/**
	 * Check if the vocabulary contains the facet `name'
	 */
	bool hasFacet(const std::string& name) const
	{
		return facets.find(name) != facets.end();
	}

	/**
	 * Check if the vocabulary contains the tag `fullname'
	 */
	bool hasTag(const std::string& fullname) const;
};

}
}

// vim:set ts=4 sw=4:
#endif
