/*
 * unity-webapps-action-tracker.c
 * Copyright (C) Canonical LTD 2011
 *
 * Author: Robert Carr <racarr@canonical.com>
 * 
 unity-webapps is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * unity-webapps is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.";
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <glib/gstdio.h>
#include <gio/gio.h>

#define WNCK_I_KNOW_THIS_IS_UNSTABLE
#include <libwnck/libwnck.h>

#include "unity-webapps-interest-tracker.h"
#include "unity-webapps-window-tracker.h"
#include "unity-webapps-interest-manager.h"

#include "../unity-webapps-debug.h"

struct _UnityWebappsInterestTrackerPrivate {
  UnityWebappsInterestManager *interest_manager;
  UnityWebappsWindowTracker *window_tracker;
  
  gint most_recent_interest;
  
  GQueue *most_recent_interest_queue;
  
  gboolean most_recent_is_active;
};

G_DEFINE_TYPE(UnityWebappsInterestTracker, unity_webapps_interest_tracker, G_TYPE_OBJECT)

enum 
{
  PROP_0,
  PROP_MOST_RECENT_INTEREST,
  PROP_WINDOW_TRACKER,
  PROP_INTEREST_MANAGER
};

enum {
  INTEREST_BECAME_ACTIVE,
  INTEREST_LOST_ACTIVE,
  LAST_SIGNAL
};

static guint signals[LAST_SIGNAL] = { 0 };

#define UNITY_WEBAPPS_INTEREST_TRACKER_GET_PRIVATE(object) (G_TYPE_INSTANCE_GET_PRIVATE ((object), UNITY_WEBAPPS_TYPE_INTEREST_TRACKER, UnityWebappsInterestTrackerPrivate))

static void
unity_webapps_interest_tracker_finalize (GObject *object)
{
  UnityWebappsInterestTracker *tracker;
  
  tracker = UNITY_WEBAPPS_INTEREST_TRACKER (object);
  
  g_queue_free (tracker->priv->most_recent_interest_queue);
}

static void
unity_webapps_interest_tracker_dispose (GObject *object)
{
  UnityWebappsInterestTracker *tracker;
  
  tracker = UNITY_WEBAPPS_INTEREST_TRACKER (object);
  
  if (tracker->priv->window_tracker)
    {
      g_object_unref (G_OBJECT (tracker->priv->window_tracker));
      tracker->priv->window_tracker = NULL;
    }
  
  if (tracker->priv->interest_manager)
    {
      g_object_unref (G_OBJECT (tracker->priv->interest_manager));
      tracker->priv->interest_manager = NULL;
    }
}

static void
unity_webapps_interest_tracker_get_property (GObject *object, guint property_id,
					   GValue *value, GParamSpec *pspec)
{
  UnityWebappsInterestTracker *self;
  
  self = UNITY_WEBAPPS_INTEREST_TRACKER (object);
  
  switch (property_id)
    {
    case PROP_MOST_RECENT_INTEREST:
      g_value_set_int (value, self->priv->most_recent_interest);
      break;
    case PROP_WINDOW_TRACKER:
      g_value_set_object (value, self->priv->window_tracker);
      break;
    case PROP_INTEREST_MANAGER:
      g_value_set_object (value, self->priv->interest_manager);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
      break;
    }
}

static void
unity_webapps_interest_tracker_set_most_recent_interest (UnityWebappsInterestTracker *tracker,
							 gint interest_id)
{
  gint old_interest_id;
  
  UNITY_WEBAPPS_NOTE (INTEREST, "Updating most recent interest to %d in response to window or tab selection",
		      interest_id);
  
  old_interest_id = tracker->priv->most_recent_interest;
  tracker->priv->most_recent_interest = interest_id;
  
  tracker->priv->most_recent_is_active = TRUE;

  g_queue_push_head (tracker->priv->most_recent_interest_queue, GINT_TO_POINTER (interest_id));

  if (old_interest_id != -1)
    g_signal_emit (tracker, signals[INTEREST_LOST_ACTIVE], 0, old_interest_id);
  if (tracker->priv->most_recent_interest != -1)
    g_signal_emit (tracker, signals[INTEREST_BECAME_ACTIVE], 0, tracker->priv->most_recent_interest);
  
  g_object_notify (G_OBJECT (tracker), "most-recent-interest");
}


static void
unity_webapps_interest_tracker_active_window_id_changed (GObject *object,
							 GParamSpec *pspec,
							 gpointer user_data)
{
  UnityWebappsInterestTracker *tracker;
  guint64 active_window_id, current_interest_window_id;
  
  tracker = (UnityWebappsInterestTracker *)user_data;
  
  active_window_id = unity_webapps_window_tracker_get_active_window_id (tracker->priv->window_tracker);
  current_interest_window_id = unity_webapps_interest_manager_get_interest_window (tracker->priv->interest_manager,
									     tracker->priv->most_recent_interest);
  
  if (current_interest_window_id == active_window_id)
    {
      if (tracker->priv->most_recent_is_active == FALSE)
	{
	  UNITY_WEBAPPS_NOTE (INTEREST, "Window for most recent interest %d became active",
			      tracker->priv->most_recent_interest);

	  tracker->priv->most_recent_is_active = TRUE;
	  g_signal_emit (tracker, signals[INTEREST_BECAME_ACTIVE], 0, tracker->priv->most_recent_interest);
	  
	  return;
	}
      else
	{
	  UNITY_WEBAPPS_NOTE (INTEREST, "Making an interest active when it already was. This may indicate an error");
	  
	  return;
	}
    }
  else
    {
      gint new_active_interest;

      if (tracker->priv->most_recent_is_active == TRUE)
	{
	  UNITY_WEBAPPS_NOTE (INTEREST, "Most recent interest %d was active but the window changed",
			      tracker->priv->most_recent_interest);

	  tracker->priv->most_recent_is_active = FALSE;
	  g_signal_emit (tracker, signals[INTEREST_LOST_ACTIVE], 0, tracker->priv->most_recent_interest);
	  
	}
      
      new_active_interest 
	= unity_webapps_interest_manager_get_active_interest_at_window (tracker->priv->interest_manager,
									active_window_id);
	  
      if (new_active_interest == -1)
	{
	  return;
	}
      
      UNITY_WEBAPPS_NOTE (INTEREST, "Found an active interest %d in a newly active window",
			  tracker->priv->most_recent_interest);
      
      unity_webapps_interest_tracker_set_most_recent_interest (tracker, new_active_interest);
    }
  
}

static void
unity_webapps_interest_tracker_ensure_interest_state (UnityWebappsInterestTracker *tracker,
						      gint interest_id)
{
  guint64 active_window_id, interest_window_id;
  gboolean is_active;
  
  is_active = unity_webapps_interest_manager_get_interest_is_active (tracker->priv->interest_manager,
								     interest_id);

  if (is_active == FALSE)
    {
      if (tracker->priv->most_recent_interest != interest_id)
	{
	  return;
	}

      if (tracker->priv->most_recent_is_active == TRUE)
	{
	  UNITY_WEBAPPS_NOTE (INTEREST, "Most recent interest (%d) lost activity due to tab selection",
			      interest_id);
	  tracker->priv->most_recent_is_active = FALSE;
	  g_signal_emit (tracker, signals[INTEREST_LOST_ACTIVE], 0, interest_id);
	}
      
      return;
    }

  
  if (interest_id == tracker->priv->most_recent_interest)
    {
      UNITY_WEBAPPS_NOTE (INTEREST, "Interest %d became active but was already the most recent interest",
			  interest_id);
      return;
    }
  
  active_window_id = unity_webapps_window_tracker_get_active_window_id (tracker->priv->window_tracker);
  
  if (active_window_id == 0)
    {
      UNITY_WEBAPPS_NOTE (INTEREST, "Interest %d became active but we have no active window to compare it to!",
			  interest_id);
      return;
    }

  interest_window_id = unity_webapps_interest_manager_get_interest_window (tracker->priv->interest_manager, interest_id);
  
  if (interest_window_id != active_window_id)
    return;
  
  UNITY_WEBAPPS_NOTE (INTEREST, "Interest %d became active at active window", interest_id);
  
  unity_webapps_interest_tracker_set_most_recent_interest (tracker, interest_id);
}



static void
unity_webapps_interest_tracker_interest_is_active_changed (UnityWebappsInterestManager *interest_manager,
							   gint interest_id,
							   gboolean is_active,
							   gpointer user_data)
{
  UnityWebappsInterestTracker *tracker;
  
  tracker = UNITY_WEBAPPS_INTEREST_TRACKER(user_data);
  
  unity_webapps_interest_tracker_ensure_interest_state (tracker, interest_id);
}

static void
unity_webapps_interest_tracker_interest_window_changed (UnityWebappsInterestManager *interest_manager,
							gint interest_id,
							guint64 new_window,
							gpointer user_data)
{
  UnityWebappsInterestTracker *tracker;

  tracker = UNITY_WEBAPPS_INTEREST_TRACKER (user_data);
  
  unity_webapps_interest_tracker_ensure_interest_state (tracker, interest_id);
}

static void
unity_webapps_interest_tracker_interest_removed (UnityWebappsInterestManager *interest_manager,
						 UnityWebappsInterest *interest,
						 gpointer user_data)
{
  UnityWebappsInterestTracker *tracker;
  gint old_interest, interest_id;
  
  tracker = (UnityWebappsInterestTracker *)user_data;
  
  interest_id = interest->id;
  
  UNITY_WEBAPPS_NOTE (INTEREST, "Interest tracker clearing %d from queue", interest_id);
  
  g_queue_remove_all (tracker->priv->most_recent_interest_queue, GINT_TO_POINTER (interest_id));
  
  if (interest_id != tracker->priv->most_recent_interest)
    return;
  
  old_interest = tracker->priv->most_recent_interest;
  
  if (g_queue_get_length (tracker->priv->most_recent_interest_queue) == 0)
    tracker->priv->most_recent_interest = -1;
  else
    tracker->priv->most_recent_interest = GPOINTER_TO_INT (g_queue_peek_tail (tracker->priv->most_recent_interest_queue));

  UNITY_WEBAPPS_NOTE (INTEREST, "Updating most recent interest to %d in response to removal of the previous most recent interest %d",
		      tracker->priv->most_recent_interest,
		      old_interest);
  
  tracker->priv->most_recent_is_active = FALSE;
  
  if (old_interest != -1)
    g_signal_emit (tracker, signals[INTEREST_LOST_ACTIVE], 0, old_interest);
  if (tracker->priv->most_recent_interest != -1)
    g_signal_emit (tracker, signals[INTEREST_BECAME_ACTIVE], 0, tracker->priv->most_recent_interest);
  
  g_object_notify (G_OBJECT (tracker), "most-recent-interest");
}



static void
unity_webapps_interest_tracker_initialize_window_tracker (UnityWebappsInterestTracker *tracker)
{
  g_signal_connect_object (tracker->priv->window_tracker, "notify::active-window-id",
			   G_CALLBACK (unity_webapps_interest_tracker_active_window_id_changed),
			   tracker, 0);
}

static void
unity_webapps_interest_tracker_initialize_interest_manager (UnityWebappsInterestTracker *tracker)
{
  g_signal_connect_object (tracker->priv->interest_manager, "active-changed",
			   G_CALLBACK (unity_webapps_interest_tracker_interest_is_active_changed),
			   tracker, 0);
  g_signal_connect_object (tracker->priv->interest_manager, "window-changed",
			   G_CALLBACK (unity_webapps_interest_tracker_interest_window_changed),
			   tracker, 0);
  g_signal_connect_object (tracker->priv->interest_manager, "interest-removed",
			   G_CALLBACK (unity_webapps_interest_tracker_interest_removed),
			   tracker, 0);
}





static void
unity_webapps_interest_tracker_set_property (GObject *object, guint property_id,
					     const GValue *value, GParamSpec *pspec)
{
  UnityWebappsInterestTracker *self;
  
  self = UNITY_WEBAPPS_INTEREST_TRACKER (object);
  
  switch (property_id)
    {
    case PROP_WINDOW_TRACKER:
      g_return_if_fail (self->priv->window_tracker == NULL);
      self->priv->window_tracker = UNITY_WEBAPPS_WINDOW_TRACKER (g_value_dup_object (value));
      unity_webapps_interest_tracker_initialize_window_tracker (self);
      break;
    case PROP_INTEREST_MANAGER:
      g_return_if_fail (self->priv->interest_manager == NULL);
      self->priv->interest_manager = UNITY_WEBAPPS_INTEREST_MANAGER (g_value_dup_object (value));
      unity_webapps_interest_tracker_initialize_interest_manager (self);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
      break;
      
    }
}




static void
unity_webapps_interest_tracker_class_init (UnityWebappsInterestTrackerClass *klass)
{
  GParamSpec *pspec;
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  
  object_class->finalize = unity_webapps_interest_tracker_finalize;
  object_class->dispose = unity_webapps_interest_tracker_dispose;

  object_class->get_property = unity_webapps_interest_tracker_get_property;
  object_class->set_property = unity_webapps_interest_tracker_set_property;
  
  g_type_class_add_private (object_class, sizeof(UnityWebappsInterestTrackerPrivate));
  
  pspec = g_param_spec_int ("most-recent-interest", "Most recent interest", "Most recently used interest",
			       0, G_MAXINT, 0, G_PARAM_READABLE);
  g_object_class_install_property (object_class, PROP_MOST_RECENT_INTEREST, pspec);
  
  pspec = g_param_spec_object("window-tracker", "Window tracker", "The window tracker to use to track interest activity",
			      UNITY_WEBAPPS_TYPE_WINDOW_TRACKER,
			      G_PARAM_READWRITE |
			      G_PARAM_CONSTRUCT_ONLY |
			      G_PARAM_STATIC_STRINGS);
  g_object_class_install_property (object_class, PROP_WINDOW_TRACKER, pspec);

  pspec = g_param_spec_object("interest-manager", "Interest Manager", "The interest manager to use as a tracking model",
			      UNITY_WEBAPPS_TYPE_INTEREST_MANAGER,
			      G_PARAM_READWRITE |
			      G_PARAM_CONSTRUCT_ONLY |
			      G_PARAM_STATIC_STRINGS);
  g_object_class_install_property (object_class, PROP_INTEREST_MANAGER, pspec);
  
  signals[INTEREST_BECAME_ACTIVE] = g_signal_new ("interest-became-active",
						  UNITY_WEBAPPS_TYPE_INTEREST_TRACKER,
						  G_SIGNAL_RUN_LAST,
						  0,
						  NULL,
						  NULL,
						  NULL,
						  G_TYPE_NONE,
						  1,
						  G_TYPE_INT);
  signals[INTEREST_LOST_ACTIVE] = g_signal_new ("interest-lost-active",
						  UNITY_WEBAPPS_TYPE_INTEREST_TRACKER,
						  G_SIGNAL_RUN_LAST,
						  0,
						  NULL,
						  NULL,
						  NULL,
						  G_TYPE_NONE,
						  1,
						  G_TYPE_INT);
}


static void
unity_webapps_interest_tracker_init (UnityWebappsInterestTracker *tracker)
{
  tracker->priv = UNITY_WEBAPPS_INTEREST_TRACKER_GET_PRIVATE (tracker);
  
  tracker->priv->most_recent_interest = -1;
  tracker->priv->most_recent_interest_queue = g_queue_new ();
  tracker->priv->most_recent_is_active = FALSE;
  tracker->priv->window_tracker = NULL;
  tracker->priv->interest_manager = NULL;
}

UnityWebappsInterestTracker *
unity_webapps_interest_tracker_new (UnityWebappsWindowTracker *window_tracker, UnityWebappsInterestManager *interest_manager)
{
  return g_object_new (UNITY_WEBAPPS_TYPE_INTEREST_TRACKER,
		       "window-tracker", window_tracker,
		       "interest-manager", interest_manager,
		       NULL);
}

gint
unity_webapps_interest_tracker_get_most_recent_interest (UnityWebappsInterestTracker *tracker)
{
  return tracker->priv->most_recent_interest;
}

GList*
unity_webapps_interest_tracker_get_most_recent_interests (UnityWebappsInterestTracker *tracker)
{
  return tracker->priv->most_recent_interest_queue->head;
}

gboolean
unity_webapps_interest_tracker_get_interest_is_active (UnityWebappsInterestTracker *tracker,
						       gint interest_id)
{
  return (tracker->priv->most_recent_interest == interest_id) && tracker->priv->most_recent_is_active;
}

void
unity_webapps_interest_tracker_raise_most_recent (UnityWebappsInterestTracker *tracker, guint32 timestamp)
{
  WnckWindow *window;
  gint interest_id;
  guint64 window_id;
  
  interest_id = unity_webapps_interest_tracker_get_most_recent_interest (tracker);
  if (interest_id <= 0)
    return;
  
  window_id = unity_webapps_interest_manager_get_interest_window (tracker->priv->interest_manager, interest_id);
  
  if (window_id == 0)
    return;
  
  window = wnck_window_get (window_id);

  wnck_window_activate (window, timestamp);
}

UnityWebappsInterestManager *
unity_webapps_interest_tracker_get_interest_manager (UnityWebappsInterestTracker *tracker)
{
  g_return_val_if_fail (UNITY_WEBAPPS_IS_INTEREST_TRACKER (tracker), NULL);
  return tracker->priv->interest_manager;
}
