/*
 * unity-webapps-music_player-context.c
 * Copyright (C) Canonical LTD 2011
 *
 * Author: Robert Carr <racarr@canonical.com>
 * 
 unity-webapps is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * unity-webapps is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.";
 */

#include "unity-webapps-music-player-context.h"

#include "unity-webapps-dbus-defs.h"

#include "unity-webapps-resource-cache.h"

#include "unity-webapps-debug.h"

#include "config.h"

static void emit_music_player_signal (GDBusConnection *connection, const gchar *signal_name);

static void player_raise_callback (UnityMusicPlayer *player, gpointer user_data);
static void player_next_callback (UnityMusicPlayer *player, gpointer user_data);
static void player_previous_callback (UnityMusicPlayer *player, gpointer user_data);
static void player_play_pause_callback (UnityMusicPlayer *player, gpointer user_data);


static void
emit_playlist_activated_signal (UnityWebappsMusicPlayerContext *indicator_context,
				const gchar *playlist)
{
  GError *error;

  UNITY_WEBAPPS_NOTE (INDICATOR, "Emitting PlaylistActivated signal (%s)", playlist);

  error = NULL;
  g_dbus_connection_emit_signal (indicator_context->connection,
				 NULL,
				 UNITY_WEBAPPS_MUSIC_PLAYER_PATH,
				 UNITY_WEBAPPS_MUSIC_PLAYER_IFACE,
				 "PlaylistActivated",
				 g_variant_new ("(s)", (gchar *)playlist, NULL),
				 &error);
  if (error != NULL)
    {
      g_warning ("Error emitting PlaylistActivated signal (from sound menu) in music player context: %s", error->message);
      
      g_error_free (error);
    }
}

static void
unity_webapps_music_player_context_playlist_activated (UnityMusicPlayer *player,
						       gchar *playlist_id,
						       gpointer user_data)
{
  UnityWebappsMusicPlayerContext *context;
  const gchar *playlist_name;
  
  context = (UnityWebappsMusicPlayerContext *)user_data;
  playlist_name = (const gchar *)g_hash_table_lookup (context->playlist_names_by_id, playlist_id);
  if (playlist_name == NULL)
    {
      UNITY_WEBAPPS_NOTE (MUSIC_PLAYER, "Got playlist activation for playlist we don't know about");
    }

  emit_playlist_activated_signal (context, playlist_name);
}




static UnityMusicPlayer *
get_unity_music_player (UnityWebappsMusicPlayerContext *context)
{
  UnityMusicPlayer *player;
  
  if (context->music_player == NULL)
    {
      UNITY_WEBAPPS_NOTE (MUSIC_PLAYER, "Initializing UnityMusicPlayer");
      player = unity_music_player_new (context->desktop_name);
      
      g_signal_connect (player, "activate-playlist",
			G_CALLBACK (unity_webapps_music_player_context_playlist_activated),
			context);
      
      unity_music_player_set_is_blacklisted (player, FALSE);
      
      g_signal_connect (player, "raise", G_CALLBACK (player_raise_callback), context);
      g_signal_connect (player, "play_pause", G_CALLBACK (player_play_pause_callback), context);
      g_signal_connect (player, "next", G_CALLBACK (player_next_callback), context);
      g_signal_connect (player, "previous", G_CALLBACK (player_previous_callback), context);
#ifdef HAVE_UNITY_MUSIC_PLAYER_EXPORT
      unity_music_player_export (player);
#endif
      unity_music_player_set_title (player, context->label);
      
      context->music_player = player;

      unity_webapps_gen_music_player_set_playback_state (context->music_player_service_interface, 0);

    }
  
  return context->music_player;
}


static void
real_set_track (UnityWebappsMusicPlayerContext *music_player_context,
		const gchar *artist,
		const gchar *album,
		const gchar *title,
		const gchar *art_location)
{
  UnityTrackMetadata  *metadata;
  GFile *artwork_file = NULL;
  
  metadata = unity_track_metadata_new ();
  unity_track_metadata_set_artist (metadata, artist);
  unity_track_metadata_set_album (metadata, album);
  unity_track_metadata_set_title (metadata, title);
  
  if (art_location != NULL)
    {
      artwork_file = g_file_new_for_path (art_location);
      unity_track_metadata_set_art_location (metadata, artwork_file);
    }
  
  unity_music_player_set_current_track (get_unity_music_player (music_player_context), metadata);
  
  if (artwork_file != NULL)
    {
      g_object_unref (G_OBJECT (artwork_file));
    }
  g_object_unref (G_OBJECT (metadata));
  
  UNITY_WEBAPPS_NOTE (MUSIC_PLAYER, "Set track: %s", title);
}

static void
player_raise_callback (UnityMusicPlayer *player, gpointer user_data)
{
  UnityWebappsMusicPlayerContext *music_player_context;

  UNITY_WEBAPPS_NOTE (MUSIC_PLAYER, "Got Raise callback");
  
  music_player_context = (UnityWebappsMusicPlayerContext *)user_data;
  
  unity_webapps_context_daemon_emit_raise (music_player_context->connection, -1, NULL);

}

static void
emit_music_player_signal (GDBusConnection *connection, const gchar *signal_name)
{
  GError *error;
  
  UNITY_WEBAPPS_NOTE (MUSIC_PLAYER, "Emitting signal %s", signal_name);
  
  error = NULL;
  
  g_dbus_connection_emit_signal (connection,
				 NULL,
				 UNITY_WEBAPPS_MUSIC_PLAYER_PATH,
				 UNITY_WEBAPPS_MUSIC_PLAYER_IFACE,
				 signal_name,
				 NULL /* Params */,
				 &error);
  
  if (error != NULL)
    {
      g_warning ("Error emitting %s signal in music player context", signal_name);
      g_error_free (error);

      return;
    }
				 
}

static void
player_play_pause_callback (UnityMusicPlayer *player, gpointer user_data)
{
  UnityWebappsMusicPlayerContext *music_player_context;
  
  UNITY_WEBAPPS_NOTE (MUSIC_PLAYER, "Got PlayPause callback");
  
  music_player_context = (UnityWebappsMusicPlayerContext *)user_data;
  
  emit_music_player_signal (music_player_context->connection, "PlayPause");
}

static void
player_previous_callback (UnityMusicPlayer *player, gpointer user_data)
{
  UnityWebappsMusicPlayerContext *music_player_context;
  
  UNITY_WEBAPPS_NOTE (MUSIC_PLAYER, "Got Previous callback");
  
  music_player_context = (UnityWebappsMusicPlayerContext *)user_data;
  
  emit_music_player_signal (music_player_context->connection, "Previous");
}

static void
player_next_callback (UnityMusicPlayer *player, gpointer user_data)
{
  UnityWebappsMusicPlayerContext *music_player_context;
  
  UNITY_WEBAPPS_NOTE (MUSIC_PLAYER, "Got Next callback");
  
  music_player_context = (UnityWebappsMusicPlayerContext *)user_data;
  
  emit_music_player_signal (music_player_context->connection, "Next");
}

static void
unity_webapps_music_player_context_clear_playlists (UnityWebappsMusicPlayerContext *context)
{
  UnityMusicPlayer *player;
  UnityPlaylist **playlists, *p;
  gint num_playlists, i;
  
  player = get_unity_music_player (context);
  
  playlists = unity_music_player_get_playlists (player, &num_playlists);
  
  for (i = 0; i < num_playlists; i++)
    {
      p = playlists[i];
      unity_music_player_remove_playlist (player, p);
    }
  
  g_hash_table_remove_all (context->playlist_names_by_id);

  g_free (playlists);
}

static void
unity_webapps_music_player_context_set_playlists (UnityWebappsMusicPlayerContext *context,
						  const gchar *const *playlists)
{
  GIcon *icon;
  UnityMusicPlayer *player;
  gint i, len;
  GError *error = NULL;
  
  player = get_unity_music_player (context);
  len = g_strv_length ((gchar **)playlists);
  icon = g_icon_new_for_string ("playlist-symbolic", &error);
  
  if (error != NULL)
    {
      UNITY_WEBAPPS_NOTE (MUSIC_PLAYER, "Failed to load playlist icon");
      icon = NULL;
    }
  
  for (i = 0; i < len; i++)
    {
      UnityPlaylist *p;
      const gchar *playlist_name;
      gchar *id;
      
      playlist_name = playlists[i];
      id = g_strdup_printf("/Playlist%d", i);
      p = unity_playlist_new(id);

      g_hash_table_insert (context->playlist_names_by_id, g_strdup(id),
			   g_strdup (playlist_name));

      g_free (id);
      
      unity_playlist_set_icon (p, icon);
      unity_playlist_set_name (p, playlist_name);
      
      unity_music_player_add_playlist (player, p);
    }
  
  UNITY_WEBAPPS_NOTE (MUSIC_PLAYER, "Added %d playlists", len);
}



static gboolean
on_handle_set_playlists (UnityWebappsGenMusicPlayer *music_player,
			 GDBusMethodInvocation *invocation,
			 const gchar *const *playlists,
			 gpointer user_data)
{
  UnityWebappsMusicPlayerContext *music_player_context;
  UNITY_WEBAPPS_NOTE (MUSIC_PLAYER, "Handling SetPlaylists call");
  
  music_player_context = (UnityWebappsMusicPlayerContext *)user_data;

  unity_webapps_music_player_context_clear_playlists (music_player_context);
  unity_webapps_music_player_context_set_playlists (music_player_context, playlists);

  g_dbus_method_invocation_return_value (invocation, NULL);
  
  return TRUE;
}

static gboolean
on_handle_set_track (UnityWebappsGenMusicPlayer *indicator,
		     GDBusMethodInvocation *invocation,
		     const gchar *artist,
		     const gchar *album,
		     const gchar *title,
		     const gchar *icon_url,
		     gpointer user_data)
{
  UnityWebappsMusicPlayerContext *context;
  
  UNITY_WEBAPPS_NOTE (MUSIC_PLAYER, "Handling set track: %s by %s from %s", title, artist, album);
  
  context = (UnityWebappsMusicPlayerContext *)user_data;
  
  if (g_strcmp0 (icon_url, "") == 0)
    {
      // TODO: DEFAULT?
      real_set_track (context, artist, album, title, NULL);
    }
  else
    {
      gchar *tmp_file;
      
      tmp_file = unity_webapps_resource_cache_lookup_uri (icon_url);
      
      real_set_track (context, artist, album, title, tmp_file);
      
      g_free (tmp_file);
    }
  
  g_dbus_method_invocation_return_value (invocation, NULL);
  
  return TRUE;
}

static void
on_notify_playback_state (GObject *object,
			  GParamSpec *param,
			  gpointer user_data)
{
  UnityWebappsMusicPlayerContext *context;
  UnityWebappsGenMusicPlayer *music_player_service_interface;
  gint playback_state;
  
  UNITY_WEBAPPS_NOTE (MUSIC_PLAYER, "Playback state changed");

  context = (UnityWebappsMusicPlayerContext *)user_data;

  music_player_service_interface = UNITY_WEBAPPS_GEN_MUSIC_PLAYER (object);
  
  playback_state 
    = unity_webapps_gen_music_player_get_playback_state (music_player_service_interface);
  
  UNITY_WEBAPPS_NOTE (MUSIC_PLAYER, "Playback state is now %s", playback_state ? "playing" : "paused");
  

  if (context->music_player != NULL)
    {
      unity_music_player_set_playback_state (get_unity_music_player (context), (UnityPlaybackState) playback_state);
    }
  
}

static void
on_notify_can_go_next (GObject *object,
		       GParamSpec *param,
		       gpointer user_data)
{
  UnityWebappsMusicPlayerContext *context;
  UnityWebappsGenMusicPlayer *music_player_service_interface;
  gboolean can;
  
  UNITY_WEBAPPS_NOTE (MUSIC_PLAYER, "Can go next changed");

  context = (UnityWebappsMusicPlayerContext *)user_data;
  
  music_player_service_interface = UNITY_WEBAPPS_GEN_MUSIC_PLAYER (object);
  
  can
    = unity_webapps_gen_music_player_get_can_go_next (music_player_service_interface);
  
  UNITY_WEBAPPS_NOTE (MUSIC_PLAYER, "Can go next is now: %s", can ? "true" : "false");
  
  if (context->music_player != NULL)
    {
      unity_music_player_set_can_go_next (get_unity_music_player (context), can);
    }
  
  
}


static void
on_notify_can_go_previous (GObject *object,
		       GParamSpec *param,
		       gpointer user_data)
{
  UnityWebappsMusicPlayerContext *context;
  UnityWebappsGenMusicPlayer *music_player_service_interface;
  gboolean can;
  
  UNITY_WEBAPPS_NOTE (MUSIC_PLAYER, "Can go previous changed");

  context = (UnityWebappsMusicPlayerContext *)user_data;
  
  music_player_service_interface = UNITY_WEBAPPS_GEN_MUSIC_PLAYER (object);
  
  can
    = unity_webapps_gen_music_player_get_can_go_previous (music_player_service_interface);

  UNITY_WEBAPPS_NOTE (MUSIC_PLAYER, "Can go previous is now: %s", can ? "true" : "false");

  if (context->music_player != NULL)
    {
      unity_music_player_set_can_go_previous (get_unity_music_player (context), can);
    }
  
  
}

static void
on_notify_can_play (GObject *object,
		       GParamSpec *param,
		       gpointer user_data)
{
  UnityWebappsMusicPlayerContext *context;
  UnityWebappsGenMusicPlayer *music_player_service_interface;
  gboolean can;
  
  UNITY_WEBAPPS_NOTE (MUSIC_PLAYER, "Can play changed");

  context = (UnityWebappsMusicPlayerContext *)user_data;
  
  music_player_service_interface = UNITY_WEBAPPS_GEN_MUSIC_PLAYER (object);
  
  can
    = unity_webapps_gen_music_player_get_can_play (music_player_service_interface);
  
  UNITY_WEBAPPS_NOTE (MUSIC_PLAYER, "Can play is now: %s", can ? "true" : "false");
  
  if (context->music_player != NULL)
    {
      unity_music_player_set_can_play (get_unity_music_player (context), can);
    }
  
  
}

static void
on_notify_can_pause (GObject *object,
		       GParamSpec *param,
		       gpointer user_data)
{
  UnityWebappsMusicPlayerContext *context;
  UnityWebappsGenMusicPlayer *music_player_service_interface;
  gboolean can;
  
  UNITY_WEBAPPS_NOTE (MUSIC_PLAYER, "Can pause changed");

  context = (UnityWebappsMusicPlayerContext *)user_data;
  
  music_player_service_interface = UNITY_WEBAPPS_GEN_MUSIC_PLAYER (object);
  
  can
    = unity_webapps_gen_music_player_get_can_pause (music_player_service_interface);
  
  UNITY_WEBAPPS_NOTE (MUSIC_PLAYER, "Can pause is now: %s", can ? "true" : "false");
  
  if (context->music_player != NULL)
    {
      unity_music_player_set_can_pause (get_unity_music_player (context), can);
    }
  
  
}

static void
export_object (GDBusConnection *connection,
	       UnityWebappsMusicPlayerContext *music_player_context)
{
  GError *error;
  
  music_player_context->music_player_service_interface = unity_webapps_gen_music_player_skeleton_new ();
  
  g_signal_connect (music_player_context->music_player_service_interface,
		    "handle-set-track",
		    G_CALLBACK (on_handle_set_track),
		    music_player_context);
  g_signal_connect (music_player_context->music_player_service_interface,
		    "handle-set-playlists",
		    G_CALLBACK (on_handle_set_playlists),
		    music_player_context);

  unity_webapps_gen_music_player_set_playback_state (music_player_context->music_player_service_interface, 0);
  unity_webapps_gen_music_player_set_can_go_previous (music_player_context->music_player_service_interface, FALSE);
  unity_webapps_gen_music_player_set_can_go_next (music_player_context->music_player_service_interface, FALSE);
  unity_webapps_gen_music_player_set_can_play (music_player_context->music_player_service_interface, FALSE);
  unity_webapps_gen_music_player_set_can_pause (music_player_context->music_player_service_interface, FALSE);

  
  g_signal_connect (music_player_context->music_player_service_interface,
		    "notify::playback-state",
		    G_CALLBACK (on_notify_playback_state),
		    music_player_context),
  g_signal_connect (music_player_context->music_player_service_interface,
		    "notify::can-go-next",
		    G_CALLBACK (on_notify_can_go_next),
		    music_player_context),
  g_signal_connect (music_player_context->music_player_service_interface,
		    "notify::can-go-previous",
		    G_CALLBACK (on_notify_can_go_previous),
		    music_player_context);
  g_signal_connect (music_player_context->music_player_service_interface,
		    "notify::can-play",
		    G_CALLBACK (on_notify_can_play),
		    music_player_context);
  g_signal_connect (music_player_context->music_player_service_interface,
		    "notify::can-pause",
		    G_CALLBACK (on_notify_can_pause),
		    music_player_context);

  error = NULL;

  g_dbus_interface_skeleton_export (G_DBUS_INTERFACE_SKELETON (music_player_context->music_player_service_interface),
				    connection,
				    UNITY_WEBAPPS_MUSIC_PLAYER_PATH,
				    &error);
  
  if (error != NULL)
    {
      g_error ("Error exporting Unity Webapps Music Player object: %s", error->message);
      
      g_error_free (error);
      
      return;
    }
  
  UNITY_WEBAPPS_NOTE (MUSIC_PLAYER, "Exported Music Player object");
  
}


UnityWebappsMusicPlayerContext *
unity_webapps_music_player_context_new (GDBusConnection *connection, const gchar *desktop_name, const gchar *canonical_name,
					const gchar *label)
{
  UnityWebappsMusicPlayerContext *music_player_context;
  
  UNITY_WEBAPPS_NOTE (MUSIC_PLAYER, "Creating new UnityWebappsMusicPlayerContext object");
  
  music_player_context = g_malloc0 (sizeof (UnityWebappsMusicPlayerContext));
  
  music_player_context->connection = g_object_ref (connection);

  music_player_context->playlist_names_by_id = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_free);
  
  export_object (connection, music_player_context);
  
  music_player_context->desktop_name = g_strdup (desktop_name);
  music_player_context->canonical_name = g_strdup (canonical_name);
  music_player_context->label = g_strdup (label);
  
  return music_player_context;
}

void
unity_webapps_music_player_context_free (UnityWebappsMusicPlayerContext *music_player_context)
{
  UNITY_WEBAPPS_NOTE (MUSIC_PLAYER, "Finalizing UnityWebappsMusicPlayerContext object");
  g_object_unref (G_OBJECT (music_player_context->music_player_service_interface));

  g_free (music_player_context->desktop_name);
  g_free (music_player_context->canonical_name);
  g_free (music_player_context->label);
  
  g_hash_table_destroy (music_player_context->playlist_names_by_id);

  if (music_player_context->music_player)
    {
      unity_music_player_set_is_blacklisted (music_player_context->music_player, TRUE);
      
#ifdef HAVE_UNITY_MUSIC_PLAYER_EXPORT
      unity_music_player_unexport (music_player_context->music_player);
#endif
      g_object_unref (G_OBJECT (music_player_context->music_player));
    }
  g_object_unref (G_OBJECT (music_player_context->connection));


  g_free (music_player_context);
}

