package com.tildemh.debbug;

import java.io.IOException;
import java.io.InputStream;
import java.net.HttpURLConnection;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.Calendar;
import java.util.Date;
import java.util.LinkedList;
import java.util.TimeZone;

import org.apache.oro.text.perl.Perl5Util;

/**
 * Gets bug reports in html format from bugs.debian.org and regexp's them to our
 * needs. This is (hopefully) only a temporary measure - Once the significance
 * of this library are apps which use it is known, we will try to convince the
 * bts maintainer to adopt a more friendly way fo working.
 *
 * <p>This is released under the terms of the GNU Lesser General Public License
 * (LGPL). See the COPYING file for details.
 *
 * @version $Id: WWWBugParser.java,v 1.43 2004/03/06 14:53:25 mh Exp $
 * @author &copy; Mark Howard &lt;mh@debian.org&gt; 2002
 */
public class WWWBugParser{

	public static final boolean DEBUG = false;

	/**
	 * Determines whether a bug has been modified, by looking at the
	 * Last-Modified header of bugreport.cgi, when using HTTP head.
	 * @param number the bug number to test
	 * @param lastCheck the last time the report was checked.
	 */
	public static boolean checkModified(Bug bug) throws MalformedURLException, IOException{
		if (DEBUG) System.out.println( System.currentTimeMillis() + " start chck modified");
		long lastCheck = bug.getLastUpdated();
		if (lastCheck < 0 || ! bug.getComplete()) return true;

		int number = bug.getNumber();
		URL url = new URL("http://bugs.debian.org/cgi-bin/bugreport.cgi?bug="+number);
		HttpURLConnection inConnection = (HttpURLConnection) url.openConnection();
		HttpURLConnection.setFollowRedirects(true);
		inConnection.setRequestMethod("HEAD");
		inConnection.connect();
		if (DEBUG) System.out.println( inConnection.getHeaderField("Last-Modified") );
		try{
			long date = convertDate( inConnection.getHeaderField("Last-Modified") ).getTime();
			if (DEBUG) System.out.println(System.currentTimeMillis() +" done checkmodified" );

			if (DEBUG) System.out.println(date + " > "+lastCheck);
			if ( date > lastCheck ){
				return true;
			}else{
				// make a note that we've checked with the server & save report.
				bug.setUpdated();
				(Cache.getInstance()).store(bug);
				return false;
			}
		}catch(Exception e){
			e.printStackTrace();
			return true;
		}

	}
			
	/** 
	 * Downloads a bug from  bugs.debian.org.
	 */
	public static void downloadBug(String number, Bug bug, boolean verbose) throws MalformedURLException, IOException, BugNotFound{
		if (DEBUG) System.out.println( System.currentTimeMillis() + " start download bug");
		//System.out.println("Downloading bug #"+number);
		URL url = new URL("http://bugs.debian.org/cgi-bin/bugreport.cgi?bug="+number);
		if (DEBUG) System.out.println("http://bugs.debian.org/cgi-bin/bugreport.cgi?bug="+number);
		HttpURLConnection inConnection = (HttpURLConnection) url.openConnection();
		InputStream in = inConnection.getInputStream();
		
		StringBuffer htmlPage = new StringBuffer();
		int ch =0;
		int STEP = 100;
		long lastPulsed = System.currentTimeMillis();
		int loop = STEP;
		while ((ch = in.read()) != -1){
			htmlPage.append((char)ch);
			if (loop == 0){
				long diff = System.currentTimeMillis() - lastPulsed;
				if ( diff > 200 ){
					STEP *= 0.8;
				}
				if ( diff > 100 ){
					lastPulsed = System.currentTimeMillis();
					bug.eventRetrievingBug();
					loop = STEP;
				}else{
					STEP *= 1.2;
					loop += 50;
				}
			}
			loop--;
		}
		bug.eventDownloadedBug();

		makeBug( htmlPage.toString(), bug);
		if (DEBUG) System.out.println(System.currentTimeMillis()+ " done download bug" );
	}

	
	/** 
	 * Constructs a bug from a html page from bugs.debian.org
	 */
	public static void makeBug(String htmlPage, Bug bug) throws BugNotFound{
		if (DEBUG) System.out.println( System.currentTimeMillis()  + " start parsebug");;
		if (htmlPage.indexOf("There is no record of Bug") >=0
				|| htmlPage.indexOf("An error occurred. Dammit.") >= 0 ){
			
			throw new BugNotFound( );
		}

		Perl5Util perl;
		perl = new Perl5Util();
		while(perl.match("/&lt;/", htmlPage))
			htmlPage = perl.substitute("s/&lt;/</", htmlPage);
		while(perl.match("/&gt;/", htmlPage))
			htmlPage = perl.substitute("s/&gt;/>/", htmlPage);
		while(perl.match("/&amp;/", htmlPage))
			htmlPage = perl.substitute("s/&amp;/&/", htmlPage);
		while(perl.match("/&quot;/", htmlPage))
			htmlPage = perl.substitute("s/&quot;/\"/", htmlPage);

		
		if (bug == null) 
			throw new RuntimeException("Bug == null!!!!"); //

		String start = htmlPage;
		if (htmlPage.indexOf("<pre>")>0)
			 start = htmlPage.substring(0,  htmlPage.indexOf("<pre>") );
			
		try{
			bug.setNumber( Integer.parseInt( start.substring( start.indexOf("logs - #")+8, start.indexOf("</TITLE>")) ));
		}catch( Exception e){
		}
		bug.setTitle( start.substring( start.indexOf("<BR>")+4, start.indexOf("</H1>") ));
		if (start.indexOf(">Severity:")<=0){
			bug.setSeverity(Severity.NORMAL);
		}else{
			if (DEBUG) System.out.println("About to parse severity "+start.substring( start.indexOf("Severity: <em>")+14, start.indexOf("</em") ) );
			bug.setSeverity( Severity.parse(
					start.substring( start.indexOf("Severity: <em>")+14, start.indexOf("</em") ) ));
		}
		if (DEBUG) System.out.println("WWWBUGPARSER - Severity set to "+bug.getSeverity());
		String pkg = start.substring( start.indexOf("Package: <a")+12, start.indexOf("Reported by:") );
		bug.setPackageName(
				pkg.substring( pkg.indexOf(">")+1, pkg.indexOf("<") ) );
		// want reported by?
		
		try{
			String date = start.substring( start.indexOf("Date: ")+6 );
			date = date.substring(0,  date.indexOf(";")  );
			bug.setCreated( convertDate(date).getTime() );
		
		}catch(Exception e){
			if (DEBUG) System.out.println("Error converting date");
			e.printStackTrace();
			bug.setCreated(-1);
		}

		
		if ( start.indexOf("Tags:") >= 0){
			String tags = start.substring( start.indexOf("Tags:"), start.indexOf("Maintainer"));
			if (DEBUG) System.out.println("==\n"+tags+"\n==\n");
			tags = tags.substring( tags.indexOf("<strong>")+8, tags.indexOf("</strong>") );
			if (DEBUG) System.out.println("==\n"+tags+"\n==\n");
			bug.parseTags( tags );
		}
		bug.setMerged( new int[0] );
		if ( start.indexOf("merged with") >= 0){
			start = start.substring( start.indexOf("merged with ") );
			String merge = start.substring( 0, start.indexOf(";"));
			while (merge.indexOf("=\"bugreport.cgi?bug=") >= 0){
				if (DEBUG) System.out.println("******** MERGE **************");
				merge = merge.substring(  merge.indexOf("<a href=\"bugreport.cgi?bug=")+27 );
				if (DEBUG) System.out.println( merge );
				if (DEBUG) System.out.println( merge.substring( 0, merge.indexOf("\"") ) );
				if (DEBUG) System.out.println( ""+Integer.parseInt(merge.substring( 0, merge.indexOf("\"") ))  );
				bug.addMerged( Integer.parseInt(merge.substring( 0, merge.indexOf("\"") )) ); 
			}
		}

		if (start.indexOf("Forwarded") >0){
			start = start.substring( start.indexOf("Forwarded")+22);
			String fwd = start.substring( start.indexOf("\">") + 2 , start.indexOf("</a>"));
			bug.setForwarded(fwd);
		}
		if (start.indexOf("Done:") >0){
			start = start.substring( start.indexOf("Done:") + 15);
			bug.setDone( start.substring(0, start.indexOf(";") )  );
		}


		// for the comments, we'll just get the bits enclosed by <pre>.
		LinkedList comments = new LinkedList();

		String lastBody = "";
		while (htmlPage.indexOf("<pre>") > 0){
			String comment = htmlPage.substring( htmlPage.indexOf("<pre>")+5, htmlPage.indexOf("</pre>"));

			BugComment cmnt = new BugComment( comment );
			// We often get duplicate entries in the bts - they're ugly, so
			// we'll get rid of them.
			if (!cmnt.getBody().equals(lastBody))			comments.addLast( cmnt ) ;
			lastBody = cmnt.getBody();			
			htmlPage = htmlPage.substring( htmlPage.indexOf("</pre>")+6 );
		}
		bug.setComments( comments );

		bug.setUpdated();
		bug.setComplete(true);
		bug.setStatus(Status.UPDATED);
		Cache.getInstance().store(bug);

		if (DEBUG) System.out.println(System.currentTimeMillis() +" bug parsed"  );
	}

	
	/**
	 * Converts a date, as usually found on the current output from debbugs. 
	 * Can't wait for debbugs 2.0.
	 */
	public static Date convertDate(String date)throws Exception{
		if (DEBUG) System.out.println("|"+date+"|");
		
		date = date.substring(  date.indexOf(" ") +1  );
		if( date.charAt(0) == ' ') date = date.substring(1);
		int day = Integer.parseInt( date.substring( 0,  date.indexOf(" ")  ));
		
		date = date.substring(  date.indexOf(" ") +1  );
		String month = date.substring( 0,  date.indexOf(" ")  );
		int mnth =0;
		if (month.equals("Jan")){ mnth = 0; 
		}else if (month.equals("Feb")){ mnth = 1; 
		}else if (month.equals("Mar")){ mnth = 2; 
		}else if (month.equals("Apr")){ mnth = 3; 
		}else if (month.equals("May")){ mnth = 4; 
		}else if (month.equals("Jun")){ mnth = 5; 
		}else if (month.equals("Jul")){ mnth = 6; 
		}else if (month.equals("Aug")){ mnth = 7; 
		}else if (month.equals("Sep")){ mnth = 8; 
		}else if (month.equals("Oct")){ mnth = 9; 
		}else if (month.equals("Nov")){ mnth = 10; 
		}else if (month.equals("Dec")){ mnth = 11; 
		}
		
		date = date.substring(  date.indexOf(" ") +1  );
		int year = Integer.parseInt( date.substring( 0,  date.indexOf(" ")  ));
		
		date = date.substring(  date.indexOf(" ") +1  );
		int hr = Integer.parseInt( date.substring( 0,  date.indexOf(":")  ));
		
		date = date.substring(  date.indexOf(":") +1  );
		int min = Integer.parseInt( date.substring( 0,  date.indexOf(":")  ));
		
		date = date.substring(  date.indexOf(":") +1  );
		int sec = Integer.parseInt( date.substring( 0,  date.indexOf(" ")  ));
		
		Calendar cal = Calendar.getInstance( TimeZone.getTimeZone("UTF") );
		cal.set(year, mnth, day, hr, min, sec );
		return cal.getTime() ;
	}
}
