/*
 *  Jajuk
 *  Copyright (C) 2006 bflorat
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 *  $Revision: 1.7 $
 */


package org.qdwizard;

import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.ComponentEvent;
import java.awt.event.ComponentListener;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Locale;

import javax.swing.BorderFactory;
import javax.swing.ImageIcon;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPanel;
import javax.swing.JScrollPane;
import javax.swing.Timer;

/**
 *  A Wizard dialog displaying one to many screens
 *
 * @author     Bertrand Florat
 * @created    1 may 2006
 */
public abstract class Wizard implements ActionListener{
    /**Wizard name*/
    String sName;
    /**Initial screen*/
    Class<Screen> initial;
    /**Current screen*/
    Screen current;
    /**Wizard left side icon*/
    ImageIcon icon;
    /**Wizard data*/    
    public static HashMap<String,Object> data;
    /**Wizard header*/
    Header header;
    /**Wizard action Panel*/
    ActionsPanel actions;
    /**Wizard dialog*/
    JDialog dialog;
    /**Wizard timer used to refresh actions states*/
    Timer timer;    
    /**Parent window*/
    Frame parentWindow;
    /**Locale*/
    Locale locale;
    /**Screens instance repository*/
    HashMap<Class,Screen> hmClassScreens = new HashMap<Class,Screen>(10);
    /**Default Wizard size*/
    private static final int DEFAULT_H_SIZE = 700;
    private static final int DEFAULT_V_SIZE = 500;
    
    
    
    /**
     * Wizard constructor
     * @param sName Wizard name displayed in dialog title
     * @param initial Initial screen class
     * @param icon Wizard icon (null if no icon)
     * @param parentWindow wizard parent window
     * @param locale Wizard locale
     * @param iHSize Horizontal size
     * @param iVSize Vertical size
     */
     @SuppressWarnings("unchecked")
	public Wizard(String sName,Class initial,ImageIcon icon,Frame parentWindow,
             Locale locale,int iHSize,int iVSize) {
         this.sName = sName;
        this.initial = initial;
        this.parentWindow = parentWindow;
        if (locale != null){
            this.locale = locale;
        }
        else{
            this.locale = Locale.getDefault();
        }
        data = new HashMap<String,Object> (10);
        this.icon = icon;
        timer  = new Timer(20,new ActionListener() {
            //Refresh button states 
            public void actionPerformed(ActionEvent arg0) {
                if (current != null){
                    boolean isFirst = Wizard.this.initial.getClass().equals(Screen.class);
                    //can go previous if screen allow it and if the screen is not the first one
                    boolean bPrevious = current.canGoPrevious();
                    boolean bNext = current.canGoNext();
                    boolean bFinish = current.canFinish();
                    actions.setState(bPrevious,bNext,bFinish);
                    actions.setProblem(current.getProblem());
                }
            }
        });
        createUI();
        setScreen(initial);
        current.onEnter();
        timer.start();
        dialog.setSize(iHSize,iVSize);

      }

    public void show()
    {
       dialog.setVisible(true);
    }
    /**
     * Wizard constructor
     * @param sName Wizard name displayed in dialog title
     * @param initial Initial screen class
     * @param icon Wizard icon (null if no icon)
     * @param parentWindow wizard parent window
     * @param locale Wizard locale
     */
     public Wizard(String sName,Class initial,ImageIcon icon,Frame parentWindow,Locale locale){
         this(sName,initial,icon,parentWindow,locale,DEFAULT_H_SIZE,DEFAULT_V_SIZE);
     }
    
     /**
     * Wizard constructor (uses default locale)
     * @param sName Wizard name displayed in dialog title
     * @param initial Initial screen class
     * @param icon Wizard icon (null if no icon)
     * @param parentWindow wizard parent window
     */
     public Wizard(String sName,Class initial,ImageIcon icon,Frame parentWindow) {
         this(sName,initial,icon,parentWindow,Locale.getDefault());
     }

    /**
     *  access to the jdailog of the wizard, in case we need it (for instance to set a glasspane
     * when waiting)
     * @return the wizard dialog
     */
    public JDialog getDialog()
    {
        return dialog;
    }

    /**
     * UI manager
     */
    private void createUI(){
        dialog = new JDialog(parentWindow,true);//modale
        //Set default size
        dialog.setSize(DEFAULT_H_SIZE,DEFAULT_V_SIZE);
        dialog.setTitle(sName);
        header = new Header();
        actions = new ActionsPanel(this,locale);
        display();
        dialog.setLocationRelativeTo(parentWindow);
     }
    
    /**
     * Display the wizard 
     * @return wizard data
     */
    private HashMap showWizard(){
        //check initial screen is not null
        if (initial == null){
            return null;
        }
        createUI();
        return data; 
    }

    /* (non-Javadoc)
     * @see java.awt.event.ActionListener#actionPerformed(java.awt.event.ActionEvent)
     */
    public void actionPerformed(ActionEvent ae) {
        dialog.setCursor(new Cursor(Cursor.WAIT_CURSOR));
        try
        {
            //Previous required. Note that the prev button is enabled only if
            //the user can go previous
            if (ae.getActionCommand().equals("Prev")){ //$NON-NLS-1$
                setScreen(getPreviousScreen(current.getClass()));
            }
            else if (ae.getActionCommand().equals("Next")){ //$NON-NLS-1$
                current.onLeave();
                setScreen(getNextScreen(current.getClass()));
                current.onEnter();
            }
            else if (ae.getActionCommand().equals("Cancel")){ //$NON-NLS-1$
                timer.stop();
            	cancel();
                dialog.dispose();
            }
            else if (ae.getActionCommand().equals("Finish")){ //$NON-NLS-1$
                timer.stop();
                finish();
                dialog.dispose();
            }
        }
        finally
        {
             dialog.setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
        }      
    }
    
    /**
     * Set the screen to display a a class
     * @param screenClass
     */
    private void setScreen(Class screenClass) {
        Screen screen = null;
        try {
        	//If the class is an clear point, we clean up all previous screens
        	if (Arrays.asList(screenClass.getInterfaces()).contains(ClearPoint.class)){
        		clearScreens();
        		screen = (Screen)screenClass.newInstance();
                screen.setWizard(this);
            }
        	//otherwise, try to get a screen from buffer or create it if needed
        	else{
        		if (!hmClassScreens.containsKey(screenClass)){
        			screen = (Screen)screenClass.newInstance();
                    screen.setWizard(this);
                    hmClassScreens.put(screenClass,screen);
        		}
        		screen = hmClassScreens.get(screenClass);
        	}
        }
        catch (Exception e) {
            throw new RuntimeException("setScreen "+screenClass+ " caused "+e.toString(), e);
        }
        current = screen;
        current.setCanGoPrevious((getPreviousScreen(screenClass) != null));
        current.setCanGoNext((getNextScreen(screenClass) != null));
        String sDesc = screen.getDescription(); 
        if (sDesc != null){
        header.setText("<html><b>&nbsp;"+screen.getName()+ //$NON-NLS-1$
            "</b><p><p>&nbsp;"+sDesc); //$NON-NLS-1$
        }
        else{
        header.setText("<html><b>&nbsp;"+screen.getName()+ //$NON-NLS-1$
            "</b>"); //$NON-NLS-1$
        }
        display();
    }
    
    /**
     * Called at each screen refresh
     */
    private void display(){
        ((JPanel)dialog.getContentPane()).removeAll();
        dialog.setLayout(new BorderLayout(5,5));
        final JLabel jl = new JLabel(icon);
        dialog.add(jl,BorderLayout.WEST);
        //Add a listener to resize left side image if wizard window is resized
        jl.addComponentListener(new ComponentListener() {
        
            public void componentShown(ComponentEvent e) {
            }
        
            public void componentResized(ComponentEvent e) {
                Wizard.this.icon = getResizedImage(icon, jl.getWidth(), jl.getHeight());
                jl.setIcon( Wizard.this.icon);
            }
        
            public void componentMoved(ComponentEvent e) {
            }
        
            public void componentHidden(ComponentEvent e) {
            }
        
        });
        dialog.add(actions,BorderLayout.SOUTH);
        JScrollPane jsp = new JScrollPane(header);
        jsp.setHorizontalScrollBarPolicy(JScrollPane.HORIZONTAL_SCROLLBAR_NEVER);
        jsp.setVerticalScrollBarPolicy(JScrollPane.VERTICAL_SCROLLBAR_NEVER);
        jsp.setBorder(BorderFactory.createLineBorder(Color.BLACK));
        
        dialog.add(jsp,BorderLayout.NORTH);
        if (current != null){
            dialog.add(current,BorderLayout.CENTER);
        }
        
        dialog.getRootPane().setDefaultButton(actions.jbNext);
        ((JPanel)dialog.getContentPane()).revalidate();
        dialog.getContentPane().repaint();
    }
    
     /**
     * @return previous screen class
     */
    abstract public Class getPreviousScreen(Class screen);
    
    
    /**
     * Clear screens history
     */
    public void clearScreens(){
    	hmClassScreens.clear();
    }
    
    /**
     * 
     * @return next screen class
     */
    abstract public Class getNextScreen(Class screen);

    /**
     * Get curent screen
     * @return current screen class
     */
    public Class getCurrentScreen() {
        return this.current.getClass();
    }
    
    /**
     * Finish action. Called when user clicks on "finish"
     */
    abstract public void finish();

     /**
     * Empty cancel action. Called when user clicks on "cancel". Override it if
     * you want to do something in cancel.
     */
    public void cancel()
     {
     }
    /**
     * Icon resizing
     * @param img
     * @param iNewWidth
     * @param iNewHeight
     * @return resized icon
     */
    private static ImageIcon getResizedImage(ImageIcon img, int iNewWidth,
            int iNewHeight) {
        if (img== null) return null;
        ImageIcon iiNew = new ImageIcon();
        Image image = img.getImage();
        Image scaleImg = image.getScaledInstance(iNewWidth, iNewHeight,
                Image.SCALE_AREA_AVERAGING);
        iiNew.setImage(scaleImg);
        return iiNew;
    }

   
}
