// Copyright 2016 The Chromium Authors. All rights reserved.
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#include "net/ssl/ssl_platform_key_util.h"

#include <stddef.h>

#include "base/memory/ref_counted.h"
#include "net/cert/x509_certificate.h"
#include "net/ssl/ssl_private_key.h"
#include "net/test/cert_test_util.h"
#include "net/test/test_data_directory.h"
#include "testing/gtest/include/gtest/gtest.h"
#include "third_party/boringssl/src/include/openssl/ecdsa.h"

namespace net {

namespace {

bool GetClientCertInfoFromFile(const char* filename,
                               SSLPrivateKey::Type* out_type,
                               size_t* out_max_length) {
  scoped_refptr<X509Certificate> cert =
      ImportCertFromFile(GetTestCertsDirectory(), filename);
  if (!cert) {
    ADD_FAILURE() << "Could not read " << filename;
    return false;
  }

  return GetClientCertInfo(cert.get(), out_type, out_max_length);
}

}  // namespace

TEST(SSLPlatformKeyUtil, GetClientCertInfo) {
  SSLPrivateKey::Type type;
  size_t max_length;

  ASSERT_TRUE(GetClientCertInfoFromFile("client_1.pem", &type, &max_length));
  EXPECT_EQ(SSLPrivateKey::Type::RSA, type);
  EXPECT_EQ(2048u / 8u, max_length);

  ASSERT_TRUE(GetClientCertInfoFromFile("client_4.pem", &type, &max_length));
  EXPECT_EQ(SSLPrivateKey::Type::ECDSA_P256, type);
  EXPECT_EQ(ECDSA_SIG_max_len(256u / 8u), max_length);
}

}  // namespace net
